package QQ::weixin::work::media;

=encoding utf8

=head1 Name

QQ::weixin::work::media

=head1 DESCRIPTION

=cut

use strict;
use base qw(QQ::weixin::work);
use Encode;
use LWP::UserAgent;
use JSON;
use utf8;

our $VERSION = '0.08';
our @EXPORT = qw/ get upload /;

=head1 FUNCTION

=head2 get(access_token, media_id);

获取临时素材

=head2 SYNOPSIS

L<https://developer.work.weixin.qq.com/document/path/90254>

=head3 请求说明：

=head4 参数说明：

    参数	            必须	说明
    access_token	是	调用接口凭证
    media_id	是	媒体文件id, 见上传临时素材

=head4 权限说明：

完全公开，media_id在同一企业内所有应用之间可以共享。

=head3 RETURN 返回结果：

正确时返回（和普通的http下载相同，请根据http头做相应的处理）：

	HTTP/1.1 200 OK
	Connection: close
	Content-Type: image/jpeg 
	Content-disposition: attachment; filename="MEDIA_ID.jpg"
	Date: Sun, 06 Jan 2013 10:20:18 GMT
	Cache-Control: no-cache, must-revalidate
	Content-Length: 339721
   
	Xxxx

错误时返回（这里省略了HTTP首部）：

    {
    	"errcode": 40007,
    	"errmsg": "invalid media_id"
    }

=head4 RETURN 参数说明：

    参数	        说明
    errcode	    出错返回码
    errmsg	对返回码的文本描述内容

=cut

sub get {
    if ( @_ && $_[0] && $_[1] ) {
        my $access_token = $_[0];
        my $media_id = $_[1];
        my $ua = LWP::UserAgent->new;
        $ua->timeout(30);
        $ua->env_proxy;

        my $response = $ua->get("https://qyapi.weixin.qq.com/cgi-bin/media/get?access_token=$access_token&media_id=$media_id");
        if ($response->is_success) {
            my $reply;
            $reply->{"content-type"} = $response->content_type;
            $reply->{filename} = $response->filename;
            $reply->{data} = $response->decoded_content;
            return $reply;
#            return $response->decoded_content;
        }

    }
    return 0;
}

=head2 upload(access_token, type, media);

上传临时素材

=head2 SYNOPSIS

L<https://developer.work.weixin.qq.com/document/path/90389>

=head3 请求说明：

使用multipart/form-data POST上传文件， 文件标识名为"media"

=head4 参数说明：

    参数	            必须	说明
    access_token	是	调用接口凭证
	type	是	媒体文件类型，分别有图片（image）、语音（voice）、视频（video），普通文件（file）

=head4 权限说明：

素材上传得到media_id，该media_id仅三天内有效
完全公开，media_id在同一企业内所有应用之间可以共享。

POST的请求包中，form-data中媒体文件标识，应包含有 filename、filelength、content-type等信息
filename标识文件展示的名称。比如，使用该media_id发消息时，展示的文件名由该字段控制

=head3 RETURN 返回结果：

请求示例：

POST https://qyapi.weixin.qq.com/cgi-bin/media/upload?access_token=accesstoken001&type=file HTTP/1.1
Content-Type: multipart/form-data; boundary=-------------------------acebdf13572468
Content-Length: 220

---------------------------acebdf13572468
Content-Disposition: form-data; name="media";filename="wework.txt"; filelength=6
Content-Type: application/octet-stream

mytext
---------------------------acebdf13572468--


返回数据：

	{
	   "errcode": 0,
	   "errmsg": ""，
	   "type": "image",
	   "media_id": "1G6nrLmr5EC3MMb_-zK1dDdzmd0p7cNliYu9V5w7o8K0",
	   "created_at": "1380000000"
	}


=head4 RETURN 参数说明：

    参数	        说明
    type	媒体文件类型，分别有图片（image）、语音（voice）、视频（video），普通文件(file)
	media_id	媒体文件上传后获取的唯一标识，3天内有效
	created_at	媒体文件上传时间戳

上传的媒体文件限制
所有文件size必须大于5个字节

图片（image）：10MB，支持JPG,PNG格式
语音（voice） ：2MB，播放长度不超过60s，仅支持AMR格式
视频（video） ：10MB，支持MP4格式
普通文件（file）：20MB

=cut

sub upload {
    if ( @_ && $_[0] && $_[1] ) {
        my $access_token = $_[0];
        my $media_id = $_[1];
        my $ua = LWP::UserAgent->new;
        $ua->timeout(30);
        $ua->env_proxy;

        my $response = $ua->post("https://qyapi.weixin.qq.com/cgi-bin/media/upload",[access_token => $access_token, type => $type, media => [$file]],Content_Type => 'form-data');
        if ($response->is_success) {
            return from_json($response->decoded_content,{utf8 => 1, allow_nonref => 1});
        }

    }
    return 0;
}

1;
__END__
