/*
 * A GNOME front end for LinuxConf
 *
 * (C) 1997 the Free Software Foundation
 * (C) 1998 Michael K. Johnson
 *
 * Authors: Miguel de Icaza (miguel@gnu.org)
 *          Michael K. Johnson <johnsonm@redhat.com>
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "gnome.h"

#define BSIZE 512
#define NFORM_STACK 16
#define GNOME_WINDOW "GNOME-Window"

enum form_type {
	FORM_HBOX,
	FORM_TABLE
};


/* W_NODUMP widgets are ignored, all other widgets are printed according
 * to the widget type.
 */
typedef enum widget_type_e {
	W_NODUMP,
	W_CHECKBOX,
	W_CHOICE,
	W_COMBO,
	W_LIST,
	W_RADIO,
	W_STRING
} widget_type;


/* The widget_info structure serves two purposes:
 *   o It arranges widgets in a tree in the order that the remadmin
 *     protocol refers to them, including which ones should have the
 *     data dumped
 *   o It keeps the widgets in a structure that makes it convenient
 *     to dispose of them when we are done with them
 *
 * widget_info contract:
 * All strings are copies that should be freed when the structure
 * is freed.
 * Whenever a child widget is created, it's path is filled in
 * from the parent, and the parent's child_list is filled in with
 * a pointer to the child widget proper (not its widget_info).
 */
typedef struct widget_info_s {
	widget_type type;  /* widget type */
	GtkWidget *form;   /* widget's toplevel (protocol, not GTK!) form */
	GtkWidget *parent; /* widget's protocol (not GTK!) parent */
	char	  *id;     /* widget's protocol (not GTK!) ID */
	char	  *path;   /* full widget path, not including this w's ID */
	GSList    *child_list; /* list of children */
	void	  *data;   /* data, only used when the data for the
			    * widget cannot conveniently be retrieved
			    * from the widget itself
			    */
} widget_info;



static char *widget_info_key = "lc_widget_info";
static char *combo_list_key = "lc_list_key";

static struct {
	GtkWidget  *table;
	int        column;
	int        line;
	int        type;
	GHashTable *hash;
} form_stack [NFORM_STACK];

static int stack_top = 0;

#define current_table  form_stack [stack_top].table
#define current_column form_stack [stack_top].column
#define current_line   form_stack [stack_top].line
#define current_hash   form_stack [stack_top].hash
#define current_type   form_stack [stack_top].type

/* toplevels is a hash from remadmin name to toplevel widget */
static GHashTable *toplevels;

/* current_window contains the current mainform window
 * current_book contains the current notebook entry
 * current_form contains the current parent form for the widget_info tree
 */
static GtkWidget *current_window, *current_book, *current_form;

/* last_* are because some commands in the remadmin protocol apply
 * to the previous widget defined
 */
static GtkWidget *last_widget, *last_menu;

/* string accumulator */
GString *reg = NULL;

static FILE *infile;
static guint input_tag;

#define bind_sym(x,y) g_hash_table_insert (current_hash, g_strdup (x), y);
#define unbind_sym(x) g_hash_table_remove (current_hash, x);

char **
split_string (char *str, int items)
{
	char **result;
	int  i;

	result = g_malloc (sizeof (char *) * (items + 1));

	for (i = 0; i < items; i++){
		result [i] = str;

		if (*str == '"') {
			/* handle quoted string with embedded spaces */

			result [i] = ++str;
			while (*str && *str != '"')
				str++;

		} else {
			/* spaces demarcate substrings */

			while (*str && (!isspace (*str)))
				str++;
		
		}

		if (*str) {
			*str = 0;
			str++;
		}

		while (isspace (*str))
			str++;

	}
	result [i] = str;
	return result;
}

widget_info *
get_widget_info(GtkWidget *widget)
{
	if (widget)
	    return gtk_object_get_data(GTK_OBJECT(widget), widget_info_key);
	else return NULL;
}

void
set_widget_info(GtkWidget *widget, widget_info *w)
{
	gtk_object_set_data(GTK_OBJECT(widget), widget_info_key, w);
}


/* fill_in_widget_info MUST be called for every widget.
 * parent is NULL for mainforms (and other toplevels?).
 * data is optional.
 */
void
fill_in_widget_info(GtkWidget *widget, GtkWidget *parent, char *this_id,
		    widget_type type, void *data)
{
	widget_info *w, *p;

	w = g_malloc0(sizeof(widget_info));
	w->type = type;
	w->parent = parent;
	w->id = g_strdup(this_id);
	w->data = data;

	/* now manage widget hierarchy info */
	p = get_widget_info(parent);
	if (p) {
	    /* is a child of another widget */
	    if (p->path) {
		w->path = g_malloc(strlen(p->path) + strlen(p->id) + 2);
		sprintf(w->path, "%s.%s", p->path, p->id);
	    } else {
		w->path = g_strdup(p->id);
	    }
	    p->child_list = g_slist_append(p->child_list, widget);
	    w->form = p->form;
	} else {
	    /* must be the top of the hierarchy */
	    w->form = widget;
	}

	set_widget_info(widget, w);
}





GtkWidget *
new_table (void)
{
	GtkWidget *table;

	table = gtk_table_new (100, 100, 0);
	gtk_container_border_width (GTK_CONTAINER (table), 6);
#if 0
	gtk_table_set_row_spacings (GTK_TABLE(table), 4);
	gtk_table_set_col_spacings (GTK_TABLE(table), 4);
#endif
	return table;
}

GtkWidget *
new_hbox (void)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (1, 0);
	gtk_container_border_width (GTK_CONTAINER (hbox), 6);
	return hbox;
}

void
maybe_skipit (void *data, void *data2)
{
	GtkTableChild *child = data;

	if (!((current_line >= child->top_attach) &&
	      (current_line < child->bottom_attach)))
		return;
	
	if ((current_column >= child->left_attach) &&
	    (current_column < child->right_attach))
		current_column = child->right_attach;
}


void
walk_over_allocated_space ()
{
	GtkTable *table = GTK_TABLE(current_table);
	GList *p;

	p = table->children;
	if (!p)
		return;

	g_list_foreach (table->children, maybe_skipit, NULL);
	return;
}

void
place (GtkWidget *widget)
{
	if (current_type == FORM_TABLE){
		walk_over_allocated_space ();
		last_widget = widget;
		gtk_table_attach (GTK_TABLE (current_table), widget,
				  current_column, current_column + 1,
				  current_line,   current_line + 1,
				  GTK_SHRINK|GTK_FILL, 
				  GTK_SHRINK|GTK_FILL,
				  0, 0);
		current_column++;
	} else
		gtk_box_pack_start (GTK_BOX (current_table), widget, 0, 0, 0);
}

void
create_form_type (enum form_type ft, char *str)
{
	GtkWidget  *window, *table;
	GHashTable *wtab;
	char **strs;

	if (!form_stack [0].table){
		strs = split_string (str, 1);
		window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title (GTK_WINDOW(window), strs [1]);
		if (ft == FORM_TABLE)
			table = new_table ();
		else
			table = new_hbox ();
		gtk_widget_show (table);
		gtk_container_add (GTK_CONTAINER(window), table);
		g_hash_table_insert (toplevels, strs [0], window);
		fill_in_widget_info(window, NULL, strs [0], W_NODUMP, NULL);
		current_form = window;
		
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		g_hash_table_insert (wtab, GNOME_WINDOW, window);
		current_hash   = wtab;
		current_window = window;
	} else {
		strs = split_string (str, 0);
		if (ft == FORM_TABLE)
			table = new_table ();
		else
			table = new_hbox ();
		
		gtk_widget_show (table);
		bind_sym (strs [0], table);
		fill_in_widget_info(table, current_form, strs [0], W_NODUMP, NULL);
		current_form = table;
		place (table);

		/* nest */
		wtab = g_hash_table_new (g_string_hash, g_string_equal);
		gtk_object_set_user_data (GTK_OBJECT (table), wtab);
		stack_top++;
	}
	current_table  = table;
	current_column = current_line = 0;
	current_type = ft;
	free (strs);
}

void
create_form (char *str)
{
	create_form_type (FORM_TABLE, str);
}

void
create_form_button (char *str)
{
	create_form_type (FORM_HBOX, str);
}
	
void
create_book (char *str)
{
	GtkWidget *book;
	char **strs;

	strs = split_string (str, 1);
	book = gtk_notebook_new ();
	gtk_widget_show (book);
	bind_sym (strs [0], book);
	fill_in_widget_info(book, current_form, strs [0], W_NODUMP, NULL);
	current_form = book;
	place (book);
	current_book = book;
	free (strs);
}

void
report_action (GtkWidget *widget, gpointer data)
{
	widget_info *w = get_widget_info(widget);

	printf ("action %s %s\n", w->path, w->id);
}

void
widget_dump_tree(GtkWidget *top)
{
	widget_info *w = get_widget_info(top);

	/* if this widget is dumpable, print its value */
	switch (w->type) {
	case W_NODUMP:
		break;
	case W_CHECKBOX:
		printf ("dump %s %s %d\n", w->path, w->id,
			GTK_TOGGLE_BUTTON(top)->active);
		break;
	case W_CHOICE:
		printf ("dump %s %s %s\n", w->path, w->id, (char *) w->data);
		break;
	case W_COMBO: /* fall-through */
	case W_STRING:
		printf ("dump %s %s %s\n", w->path, w->id,
			gtk_entry_get_text(GTK_ENTRY(top)));
		break;
	case W_LIST:
		printf ("dump %s %s; list unimplemented", w->path, w->id);
	case W_RADIO:
		/* only print the value of the currently-selected button */
		if (GTK_TOGGLE_BUTTON(top)->active)
			printf ("dump %s %s %s", w->path, w->id, w->data);
	default:
		fprintf(stderr, "Illegal widget type 0x%x\n", w->type);
		break;
	}

	/* recurse */
	g_slist_foreach (w->child_list, (GFunc) widget_dump_tree, NULL);
}

void
button_dump_form (GtkWidget *widget, gpointer data)
{
	widget_dump_tree(get_widget_info(widget)->form);
	report_action (widget, data);
}

void
generic_button_setup (char *sym, GtkWidget *widget, char *id, int should_dump)
{
	bind_sym (sym, widget);
	fill_in_widget_info(widget, current_form, id, W_NODUMP, NULL);
	place (widget);

	if (should_dump)
		gtk_signal_connect (GTK_OBJECT (widget), "clicked",
				    (GtkSignalFunc) button_dump_form, 0);
	else
		gtk_signal_connect (GTK_OBJECT (widget), "clicked",
				    (GtkSignalFunc) report_action, 0);

}

		      
void
create_button (char *str)
{
	GtkWidget *button;
	char **strs;

	strs = split_string (str, 2);
	button = gtk_button_new_with_label (strs [2]);
	gtk_widget_show (button);
	generic_button_setup (strs [0], button, strs[0], strs [1][0] == '1');
	free (strs);
}

void
create_button_fill (char *str)
{
	GtkWidget *button, *label;
	char **strs;

	strs = split_string (str, 1);
	button = gtk_button_new ();
	label = gtk_label_new (strs [1]);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_container_add (GTK_CONTAINER(button), label);
	gtk_widget_show (label);
	gtk_widget_show (button);
	generic_button_setup (strs [0], button, strs[0], 0);
	free (strs);
}

void
internal_create_button_xpm (char *id, char *fname, char *text)
{
	GtkWidget *button, *pixmap, *box, *container, *label;

	button = gtk_button_new ();
	/* We have a text */
	if (text [0]){
		container = box = gtk_vbox_new (0, 0);
		label = gtk_label_new (text);
		gtk_widget_show (label);
	} else {
		container = button;
		label = NULL;
		box = NULL;
	}
	pixmap = gnome_create_pixmap_widget (current_window, container, fname);
	unlink (fname);
	if (label){
		gtk_box_pack_start_defaults (GTK_BOX(box), pixmap);
		gtk_box_pack_start_defaults (GTK_BOX(box), label);
		gtk_container_add (GTK_CONTAINER(button), box);
		gtk_widget_show (container);
	} else
		gtk_container_add (GTK_CONTAINER(button), pixmap);
	gtk_widget_show (pixmap);
	gtk_widget_show (button);
	generic_button_setup (id, button, id, 0);
}

void
create_button_xpm (char *str)
{
	char **strs;
	char *fname;
	
	strs = split_string (str, 2);
	fname = g_hash_table_lookup (current_hash, strs [1]);
	if (!fname){
		printf ("Pixmap id %s not defined\n", strs [1]);
		exit (1);
	}
	internal_create_button_xpm (strs [0], fname, strs [2]);
	free (strs);
}

void
create_button_xpmf (char *str)
{
	char **strs;

	strs = split_string (str, 2);
	internal_create_button_xpm (strs [0], strs [1], strs [2]);
	free (strs);
}

void
create_combo (char *str)
{
	GtkWidget *entry;
	char **strs;
	int maxlen;
	
	strs = split_string (str, 3);
	maxlen = atoi (strs [1]);
	entry = gtk_combo_new ();
	gtk_object_set_data(GTK_OBJECT(entry), combo_list_key, NULL);
	gtk_widget_show (entry);
	gtk_entry_set_text (GTK_ENTRY (entry), strs [2]);
	place (entry);
	bind_sym (strs [0], entry);
	fill_in_widget_info(entry, current_form, strs [0], W_COMBO, NULL);
	free (strs);
}

void
combo_item (char *str)
{
	GtkWidget *box;
	char **strs;
	GList *boxlist;

	strs = split_string (str, 3);
	box = g_hash_table_lookup (current_hash, strs [0]);
	if (!box) {
		fprintf (stderr, "Can not find %s\n", strs [0]);
		return;
	}

	boxlist = gtk_object_get_data(GTK_OBJECT(box), combo_list_key);
	/* FIXME: perhaps we shouldn't be using the combobox here:
	 * we're supposed to find in strs[2] an "explanation" and
	 * show both strs[1] and strs[2], but only dump strs[1]
	 */
	boxlist = g_list_append(boxlist, g_strdup(strs [1]));
	gtk_object_set_data(GTK_OBJECT(box), combo_list_key, boxlist);
	gtk_combo_set_popdown_strings(box, boxlist);
	free (strs);
}

void
resize_option_menu(GtkWidget *widget, gpointer data)
{
	GtkOptionMenu *omenu;

	omenu = GTK_OPTION_MENU (data);
	gtk_option_menu_set_menu (omenu, widget);
}

void
create_choice (char *str)
{
	GtkWidget *choice, *menu;
	char **strs;
	
	strs = split_string (str, 1);
	choice = gtk_option_menu_new ();
	menu   = gtk_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (choice), menu);
	place (choice);
	bind_sym (strs [0], menu);
	fill_in_widget_info(menu, current_form, strs [0], W_CHOICE, NULL);
	printf ("binding: %s to %p\n", strs [0], menu);
	gtk_widget_show (choice);

	/* FIXME: This is a hack to work around a bug in Gtk: When
	 * menu items are added to a menu which is attached to an
	 * option_menu, *after* the attachment has been made, the
	 * option_menu is not resized.  This should be done inside
	 * GtkOptionMenu.
	 */
	
	gtk_signal_connect(GTK_OBJECT(menu), "need_resize",
			   (GtkSignalFunc) resize_option_menu,
			   choice);
	free (strs);
}

void
set_choice_value (GtkWidget *widget, char *value)
{
	widget_info *w = get_widget_info(widget);

	w = get_widget_info(w->parent);
	w->data = value;
}

void
choice_item (char *str)
{
	GtkWidget *menu, *item;
	char **strs;

	strs = split_string (str, 1);
	menu = g_hash_table_lookup (current_hash, strs [0]);
	if (!menu){
		printf ("Can not find %s\n", strs [0]);
		return;
	}
	item = gtk_menu_item_new_with_label (strs [1]);
	gtk_signal_connect_object (GTK_OBJECT(item), "activate",
		GTK_SIGNAL_FUNC(set_choice_value), g_strdup(strs [1]));
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);
	fill_in_widget_info(item, current_form, strs [0], W_NODUMP, NULL);
	free (strs);
}

void
create_checkbox (char *str)
{
	GtkWidget *check;
	char **strs;

	strs = split_string (str, 2);
	check = gtk_check_button_new_with_label (strs [2]);
	gtk_widget_show (check);
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(check), *strs [1] == '0' ? 0 : 1);
	bind_sym (strs [0], check);
	fill_in_widget_info(check, current_form, strs [0], W_CHECKBOX, NULL);
	place (check);
	free (strs);
}

void
create_radio (char *str)
{
	GtkWidget *radiogroup, *radio;
	char **strs;

	strs = split_string (str, 4);

	radiogroup = g_hash_table_lookup (current_hash, strs [0]);
	if (!radiogroup){
		radio = gtk_radio_button_new_with_label (NULL, strs [3]);
	} else {
		radio = gtk_radio_button_new_with_label (
			gtk_radio_button_group (GTK_RADIO_BUTTON (radiogroup)),
		strs [3]);
	}
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON(radio),
		strs [1][0] == '0' ? 0 : 1);
	if (!radiogroup) {
		bind_sym (strs [0], radio);
	}
	fill_in_widget_info(radio, current_form, strs [0], W_RADIO,
		g_strdup(strs [2]));
	place(radio);

	free (strs);
}

void
create_group (char *str)
{
	GtkWidget *group, *table;
	GHashTable *wtab;
	char **strs;

	strs = split_string (str, 1);
	group = gtk_frame_new (strs [1][0] ? strs [1] : NULL);
	gtk_widget_show (group);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	fill_in_widget_info(table, current_form, strs [0], W_NODUMP, NULL);
	current_form  = table;
	place (group);
	gtk_container_add (GTK_CONTAINER(group), table);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT(table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
	current_type = FORM_TABLE;

	free (strs);
}

void
create_label (char *str)
{
	GtkWidget *label;

	label = gtk_label_new (str);
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_widget_show (label);
	fill_in_widget_info(label, current_form, str, W_NODUMP, NULL);
	place (label);
}

void
create_page (char *str)
{
	GtkWidget *label, *table;
	GHashTable *wtab; 
	char **strs;

	strs = split_string (str, 1);
	label = gtk_label_new (strs [1]);
	gtk_widget_show (label);
	table = new_table ();
	gtk_widget_show (table);
	bind_sym (strs [0], table);
	fill_in_widget_info(table, current_form, strs [0], W_NODUMP, NULL);
	current_form  = table;
	place (table);
	gtk_notebook_append_page (GTK_NOTEBOOK(current_book), table, label);

	/* Nest */
	wtab = g_hash_table_new (g_string_hash, g_string_equal);
	gtk_object_set_user_data (GTK_OBJECT (table), wtab);
	stack_top++;
	current_table = table;
	current_column = current_line = 0;
	current_type = FORM_TABLE;
	current_hash = wtab;

	free (strs);
}

void create_entry (char *str, int visible)
{
	GtkWidget *entry;
	char **strs;
	int  maxlen;
	
	strs = split_string (str, 2);
	maxlen = atoi (strs [1]);
	entry = gtk_entry_new ();
	gtk_entry_set_visibility(GTK_ENTRY(entry), visible);
	gtk_widget_show (entry);
	gtk_entry_set_text (GTK_ENTRY(entry), strs [2]);
	place (entry);
	bind_sym (strs [0], entry);
	fill_in_widget_info(entry, current_form, strs [0], W_STRING, NULL);
	free (strs);
}

void
create_string (char *str)
{
	create_entry(str, 1);
}

void
create_password (char *str)
{
	create_entry(str, 0);
}

void
create_hline (char *str)
{
	GtkWidget *sep;
	
	sep = gtk_hseparator_new ();
	gtk_widget_show (sep);
	fill_in_widget_info(sep, current_form, "", W_NODUMP, NULL);
	place (sep);
}

void
newline (char *str)
{
	current_line++;
	current_column = 0;
}

void
end (char *str)
{
	if (current_form)
		current_form = get_widget_info(current_form)->parent;

	if (stack_top)
		stack_top--;
	else {
		gtk_widget_realize (current_window);
		gtk_widget_show (current_window);
		/* bad hack: force MainForm  :-) */
		/* form_stack [0].table = 0; */
	}
}

void
skip (char *str)
{
	current_column += atoi (str);
}

float
h_align (char *str)
{
	if (*str == 'l')
		return 0.0;
	if (*str == 'r')
		return 1.0;
	return 0.5;
}

float
v_align (char *str)
{
	if (*str == 't')
		return 0.0;
	if (*str == 'b')
		return 1.0;
	return 0.5;
}

/*
 * When we receive this command, it means that we have to change the layout of
 * the last inserted widget.
 * This also provides alignement contraints.
 */
void
dispolast (char *str)
{
	GtkWidget *align;
	char **strs;
	int  column;

	column = current_column - 1;
	strs = split_string (str, 4);
	gtk_container_remove (GTK_CONTAINER(current_table), last_widget);
	
	align = gtk_alignment_new (h_align (strs [0]), v_align (strs [2]), 1.0, 1.0);
	gtk_widget_show (align);
	gtk_table_attach_defaults (GTK_TABLE (current_table), align,
				   column,       column + atoi (strs [1]),
				   current_line, current_line + atoi (strs [3]));
	gtk_container_add (GTK_CONTAINER(align), last_widget);
	free (strs);
}

void
dump (char *str)
{
	GtkWidget *form;

	form = g_hash_table_lookup (toplevels, str);
	if (!form) {
		fprintf (stderr, "widget %s does not exist\n", str);
		return;
	}
	widget_dump_tree (form);
}

/* delete_tree deletes a widget_info tree, but does NOT destroy any widgets */
void
delete_tree (GtkWidget *widget)
{
	widget_info *w = get_widget_info(widget);

	g_slist_foreach (w->child_list, (GFunc) delete_tree, NULL);
	g_slist_free (w->child_list);

	/* linuxconf doesn't do it, but ID's can be reused in the
	 * remadmin protocol, so get rid of the binding
	 */
	unbind_sym(w->id);

	g_free (w->id);
	if (w->data)
		g_free (w->data);
	if (w->path)
		g_free (w->path);
	g_free (w);

	set_widget_info(widget, NULL);
}

void
delete (char *str)
{
	GtkWidget *del;

	del = g_hash_table_lookup (toplevels, str);
	if (!del) {
		fprintf (stderr, "widget %s does not exist\n", str);
		return;
	}

	delete_tree(del);

	/* destroys the whole tree */
	gtk_widget_destroy (del);
}

void
unimplemented (char *str)
{
}

void
str (char *str)
{
	if (reg)
		g_string_append (reg, str);
	else
		reg = g_string_new (str);
}

void
xfer_xpm (char *str)
{
	FILE *f;
	char *fname = tmpnam (NULL);

	f = fopen (fname, "w");
	if (f == NULL)
		return;
	fwrite (reg->str, 1, reg->len, f);
	fclose (f);

	bind_sym (str, g_strdup (fname));
	g_string_free (reg, TRUE);
	reg = NULL;
}

void
sidetitle (char *str)
{
	/* nothing */
}

void
warning_dialog(char *string)
{
	GtkDialog *dialog;
	GtkWidget *label;
	GtkWidget *button;

	gtk_window_set_title(GTK_WINDOW(dialog), "Warning");
	/* the following is a hack until we have a proper text display widget
	 * besides a label
	 */
	gtk_widget_set_usize (GTK_WIDGET(dialog), 400, 150);
	label = gtk_label_new(string);
	gtk_box_pack_start(GTK_BOX(dialog->vbox), label, TRUE, TRUE, 0);
	gtk_widget_show (label);

	button = gtk_button_new_with_label ("close");
	gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
		GTK_SIGNAL_FUNC (gtk_exit), GTK_OBJECT (0));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
		button, TRUE, TRUE, 0);
	gtk_widget_grab_default (GTK_WIDGET(button));
	gtk_widget_show (GTK_WIDGET(button));
	gtk_widget_show (GTK_WIDGET(dialog));

	gtk_main();
}

void
check_version (char *str)
{
	char **strs;
	int i;

	strs = split_string (str, 2);
	i = atoi (strs [0]);
	if (i != 1) {
		warning_dialog(strs [1]);

		/* tell the other end about the problem */
		printf("action badver 1");
	}

	free(strs);
}

struct {
	char *name;
	void (*cmd)(char *str);
} cmds [] = {
	{ "Book",        create_book },
	{ "Button",      create_button },
	{ "ButtonFill",  create_button_fill },
	{ "Checkbox",    create_checkbox },
	{ "Choice",      create_choice },
	{ "Choice_item", choice_item }, 
	{ "Combo",       create_combo },
	{ "Combo_item",  combo_item },
	{ "Delete",      delete },
	{ "Dispolast",   dispolast },
	{ "Dump",        dump },
	{ "End",         end },
	{ "Form",        create_form },
	{ "FormButton",  create_form_button },
	{ "Group",       create_group },
	{ "Hline",       create_hline },
	{ "Label",       create_label },
	{ "MainForm",    create_form  },
	{ "Newline",     newline },
	{ "Page",        create_page },
	{ "Password",	 create_password },
	{ "Radio",	 create_radio },
	{ "Sidetitle",   sidetitle },
	{ "Skip",        skip },
	{ "Str",         str },
	{ "String",      create_string },
	{ "Version",	 check_version },
	{ "Xfer_xpm",    xfer_xpm },
	{ NULL, NULL }
};

struct {
	void (*cmd)(char *str);
} cmds_bynumber [] = {
	{ unimplemented },	/* 0. indexing starts at 1 */
	{ str },		/* 1 str */
	{ newline },		/* 2 */
	{ skip },		/* 3 */
	{ create_hline },	/* 4 */
	{ dispolast },		/* 5 */
        { create_label },	/* 6 */
	{ create_string },	/* 7 */
	{ create_checkbox },	/* 8 */
	{ create_radio },	/* 9 radio */
	{ create_button },	/* 10 */
	{ create_button_xpm },	/* 11 button xpm */
	{ create_button_xpmf },	/* 12 button xpmf */
	{ create_choice },	/* 13 */
	{ choice_item },	/* 14 */
	{ unimplemented },	/* 15 list */
	{ unimplemented },	/* 16 list item */
	{ create_combo },	/* 17 */
	{ combo_item },		/* 18 */
	{ create_group },	/* 19 */
	{ create_form },	/* 20 */
	{ create_page },	/* 21 */
	{ create_book },	/* 22 */
	{ create_form },	/* 23 */
	{ end },		/* 24 */
	{ delete },		/* 25 */
	{ dump },		/* 26 */
	{ unimplemented },	/* 27 icon xpm */
	{ xfer_xpm },		/* 28 xfer xpm */
	{ sidetitle },		/* 29 sidetitle */
	{ unimplemented },	/* 30 fill */
	{ create_form_button },	/* 31 formbutton */
	{ unimplemented },	/* 32 groupfit */
	{ unimplemented },	/* 33 setweightlast */
	{ create_label },	/* 34 FIXME: richtext with text or xm-html */
	{ create_password },	/* 35 password */
	{ create_button_fill },	/* 36 buttonfill */
	{ unimplemented },	/* 37 enteraction */
	{ unimplemented },	/* 38 curfield */
	{ check_version },	/* 39 version */
	{ unimplemented }	/* 40 last */
};
	
void
process (char *str)
{
	char *space;
	int i;
		
	while (*str && (*str == ' ' || *str == '\t'))
		str++;

	space = strchr (str, ' ');
	if (space)
		*space = 0;

	if (isdigit (str [0])){
		i = atoi (str);
		if (cmds_bynumber [i].cmd == unimplemented)
			printf ("unimp: %d\n", i);
		(*cmds_bynumber [i].cmd) (space+1);
	} else 
		for (i = 0; cmds [i].name; i++){
			if (strcmp (cmds [i].name, str))
				continue;
			
			(*cmds [i].cmd) (space+1);
		}
}

void
read_string(gpointer data, gint fd, GdkInputCondition condition)
{
	static char buffer [BSIZE];
	char *s;

	if (fgets (buffer, BSIZE, infile)) {
		if ((s = strchr (buffer, '\n')))
			*s = 0;
		process (buffer);
	} else {
		gtk_input_remove (input_tag);
		/* gtk_exit (0); */
	}
}

int
main (int argc, char *argv [])
{
	
	gnome_init (&argc, &argv);
	toplevels = g_hash_table_new (g_string_hash, g_string_equal);
	
	infile = stdin;
	input_tag = gdk_input_add (STDIN_FILENO, GDK_INPUT_READ,
				read_string, NULL);
	gtk_main ();
	return 0;
}
