/*
    bitmap.c -- reiserfs bitmap functions
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <string.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#if ENABLE_NLS
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

#define reiserfs_bm_range_check(bm, blk, action) \
	do { \
		if (blk >= bm->bm_blocks) { \
	    	libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, \
			_("Block %lu is out of range (0-%lu)"), blk, bm->bm_blocks); \
	    	action; \
		} \
	} while (0); \
    

/* Mark given block as used */
void reiserfs_bm_use_block(reiserfs_bitmap_t *bm, blk_t blk) {
    ASSERT(bm != NULL, return);

    reiserfs_bm_range_check(bm, blk, return);
    if (reiserfs_tools_test_bit(blk, bm->bm_map))
		return;
	
    reiserfs_tools_set_bit(blk, bm->bm_map);
    bm->bm_used_blocks++;
}

/* Mark given block as unused */
void reiserfs_bm_unuse_block(reiserfs_bitmap_t *bm, blk_t blk) {
    ASSERT(bm != NULL, return);

    reiserfs_bm_range_check(bm, blk, return);
    if (!reiserfs_tools_test_bit(blk, bm->bm_map))
		return;
	
    reiserfs_tools_clear_bit(blk, bm->bm_map);
    bm->bm_used_blocks--;
}

/* Check for block usage */
int reiserfs_bitmap_test_block(reiserfs_bitmap_t *bm, blk_t blk) {
    ASSERT(bm != NULL, return 0);

    reiserfs_bm_range_check(bm, blk, return 0);
    return reiserfs_tools_test_bit(blk, bm->bm_map);
}

/* Find first unused block */
blk_t reiserfs_bm_find_free_block(reiserfs_bitmap_t *bm, blk_t start) {
    blk_t bit;
    
    ASSERT(bm != NULL, return 0);
    
    reiserfs_bm_range_check(bm, start, return 0);
    if ((bit = reiserfs_tools_find_next_zero_bit(bm->bm_map, 
			bm->bm_blocks, start)) >= bm->bm_blocks)
		return 0;

    return bit;
}

/* Calculating blocks usage */
static blk_t reiserfs_bm_calc(reiserfs_bitmap_t *bm, blk_t start, blk_t end, 
    int flag) 
{
    blk_t i, blocks = 0;
    
    ASSERT(bm != NULL, return 0);
    
    reiserfs_bm_range_check(bm, start, return 0);
    reiserfs_bm_range_check(bm, end - 1, return 0);
    
    for (i = start; i < end; ) {
#if defined(__GNUC__) || (!defined(__sparc__) && !defined(__sparc_v9__))
		uint64_t *block64 = (uint64_t *)(bm->bm_map + (i >> 0x3));
		size_t bits = sizeof(uint64_t) * 8;
		
		if (i + bits < end && i % 0x8 == 0 &&
			*block64 == (flag == 0 ? 0xffffffffffffffffLL : 0)) 
		{
			blocks += bits;
			i += bits;
		} else {
			if ((flag == 0 ? reiserfs_bitmap_test_block(bm, i) : 
					!reiserfs_bitmap_test_block(bm, i)))
				blocks++;
			i++;	
		}
#else /* sparc but not GNU C */
		if ((flag == 0 ? reiserfs_bitmap_test_block(bm, i) : !reiserfs_bitmap_test_block(bm, i)))
			blocks++;
		i++;	
#endif
    }
    
    return blocks;
}

blk_t reiserfs_bm_calc_used(reiserfs_bitmap_t *bm) {
    return reiserfs_bm_calc(bm, 0, bm->bm_blocks, 0);
}

blk_t reiserfs_bm_calc_unused(reiserfs_bitmap_t *bm) {
    return reiserfs_bm_calc(bm, 0, bm->bm_blocks, 1);
}

blk_t reiserfs_bm_calc_used_in_area(reiserfs_bitmap_t *bm, 
    blk_t start, blk_t end) 
{
    ASSERT(bm != NULL, return 0);
    return reiserfs_bm_calc(bm, start, end, 0);
}

blk_t reiserfs_bm_calc_unused_in_area(reiserfs_bitmap_t *bm, 
    blk_t start, blk_t end) 
{
    ASSERT(bm != NULL, return 0);
    return reiserfs_bm_calc(bm, start, end, 1);
}

blk_t reiserfs_bm_used(reiserfs_bitmap_t *bm) {
    
    ASSERT(bm != NULL, return 0);
    return bm->bm_used_blocks;
}

blk_t reiserfs_bm_unused(reiserfs_bitmap_t *bm) {

    ASSERT(bm != NULL, return 0);
    ASSERT(bm->bm_blocks - bm->bm_used_blocks > 0, return 0);
    
    return bm->bm_blocks - bm->bm_used_blocks;
}

int reiserfs_bm_check(reiserfs_bitmap_t *bm) {
    
    ASSERT(bm != NULL, return 0);
    
    if (reiserfs_bm_calc_used(bm) != bm->bm_used_blocks)
		return 0;
	
    return 1;
}

/* Allocating and creating functions  */
reiserfs_bitmap_t *reiserfs_bm_alloc(blk_t fs_len) {
    reiserfs_bitmap_t *bm;
    
    ASSERT(fs_len > 0, goto error);
    
    if (!(bm = (reiserfs_bitmap_t *)libreiserfs_calloc(sizeof(*bm), 0)))
		goto error;
    
    bm->bm_blocks = fs_len;
    bm->bm_size = (fs_len + 7) / 8;
    bm->bm_used_blocks = 0;
    
    if (!(bm->bm_map = (char *)libreiserfs_calloc(bm->bm_size, 0)))
		goto error_free_bm;
    
    return bm;
    
error_free_bm:
    reiserfs_bm_free(bm);    
error:
    return NULL;    
}

/* Fetching bitmap */
static int reiserfs_bm_fetch(dal_t *dal, reiserfs_bitmap_t *bm) {
    char *p;
    size_t left;
    reiserfs_block_t *block;
    int i, last_byte_unused_bits;
    
    blk_t blk;

    ASSERT(bm != NULL, return 0);
    ASSERT(dal != NULL, return 0);
        
    p = bm->bm_map;
    for (left = (bm->bm_blocks + 7) / 8, blk = bm->bm_start; left > 0; 
		left -= (left < dal_block_size(dal) ? left : dal_block_size(dal))) 
    {
		if (!(block = reiserfs_block_read(dal, blk)))
			reiserfs_block_reading_failed(blk, return 0);

		memcpy (p, block->data, 
			(left < dal_block_size(dal) ? left : dal_block_size(dal)));
			
		reiserfs_block_free(block);
		
		p += (left < dal_block_size(dal) ? left : dal_block_size(dal));

		if (bm->bm_start > (DEFAULT_SUPER_OFFSET / dal_block_size(dal)))
			blk = (blk / (dal_block_size(dal) * 8) + 1) * (dal_block_size(dal) * 8);
		else
			blk++;
    }

    /* 
		On disk bitmap has bits out of sb_block_count set to 1, where as
        reiserfs_bitmap_t has those bits set to 0.
    */
    last_byte_unused_bits = bm->bm_size * 8 - bm->bm_blocks;
    for (i = 0; i < last_byte_unused_bits; i++)
		reiserfs_tools_clear_bit(bm->bm_blocks + i, bm->bm_map);

    return ((bm->bm_used_blocks = reiserfs_bm_calc_used(bm)) > 0);
}

/* Flushing bitmap */
static int reiserfs_bm_flush(dal_t *dal, reiserfs_bitmap_t *bm) {
    char *p;
    size_t left;
    reiserfs_block_t *block;
    int i, last_byte_unused_bits;
    
    blk_t blk;
    
    ASSERT(bm != NULL, return 0);
    ASSERT(dal != NULL, return 0);
    
    p = bm->bm_map;
    for (left = bm->bm_size, blk = bm->bm_start; left > 0; 
		left -= (left < dal_block_size(dal) ? left : dal_block_size(dal)))
    {	
		if (!(block = reiserfs_block_alloc(dal, blk, 0xff)))
			return 0;
		
		memcpy(block->data, p, 
			(left < dal_block_size(dal) ? left : dal_block_size(dal)));
		
		if (left == dal_block_size(dal)) {
			/* Set unused bits of last byte of the bitmap to 1 */
			last_byte_unused_bits = bm->bm_size * 8 - bm->bm_blocks;
			for (i = 0; i < last_byte_unused_bits; i ++)
				reiserfs_tools_set_bit((bm->bm_blocks % (dal_block_size(dal) * 8)) + i, 
					block->data);
		}
		
		if (!reiserfs_block_write(dal, block)) {
			reiserfs_block_writing_failed(reiserfs_block_location(block), 
				goto error_free_block);
		}		
		
		reiserfs_block_free(block);
		
		p += (left < dal_block_size(dal) ? left : dal_block_size(dal));

		if (bm->bm_start > DEFAULT_SUPER_OFFSET / dal_block_size(dal))
			blk = (blk / (dal_block_size(dal) * 8) + 1) * (dal_block_size(dal) * 8);
		else
			blk++;
    }
    
    return 1;
	
error_free_block:
	reiserfs_block_free(block);
error:
	return 0;
}

/* Open bitmap. Allocating and fetching it */
reiserfs_bitmap_t *reiserfs_bm_open(dal_t *dal, blk_t fs_len, blk_t bm_start) {
    reiserfs_bitmap_t *bm;
        
    ASSERT(dal != NULL, return NULL);
    
    if(!(bm = reiserfs_bm_alloc(fs_len)))
		goto error;
    
    bm->bm_start = bm_start;
    
    if (!reiserfs_bm_fetch(dal, bm))
		goto error_free_bm;
	
    return bm;
    
error_free_bm:
    reiserfs_bm_free(bm);
error:
    return NULL;
}

/* Creating bitmap */
reiserfs_bitmap_t *reiserfs_bm_create(blk_t fs_len, blk_t bm_start, size_t blocksize) {
    blk_t i, bmap_blknr;
    reiserfs_bitmap_t *bm;
    
    if (!(bm = reiserfs_bm_alloc(fs_len)))
		return NULL;
    
    bm->bm_start = bm_start;
    
    reiserfs_bm_use_block(bm, (DEFAULT_SUPER_OFFSET / blocksize) + 1);
  
    /* Setting up other bitmap blocks */
    bmap_blknr = (fs_len - 1) / (blocksize * 8) + 1;
    for (i = 1; i < bmap_blknr; i++)
		reiserfs_bm_use_block(bm, i * blocksize * 8);

    return bm;
}

/* Realloc bitmap corresponding to given fs_len value */
int reiserfs_bm_resize(reiserfs_bitmap_t *bm, blk_t fs_len, size_t blocksize) {
    blk_t i, bmap_old_blknr, bmap_new_blknr;
    int bm_size = ((fs_len + 7) / 8), chunk;
    
    ASSERT(bm != NULL && fs_len > 0, return 0);
    
    if (bm_size - bm->bm_size == 0)
		return 1;

    if (!libreiserfs_realloc((void **)&bm->bm_map, bm_size))
		return 0;
	
    if ((chunk = bm_size - bm->bm_size) > 0)
		memset(bm->bm_map + bm->bm_size, 0, chunk);
    
    bmap_old_blknr = bm->bm_size / blocksize;
    bmap_new_blknr = (fs_len - 1) / (blocksize * 8) + 1;

    bm->bm_size = bm_size;
    bm->bm_blocks = fs_len;
    
    /* Marking new bitmap blocks as used */
    if (bmap_new_blknr - bmap_old_blknr > 0) {
        for (i = bmap_old_blknr; i < bmap_new_blknr; i++)
			reiserfs_bm_use_block(bm, i * blocksize * 8);
    }

    return 1;
}

blk_t reiserfs_bm_copy(reiserfs_bitmap_t *dst_bm, reiserfs_bitmap_t *src_bm, 
	blk_t fs_len, size_t blocksize) 
{
    
    ASSERT(dst_bm != NULL, return 0);
    ASSERT(src_bm != NULL, return 0);

    if (!fs_len) return 0;
    
    if (!reiserfs_bm_resize(dst_bm, (fs_len > src_bm->bm_blocks ? 
			src_bm->bm_blocks : fs_len), blocksize))
        return 0;
	
    memcpy(dst_bm->bm_map, src_bm->bm_map, dst_bm->bm_size);
    dst_bm->bm_used_blocks = reiserfs_bm_used(dst_bm);
    
    return dst_bm->bm_blocks;
}

reiserfs_bitmap_t *reiserfs_bm_clone(reiserfs_bitmap_t *src_bm) {
    reiserfs_bitmap_t *dst_bm;

    ASSERT(src_bm != NULL, return 0);	

    if (!(dst_bm = reiserfs_bm_alloc(src_bm->bm_blocks)))
		return NULL;
	
    memcpy(dst_bm->bm_map, src_bm->bm_map, dst_bm->bm_size);
    dst_bm->bm_used_blocks = reiserfs_bm_used(dst_bm);
    
    return dst_bm;
}

/* Synchronizing bitmap */
int reiserfs_bm_sync(dal_t *dal, reiserfs_bitmap_t *bm) {

    ASSERT(dal != NULL, return 0);
    ASSERT(bm != NULL, return 0);

    if (!reiserfs_bm_flush(dal, bm))
		return 0;
	
    return 1;
}

/* Free given bitmap */
void reiserfs_bm_free(reiserfs_bitmap_t *bm) {

    ASSERT(bm != NULL, return);
    
    if (bm->bm_map)
		libreiserfs_free(bm->bm_map);

    libreiserfs_free(bm);
}

/* Closing bitmap. Free all allocated memory */
void reiserfs_bm_close(reiserfs_bitmap_t *bm) {
    reiserfs_bm_free(bm);
}

/* Reopen bitmap */
reiserfs_bitmap_t *reiserfs_bm_reopen(dal_t *dal, reiserfs_bitmap_t *bm, blk_t fs_len) {
    blk_t bm_start;
    
    ASSERT(bm != NULL, return NULL);

    bm_start = bm->bm_start;
    reiserfs_bm_close(bm);
    
    return reiserfs_bm_open(dal, fs_len, bm_start);
}

char *reiserfs_bm_map(reiserfs_bitmap_t *bm) {
    ASSERT(bm != NULL, return NULL);
    return bm->bm_map;
}

