/*
    geom.c -- filesystem relocation functions
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <string.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#include <dal/dal.h>

#define N_(String) (String)
#if ENABLE_NLS
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

struct reiserfs_reloc {
    reiserfs_geom_t *dst_geom;
	reiserfs_fs_t *dst_fs;
    reiserfs_geom_t *src_geom;
	reiserfs_fs_t *src_fs;
	
    reiserfs_gauge_t *gauge;
	blk_t counter;
};

static int generic_node_write(reiserfs_fs_t *fs, reiserfs_block_t *node, blk_t blk) {

    reiserfs_block_set_location(node, blk);
    reiserfs_fs_bitmap_use_block(fs, blk);
    
    if (!reiserfs_block_write(fs->host_dal, node))
		reiserfs_block_writing_failed(blk, return 0);
    
    return 1;
}

static blk_t generic_node_move(struct reiserfs_reloc *reloc, blk_t src_blk) {
    blk_t dst_blk;
    reiserfs_block_t *node;
    reiserfs_fs_t *src_fs, *dst_fs;
    
    src_fs = reloc->src_fs;
    dst_fs = reloc->dst_fs;

    if (dal_equals(dst_fs->host_dal, src_fs->host_dal)) {
		if (reiserfs_geom_test_inside(reloc->dst_geom, src_blk))
			return src_blk;
    }
    
    libreiserfs_gauge_set_value(reloc->gauge, (reloc->counter++ * 100) / 
		reiserfs_geom_len(reloc->src_geom));

    if (dal_equals(dst_fs->host_dal, src_fs->host_dal))
		reiserfs_fs_bitmap_unuse_block(src_fs, src_blk);
    
    /* Finding free block */
    dst_blk = reiserfs_fs_bitmap_find_free_block(dst_fs, reloc->dst_geom->start);

    /* Checking whether found free block is inside allowed find area */
    if (!reiserfs_geom_test_inside(reloc->dst_geom, dst_blk)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't find free block inside allowed area (%lu - %lu)."), 
			reloc->dst_geom->start, reloc->dst_geom->end);
		goto error;
    }	

    if (!(node = reiserfs_block_read(src_fs->host_dal, src_blk)))
		reiserfs_block_reading_failed(src_blk, goto error);
    
    if (!generic_node_write(dst_fs, node, dst_blk))
		goto error_free_node;
    
    reiserfs_block_free(node);
    return dst_blk;
    
error_free_node:
    reiserfs_block_free(node);
error:
    return 0;
}

/* Calback functions */
static long callback_node_check(reiserfs_block_t *node, struct reiserfs_reloc *reloc) {
    /* 
		Some node checks must be here. This is because traverse function 
		performs only basic checks like tree node level check.
	*/
    return 1;
}    

static long callback_node_setup(reiserfs_block_t *node, struct reiserfs_reloc *reloc) {
    reiserfs_item_head_t *item;
    reiserfs_fs_t *dst_fs, *src_fs;
    
    src_fs = reloc->src_fs;
    dst_fs = reloc->dst_fs;
    
    libreiserfs_gauge_set_value(reloc->gauge, (reloc->counter++ * 100) / 
        reiserfs_geom_len(reloc->src_geom));
    
    if (is_leaf_node(node)) {
		uint32_t i;

		for (i = 0; i < get_blkh_nr_items(GET_BLOCK_HEAD(node)); i++) {
    	    item = GET_ITEM_HEAD(node, i);

    	    if (!dal_equals(src_fs->host_dal, dst_fs->host_dal)) {
				if (is_stat_data_ih(item))
	    	    	reiserfs_object_use(dst_fs, get_key_objectid(&item->ih_key));
	    	}
	    
	    	/* Moving the all pieces of a big file */
	    	if (is_indirect_ih(item)) {
				uint32_t j, *blocks = (uint32_t *)GET_ITEM_BODY(node, item);
	    
				for (j = 0; j < IH_UNFM_NUM(item); j++) {
	    	    	blk_t blk;

	    	    	if (blocks[j] != 0) {
						if (!(blk = generic_node_move(reloc, LE32_TO_CPU(blocks[j]))))
			    			return 0;
						blocks[j] = CPU_TO_LE32(blk);
		    		}
				}
	    	}
		}
    }
    
    return reiserfs_block_location(node);
}

static long callback_chld_setup(reiserfs_block_t *node, uint32_t chld, long chld_blk, 
	struct reiserfs_reloc *reloc)
{
    set_dc_child_blocknr(GET_DISK_CHILD(node, chld), (blk_t)chld_blk);
    return 1;
}

static long callback_node_write(reiserfs_block_t *node, struct reiserfs_reloc *reloc) {
    blk_t dst_blk;
    reiserfs_fs_t *src_fs, *dst_fs;
    
    src_fs = reloc->src_fs;
    dst_fs = reloc->dst_fs;
    
    if (dal_equals(dst_fs->host_dal, src_fs->host_dal))
		reiserfs_fs_bitmap_unuse_block(src_fs, reiserfs_block_location(node));
	
    /* Finding free block */
    dst_blk = reiserfs_fs_bitmap_find_free_block(dst_fs, reloc->dst_geom->start);
    
    /* Checking whether found free block lies inside allowed find area */
    if (!reiserfs_geom_test_inside(reloc->dst_geom, dst_blk)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
	    	_("Couldn't find free block inside allowed area (%lu - %lu)."), 
	    	reloc->dst_geom->start, reloc->dst_geom->end);
		return 0;
    }

    /* Moving given node to found free location */
    if (!generic_node_write(dst_fs, node, dst_blk))
		return 0;
    
    return dst_blk;
}

/* Geometry functions */
reiserfs_geom_t *reiserfs_geom_create(dal_t *dal, blk_t start, blk_t end) {
	reiserfs_geom_t *geom;
    
    if (!(geom = libreiserfs_calloc(sizeof(*geom), 0)))
		return NULL;

    if (!reiserfs_geom_init(geom, dal, start, end))
		goto error_free_geom;
    
    return geom;
	
error_free_geom:
	libreiserfs_free(geom);
error:
	return NULL;
}
    
int reiserfs_geom_init(reiserfs_geom_t *geom, dal_t *dal, blk_t start, blk_t end) {
	ASSERT(dal != NULL, return 0);
    ASSERT(geom != NULL, return 0);
    ASSERT(start < end, return 0);
    
    geom->dal = dal;
    geom->start = start;
    geom->end = end;

	return 1;
}
    
void reiserfs_geom_free(reiserfs_geom_t *geom) {
    ASSERT(geom != NULL, return);
    libreiserfs_free(geom);
}

blk_t reiserfs_geom_relocate(reiserfs_fs_t *dst_fs, reiserfs_geom_t *dst_geom, 
    reiserfs_fs_t *src_fs, reiserfs_geom_t *src_geom, reiserfs_gauge_t *gauge) 
{
    struct reiserfs_reloc reloc;
    
    ASSERT(dst_geom != NULL, return 0);
    ASSERT(src_geom != NULL, return 0);
    ASSERT(dst_fs != NULL, return 0);
    ASSERT(src_fs != NULL, return 0);
    
	/* 
		Checking whether src and dst geoms are 
		overlaped.
	*/
	if (reiserfs_geom_test_overlap(dst_geom, src_geom)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Overlaped geoms are detected. Destination geom (%lu - %lu) "
			"overlaps with source geom (%lu - %lu)."), dst_geom->start, 
			dst_geom->end, src_geom->start, src_geom->end);
		return 0;	
	}

    reloc.dst_geom = dst_geom;
    reloc.src_geom = src_geom;
	reloc.dst_fs = dst_fs;
	reloc.src_fs = src_fs;
	reloc.gauge = gauge;
	
    reloc.counter = 0;

	return reiserfs_tree_traverse(reiserfs_fs_tree(src_fs), &reloc, 
		(reiserfs_edge_traverse_func_t)callback_node_check, 
		(reiserfs_node_func_t)callback_node_setup, 
		(reiserfs_chld_func_t)callback_chld_setup, 
		(reiserfs_edge_traverse_func_t)callback_node_write);
}

int reiserfs_geom_test_inside(reiserfs_geom_t *geom, blk_t blk) {

    ASSERT(geom != NULL, return 0);
    return (blk >= geom->start && blk < geom->end);
}

int reiserfs_geom_test_overlap(reiserfs_geom_t *geom1, reiserfs_geom_t *geom2) {

    ASSERT(geom1 != NULL, return 0);
    ASSERT(geom2 != NULL, return 0);
   
	if (dal_equals(geom1->dal, geom2->dal)) {
		if (geom1->start < geom2->start)
			return geom1->end > geom2->start;
		else
			return geom2->end > geom1->start;
	}
	
	return 0;
}

int reiserfs_geom_fill(reiserfs_geom_t *geom, char c, reiserfs_gauge_t *gauge) {
	blk_t i;
	reiserfs_block_t *block;
	
	ASSERT(geom != NULL, return 0);

	for (i = 0; i < reiserfs_geom_len(geom); i++) {

		if (gauge)
			libreiserfs_gauge_set_value(gauge, (unsigned int)((i + 1) * 100) / 
			reiserfs_geom_len(geom));
			
		if (!(block = reiserfs_block_alloc(geom->dal, geom->start + i, c)))
			return 0;
		
		if (!reiserfs_block_write(geom->dal, block))
			reiserfs_block_writing_failed(geom->start + i, goto error_free_block);
		
		reiserfs_block_free(block);
	}
	
	if (gauge)
		libreiserfs_gauge_done(gauge);
	
	return 1;
	
error_free_block:
	reiserfs_block_free(block);
error:
	return 0;
}

blk_t reiserfs_geom_len(reiserfs_geom_t *geom) {
    ASSERT(geom != NULL, return 0);
    return geom->end - geom->start;
}

