/*
    tree.c -- reiserfs balanced tree implementation
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>

#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#define N_(String) (String)
#if ENABLE_NLS
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

#define get_st_blocks(size) ((size + 511) / 512)

static int reiserfs_tree_internal_insert(reiserfs_block_t *node, 
    uint32_t pos, struct key *key, reiserfs_disk_child_t *child);

static reiserfs_block_t *reiserfs_tree_node_alloc(reiserfs_tree_t *tree, 
    uint32_t level) 
{
    blk_t blk;
    reiserfs_block_t *node;

    if (!(blk = reiserfs_fs_bitmap_find_free_block(tree->fs, 1))) {
        libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
    	    _("Couldn't find free block."));
		return 0;
    }
	
    if (!(node = reiserfs_block_alloc(reiserfs_tree_dal(tree), blk, 0)))
        return 0;
	    
    set_blkh_level(GET_BLOCK_HEAD(node), level);
    set_blkh_nr_items(GET_BLOCK_HEAD(node), 0);
    set_blkh_free_space(GET_BLOCK_HEAD(node), reiserfs_fs_block_size(tree->fs) - 
		BLKH_SIZE);
	
    return node;
}

static reiserfs_block_t *reiserfs_tree_grow(reiserfs_tree_t *tree) {
    blk_t new_root_blk;
    reiserfs_disk_child_t child;
    reiserfs_item_head_t *item_head;
    reiserfs_block_t *new_root, *old_root;
	
    if (reiserfs_tree_height(tree) == MAX_HEIGHT) {
        libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
    	    _("Tree already has max height %d."), MAX_HEIGHT);
		goto error;
    }
	
    if (!(new_root = reiserfs_tree_node_alloc(tree, 
        reiserfs_tree_height(tree)))) 
    {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
			_("Couldn't allocate new block."));
		goto error;
    }
	
    /* Getting old root */
    if (reiserfs_tree_root(tree) && reiserfs_tree_height(tree) > 1) {

		if (!(old_root = reiserfs_block_read(reiserfs_tree_dal(tree), 
		  reiserfs_tree_root(tree))))
		{  
			reiserfs_block_reading_failed(reiserfs_tree_root(tree), 
			  goto error_free_new_root);
		}	
		
		/* Inserting old root as child of new root */
		item_head = GET_ITEM_HEAD(old_root, 0);
		
		memset(&child, 0, sizeof(child));
		
		child.dc_block_number = reiserfs_block_location(old_root);
		child.dc_size = get_blkh_free_space(GET_BLOCK_HEAD(old_root));

		if (!reiserfs_tree_internal_insert(new_root, 0, &item_head->ih_key, 
			&child)) 
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't insert old root as child into new root."));
			goto error_free_old_root;
		}
		reiserfs_block_free(old_root);
    }
    
    new_root_blk = reiserfs_block_location(new_root);
    
    if (!reiserfs_block_write(reiserfs_tree_dal(tree), new_root))
		reiserfs_block_writing_failed(new_root_blk, goto error_free_new_root);
    
    reiserfs_fs_bitmap_use_block(tree->fs, new_root_blk);

    /* Updating super */
    reiserfs_tree_set_root(tree, new_root_blk);
    reiserfs_tree_set_height(tree, reiserfs_tree_height(tree) + 1);

    return new_root;
    
error_free_old_root:
    reiserfs_block_free(old_root);
error_free_new_root:
    reiserfs_block_free(new_root);
error:
    return NULL;
}

dal_t *reiserfs_tree_dal(reiserfs_tree_t *tree) {
    
    ASSERT(tree != NULL, return NULL);
    return tree->fs->host_dal;
}

blk_t reiserfs_tree_root(reiserfs_tree_t *tree) {
    
    ASSERT(tree != NULL, return 0);
    return get_sb_root_block(tree->fs->super);
}

void reiserfs_tree_set_root(reiserfs_tree_t *tree, blk_t root) {
    
    ASSERT(tree != NULL, return);
    
    set_sb_root_block(tree->fs->super, root);
    mark_super_dirty(tree->fs);
}

uint32_t reiserfs_tree_height(reiserfs_tree_t *tree) {

    ASSERT(tree != NULL, return 0);
    return get_sb_tree_height(tree->fs->super);
}

void reiserfs_tree_set_height(reiserfs_tree_t *tree, uint32_t height) {
    
    ASSERT(tree != NULL, return);
    ASSERT(height < MAX_HEIGHT, return);
    
    set_sb_tree_height(tree->fs->super, height);
    mark_super_dirty(tree->fs);
}

reiserfs_tree_t *reiserfs_tree_open(reiserfs_fs_t *fs) {
    reiserfs_tree_t *tree;
    
    ASSERT(fs != NULL, return NULL);
    
    if (!(tree = (reiserfs_tree_t *)libreiserfs_calloc(sizeof(*tree), 0)))
		return NULL;
	
    tree->fs = fs;
    return tree;
}

static inline void make_empty_direntry(reiserfs_de_head_t *deh, int format, 
	uint32_t dirid, uint32_t objid, uint32_t par_dirid, uint32_t par_objid)
{
    size_t dir_size;

    dir_size = (size_t)(format == FS_FORMAT_3_6 ? EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE);
    
    memset(deh, 0, dir_size);
    
    /* Direntry header of "." */
    set_de_offset(&deh[0], DOT_OFFSET);
    set_de_dirid(&deh[0], dirid);
    set_de_objectid(&deh[0], objid);

    if (format == FS_FORMAT_3_6)
		set_de_location(&deh[0], (EMPTY_DIR_V2_SIZE - ROUND_UP(strlen ("."))));
    else	
		set_de_location(&deh[0], (EMPTY_DIR_V1_SIZE - strlen(".")));
	
    set_de_state(&deh[0], 0);
    mark_de_visible(&deh[0]);
  
    /* Direntry header of ".." */
    set_de_offset(&deh[1], DOT_DOT_OFFSET);
    
    /* Key of ".." for the directory */
    set_de_dirid(&deh[1], par_dirid);
    set_de_objectid(&deh[1], par_objid);
    
    if (format == FS_FORMAT_3_6)
		set_de_location(&deh[1], (get_de_location(&deh[0]) - ROUND_UP(strlen(".."))));
    else	
		set_de_location(&deh[1], (get_de_location(&deh[0]) - strlen("..")));
	
    set_de_state(&deh[1], 0);
    mark_de_visible(&deh[1]);

    memcpy((void *)deh + get_de_location(&deh[0]), ".", 1);
    memcpy((void *)deh + get_de_location(&deh[1]), "..", 2);
}

static void make_empty_dir(void *body, int format, size_t blocksize, 
    uint32_t dirid, uint32_t objid, uint32_t par_dirid, uint32_t par_objid) 
{
    reiserfs_item_head_t *ih;
    reiserfs_sd_v1_t *sd_v1;
    reiserfs_sd_v2_t *sd_v2;

    ASSERT(body != NULL, return);
    
    /* First item is stat data item of the directory */
    ih = (reiserfs_item_head_t *)body;
    set_key_dirid(&ih->ih_key, dirid);
    set_key_objectid(&ih->ih_key, objid);

    if (format == FS_FORMAT_3_6) {
		set_ih_item_format(ih, ITEM_FORMAT_2);
		set_key_v2_offset(&ih->ih_key, SD_OFFSET);
		set_key_v2_type(&ih->ih_key, KEY_TYPE_SD);
    } else {
		set_ih_item_format(ih, ITEM_FORMAT_1);
		set_key_v1_offset(&ih->ih_key, SD_OFFSET);
		set_key_v1_type(&ih->ih_key, KEY_UNIQ_SD);
    }	
    set_ih_item_len(ih, (format == FS_FORMAT_3_6 ?  SD_V2_SIZE : SD_V1_SIZE));
    set_ih_item_location(ih, blocksize - (format == FS_FORMAT_3_6 ? SD_V2_SIZE : 
		SD_V1_SIZE));

    set_ih_free_space(ih, MAX_US_INT);

    /* Fill new stat data */
    if (format == FS_FORMAT_3_6) {
		sd_v2 = (reiserfs_sd_v2_t *)(body + get_ih_item_location(ih) - BLKH_SIZE);
		
		set_sd_v2_mode(sd_v2, S_IFDIR + 0755);
		set_sd_v2_nlink(sd_v2, 3);
		set_sd_v2_uid(sd_v2, getuid());
		set_sd_v2_gid(sd_v2, getgid());
		
		set_sd_v2_size(sd_v2, EMPTY_DIR_V2_SIZE);
		
		set_sd_v2_atime(sd_v2, time(NULL));
		set_sd_v2_ctime(sd_v2, time(NULL));
		set_sd_v2_mtime(sd_v2, time(NULL));
		
		set_sd_v2_blocks(sd_v2, get_st_blocks(EMPTY_DIR_V2_SIZE));
		set_sd_v2_rdev(sd_v2, 0);
    } else {
		sd_v1 = (reiserfs_sd_v1_t *)(body + get_ih_item_location(ih) - BLKH_SIZE);
		
		set_sd_v1_mode(sd_v1, S_IFDIR + 0755);
		set_sd_v1_nlink(sd_v1, 3);
		set_sd_v1_uid(sd_v1, getuid());
		set_sd_v1_gid(sd_v1, getgid());
		
		set_sd_v1_size(sd_v1, EMPTY_DIR_V1_SIZE);
		
		set_sd_v1_atime(sd_v1, time(NULL));
		set_sd_v1_ctime(sd_v1, time(NULL));
		set_sd_v1_mtime(sd_v1, time(NULL));
		
		set_sd_v1_blocks(sd_v1, get_st_blocks(EMPTY_DIR_V1_SIZE));
    }

    /* Second item is directory item, containing "." and ".." */
    ih++;
    set_key_dirid(&ih->ih_key, dirid);
    set_key_objectid(&ih->ih_key, objid);
    
    if (format == FS_FORMAT_3_6) {
		set_ih_item_format(ih, ITEM_FORMAT_2);
		set_key_v2_offset(&ih->ih_key, DOT_OFFSET);
		set_key_v2_type(&ih->ih_key, KEY_TYPE_DR);
    } else {
		set_ih_item_format(ih, ITEM_FORMAT_1);
		set_key_v1_offset(&ih->ih_key, DOT_OFFSET);
		set_key_v1_type(&ih->ih_key, KEY_UNIQ_DR);
    }	
	
    set_ih_item_len(ih, 
		(format == FS_FORMAT_3_6 ? EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE));
    set_ih_item_location(ih, get_ih_item_location(ih - 1) - get_ih_item_len(ih));
    set_ih_entry_count(ih, 2);

    /* Compose item */
    make_empty_direntry((reiserfs_de_head_t *)(body + get_ih_item_location(ih) - 
		BLKH_SIZE), format, dirid, objid, 0, dirid);
}

reiserfs_tree_t *reiserfs_tree_create(reiserfs_fs_t *fs) {
    int format;
    blk_t root_blk;
    size_t blocksize;
    reiserfs_tree_t *tree;
    reiserfs_block_t *root;
    reiserfs_block_head_t *blkh;
    
	ASSERT(fs != NULL, return NULL);
    
    if (!(tree = (reiserfs_tree_t *)libreiserfs_calloc(sizeof(*tree), 0)))
		return NULL;

    tree->fs = fs;
    
    if (!(root = reiserfs_tree_node_alloc(tree, 2)))
		goto error_free_tree;
    
    blocksize = get_sb_block_size(fs->super);
    format = get_sb_format(fs->super);
    
    /* Block head */
    blkh = (reiserfs_block_head_t *)root->data;
    set_blkh_level(blkh, LEAF_LEVEL);
    set_blkh_nr_items(blkh, 2);
        
    set_blkh_free_space(blkh, blocksize - BLKH_SIZE - 2 * IH_SIZE - 
		(format == FS_FORMAT_3_6 ? SD_V2_SIZE : SD_V1_SIZE) - 
		(format == FS_FORMAT_3_6 ? EMPTY_DIR_V2_SIZE : EMPTY_DIR_V1_SIZE));

    make_empty_dir(root->data + BLKH_SIZE, format, 
		blocksize, ROOT_DIR_ID, ROOT_OBJ_ID, 0, ROOT_DIR_ID);
	
    if (!reiserfs_block_write(reiserfs_tree_dal(tree), root))
		goto error_free_root;
	
    root_blk = reiserfs_block_location(root);
    reiserfs_fs_bitmap_use_block(tree->fs, root_blk);
    
    reiserfs_object_use(fs, ROOT_DIR_ID);
    reiserfs_object_use(fs, ROOT_OBJ_ID);
    
    reiserfs_tree_set_height(tree, 2);
    reiserfs_tree_set_root(tree, root_blk);
    
    reiserfs_block_free(root);
    return tree;

error_free_root:
    reiserfs_block_free(root);    
error_free_tree:
    libreiserfs_free(tree);    
error:
    return NULL;    
}

static size_t reiserfs_tree_leaf_body_size(reiserfs_block_t *node, 
    uint32_t from, uint32_t to) 
{
    uint32_t i, size = 0;
    
    for (i = from; i < to; i++)
		size += get_ih_item_len(GET_ITEM_HEAD(node, i));
    
    return size;
}

static int reiserfs_tree_leaf_insert(reiserfs_block_t *node, uint32_t pos, 
    reiserfs_item_head_t *item_head, void *item_body)
{
    int i;
    size_t item_size, body_size;
    void *dst_item_body, *src_item_body;
    reiserfs_item_head_t *dst_item_head, *src_item_head;

    ASSERT(node != NULL, return 0);
    ASSERT(item_head != NULL, return 0);
    
    if (pos > get_blkh_nr_items(GET_BLOCK_HEAD(node)))
		return 0;

    /* Source and destination item head pointers */
    src_item_head = GET_ITEM_HEAD(node, pos);
    src_item_body = GET_ITEM_BODY(node, GET_ITEM_HEAD(node, 
	get_blkh_nr_items(GET_BLOCK_HEAD(node)) - 1));
    
    /* Source and destination item body pointers */
    dst_item_head = src_item_head + 1;
    dst_item_body = src_item_body - get_ih_item_len(item_head);
    
    set_ih_item_location(item_head, src_item_head->ih_item_location);

    if (pos < get_blkh_nr_items(GET_BLOCK_HEAD(node))) {
		item_size = (get_blkh_nr_items(GET_BLOCK_HEAD(node)) - pos) * IH_SIZE;
		body_size = reiserfs_tree_leaf_body_size(node, pos, 
			get_blkh_nr_items(GET_BLOCK_HEAD(node)));
		
		memmove(dst_item_head, src_item_head, item_size);
		memmove(dst_item_body, src_item_body, body_size);
    }
    memcpy(src_item_head, item_head, IH_SIZE);
    memcpy(src_item_body, item_body, get_ih_item_len(item_head));
    
    /* Setting up the leaf */
    set_blkh_nr_items(GET_BLOCK_HEAD(node), get_blkh_nr_items(GET_BLOCK_HEAD(node)) + 1);
    set_blkh_free_space(GET_BLOCK_HEAD(node), get_blkh_free_space(GET_BLOCK_HEAD(node)) - 
		(IH_SIZE + get_ih_item_len(item_head)));
    
    /* Setting up item locations */
    for (i = pos + 1; i < get_blkh_nr_items(GET_BLOCK_HEAD(node)) - 1; i++) {
		reiserfs_item_head_t *ih = GET_ITEM_HEAD(node, i);
		set_ih_item_location(ih, ih->ih_item_location - get_ih_item_len(item_head));
    }
    
    return 1;
}

/* 
    Inserting given item (stat data item, direntry item, direct 
    item or indirect item) into the tree corresponding to item's 
    key. If given item is stat data then it objectid will be 
    marked as used in objectid map. Inserted item must be marked
    as not used.
*/
int reiserfs_tree_insert_item(reiserfs_tree_t *tree, reiserfs_item_head_t *item_head, 
    void *item_body) 
{
    struct key *key;
    reiserfs_path_t *path;
    reiserfs_path_node_t *path_node;
    reiserfs_block_t *node;

    ASSERT(item_head != NULL, return 0);
    ASSERT(item_body != NULL, return 0);
    
    key = &item_head->ih_key;
    
    if (reiserfs_object_test((reiserfs_fs_t *)tree->fs, get_key_objectid(key)) &&
		is_stat_data_key(key))
    {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
			_("Stat data item for object %lu already exists."), get_key_objectid(key));
		return 0;    
    }	
    
    if (!reiserfs_object_test((reiserfs_fs_t *)tree->fs, get_key_objectid(key)) &&
		!is_stat_data_key(key))
    {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
			_("Couldn't find stat data item for object %lu."), get_key_objectid(key));
		return 0;    
    }

    /* Trying to find where we must insert given item */
    path = reiserfs_path_create(MAX_HEIGHT);
    reiserfs_tree_lookup_leaf(tree, reiserfs_tree_root(tree), 
		reiserfs_key_comp_two_components, key, path);
    
    /* Checking whether tree isn't empty */
    if (path->length == 0) {
		reiserfs_block_t *root_node;
		
		if (!(root_node = reiserfs_tree_grow(tree)))
			goto error_free_path;

		reiserfs_path_inc(path, reiserfs_path_node_create(NULL, root_node, 0));
    }

    path_node = reiserfs_path_last(path);
    node = path_node->node;
    
    /* 
		Node was found. If it has enought free space, then we'll 
		trying to insert given item inside it. If node doesn't has 
		enought free space for new item we'll prepare new leaf and
		insert it into the tree.
    */
    if (get_blkh_free_space(GET_BLOCK_HEAD(node)) >= IH_SIZE + 
		get_ih_item_len(item_head)) 
    {
		blk_t node_blk = reiserfs_block_location(node);
		
		if (!reiserfs_tree_leaf_insert(node, path_node->pos, item_head, item_body)) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
				_("Couldn't insert item into leaf %lu."), node_blk);
			goto error_free_path;
		}
		
		if (!reiserfs_block_write(reiserfs_tree_dal(tree), node))
			reiserfs_block_writing_failed(node_blk, goto error_free_path);
    } else {
		/*
			Preparing new tree node and inserting them into the tree 
			by reiserfs_tree_insert 
		*/
		reiserfs_block_t *new_node;
		
		if (!(new_node = reiserfs_tree_node_alloc(tree, LEAF_LEVEL))) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
				_("Couldn't allocate new block."));
			goto error_free_path;
		}
		
		if (!reiserfs_tree_leaf_insert(new_node, 0, item_head, item_body)) {
			reiserfs_block_free(new_node);
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
				_("Couldn't insert item into new block."));
			goto error_free_path;
		}
		
		if (!reiserfs_tree_insert(tree, new_node)) {
			reiserfs_block_free(new_node);
			goto error_free_path;
		}
    }

    /* Marking object as used */
    if (is_stat_data_key(key))
		reiserfs_object_use((reiserfs_fs_t *)tree->fs, get_key_objectid(key));
    
    reiserfs_path_free(path);
    return 1;
    
error_free_path:
    reiserfs_path_free(path);
error:
    return 0;
}

int reiserfs_tree_remove_item(reiserfs_tree_t *tree, reiserfs_item_head_t *item_head) {
	/* Not implemented yet */
    return 1;
}

static int reiserfs_tree_internal_insert(reiserfs_block_t *node, 
    uint32_t pos, struct key *key, reiserfs_disk_child_t *child) 
{
    uint32_t item_num;
    void *src, *dst;

    ASSERT(node != NULL, return 0);
    ASSERT(child != NULL, return 0);

    if (pos > get_blkh_nr_items(GET_BLOCK_HEAD(node)))
		return 0;
	
    src = node->data + BLKH_SIZE + pos * FULL_KEY_SIZE;
    dst = src + FULL_KEY_SIZE;
    item_num = get_blkh_nr_items(GET_BLOCK_HEAD(node)) - pos;
	
    memmove(src, dst, item_num * FULL_KEY_SIZE + 
	get_blkh_nr_items(GET_BLOCK_HEAD(node)) * DC_SIZE);
    memcpy(src, key, FULL_KEY_SIZE);
	
    src = GET_DISK_CHILD(node, pos) + DC_SIZE;
    dst = src + DC_SIZE;
	
    memmove(src, dst, item_num * DC_SIZE);
    memcpy(src, child, DC_SIZE);
    
    set_blkh_nr_items(GET_BLOCK_HEAD(node), get_blkh_nr_items(GET_BLOCK_HEAD(node)) + 1);
    set_blkh_free_space(GET_BLOCK_HEAD(node),
        get_blkh_free_space(GET_BLOCK_HEAD(node)) - (FULL_KEY_SIZE + DC_SIZE));

    return 1;	
}

static int reiserfs_tree_node_split(reiserfs_tree_t *tree, 
    reiserfs_block_t *node, reiserfs_block_t **left, 
    reiserfs_block_t **right)
{
    void *right_body;
    struct key *median_key;
    uint32_t left_key_num, right_key_num;

    ASSERT(tree != NULL, return 0);
    ASSERT(node != NULL, return 0);
    ASSERT(right != NULL, return 0);
    ASSERT(left != NULL, return 0);

    left_key_num = get_blkh_nr_items(GET_BLOCK_HEAD(node)) / 2;
    median_key = GET_KEY(node, left_key_num);

    if (!(*right = reiserfs_tree_node_alloc(tree, 
		get_blkh_level(GET_BLOCK_HEAD(node))))) 
	{
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_IGNORE, 
    	    _("Couldn't allocate new block."));
    	return 0;
    }
    
    /* Preparing right neighbor */
    right_body = (*right)->data + BLKH_SIZE;
    right_key_num = get_blkh_nr_items(GET_BLOCK_HEAD(node)) - left_key_num;
    memcpy(right_body, median_key, right_key_num * FULL_KEY_SIZE);
    memcpy(right_body + right_key_num * FULL_KEY_SIZE, 
	GET_DISK_CHILD(node, 0), right_key_num * DC_SIZE);
    set_blkh_nr_items(GET_BLOCK_HEAD(*right), right_key_num);
    set_blkh_free_space(GET_BLOCK_HEAD(*right), reiserfs_fs_block_size(tree->fs) - 
	BLKH_SIZE - (right_key_num * FULL_KEY_SIZE) - (right_key_num * DC_SIZE));
    
    /* Preparing left neighbor */
    *left = node;
    memmove(median_key, GET_DISK_CHILD(*left, 0), right_key_num * FULL_KEY_SIZE);
    set_blkh_nr_items(GET_BLOCK_HEAD(*left), left_key_num);
    set_blkh_free_space(GET_BLOCK_HEAD(*left), reiserfs_fs_block_size(tree->fs) - 
	BLKH_SIZE - (left_key_num * FULL_KEY_SIZE) - (left_key_num * DC_SIZE));
    
    memset(GET_DISK_CHILD(*left, left_key_num - 1), 0, 
	get_blkh_free_space(GET_BLOCK_HEAD(*left)));
    
    return 1;
}

/* Inserting given node corresponding to balancing rules */
static int reiserfs_tree_insert_node(reiserfs_tree_t *tree, 
    reiserfs_path_t *path, reiserfs_block_t *node) 
{
    blk_t node_blk;
    reiserfs_disk_child_t child;
    reiserfs_item_head_t *item_head;
    reiserfs_path_node_t *curr_node = NULL;
    
    ASSERT(tree != NULL, return 0);
    ASSERT(path != NULL, return 0);
    ASSERT(node != NULL, return 0);
    
    /* Checking whether we are at root. If so, making new root */
    if (path->length == 0) {
		reiserfs_block_t *new_root;
	
		if (!(new_root = reiserfs_tree_grow(tree)))
			return 0;
	
		reiserfs_path_insert(path, 0, reiserfs_path_node_create(NULL, new_root, 0));
    }
    
    item_head = GET_ITEM_HEAD(node, 0);
    
    if (!(curr_node = reiserfs_path_last(path))) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Invalid path detected. Path length is %d."), path->length);
		return 0;
    }
    
    node_blk = reiserfs_block_location(node);
    
    /* Preparing new child */
    memset(&child, 0, sizeof(child));
	
    child.dc_block_number = node_blk;
    child.dc_size = get_blkh_free_space(GET_BLOCK_HEAD(node));

    /* 
		Checking whether internal node has enought free space 
		to insert yet another child. 
    */
    if (get_blkh_free_space(GET_BLOCK_HEAD(curr_node->node)) >= FULL_KEY_SIZE + DC_SIZE) {
		if (!reiserfs_tree_internal_insert(curr_node->node, curr_node->pos, 
			&item_head->ih_key, &child)) 
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't insert disk child into internal node."));
			return 0;
		}
		
		/* Writing given node to disk, internal node and updating bitmap. */
		if (!reiserfs_block_write(reiserfs_tree_dal(tree), node))
			reiserfs_block_writing_failed(node_blk, return 0);
		
		reiserfs_fs_bitmap_use_block(tree->fs, node_blk);
		
		if (!reiserfs_block_write(reiserfs_tree_dal(tree), curr_node->node)) {
			reiserfs_block_writing_failed(
				reiserfs_block_location(curr_node->node), return 0);
		}
    } else {
		uint32_t pos;
		blk_t curr_node_blk;
		reiserfs_block_t *right_node, *left_node, *target_node;
		
		struct key *median_key = GET_KEY(curr_node->node, 
			get_blkh_nr_items(GET_BLOCK_HEAD(curr_node->node)) / 2);
		
		curr_node_blk = reiserfs_block_location(curr_node->node);
		
		/* Spliting internal node */
		if (!reiserfs_tree_node_split(tree, curr_node->node, &left_node, &right_node)) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't split node %lu."), curr_node_blk);
			return 0;
		}
		
		/* 
			Inserting corresponding key and disk pointer 
			into left or right neighbor.
		*/
		target_node = (reiserfs_key_comp_four_components(&item_head->ih_key, 
			median_key) < 0 ? left_node : right_node);
		
		reiserfs_tools_fast_search(&item_head->ih_key, GET_ITEM_HEAD(target_node, 0), 
			get_blkh_nr_items(GET_BLOCK_HEAD(target_node)), FULL_KEY_SIZE,
			reiserfs_key_comp_four_components, &pos);
		
		if (!reiserfs_tree_internal_insert(target_node, pos, &item_head->ih_key, &child))
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't insert into internal node."));
			return 0;	
		}
		
		/* Saving left node */
		if (!reiserfs_block_write(reiserfs_tree_dal(tree), left_node))
			reiserfs_block_writing_failed(reiserfs_block_location(left_node), return 0);
		
		/* Decreasing path by one level */	
		reiserfs_path_dec(path);
		
		/* 
			Recursive call to insert the right neighbor node into corresponding 
			internal node. In theory this my cause splitting parent node too 
			and finally to increase tree height.
		*/
		if (!reiserfs_tree_insert_node(tree, path, right_node)) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Couldn't insert right neighbor into parent internal node."));
			return 0;
		}
    }
    
    return 1;
}

int reiserfs_tree_insert(reiserfs_tree_t *tree, reiserfs_block_t *node) {
    reiserfs_block_t *internal = NULL;
    reiserfs_item_head_t *item_head;
    reiserfs_path_t *path;

    ASSERT(tree != NULL, return 0);
    ASSERT(node != NULL, return 0);
    
    item_head = (reiserfs_item_head_t *)(node->data + BLKH_SIZE);
    path = reiserfs_path_create(MAX_HEIGHT);
    
    if (!reiserfs_tree_lookup_internal(tree, reiserfs_tree_root(tree), 
		reiserfs_key_comp_four_components, &item_head->ih_key, path))
    {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Couldn't find corresponding internal node."));
		goto error_free_path;
    }
    
    if (!reiserfs_tree_insert_node(tree, path, node))
		goto error_free_path;
    
    reiserfs_path_free(path);
    return 1;
    
error_free_path:
    reiserfs_path_free(path);
error:
    return 0;
} 

int reiserfs_tree_remove(reiserfs_tree_t *tree, reiserfs_block_t *node) {
	/* Not implemented yet */
    return 1;
}

static int reiserfs_tree_node_lookup(reiserfs_tree_t *tree, blk_t blk, 
	reiserfs_comp_func_t comp_func, struct key *key, int for_leaf, 
	reiserfs_path_t *path) 
{
    reiserfs_block_t *node;
    uint32_t level, is_found = 0, pos = 0;
    
    ASSERT(tree != NULL, return 0);
    ASSERT(key != NULL, return 0);
    
    if (!comp_func) return 0;
    
    if (path)
		reiserfs_path_clear(path);
    
    while (1) {
	
		if (!(node = reiserfs_block_read(reiserfs_tree_dal(tree), blk)))
			reiserfs_block_reading_failed(blk, return 0);
		
		if ((level = get_blkh_level((reiserfs_block_head_t *)node->data)) > 
			(uint32_t)reiserfs_tree_height(tree) - 1)
		{
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Invalid node level. Found %d, expected less than %d."), 
				level, reiserfs_tree_height(tree));
			return 0;
		}
		
		if (!for_leaf && is_leaf_node(node))
			return 0;
			
		is_found = reiserfs_tools_fast_search(key, GET_ITEM_HEAD(node, 0), 
			get_blkh_nr_items(GET_BLOCK_HEAD(node)), (is_leaf_node(node) ? 
			IH_SIZE : FULL_KEY_SIZE), comp_func, &pos);
		
		if (path) {
			reiserfs_path_inc(path, reiserfs_path_node_create(reiserfs_path_last(path), 
				node, (is_found && is_internal_node(node) ? pos + 1 : pos)));
		}
		
		if (is_leaf_node(node))
			return is_found;
			
		if (level == 2 && !for_leaf)
			return 1;
			
		if (is_found) pos++;
		
		blk = get_dc_child_blocknr(GET_DISK_CHILD(node, pos));
    }
	
    return 0;
}

reiserfs_path_node_t *reiserfs_tree_lookup_internal(reiserfs_tree_t *tree, blk_t from,
    reiserfs_comp_func_t comp_func, struct key *key, reiserfs_path_t *path) 
{
    if (tree && reiserfs_tree_height(tree) < 2)
		return NULL;

    return (reiserfs_tree_node_lookup(tree, from, comp_func, key, 0, path) ? 
		reiserfs_path_last(path) : NULL);
}

reiserfs_path_node_t *reiserfs_tree_lookup_leaf(reiserfs_tree_t *tree, blk_t from,
    reiserfs_comp_func_t comp_func, struct key *key, reiserfs_path_t *path) 
{
    if (tree && reiserfs_tree_height(tree) < 2)
		return NULL;
    
    return (reiserfs_tree_node_lookup(tree, from, comp_func, key, 1, path) ? 
		reiserfs_path_last(path) : NULL);
}

static long reiserfs_tree_node_traverse(reiserfs_tree_t *tree, blk_t blk, void *data,
    reiserfs_edge_traverse_func_t before_node_func, reiserfs_node_func_t node_func, 
    reiserfs_chld_func_t chld_func, reiserfs_edge_traverse_func_t after_node_func)
{
    uint32_t block_level;
    long call_result = 0;
    reiserfs_block_t *node;
    
    if (!node_func) return 0;

    if (!(node = reiserfs_block_read(reiserfs_tree_dal(tree), blk)))
		reiserfs_block_writing_failed(blk, goto error);
    
    if (!is_leaf_node(node) && !is_internal_node(node)) {
        libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
    	    _("Invalid node detected (%lu). Unknown type."), blk);
        goto error_free_node;
    }
    
    /* This callback function may be using to perform some checks */
    if (before_node_func && !(call_result = before_node_func(node, data)))
    	goto error_free_node;
	
    if (!(call_result = node_func(node, data)))
		goto error_free_node;
	
    if (is_internal_node(node)) {
		uint32_t i;
		
		for (i = 0; i <= get_blkh_nr_items(GET_BLOCK_HEAD(node)); i++) {
			blk_t chld_blk = get_dc_child_blocknr(GET_DISK_CHILD(node, i));
			
			if (!(call_result = reiserfs_tree_node_traverse(tree, chld_blk, 
					data, before_node_func, node_func, chld_func, after_node_func)))
				goto error_free_node;
			
			if (chld_func && !chld_func(node, i, call_result, data)) 
				goto error_free_node;
		}
    }

    /* This callback function may be using to save changed node */
    if (after_node_func && !(call_result = after_node_func(node, data)))
    	goto error_free_node;
	
    reiserfs_block_free(node);
    return call_result;
    
error_free_node:
    reiserfs_block_free(node);    
error:
    return call_result;
}

long reiserfs_tree_simple_traverse(reiserfs_tree_t *tree, void *data,
    reiserfs_node_func_t node_func)
{
    if (reiserfs_tree_root(tree) < 2)
		return 1;
    
    return reiserfs_tree_node_traverse(tree, reiserfs_tree_root(tree), data, 
        NULL, node_func, NULL, NULL);
}
    
long reiserfs_tree_traverse(reiserfs_tree_t *tree, void *data,
    reiserfs_edge_traverse_func_t before_node_func, reiserfs_node_func_t node_func, 
    reiserfs_chld_func_t chld_func, reiserfs_edge_traverse_func_t after_node_func)
{
    if (reiserfs_tree_height(tree) < 2)
		return 1;
	
    return reiserfs_tree_node_traverse(tree, reiserfs_tree_root(tree), data,
        before_node_func, node_func, chld_func, after_node_func);
}


void reiserfs_tree_free(reiserfs_tree_t *tree) {
    if (!tree) return;
    	libreiserfs_free(tree);
}

