/*
 * GTK interface for pptp-client
 * 
 * Copyright 2006 Andrew Kozin <stanson@btv.ru> under the terms of the
 * GNU GPL.
 *
 * Very simple frontend for VPN connections
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <math.h>
#include <signal.h>

GPid		pptp_pid = -1;
GtkTextTag	*tag_error;
GtkTextTag	*tag_info;
GtkTextMark	*end_mark;

typedef struct _widgets {
	GtkWidget *server;
	GtkWidget *login;
	GtkWidget *passwd;
	GtkWidget *start;
	GtkWidget *stop;
	GtkWidget *log;
} GpptpWidgets;

void add_entries_from_file(GtkWidget *combo_box, char * filename, char * stops)
{
  FILE		*f;
  char		buf[256], *s, *e;
	f = fopen(filename,"r");
	if(f)
	{
		while(fgets(buf,sizeof(buf),f))
		{
			s = buf;
			while((*s == ' ') || (*s == '\t')) s++;
			if((!*s) || (*s == '#') || (*s == '\n')) continue;
			e = s;
			while((*e) && !strchr(stops, *e)) e++;
			*e = 0;
			if(e != s)
			{
				gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box), s);
			}
		}
		fclose(f);
	}
}

void log_add(GtkWidget *w, GtkTextTag *tag, gchar *str)
{
  GtkTextBuffer	*buf;
  GtkTextIter	end_iter;

	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (w));
	gtk_text_buffer_get_end_iter (buf, &end_iter);
	gtk_text_buffer_insert_with_tags (buf, &end_iter, str, -1, tag, NULL);
	gtk_text_buffer_get_end_iter (buf, &end_iter);
	gtk_text_buffer_move_mark_by_name (buf, "end", &end_iter);
	gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (w),
				gtk_text_buffer_get_mark (buf, "end"));
}

gboolean on_pptp_out (GIOChannel *src, GIOCondition cond, gpointer data)
{
  GpptpWidgets	*w = (GpptpWidgets *)data;
  gchar		*line;
  GIOStatus	status;

	status = g_io_channel_read_line (src, &line, NULL, NULL, NULL);
	
	if (status != G_IO_STATUS_NORMAL)
		return FALSE;
	
	log_add(w->log, NULL, line);
	
	g_free (line);
	
	return TRUE;
}

gboolean on_pptp_err (GIOChannel *src, GIOCondition cond, gpointer data)
{
  GpptpWidgets	*w = (GpptpWidgets *)data;
  gchar		*line;
  GIOStatus	status;

	status = g_io_channel_read_line (src, &line, NULL, NULL, NULL);
	
	if (status != G_IO_STATUS_NORMAL)
		return FALSE;
	
	log_add(w->log, tag_error, line);
	
	g_free (line);
	
	return TRUE;
}

void on_pptp_exit (GPid pid, gint status, gpointer data)
{
  GpptpWidgets	*w = (GpptpWidgets *)data;
  gchar		str[256];

  	snprintf(str, sizeof(str), "PPTP client (%d) exited with %d\n", pid, status);
	log_add(w->log, tag_info, str);
	gtk_widget_set_sensitive (GTK_WIDGET (w->start), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET (w->stop), FALSE);
	pptp_pid = -1;
}

static void pptp_connect(GtkWidget *widget, GpptpWidgets *w)
{
  char 		server[256], login[265], passwd[256], *argv[256];
  char 		str[1024];
  int		i;
  GError	*err;
  GIOChannel	*pptp_out, *pptp_err;
  gint		pptp_out_pipe, pptp_err_pipe;
  gint		pptp_out_tag, pptp_err_tag;
  GtkTextBuffer	*buf;
  GtkTextIter	end_iter;

	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (w->log));

	gtk_text_buffer_set_text (buf, "", 0);
	log_add(w->log, tag_info, "Connecting to VPN server...\n");

	strncpy(server, gtk_entry_get_text (GTK_ENTRY (w->server)), sizeof(server));
	strncpy(login, gtk_entry_get_text (GTK_ENTRY (w->login)), sizeof(login));
	strncpy(passwd, gtk_entry_get_text (GTK_ENTRY (w->passwd)), sizeof(passwd));

	i = 0;
	argv[i++] = "/usr/sbin/pptp";
	argv[i++] = server;
	argv[i++] = "file";
	argv[i++] = "/etc/ppp/options.pptp";
	argv[i++] = "user";
	argv[i++] = login;
	if(passwd[0])
	{
	  argv[i++] = "password";
	  argv[i++] = passwd;
	}
	argv[i++] = NULL;

	if(! g_spawn_async_with_pipes ("/etc/ppp", argv, NULL,
					G_SPAWN_DO_NOT_REAP_CHILD,
					NULL, NULL,
					&pptp_pid,
					NULL, &pptp_out_pipe, &pptp_err_pipe,
				  	&err))
	{
		snprintf(str, sizeof(str), "gpptp: Command failed: %s\n", err->message);
		log_add(w->log, tag_error, str);
		g_error_free (err);
		err = NULL;
		pptp_pid = -1;
	}
	else
	{
		g_child_watch_add(pptp_pid, on_pptp_exit, w);

		/* stdout */
		pptp_out = g_io_channel_unix_new (pptp_out_pipe);
		g_io_channel_set_encoding (pptp_out, NULL, NULL);
		pptp_out_tag = g_io_add_watch (pptp_out,
						(G_IO_IN | G_IO_HUP | G_IO_ERR),
						on_pptp_out,
						w);
		
		g_io_channel_unref (pptp_out);

		/* error */
		pptp_err = g_io_channel_unix_new (pptp_err_pipe);
		g_io_channel_set_encoding (pptp_err, NULL, NULL);
		pptp_err_tag = g_io_add_watch (pptp_err,
						(G_IO_IN | G_IO_HUP | G_IO_ERR),
						on_pptp_err,
						w);
		
		g_io_channel_unref (pptp_err);

		gtk_widget_set_sensitive (GTK_WIDGET (w->start), FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (w->stop), TRUE);
	}
}

static void pptp_disconnect(GtkWidget *widget, GpptpWidgets *w)
{
	log_add(w->log, tag_info, "Disconnecting from VPN server...\n");
	kill (pptp_pid, SIGTERM);
}

static void destroy(GtkWidget *widget, gpointer *cb)
{
	gtk_main_quit();
	exit(0);
}

int main(int argc, char *argv[])
{
  GtkWidget	*window,
			*vbox,
				*hbox_server,
					*label_server,
					*entry_server,
				*hbox_login, 
					*label_login,
					*entry_login,
				*hbox_passwd,
					*label_passwd,
					*entry_passwd,
				*hbox_conn,
					*button_start,
					*button_stop,
				*scroll_log,
					*text_log;

  GpptpWidgets		w;

  GtkRequisition	req;
  GtkTextBuffer		*buffer;
  GtkTextIter		end_iter;

	gtk_init(&argc, &argv);

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "VPN Connection");
	gtk_container_set_border_width(GTK_CONTAINER(window), 2);

	vbox = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	/* VPN server field */
	hbox_server = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox_server, FALSE, FALSE, 0);

	label_server = gtk_label_new("VPN Server:");
	gtk_misc_set_alignment(GTK_MISC(label_server), 1, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox_server), label_server, TRUE, TRUE, 0);

	entry_server = gtk_combo_box_entry_new_text();
	add_entries_from_file(entry_server, "/etc/ppp/vpn_servers", "# \n");
	gtk_box_pack_start(GTK_BOX(hbox_server), entry_server, FALSE, FALSE, 0);

	/* Login field */
	hbox_login = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox_login, FALSE, FALSE, 0);

	label_login = gtk_label_new("Login:");
	gtk_misc_set_alignment(GTK_MISC(label_login), 1, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox_login), label_login, TRUE, TRUE, 0);

	entry_login = gtk_combo_box_entry_new_text();
	add_entries_from_file(entry_login, "/etc/ppp/chap-secrets", "*# \n");
	gtk_box_pack_start(GTK_BOX(hbox_login), entry_login, FALSE, FALSE, 0);

	/* Password field */
	hbox_passwd = gtk_hbox_new(FALSE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox_passwd, FALSE, FALSE, 0);

	label_passwd = gtk_label_new("Password:");
	gtk_misc_set_alignment(GTK_MISC(label_passwd), 1, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox_passwd), label_passwd, TRUE, TRUE, 0);

	entry_passwd = gtk_entry_new();
	gtk_entry_set_visibility(GTK_ENTRY(entry_passwd), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox_passwd), entry_passwd, FALSE, FALSE, 0);

	/* Start/stop buttons */
	hbox_conn = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox_conn, FALSE, FALSE, 0);

	button_start = gtk_button_new_with_label("Connect");
	gtk_button_set_image (GTK_BUTTON (button_start),
				gtk_image_new_from_stock (GTK_STOCK_CONNECT,
				GTK_ICON_SIZE_BUTTON));
	gtk_box_pack_start(GTK_BOX(hbox_conn), button_start, TRUE, TRUE, 0);

	button_stop = gtk_button_new_with_label("Disconnect");
	gtk_button_set_image (GTK_BUTTON (button_stop),
				gtk_image_new_from_stock (GTK_STOCK_DISCONNECT,
				GTK_ICON_SIZE_BUTTON));
	gtk_widget_set_sensitive (GTK_WIDGET (button_stop), FALSE);
	gtk_box_pack_start(GTK_BOX(hbox_conn), button_stop, TRUE, TRUE, 0);

	/* Log window */
	scroll_log = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_log),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scroll_log, TRUE, TRUE, 0);
									      
	text_log = gtk_text_view_new();
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (text_log), GTK_WRAP_WORD);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text_log), FALSE);
	gtk_container_add (GTK_CONTAINER (scroll_log), text_log);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text_log));
	gtk_text_buffer_set_text (buffer, "", 0);
	gtk_text_buffer_get_end_iter (buffer, &end_iter);
	gtk_text_buffer_create_mark (buffer, "end", &end_iter, FALSE);

	tag_error = gtk_text_buffer_create_tag (buffer, NULL, 
						"foreground", "#C00000", NULL);
	tag_info  = gtk_text_buffer_create_tag (buffer, NULL,
						"foreground", "#00C000", NULL);

	/* Adjust entry sizes */
	gtk_widget_size_request(entry_passwd, &req);
	req.width = 150;
	gtk_widget_set_size_request(entry_server, req.width, req.height);
	gtk_widget_set_size_request(entry_login, req.width, req.height);
	gtk_widget_set_size_request(entry_passwd, req.width, req.height);

	w.server = gtk_bin_get_child(GTK_BIN(entry_server));
	w.login  = gtk_bin_get_child(GTK_BIN(entry_login));
	w.passwd = entry_passwd;
	w.start  = button_start;
	w.stop   = button_stop;
	w.log    = text_log;

	gtk_signal_connect(GTK_OBJECT(button_start), "clicked", G_CALLBACK(pptp_connect), &w);
	gtk_signal_connect(GTK_OBJECT(button_stop), "clicked", G_CALLBACK(pptp_disconnect), &w);

	/* Show widgets */
	gtk_widget_show_all (vbox);
	gtk_widget_show (window);

	gtk_signal_connect(GTK_OBJECT(window), "destroy", G_CALLBACK(destroy), NULL);

	gtk_main();
	return 0;
}
