package Clair::Classify;

=head1 NAME

 B<package> Clair::Classify
 Take in the model file generated by Learn.pm and then carry out the classification
 on the test data.

=head1 AUTHOR

 JB Kim
 jbremnant@gmail.com
 20070407

=head1 SYNOPSIS

 Use the test data produced by Clair::Feature.pm (in svm_light format), and the model
 file generated by Clair::Learn.pm to carry out classification on the test data.
 The underlying algorithm can be either Naive Bayes or Perceptron. (default: perceptron)

 In case of perceptron, all you have to do is to obtain the dot product between the weight
 vector (W) and the feature vector (X), and determine whether they are positive or negative.

 Here, both the "test" and "model" parameter in the constructor is required.

  use Clair::Classify;

  my $cla = new Clair::Classify(DEBUG => $DEBUG, test => "test.dat", model => "model.file");
  $cla->classify($algo);

=head1 DESCRIPTION

 The module should provide the ability to choose between different classifier
 algorithms.

=cut


# use FindBin;
# use lib "$FindBin::Bin/../lib/perl5/site_perl/5.8.5";
# use lib "$FindBin::Bin/lib/perl5/site_perl/5.8.5";

use strict;
use vars qw/$DEBUG/;

use Clair::Debug;
use Clair::Features;
use Clair::Learn;
use Data::Dumper;
use File::Path;


=head2 new

 The constructor. Initializes several container hashes for later use. 

=cut

sub new
{
	my ($proto, %args) = @_;
	my $class = ref $proto || $proto;

	my $self = bless {}, $class;
	$DEBUG = $args{DEBUG} || $ENV{MYDEBUG};
	
	$self->{test} = "output.test";
	$self->{model} = "model";

	# overrides
	while ( my($k, $v) = each %args )
	{
		$self->{$k} = $v if(defined $v);
	}


	if(-f $self->{test})
	{
		# read in our features from the training data set
		my $fea = new Clair::Features(DEBUG => $DEBUG);
		$self->{test_data} = $fea->input($self->{test});
  }

	if( -f $self->{model})
	{
		# read in the model file
		$self->{learn} = new Clair::Learn(DEBUG => $DEBUG);
		$self->{model_data} = $self->{learn}->read_model($self->{model});
	}

  return $self;
}



=head2 classify 
 
 A wrapper function for the underlying algorithms.

=cut

sub classify
{
	my ($self, $algo) = @_;

	$algo = "_classify_perceptron" unless($algo);

	$self->debugmsg("running \$self->$algo()", 2);

	return $self->$algo();
}



=head2 _classify_perceptron

 Classifies and then returns the results.
 Simple algorithm:

  if wt.f(x) > 0: classify x as positive group (+1)
  if wt.f(x) < 0: classify x as negative group (-1) 

=cut

sub _classify_perceptron
{
	my ($self) = @_;

	my $w0 = $self->{model_data}->{intercept};
	delete $self->{model_data}->{intercept};
	my $w = $self->{model_data};

	my $correct_count = 0;
	my $total_count = scalar @{ $self->{test_data} };

	my @return;
	for my $d (@{ $self->{test_data} })
	{

		my $y = $d->{class};
		my $x = $d->{features};
		my $com = $d->{comment};
		my ($junk, $grp, $path) = split /\s+/, $com;
		$com = "$grp:$path";
		# dot_product function is in Learn.pm - and we don't want to reimplement the sub.
		my $sum = $self->{learn}->dot_product($x, $w);
		my $s = $sum + $w0;
		# my $s = $sum;

		# $self->debugmsg("what??? \$s = $s",0) if($s == 0);

		my $I = ($s > 0) ? "+1" : "-1";
		my $yesno = ($I == $y) ? "y" : "n";
		$correct_count++ if($yesno eq "y");

		my @items = ($com, $s, $I, $y, $yesno);
		push @return, \@items;
	}

	return (\@return, $correct_count, $total_count);
}


1;
