/*	$NetBSD$	*/

/*
 * Copyright (c) 1997 Michael L. Hitch
 * Copyright (c) 1995 Daniel Widenfalk
 * Copyright (c) 1994 Christian E. Hopps
 * Copyright (c) 1982, 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Daniel Widenfalk
 *	and Michael L. Hitch.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1998 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This software contains code contributed to The NetBSD Foundation
 * by Mark Brinicombe.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the NetBSD
 *	Foundation, Inc. and its contributors.
 * 4. Neither the name of The NetBSD Foundation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SCSI driver for the Cumana SCSI II interface
 *
 *	from: amiga/dev/flsc.c,v 1.21 1998/07/04 22:18:16
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/device.h>
#include <sys/buf.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/queue.h>

#include <dev/scsipi/scsi_all.h>
#include <dev/scsipi/scsipi_all.h>
#include <dev/scsipi/scsiconf.h>
#include <dev/scsipi/scsi_message.h>

#include <machine/bootconfig.h>
#include <machine/cpu.h>
#include <machine/param.h>
#include <machine/io.h>

#include <dev/ic/ncr53c9xreg.h>
/*#include <dev/ic/ncr53c9xvar.h>*/
#include <arm32/dev/ncr53c9xvar.h>

#include <arm32/podulebus/podulebus.h>
#include <arm32/podulebus/podules.h>
#include <arm32/podulebus/cscvar.h>

void	cscattach	__P((struct device *, struct device *, void *));
int	cscmatch	__P((struct device *, struct cfdata *, void *));
int	csc_scsi_cmd	__P((struct scsipi_xfer *xs));

/* Linkup to the rest of the kernel */
struct cfattach csc_ca = {
	sizeof(struct csc_softc), cscmatch, cscattach
};

struct scsipi_adapter csc_switch = {
#ifdef CSC_DEBUG
	csc_scsi_cmd,
#else
	ncr53c9x_scsi_cmd,
#endif
	minphys,		/* no max at this level; handled by DMA code */
	NULL,
	NULL,
};

struct scsipi_device csc_dev = {
	NULL,			/* Use default error handler */
	NULL,			/* have a queue, served by this */
	NULL,			/* have no async handler */
	NULL,			/* Use default 'done' routine */
};

/*
 * Functions and the switch for the MI code.
 */
u_char	csc_read_reg __P((struct ncr53c9x_softc *, int));
void	csc_write_reg __P((struct ncr53c9x_softc *, int, u_char));
int	csc_dma_isintr __P((struct ncr53c9x_softc *));
void	csc_dma_reset __P((struct ncr53c9x_softc *));
int	csc_dma_intr __P((struct ncr53c9x_softc *));
int	csc_dma_setup __P((struct ncr53c9x_softc *, caddr_t *,
	    size_t *, int, size_t *));
void	csc_dma_go __P((struct ncr53c9x_softc *));
void	csc_dma_stop __P((struct ncr53c9x_softc *));
int	csc_dma_isactive __P((struct ncr53c9x_softc *));
void	csc_clear_latched_intr __P((struct ncr53c9x_softc *));
int	csc_intr __P((void *arg));

struct ncr53c9x_glue csc_glue = {
	csc_read_reg,
	csc_write_reg,
	csc_dma_isintr,
	csc_dma_reset,
	csc_dma_intr,
	csc_dma_setup,
	csc_dma_go,
	csc_dma_stop,
	csc_dma_isactive,
	NULL,
};

extern int ncr53c9x_debug;

/*
 * if we are a Cumana SCSI-2 card
 */
int
cscmatch(parent, cf, aux)
	struct device	*parent;
	struct cfdata	*cf;
	void		*aux;
{
	struct podule_attach_args *pa = (struct podule_attach_args *)aux;

	/* Look for the card */
	if (matchpodule(pa, MANUFACTURER_CUMANA, PODULE_CUMANA_SCSI2, -1) == 0)
		return(0);

	return(1);
}

/*
 * Attach this instance, and then all the sub-devices
 */
void
cscattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	struct podule_attach_args *pa = (struct podule_attach_args *)aux;
	struct csc_softc *csc = (void *)self;
	struct ncr53c9x_softc *sc = &csc->sc_ncr53c9x;
	volatile u_char *iobase;

	if (pa->pa_podule_number == -1)
		panic("Podule has disappeared !");

	/*
	 * Set up the glue for MI code early; we use some of it here.
	 */
	sc->sc_glue = &csc_glue;

	/*
	 * Set up the register pointers
	 */
	iobase = (volatile u_char *)pa->pa_podule->mod_base;
	csc->sc_reg = iobase + CSC_FASOFFSET_BASE;
	csc->sc_alatch = iobase + CSC_ALATCH;
	csc->sc_status = iobase + CSC_STATUS0;
	csc->sc_dack = (volatile u_short *)(iobase + CSC_DACK);

	sc->sc_freq = 40;		/* Clocked at 40Mhz */

	sc->sc_id = 7;
	/* Provide an override for the host id */
	(void)get_bootconf_option(boot_args, "csc.hostid",
	    BOOTOPT_TYPE_INT, &sc->sc_id);

	/*
	 * It is necessary to try to load the 2nd config register here,
	 * to find out what rev the flsc chip is, else the flsc_reset
	 * will not set up the defaults correctly.
	 */
	sc->sc_cfg1 = sc->sc_id | NCRCFG1_PARENB;
	sc->sc_cfg2 = NCRCFG2_SCSI2 | NCRCFG2_FE;
	sc->sc_cfg3 = 0x08 /*FCLK*/ | NCRESPCFG3_FSCSI | NCRESPCFG3_CDB;
	sc->sc_rev = NCR_VARIANT_FAS216;

	/*
	 * This is the value used to start sync negotiations
	 * Note that the NCR register "SYNCTP" is programmed
	 * in "clocks per byte", and has a minimum value of 4.
	 * The SCSI period used in negotiation is one-fourth
	 * of the time (in nanoseconds) needed to transfer one byte.
	 * Since the chip's clock is given in MHz, we have the following
	 * formula: 4 * period = (1000 / freq) * 4
	 */
/*	sc->sc_minsync = 1000 / sc->sc_freq;*/
	sc->sc_minsync = 0;

	/* Really no limit, but since we want to fit into the TCR... */
	sc->sc_maxxfer = 64 * 1024;

/*	ncr53c9x_debug |= 0xffff;*/

#ifndef CSC_POLL
	/*
	 * Configure interrupts.
	 */
#ifdef CSC_DEBUG
	csc->sc_ih.ih_func = (void *)csc_intr;
#else
	csc->sc_ih.ih_func = (void *)ncr53c9x_intr;
#endif
	csc->sc_ih.ih_arg  = sc;
	csc->sc_ih.ih_level = IPL_BIO;
	csc->sc_ih.ih_name = "csc";
	csc->sc_ih.ih_maskaddr = pa->pa_podule->irq_addr;
	csc->sc_ih.ih_maskbits = pa->pa_podule->irq_mask;

	if (irq_claim(IRQ_PODULE, &csc->sc_ih))
		panic("%s: Cannot install IRQ handler\n", self->dv_xname);

	/* Enable interrupts */
	*csc->sc_alatch = CSC_INTERRUPTS_ON;
	sc->sc_features = 0;

#else
	/* Disable interrupts and flag no-interrupt mode */
	*csc->sc_alatch = CSC_INTERRUPTS_OFF;
	sc->sc_features = NCR_F_NOINTR;
	printf(" polling");
#endif

	/* Disable DMA capability */
	*csc->sc_alatch = (CSC_ALATCH_DMAEN | 0);

	/*
	 * Now try to attach all the sub-devices
	 */
	ncr53c9x_attach(sc, &csc_switch, &csc_dev);
}

/*
 * Glue functions.
 */

u_char
csc_read_reg(sc, reg)
	struct ncr53c9x_softc *sc;
	int reg;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

	return csc->sc_reg[reg * CSC_FAS_REG_SPACING];
}

void
csc_write_reg(sc, reg, val)
	struct ncr53c9x_softc *sc;
	int reg;
	u_char val;
{
	struct csc_softc *csc = (struct csc_softc *)sc;
	struct ncr53c9x_tinfo *ti;
	u_char v = val;

	if (csc->sc_piomode && reg == NCR_CMD &&
	    v == (NCRCMD_TRANS|NCRCMD_DMA)) {
		v = NCRCMD_TRANS;
	}
	/*
	 * Can't do synchronous transfers in SCSI_POLL mode:
	 * If starting SCSI_POLL command, clear defer sync negotiation
	 * by clearing the T_NEGOTIATE flag.  If starting SCSI_POLL and
	 * the device is currently running synchronous, force another
	 * T_NEGOTIATE with 0 offset.
	 */
	if (reg == NCR_SELID) {
		ti = &sc->sc_tinfo[
		    sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
		if (sc->sc_nexus->xs->flags & SCSI_POLL) {
			if (ti->flags & T_SYNCMODE) {
				ti->flags ^= T_SYNCMODE | T_NEGOTIATE;
			} else if (ti->flags & T_NEGOTIATE) {
				ti->flags ^= T_NEGOTIATE | T_SYNCHOFF;
				/* save T_NEGOTIATE in private flags? */
			}
		} else {
			/*
			 * If we haven't attempted sync negotiation yet,
			 * do it now.
			 */
			if ((ti->flags & (T_SYNCMODE | T_SYNCHOFF)) ==
			    T_SYNCHOFF &&
			    sc->sc_minsync != 0)	/* XXX */
				ti->flags ^= T_NEGOTIATE | T_SYNCHOFF;
		}
	}
	if (reg == NCR_CMD && v == NCRCMD_SETATN  &&
	    sc->sc_flags & NCR_SYNCHNEGO &&
	     sc->sc_nexus->xs->flags & SCSI_POLL) {
		ti = &sc->sc_tinfo[
		    sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
		ti->offset = 0;
	}
	csc->sc_reg[reg * CSC_FAS_REG_SPACING] = v;
}

int
csc_dma_isintr(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

/*	if (sc->sc_state == NCR_CONNECTED || sc->sc_state == NCR_SELECTING)
		*csc->sc_led = csc_LED_ON;
	else
		*csc->sc_led = csc_LED_OFF;
*/
	if (csc->sc_reg[NCR_STAT * CSC_FAS_REG_SPACING] & NCRSTAT_INT)
		return 1;

	return 0;
}

#if 0
void
csc_clear_latched_intr(sc)
	struct ncr53c9x_softc *sc;
{
/*	struct csc_softc *csc = (struct csc_softc *)sc;*/

	/* Interrupt cleared via ncr53c9x */
}
#endif

void
csc_dma_reset(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;
	struct ncr53c9x_tinfo *ti;

	if (sc->sc_nexus)
		ti = &sc->sc_tinfo[sc->sc_nexus->xs->sc_link->scsipi_scsi.target];
	else
		ti = &sc->sc_tinfo[1];	/* XXX */
	if (csc->sc_active) {
		printf("dmaaddr %p dmasize %d stat %x flags %x off %d per %d ff %x",
	            *csc->sc_dmaaddr, csc->sc_dmasize,
		    csc->sc_reg[NCR_STAT * CSC_FAS_REG_SPACING], ti->flags, ti->offset,
		    ti->period, csc->sc_reg[NCR_FFLAG * CSC_FAS_REG_SPACING]);
		printf(" intr %x\n", csc->sc_reg[NCR_INTR * CSC_FAS_REG_SPACING]);
	}
	csc->sc_active = 0;
	csc->sc_piomode = 0;
}

int
csc_dma_intr(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;
	u_char	*p;
	volatile u_char *cmdreg, *intrreg, *statreg, *fiforeg;
	u_int	cscphase, cscstat, cscintr;
	int	cnt;

	NCR_DMA(("csc_dma_intr: pio %d cnt %d int %x stat %x fifo %d ",
	    csc->sc_piomode, csc->sc_dmasize, sc->sc_espintr, sc->sc_espstat,
	    csc->sc_reg[NCR_FFLAG * CSC_FAS_REG_SPACING] & NCRFIFO_FF));
	if (csc->sc_active == 0) {
		printf("csc_intr--inactive DMA\n");
		return -1;
	}

	/* if DMA transfer, update sc_dmaaddr and sc_pdmalen, else PIO xfer */
	if (csc->sc_piomode == 0) {
		panic("csc_dma_intr");
	}

	if ((sc->sc_espintr & NCRINTR_BS) == 0) {
		csc->sc_active = 0;
		csc->sc_piomode = 0;
		NCR_DMA(("no NCRINTR_BS\n"));
		return 0;
	}

	cnt = csc->sc_dmasize;
#if 0
	if (cnt == 0) {
		printf("data interrupt, but no count left.");
	}
#endif

	p = *csc->sc_dmaaddr;
	cscphase = sc->sc_phase;
	cscstat = (u_int) sc->sc_espstat;
	cscintr = (u_int) sc->sc_espintr;
	cmdreg = csc->sc_reg + NCR_CMD * CSC_FAS_REG_SPACING;
	fiforeg = csc->sc_reg + NCR_FIFO * CSC_FAS_REG_SPACING;
	statreg = csc->sc_reg + NCR_STAT * CSC_FAS_REG_SPACING;
	intrreg = csc->sc_reg + NCR_INTR * CSC_FAS_REG_SPACING;
	NCR_DMA(("PIO %d datain %d phase %d stat %x intr %x\n",
	    cnt, csc->sc_datain, cscphase, cscstat, cscintr));
	do {
		if (csc->sc_datain) {
			*p++ = *fiforeg;
			cnt--;
			if (cscphase == DATA_IN_PHASE) {
				*cmdreg = NCRCMD_TRANS;
			} else {
				csc->sc_active = 0;
			}
	 	} else {
			NCR_DMA(("csc_dma_intr: PIO out- phase %d cnt %d active %d\n",
			    cscphase, cnt, csc->sc_active));
			if (   (cscphase == DATA_OUT_PHASE)
			    || (cscphase == MESSAGE_OUT_PHASE)) {
				int n;
				n = 16 - (csc->sc_reg[NCR_FFLAG * CSC_FAS_REG_SPACING] & NCRFIFO_FF);
				if (n > cnt)
					n = cnt;
				cnt -= n;
				while (n-- > 0)
					*fiforeg = *p++;
				*cmdreg = NCRCMD_TRANS;
			} else {
				csc->sc_active = 0;
			}
		}

		if (csc->sc_active && cnt) {
			while (!(*statreg & 0x80));
			cscstat = *statreg;
			cscintr = *intrreg;
			cscphase = (cscintr & NCRINTR_DIS)
				    ? /* Disconnected */ BUSFREE_PHASE
				    : cscstat & PHASE_MASK;
		}
	} while (cnt && csc->sc_active && (cscintr & NCRINTR_BS));
#if 1
	if (csc->sc_dmasize < 8 && cnt)
		printf("csc_dma_intr: short transfer: dmasize %d cnt %d\n",
		    csc->sc_dmasize, cnt);
#endif
	NCR_DMA(("csc_dma_intr: PIO transfer [%d], %d->%d phase %d stat %x intr %x\n",
	    *csc->sc_pdmalen, csc->sc_dmasize, cnt, cscphase, cscstat, cscintr));
	sc->sc_phase = cscphase;
	sc->sc_espstat = (u_char) cscstat;
	sc->sc_espintr = (u_char) cscintr;
	*csc->sc_dmaaddr = p;
	*csc->sc_pdmalen -= csc->sc_dmasize - cnt;
	csc->sc_dmasize = cnt;

	if (*csc->sc_pdmalen == 0) {
		sc->sc_espstat |= NCRSTAT_TC;
		csc->sc_piomode = 0;
	}
	return 0;
}

int
csc_dma_setup(sc, addr, len, datain, dmasize)
	struct ncr53c9x_softc *sc;
	caddr_t *addr;
	size_t *len;
	int datain;
	size_t *dmasize;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

	csc->sc_dmaaddr = addr;
	csc->sc_pdmalen = len;
	csc->sc_datain = datain;
	csc->sc_dmasize = *dmasize;

	csc->sc_piomode = 1;
	/* IF pio or POLL */
	if (csc->sc_piomode) {
		*dmasize = csc->sc_dmasize;
		NCR_DMA(("csc_dma_setup: PIO %p/%d [%d]\n", *addr,
		    csc->sc_dmasize, *len));
		if (datain == 0) {
			int n;
			n = csc->sc_dmasize;
			if (n > 16)
				n = 16;
			while (n-- > 0) {
				csc->sc_reg[NCR_FIFO * CSC_FAS_REG_SPACING] = **csc->sc_dmaaddr;
				(*csc->sc_pdmalen)--;
				(*csc->sc_dmaaddr)++;
				--csc->sc_dmasize;
			}
		}
		return 0;
	}
	/* ELSE DMA */
	panic("csc_dma_setup");
}

void
csc_dma_go(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

	NCR_DMA(("csc_dma_go: datain %d size %d\n", csc->sc_datain,
	    csc->sc_dmasize));
	if (sc->sc_nexus->xs->flags & SCSI_POLL) {
		csc->sc_active = 1;
		return;
	} else if (csc->sc_piomode == 0) {
		/* NO DMA */
		panic("csc_dma_go");
	} else
		csc->sc_active = 1;
}

void
csc_dma_stop(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

	csc->sc_piomode = 0;
}

int
csc_dma_isactive(sc)
	struct ncr53c9x_softc *sc;
{
	struct csc_softc *csc = (struct csc_softc *)sc;

	return csc->sc_active;
}

/* Only used for debugging */
#ifdef CSC_DEBUG
int
csc_scsi_cmd(xs)
	struct scsipi_xfer *xs;
{
	struct scsipi_link *sc_link = xs->sc_link;

	if (sc_link->scsipi_scsi.lun == 0)
	printf("id=%d lun=%d cmdlen=%d datalen=%d opcode=%02x flags=%08x status=%02x blk=%02x %02x timeo=%d\n",
	    sc_link->scsipi_scsi.target, sc_link->scsipi_scsi.lun, xs->cmdlen, xs->datalen, xs->cmd->opcode,
	    xs->flags, xs->status, xs->cmd->bytes[0], xs->cmd->bytes[1],
	    xs->timeout);

	return(ncr53c9x_scsi_cmd(xs));
}

/* Only used to cleanup trace output as interrupt is shared */
int
csc_intr(arg)
	void *arg;
{
	struct ncr53c9x_softc *sc = arg;

	if (csc_dma_isintr(sc))
		return(ncr53c9x_intr(sc));
	return(0);
}
#endif
