
/*
 * btree.c
 *
 *   July 10, 1994 -- Lawrence Kesteloot
 *
 *   Functions to handle the b*-trees of HFS.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>

#include "util.h"
#include "hfs.h"
#include "btree.h"

/*
 * keycmp()
 *
 *   Compares two keys and returns a number less than zero if
 *   key1 < key2, zero if key1 == key2, and a number greater than
 *   zero if key1 > key2.  The HFS documentation does not specify
 *   how keys are to be compared, so I just zero-extended them to
 *   256 bytes and did a memory compare on all 256 bytes.
 */

static int keycmp (uchar *key1, int len1, uchar *key2, int len2, int isext)
{
	uchar	buf1[256], buf2[256];
	int	i;


	if (isext) {
		/*
		 * For some reason, the first byte of the extents tree
		 * (0x00 for data fork and 0xff for resource fork)
		 * is treated as 0xff < 0x00.
		 */

		if (key1[0] != key2[0]) {
			if (key1[0]) {
				return -1;
			} else {
				return 1;
			}
		}
	}

	memset (buf1, '\0', 256);
	memcpy (buf1, key1, len1);
	memset (buf2, '\0', 256);
	memcpy (buf2, key2, len2);

	if (debug) {
		printf ("%d: ", len1);
		for (i = 0; i < 15; i++) {
			printf ("%02x ", buf1[i]);
		}
		printf ("\n");
		printf ("%d: ", len2);
		for (i = 0; i < 15; i++) {
			printf ("%02x ", buf2[i]);
		}
		printf ("\n");
	}

	return memcmp (buf1, buf2, 256);
}

/*
 * btreeGetLeaf()
 *
 *   Given a tree and a key, this function returns the block of the
 *   leaf in which the key resides, or -1 if it didn't find it.
 *   Maybe I should just remove the "root" parameter since it's
 *   always 0.
 */

int btreeGetLeaf (int f, int tree, int root, void *key, int keylen, int isext)
{
	uchar	buf[BLKSIZ];
	int	numrecs, i, offset;

	while (1) {
		dprintf ("root = 0x%x\n", root);
		readshortblock (f, (tree + root) * BLKSIZ, buf);

		if (buf[8] == 0xff) {
			return root;
		}
		if (buf[8] == 0x01) {
			/* First block of tree -- points to root */
			offset = *((short *)&buf[BLKSIZ] - 1);
			root = *(long *)&buf[offset + 2];
			continue;
		}
		if (buf[8] != 0x00) {
			printf ("btreeGetLeaf: don't know how to handle "
				"tree node with type %d.\n", (int)buf[9]);
			return -1;
		}
		numrecs = *(short *)(buf + 10);
		/* Want to search backwards through the fields: */
		for (i = numrecs - 1; i >= 0; i--) {
			offset = *((short *)&buf[BLKSIZ] - i - 1);
			if (keycmp (&buf[offset + 1], buf[offset], key,
				keylen, isext) <= 0) {
				dprintf ("Found at rec %d\n", i);
				break;
			}
		}
		if (i < 0) {
			printf ("btreeGetLeaf: didn't find key in node 0x%x.\n",
				root);
			return -1;
		}
		root = *(long *)&buf[offset + buf[offset] + 1];
	}
}

/*
 * btreeGetRecordNum()
 *
 *   Given a block of a tree and a key, this function returns the record
 *   in the block which corresponds to the key, or -1 if it didn't find
 *   it.
 */

int btreeGetRecordNum (int f, int tree, int blk, void *key, int keylen,
	int isext)
{
	static uchar	buf[BLKSIZ];
	int		numrecs, i, offset;

	readshortblock (f, (tree + blk) * BLKSIZ, buf);
	numrecs = *(short *)(buf + 10);
	for (i = numrecs - 1; i >= 0; i--) {
		offset = *((short *)&buf[BLKSIZ] - i - 1);
		if (keycmp (&buf[offset + 1], buf[offset], key, keylen, isext)
			<= 0) {
			return i;
		}
	}

	printf ("btreeGetRecordNum: not found\n");
	return -1;
}

/*
 * btreeGetRecord()
 *
 *   Given a block in a tree and a record into it, returns the key that
 *   corresponds to that record including the length (if key != NULL)
 *   and the data for that record (if info != NULL).  If the record is
 *   past the end of the block, then the next block in the list is found
 *   and *blk and *rec are updated.  Returns 0 on failure, 1 on success.
 */

int btreeGetRecord (int f, int tree, ulong *blk, ulong *rec, void *info,
	void *key)
{
	static uchar	buf[BLKSIZ];
	int		numrecs, offset, len, keylen;

	while (1) {
		dprintf ("btreeGetRecord: 0x%x, %d\n", *blk, *rec);
		readshortblock (f, (tree + *blk) * BLKSIZ, buf);
		numrecs = *(short *)(buf + 10);
		if (*rec >= numrecs) {
			*blk = *(ulong *)&buf[0];
			*rec = 0;
			if (*blk == 0) {
				return 0;
			}
		} else {
			break;
		}
	}
	offset = *((short *)&buf[BLKSIZ] - *rec - 1);
	if (info != NULL) {
		keylen = (buf[offset] + 2) & ~1;
		len = *((short *)&buf[BLKSIZ] - *rec - 2) - offset - keylen;
		memcpy (info, &buf[offset + keylen], len);
	}
	if (key != NULL) {
		memcpy (key, &buf[offset], buf[offset] + 1);
	}

	return 1;
}
