/* session.c - Guile interface to Gnome session management code.  */

#include <assert.h>
#include <libguile.h>
#include "gnome.h"
#include "libgnomeui/gnome-session.h"

/* Some global symbols.  */
static SCM sym_none;
static SCM sym_errors;
static SCM sym_any;
static SCM sym_global;
static SCM sym_local;
static SCM sym_both;
static SCM sym_normal;
static SCM sym_error;
static SCM sym_ifrunning;
static SCM sym_anyway;
static SCM sym_immediately;
static SCM sym_never;

/* Protection code taken from guile-gtk.c.  */

static SCM protects;

static SCM
protect (SCM obj)
{
  SCM_SETCAR (protects, scm_cons (obj, SCM_CAR (protects)));
  return obj;
}

static void
unprotect (SCM obj)
{
  SCM walk;
  SCM *prev;

  for (prev = SCM_CARLOC (protects), walk = SCM_CAR (protects);
       SCM_NIMP (walk) && SCM_CONSP (walk);
       walk = SCM_CDR (walk))
    {
      if (SCM_CAR (walk) == obj)
	{
	  *prev = SCM_CDR (walk);
	  break;
	}
      else
	prev = SCM_CDRLOC (walk);
    }
}



/* Invoked as "save yourself" callback.  */
static int
save_yourself (gpointer client_data, GnomeSaveStyle save_style,
	       int is_shutdown, GnomeInteractStyle interact_style,
	       int is_fast)
{
  SCM args, proc, t, result;

  proc = (SCM) client_data;
  args = SCM_EOL;
  args = scm_cons (is_fast ? SCM_BOOL_T : SCM_BOOL_F, args);
  switch (interact_style)
    {
    GNOME_INTERACT_NONE:
      t = sym_none;
      break;
    GNOME_INTERACT_ERRORS:
      t = sym_errors;
      break;
    GNOME_INTERACT_ANY:
      t = sym_any;
      break;
    default:
      /* FIXME: report this.  */
      t = sym_none;
      break;
    }
  args = scm_cons (t, args);
  args = scm_cons (is_shutdown ? SCM_BOOL_T : SCM_BOOL_F, args);
  switch (save_style)
    {
    GNOME_SAVE_GLOBAL:
      t = sym_global;
      break;
    GNOME_SAVE_LOCAL:
      t = sym_local;
      break;
    GNOME_SAVE_BOTH:
      t = sym_both;
      break;
    default:
      /* FIXME: report this.  */
      t = sym_local;
      break;
    }
  args = scm_cons (t, args);

  result = scm_apply (proc, args, SCM_EOL);
  return (result == SCM_BOOL_F) ? 0 : 1;
}

/* Invoked as death callback.  */
static void
death_func (gpointer client_data)
{
  SCM proc = (SCM) client_data;
  scm_apply (proc, SCM_EOL, SCM_EOL);
}

SCM_PROC (s_gnome_init_session, "gnome-session-init", 2, 3, 0, init);
static SCM
init (SCM saver, SCM death, SCM previous)
{
  char *prev = NULL;
  char *ret;

  SCM_ASSERT (scm_procedure_p (saver) == SCM_BOOL_T, saver, SCM_ARG1,
	      s_gnome_init_session);
  SCM_ASSERT (scm_procedure_p (death) == SCM_BOOL_T, death, SCM_ARG2,
	      s_gnome_init_session);
  if (previous != SCM_UNDEFINED && previous != SCM_BOOL_F)
    {
      SCM_ASSERT (SCM_NIMP (previous) && SCM_STRINGP (previous),
		  previous, SCM_ARG3, s_gnome_init_session);
      SCM_COERCE_SUBSTR (previous);
      prev = SCM_CHARS (previous);
    }

  protect (saver);
  protect (death);
  ret = gnome_session_init (save_yourself, (gpointer) saver,
			    death_func, (gpointer) death,
			    prev);

  /* FIXME: memory leak.  */
  return scm_makfrom0str (ret);
}

SCM_PROC (s_gnome_set_dir, "gnome-session-set-current-directory", 1, 1, 0, setdir);
static SCM
setdir (SCM dir)
{
  SCM_ASSERT (SCM_NIMP (dir) && SCM_STRINGP (dir),
	      dir, SCM_ARG1, s_gnome_set_dir);
  SCM_COERCE_SUBSTR (dir);
  gnome_session_set_current_directory (SCM_CHARS (dir));
  return SCM_UNDEFINED;
}

typedef void help_func (int argc, char *argv[]);

/* Helper function.  */
static void
set_property_from_arglist (SCM list, char *name, help_func *func)
{
  char **argv;
  int argc, i;
  SCM l, s;

  argc = 0;
  l = list;
  while (SCM_NIMP (l))
    {
      SCM_ASSERT (SCM_CONSP (l), l, SCM_ARGn, name);
      s = SCM_CAR (l);
      SCM_ASSERT (SCM_NIMP (s) && SCM_STRINGP (s), s, SCM_ARGn, name);
      SCM_COERCE_SUBSTR (s);
      ++argc;
      l = SCM_CDR (l);
    }
  SCM_ASSERT (SCM_NULLP (l), list, SCM_ARGn, name);

  SCM_DEFER_INTS;
  argv = (char **) malloc (argc * sizeof (char *));
  for (i = 0, l = list; SCM_NIMP (l); l = SCM_CDR (l), ++i)
    argv[i] = SCM_CHARS (SCM_CAR (l));
  (*func) (argc, argv);
  free (argv);
  SCM_ALLOW_INTS;
}

SCM_PROC (s_gnome_set_discard, "gnome-session-set-discard-command", 0, 0, 1, set_discard);
static SCM
set_discard (SCM list)
{
  set_property_from_arglist (list, s_gnome_set_discard,
			     gnome_session_set_discard_command);
  return SCM_UNDEFINED;
}

SCM_PROC (s_gnome_set_restart, "gnome-session-set-restart-command", 0, 0, 1, set_restart);
static SCM
set_restart (SCM list)
{
  set_property_from_arglist (list, s_gnome_set_restart,
			     gnome_session_set_restart_command);
  return SCM_UNDEFINED;
}

SCM_PROC (s_gnome_set_clone, "gnome-session-set-clone-command", 0, 0, 1, set_clone);
static SCM
set_clone (SCM list)
{
  set_property_from_arglist (list, s_gnome_set_clone,
			     gnome_session_set_clone_command);
  return SCM_UNDEFINED;
}

SCM_PROC (s_gnome_set_init, "gnome-session-set-initialization-command", 0, 0, 1, set_init);
static SCM
set_init (SCM list)
{
  set_property_from_arglist (list, s_gnome_set_init,
			     gnome_session_set_initialization_command);
  return SCM_UNDEFINED;
}

SCM_PROC (s_gnome_set_prog, "gnome-session-set-program", 1, 1, 0, set_program);
static SCM
set_program (SCM prog)
{
  SCM_ASSERT (SCM_NIMP (prog) && SCM_STRINGP (prog),
	      prog, SCM_ARG1, s_gnome_set_prog);
  SCM_COERCE_SUBSTR (prog);
  gnome_session_set_program (SCM_CHARS (prog));
  return SCM_UNDEFINED;
}

SCM_PROC (s_gnome_set_restart_style, "gnome-session-set-restart-style", 1, 0, 0, set_rs_style);
static SCM
set_rs_style (SCM style)
{
  /* FIXME: error if no match.  */
  GnomeRestartStyle s = GNOME_RESTART_IF_RUNNING;

  if (style == sym_ifrunning)
    s = GNOME_RESTART_IF_RUNNING;
  else if (style == sym_anyway)
    s = GNOME_RESTART_ANYWAY;
  else if (style == sym_immediately)
    s = GNOME_RESTART_IMMEDIATELY;
  else if (style == sym_never)
    s = GNOME_RESTART_NEVER;

  gnome_session_set_restart_style (s);
  return SCM_UNDEFINED;
}


/* FIXME: consider rewriting interaction support so there is a single
   function that takes a callback proc as an argument.  */

SCM_PROC (s_gnome_request_interaction, "gnome-session-request-interaction", 1, 0, 0, request_interact);
static SCM
request_interact (SCM dialog_type)
{
  int result;
  GnomeDialogType type;

  if (dialog_type == sym_error)
    type = GNOME_DIALOG_ERROR;
  else /* FIXME: should error if no sym_normal.  */
    type = GNOME_DIALOG_NORMAL;
  result = gnome_session_request_interaction (type);
  return result ? SCM_BOOL_T : SCM_BOOL_F;
}

SCM_PROC (s_gnome_interaction_done, "gnome-session-interaction-done", 1, 0, 0, interact_done);
static SCM
interact_done (SCM shutdown)
{
  gnome_session_interaction_done (shutdown == SCM_BOOL_F ? 0 : 1);
  return SCM_UNDEFINED;
}


void
gnome_guile_session_init (void)
{
  protects = scm_permanent_object (scm_cons (SCM_EOL, SCM_BOOL_F));
  sym_none = scm_sysintern0 ("none");
  sym_errors = scm_sysintern0 ("errors");
  sym_any = scm_sysintern0 ("any");
  sym_global = scm_sysintern0 ("global");
  sym_local = scm_sysintern0 ("local");
  sym_both = scm_sysintern0 ("both");
  sym_normal = scm_sysintern0 ("normal");
  sym_error = scm_sysintern0 ("error");
  sym_ifrunning = scm_sysintern0 ("ifrunning");
  sym_anyway = scm_sysintern0 ("anyway");
  sym_immediately = scm_sysintern0 ("immediately");
  sym_never = scm_sysintern0 ("never");

#include "session.x"
}
