/* GnomENIUS Calculator
 * Copyright (C) 1997 George Lebl.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gmp.h>
#include "mymath.h"

/*get sin*/
void
mympf_sin(mpf_t rop, mpf_t op)
{
	mpf_t ftmp;
	mpf_t fres;
	mpf_t foldres;
	mpz_t ir;
	unsigned int i;
	int negate=TRUE;

	/*special case*/
	if(mpf_cmp_ui(op,0)==0) {
		mpf_set_ui(rop,0);
		return;
	}

	mpf_init(ftmp);
	mpf_init(fres);
	mpf_init_set(foldres,op);
	mpz_init(ir);

	for(i=3;;i+=2) {
		mpz_fac_ui(ir,i);
		mpf_set_z(ftmp,ir);
		mympf_pow_ui(fres,op,i);
		mpf_div(fres,fres,ftmp);
		if(negate)
			mpf_neg(fres,fres);
		negate= !negate;
		mpf_add(fres,fres,foldres);

		if(mpf_cmp(foldres,fres)==0)
			break;
		mpf_set(foldres,fres);
	}
	
	mpz_clear(ir);
	mpf_clear(ftmp);
	mpf_clear(foldres);

	mpf_set(rop,fres);

	mpf_clear(fres);
}

/*get cos*/
void
mympf_cos(mpf_t rop, mpf_t op)
{
	mpf_t ftmp;
	mpf_t fres;
	mpf_t foldres;
	mpz_t ir;
	unsigned int i;
	int negate=TRUE;

	/*special case*/
	if(mpf_cmp_ui(op,0)==0) {
		mpf_set_ui(rop,1);
		return;
	}

	mpf_init(ftmp);
	mpf_init(fres);
	mpf_init(foldres);
	mpf_set_ui(foldres,1);
	mpz_init(ir);

	for(i=2;;i+=2) {
		mpz_fac_ui(ir,i);
		mpf_set_z(ftmp,ir);
		mympf_pow_ui(fres,op,i);
		mpf_div(fres,fres,ftmp);
		if(negate)
			mpf_neg(fres,fres);
		negate= !negate;
		mpf_add(fres,fres,foldres);

		if(mpf_cmp(foldres,fres)==0)
			break;
		mpf_set(foldres,fres);
	}
	
	mpz_clear(ir);
	mpf_clear(ftmp);
	mpf_clear(foldres);

	mpf_set(rop,fres);

	mpf_clear(fres);
}

/*get e*/
void
mympf_gete(mpf_t rop)
{
	static mpf_t cache;
	static int iscached=FALSE;

	mpf_t fres;
	mpf_t foldres;
	mpz_t ir;
	unsigned int i;

	if(iscached)
		mpf_set(rop,cache);

	/*taylor series for x=1*/
	mpf_init(fres);
	mpf_init(foldres);
	mpf_set_ui(foldres,2);
	mpz_init(ir);

	for(i=2;;i++) {
		mpz_fac_ui(ir,i);
		mpf_set_z(fres,ir);
		mpf_ui_div(fres,1,fres);
		mpf_add(fres,fres,foldres);

		if(mpf_cmp(foldres,fres)==0)
			break;
		mpf_set(foldres,fres);
	}
	
	mpz_clear(ir);
	mpf_clear(foldres);

	iscached=TRUE;
	mpf_init_set(cache,fres);
	mpf_set(rop,fres);

	mpf_clear(fres);
}

/*get the value for pi*/
void
mympf_getpi(mpf_t rop)
{
	static mpf_t cache;
	static int iscached=FALSE;

	mpf_t fr;
	mpf_t fr2;
	mpf_t frt;

	if(iscached)
		mpf_set(rop,cache);

	/*
	 * Newton's method: Xn+1 = Xn - f(Xn)/f'(Xn)
	 */
	
	mpf_init(fr);
	mpf_init(fr2);
	mpf_init(frt);
	mpf_set_d(fr,3.14159265358979323846); /*use quite a precise guess
						as theinitial one*/
	for(;;) {
		mympf_sin(fr2,fr);

		mympf_cos(frt,fr);
		mpf_div(fr2,fr2,frt);
		mpf_neg(fr2,fr2);
		mpf_add(fr2,fr2,fr);
		
		if(mpf_cmp(fr2,fr)==0)
			break;
		mpf_set(fr,fr2);
	}
	mpf_clear(fr2);
	mpf_clear(frt);

	iscached=TRUE;
	mpf_init_set(cache,fr);
	mpf_set(rop,fr);
}

/*my own power function for floats, very simple :) */
void
mympf_pow_ui(mpf_t rop,mpf_t op,unsigned long e)
{
	mpf_t fr;

	if(rop==op) {
		mpf_init(fr);
		mpf_set_ui(fr,1);
		for(;e>0;e--)
			mpf_mul(fr,fr,op);
		mpf_set(rop,fr);
		mpf_clear(fr);
	} else { /*different numbers safe to do a quicker way*/
		mpf_set_ui(rop,1);
		for(;e>0;e--)
			mpf_mul(rop,rop,op);
	}
}

