/*
 * GnomeTalk: Protocol initialization and transactions module
 * (C) 1997 the Free Software Foundation
 *
 * Authors:
 *          Miguel de Icaza (miguel@gnu.org)
 *          Federico Mena   (quartic@gimp.org)
 */

#include "config.h"

/*
 * libc5 doesn't define struct osockaddr. But at least it _is_ the same
 * as struct sockaddr
 */

#include <sys/types.h>
#include <sys/socket.h>

#ifndef HAVE_STRUCT_OSOCKADDR
#define osockaddr sockaddr
#endif


#include <errno.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <protocols/talkd.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include "gnome.h"
#include "global.h"
#include "protocol.h"
#include "talk.h"

#define TALK_MAX_ERROR 9
static char *talk_results [TALK_MAX_ERROR];

static CTL_MSG control_msg;
static struct sockaddr_in control_addr;
static struct sockaddr_in daemon_addr;
static int control_socket;
static int daemon_port;

static int talk_machine_tag = -1;
static int retry_tag        = -1;
static int socket_tag       = -1;

int connected = 0;
int remote_id, local_id;

static enum {
	CHECK_INVITATION,
	ANNOUNCE_SENT,
	WAITING_FOR_CANCEL
} talk_state;

void
talk_report (char *msg)
{
	fprintf (stderr, "[%s]\n", msg);
}

void
talk_cancel_ctl_callback ()
{
	if (talk_machine_tag == -1)
		return;
	gdk_input_remove (talk_machine_tag);
	talk_machine_tag = -1;
}

void
talk_cancel_retry ()
{
	if (retry_tag == -1)
		return;
	gtk_timeout_remove (retry_tag);
	retry_tag = -1;
}

void
talk_abort (char *msg)
{
	if (connected)
	{
	}
	else {
		talk_cancel_ctl_callback ();
		talk_report (msg);
	}
	/* popup some error */
}

static char *
init_control_socket(void)
{
	size_t size;
	
	control_addr.sin_family = AF_INET;
	control_addr.sin_port   = 0;
	control_addr.sin_addr   = our_machine_addr;

	control_socket = socket(AF_INET, SOCK_DGRAM, 0);
	if (control_socket < 0) 
		return _("could not create socket");

	if (bind(control_socket, (struct sockaddr *) &control_addr, sizeof(control_addr)) != 0)
		return _("init_control_socket: could not bind()");

	size = sizeof(control_addr);

	if (getsockname(control_socket, (struct sockaddr *) &control_addr, &size) != 0)
		return _("getsockname() failed");
	return NULL;
}

int retry (gpointer data);

static char *
init_talk_socket(void)
{
	size_t size;
	
	talk_addr.sin_family = AF_INET;
	talk_addr.sin_port   = 0;
	talk_addr.sin_addr   = our_machine_addr;

	talk_socket = socket(AF_INET, SOCK_STREAM, 0);
	if (talk_socket < 0)
		return _("could not create socket");

	if (bind(talk_socket, (struct sockaddr *) &talk_addr, sizeof(talk_addr)) != 0)
		return _("could not bind ()");

	size = sizeof(talk_addr);

	if (getsockname(talk_socket, (struct sockaddr *) &talk_addr, &size) != 0)
		return _("getsockname() failed");
	return NULL;
}

static char *
transaction_send(struct in_addr dest, int type)
{
	int count;

	control_msg.type = type;

	daemon_addr.sin_addr = dest;
	daemon_addr.sin_port = daemon_port;

	if (retry_tag == -1)
		retry_tag = gtk_timeout_add (2 * 1000, retry, 0);

	for (;;){
		count = sendto(control_socket, (char *) &control_msg, sizeof(control_msg),
			       0, (struct sockaddr *) &daemon_addr, sizeof(daemon_addr));
		if (count != sizeof(control_msg)) {
			if (errno == EINTR)
				continue;
			return _("Error sending");
		}
		return NULL;
	}
}

static char *
transaction_get_reply (CTL_RESPONSE *response, int type)
{
	int count;

	talk_cancel_retry ();
	
	do {
		count = recv(control_socket, (char *) response, sizeof(*response), 0);
		if (count < 0) {
			if (errno == EINTR)
				continue;
			
			return _("Error when reading from talk daemon");
		}
	} while ((count == -1) ||
		 (response->vers != TALK_VERSION) ||
		 (response->type != type));
	
	response->id_num = ntohl(response->id_num);
	response->addr.sa_family = ntohs(response->addr.sa_family);
	return NULL;
}

int
retry (gpointer data)
{
	transaction_send (daemon_addr.sin_addr, control_msg.type);
	return 1;
}

static void
start_invite(void)
{
	if (listen (talk_socket, 5) != 0){
		talk_abort (_("Error trying to listen for caller"));
		return;
	}
	control_msg.id_num = htonl (-1);
	control_msg.addr = *(struct osockaddr *)&talk_addr;
	control_msg.addr.sa_family = htons (AF_INET);
	talk_report (_("Trying to connect to your party's talk daemon"));
	transaction_send (his_machine_addr, ANNOUNCE);
	talk_state = ANNOUNCE_SENT;
}

/*
 * This routine should actually have another state machine for not
 * blocking while receiving/sending the edit chars, but this is left
 * as an excercise to the reader of this code
 */
static void
set_edit_chars(void)
{
	char edit_chars [3];
	int  status;
	
	edit_chars [0] = TALK_ERASE;
	edit_chars [1] = TALK_KILL;
	edit_chars [2] = TALK_WIN_ERASE;
	status = write (talk_socket, edit_chars, 3);
	if (status != 3){
		talk_abort (_("Lost connection"));
		return;
	}
	status = read (talk_socket, &edit_chars, 3);
	if (status != 3){
		talk_abort (_("Lost connection"));
		return;
	}
	his_erase     = edit_chars [0];
	his_kill      = edit_chars [1];
	his_win_erase = edit_chars [2];
}

/* Called when we are ready for doing an accept on our talk socket */
void
socket_ready_for_accept (gpointer data, gint source, GdkInputCondition cond)
{
	int new_talk_socket;

	do {
		new_talk_socket = accept (talk_socket, 0, 0);
		if (new_talk_socket == -1 && errno != EINTR){
			talk_abort (_("Unable to connect with your party"));
			return;
		}
	} while (new_talk_socket == -1);
	close (talk_socket);
	talk_socket = new_talk_socket;
	talk_report (_("Waiting for your party to respond"));

	/* tell the daemons to delete the invitations */
	control_msg.id_num = htonl (local_id);
	transaction_send (our_machine_addr, DELETE);
	
	control_msg.id_num = htonl (remote_id);
	transaction_send (his_machine_addr, DELETE);

	set_edit_chars ();
	connected = 1;
	
	/* launch it! */
	create_talk_interface ();
	gdk_input_remove (socket_tag);
}

/*
 * Called when there is data on the control udp socket
 */
void
talk_state_machine (gpointer data, gint source, GdkInputCondition cond)
{
	CTL_RESPONSE response;
	int result;

	printf ("state: %d\n", talk_state);
	switch (talk_state){
	case CHECK_INVITATION:
		transaction_get_reply (&response, LOOK_UP);
		if (response.answer != SUCCESS){
			start_invite ();
			return;
		}
		control_msg.id_num = htonl (response.id_num);
		gdk_input_remove (talk_machine_tag);
		if (response.addr.sa_family != AF_INET){
			talk_abort (_("response uses an invalid network address"));
			return;
		}

		do {
			result = connect (talk_socket, (struct sockaddr *)&response.addr,
					  sizeof(response.addr));
			if (result == -1){
				if (errno == ECONNREFUSED){
					transaction_send (his_machine_addr, DELETE);
					close (talk_socket);
					init_talk_socket ();
					start_invite ();
				}
				if (errno == EINTR)
					continue;
			}
		} while (result == -1);

		/* Start talk */
		set_edit_chars ();
		connected = 1;
		create_talk_interface ();
		break;
		
	case ANNOUNCE_SENT:
		transaction_get_reply (&response, ANNOUNCE);
		remote_id = response.id_num;
		if (response.answer != SUCCESS){
			if (response.answer < TALK_MAX_ERROR)
				talk_abort (talk_results [response.answer-1]);
			return;
		}
		transaction_send (our_machine_addr, LEAVE_INVITE);
		/* we can do this, as this is the local machine */
		transaction_get_reply (&response, LEAVE_INVITE);
		local_id = response.id_num;
		talk_report (_("Waiting for your party to respond"));
		socket_tag = gdk_input_add (talk_socket, GDK_INPUT_READ, socket_ready_for_accept, 0);
		
		talk_state = WAITING_FOR_CANCEL;
		break;

	case WAITING_FOR_CANCEL:
		transaction_get_reply (&response, DELETE);
		printf (_("CANCEL CALLBACK:Answer:%d\n"), response.answer);
		gdk_input_remove (talk_machine_tag);
		talk_cancel_ctl_callback ();
		talk_cancel_retry ();
		break;
	}
}

char *
create_connection(char *me, char *he, char *tty)
{
	struct servent *server;
	char *err;
	
	/* Initialize talk_msg template */

	control_msg.pid = htonl(getpid());
	control_msg.vers = TALK_VERSION;
	control_msg.id_num = htonl(0);

	strncpy(control_msg.l_name, me, NAME_SIZE);
	control_msg.l_name[NAME_SIZE - 1] = 0;

	strncpy(control_msg.r_name, he, NAME_SIZE);
	control_msg.r_name[NAME_SIZE - 1] = 0;

	strncpy(control_msg.r_tty, tty, TTY_SIZE);
	control_msg.r_tty[TTY_SIZE - 1] = 0;

	if ((err = init_control_socket()) != NULL)
		return err;
	
	if ((err = init_talk_socket()) != NULL)
		return err;

	control_msg.ctl_addr = *(struct osockaddr *)&control_addr;
        control_msg.ctl_addr.sa_family = htons (AF_INET);
	control_msg.addr.sa_family = htons (AF_INET);
	
	/* Init daemon port */

	server = getservbyname("ntalk", "udp");

	/* we have a failback: it once happened to me, and I was pissed off */
	if (server)
		daemon_port = server->s_port;
	else
		daemon_port = 518;

	err = transaction_send (his_machine_addr, LOOK_UP);
	if (err)
		return err;

	talk_machine_tag = gdk_input_add (control_socket, GDK_INPUT_READ, talk_state_machine, 0);
	talk_state = CHECK_INVITATION;
	return NULL;
	
	/* Check invitation */

	if (!check_invitation())
		invite();

	set_edit_chars();

	create_talk_interface();
}

void
init_protocol (void)
{
	/* NOT_HERE */
	talk_results [0] = _("Your party is not logged on");

	/* FAILED */
        talk_results [1] = _("Target machine is too confused to talk to us");
	
	/* MACHINE_UNKNOWN */
        talk_results [2] = _("Target machine does not recognize us");
	
	/* PERMISSION_REFUSED */
        talk_results [3] = _("Your party is refusing messages");
	
	/* UNKNOWN_REQUEST */
        talk_results [4] = _("Target machine can not handle remote talk");
	
	/* BADVERSION */
        talk_results [5] = _("Target machine indicates protocol mismatch");
	
	/* BADADDR */
        talk_results [6] = _("Target machine indicates protocol botch (addr)");
	
	/* BADCTLADDR */
	talk_results [7] = _("Target machine indicates protocol botch (ctl_addr)");
}
