
/*  ----------------------------------------------------------------------

    Copyright (C) 1998  Cesar Miquel  (miquel@df.uba.ar)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    ---------------------------------------------------------------------- */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include "logview.h"
#include "logrtns.h"
#include "gtk/gtk.h"
#include "gdk/gdkkeysyms.h"

/*
 * -----------------
 * Local definitions
 * ------------------
 */

#define LOG_COL1                 7
#define LOG_COL2                 70
#define LOG_COL3                 170
#define LOG_COL4                 200
#define LOG_COL5                 300
#define LOG_COL6                 340
#define LOG_COL7                 400
#define LOG_TITLEY               10

#define MAX_NUM_LOGS             10
#define MAX_NUM_DPY_LINES        300
#define MAX_DPY_NUM_CHARS        200
#define DEF_NUM_LOGS             1

#define FRMT_HEADING             1
#define FRMT_NORMAL              2



/*
 * -------------------
 * Module variables 
 * -------------------
 */


GdkGC *gc;
GdkDrawable *canvas;

Log *loglist[MAX_NUM_LOGS];
Log *curlog;
int numlogs, curlognum;
int log_line_sep;
int open_log_visible;
char *deflognames[] =
{"/var/log/messages"};

char *month[12] =
{"January", "February", "March", "April", "May",
 "June", "July", "August", "September", "October",
 "November", "December"};

extern int loginfovisible, calendarvisible;
extern ConfigData *cfg;

/*
 * -------------------
 * Function prototypes
 * -------------------
 */

int InitPages ();
int HandleLogEvents ();
int GetLineAtCursor (int y);
int RepaintCalendar (GtkWidget * widget, GdkEventExpose * event);
int RepaintLogInfo (GtkWidget *, GdkEventExpose *);
void log_repaint (GtkWidget * cv, GdkRectangle * area);
void log_redrawcursor (int ol, int nl, Page * np);
void log_redrawdetail (Page * pg, int ln);
void ShowErrMessage (char *msg);
void DrawMonthHeader (LogLine * line, int y);
void DrawLogLine (LogLine *line, int y);
void DrawLogCursor (int y);
void EraseLogCursor (int y);
void Draw3DBox (GdkDrawable *win, GdkGC *gc, int x, int y, int w, int h, GdkColor color[3]);
void CloseApp ();
void InitCalendarData ();
Page *GetPageAtCursor (int y);


/* ----------------------------------------------------------------------
   NAME:        PointerMoved
   DESCRIPTION: The pointer moved inside the window.
   ---------------------------------------------------------------------- */

void
PointerMoved (GtkWidget * cv, GdkEventMotion * event)
{
   int cursory, nl;
   Page *np;

   /* Check that there is at least one log */
   if (curlog == NULL)
      return;

   cursory = event->y;
   if ((nl = GetLineAtCursor (cursory)) != -1 && nl != curlog->pointerln)
   {
      np = GetPageAtCursor (cursory);
      log_redrawdetail (np, nl);
      log_redrawcursor (curlog->pointerln, nl, np);
      curlog->pointerln = nl;
      curlog->pointerpg = np;
   }
}


/* ----------------------------------------------------------------------
   NAME:        NumTxtLines
   DESCRIPTION: Returns the number of log-lines to read to ocupy l
   screen lines.
   ---------------------------------------------------------------------- */

int
NumTextLines (int l)
{
   int cd, cm, ln, count;
   Page *pg;
   LogLine *line;

   cd = cm = -1;
   ln = curlog->firstline;
   pg = curlog->currentpg;
   count = 0;

   while (count < l)
   {
      line = &pg->line[ln];
      if ((cd != line->date && line->date >= 0) ||
	  (cm != line->month && line->month >= 0))
      {
	 count += 2;
	 cd = line->date;
	 cm = line->month;
      }
      ln++;
      count++;
   }

   return ((ln - curlog->firstline));
}

/* ----------------------------------------------------------------------
   NAME:        HandleLogKeyboard
   DESCRIPTION: Handle all posible keyboard actions.
   ---------------------------------------------------------------------- */

void
HandleLogKeyboard (GtkWidget * win, GdkEventKey * event_key)
{
   int ln, nl, y;
   guint key;
   Page *next;

   /* Check that there is at least one log */
   if (curlog == NULL)
      return;

   key = event_key->keyval;
   switch (key)
   {
   case GDK_R:
   case GDK_r:
      curlog->firstline = 0;
      log_repaint (win, NULL);
      break;
   case GDK_Q:
   case GDK_q:
      CloseApp ();
      break;

   case GDK_plus:
      if (numlogs == 1)
	 return;
      if (curlognum == numlogs - 1)
	 curlognum = 0;
      else
	 curlognum++;
      curlog = loglist[curlognum];
      log_repaint (win, NULL);
      if (loginfovisible)
	 RepaintLogInfo (NULL, NULL);
      if (calendarvisible)
      {
	 InitCalendarData ();
	 RepaintCalendar (NULL, NULL);
      }
      break;

   case GDK_minus:
      if (numlogs == 1)
	 return;
      if (curlognum == 0)
	 curlognum = numlogs - 1;
      else
	 curlognum--;
      curlog = loglist[curlognum];
      log_repaint (win, NULL);
      if (loginfovisible)
	 RepaintLogInfo (NULL, NULL);
      if (calendarvisible)
      {
	 InitCalendarData ();
	 RepaintCalendar (NULL, NULL);
      }
      break;

   case GDK_Page_Up:
   case GDK_KP_Page_Up:
   case GDK_R9:
   case GDK_Up:
      ln = curlog->firstline;
      ln -= (key == GDK_Up) ? 1 : NumTextLines ((LINES_P_PAGE >> 1));
      if (curlog->currentpg->isfirstpage == TRUE)
      {
	 if (ln < 0)
	    ln = 0;
	 curlog->firstline = ln < curlog->currentpg->fl ? curlog->currentpg->fl : ln;
	 log_repaint (win, NULL);
	 break;
      }
      if (ln < 0)
      {
	 ln += LINES_P_PAGE;
	 next = curlog->currentpg->prev;
	 if (next == curlog->firstpg)
	    ReadNPagesUp (curlog, next, (NUM_PAGES>>1));
	 if (curlog->currentpg->isfirstpage == FALSE)
	    curlog->currentpg = curlog->currentpg->prev;
      }
      if (curlog->currentpg->isfirstpage == TRUE)
	 curlog->firstline = MAX (ln, curlog->currentpg->fl);
      else
	 curlog->firstline = ln;
      if ((nl = GetLineAtCursor (y)) != -1)
      {
	 curlog->pointerln = nl;
	 curlog->pointerpg = GetPageAtCursor (y);
      }
      log_repaint (win, NULL);
      break;

   case GDK_Page_Down:
   case GDK_KP_Page_Down:
   case GDK_R15:
   case GDK_Down:
      if (curlog->currentpg->islastpage == TRUE)
	 break;
      ln = curlog->firstline;
      ln += (key == GDK_Down) ? 1 : NumTextLines ((LINES_P_PAGE >> 1));
      if (ln >= LINES_P_PAGE)
      {
	 ln -= LINES_P_PAGE;
	 next = curlog->currentpg->next;
	 if (next == curlog->lastpg)
	    ReadNPagesDown (curlog, next, (NUM_PAGES>>1));
	 curlog->currentpg = curlog->currentpg->next;
      }
      curlog->firstline = ln;
      if ((nl = GetLineAtCursor (y)) != -1)
      {
	 curlog->pointerln = nl;
	 curlog->pointerpg = GetPageAtCursor (y);
      }
      log_repaint (win, NULL);
      break;

   default:
      break;
   };

   return;
}

/* ----------------------------------------------------------------------
   NAME:          FileSelectCancel
   DESCRIPTION:   User selected a file.
   ---------------------------------------------------------------------- */

void
FileSelectCancel (GtkWidget * w, GtkFileSelection * fs)
{
   gtk_widget_destroy (GTK_WIDGET (fs));
   open_log_visible = FALSE;
}

/* ----------------------------------------------------------------------
   NAME:          FileSelectOk
   DESCRIPTION:   User selected a file.
   ---------------------------------------------------------------------- */

void
FileSelectOk (GtkWidget * w, GtkFileSelection * fs)
{
   char *f;
   Log *tl;

   /* Check that we haven't opened all logfiles allowed    */
   if (numlogs >= MAX_NUM_LOGS)
     {
       ShowErrMessage ("Too many open logs. Close one and try again");
       return;
     }

   f = gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs));
   gtk_widget_destroy (GTK_WIDGET (fs));
   open_log_visible = FALSE;
   if (f != NULL)
      if ((tl = OpenLogFile (f)) != NULL)
      {
	 curlog = tl;
	 loglist[numlogs] = tl;
	 numlogs++;
	 curlognum = numlogs - 1;
	 if (loginfovisible)
	    RepaintLogInfo (NULL, NULL);
	 if (calendarvisible)
	   {
	     InitCalendarData();
	     RepaintCalendar (NULL, NULL);
	   }
      }

}

/* ----------------------------------------------------------------------
   NAME:          LoadLogMenu
   DESCRIPTION:   Open a new log defined by the user.
   ---------------------------------------------------------------------- */

void
LoadLogMenu (GtkWidget * widget, gpointer user_data)
{
   GtkWidget *filesel = NULL;

   /*  Cannot open more than MAX_NUM_LOGS */
   if (numlogs == MAX_NUM_LOGS)
      return;
   
   /*  Cannot have more than one fileselect window */
   /*  at one time. */
   if (open_log_visible)
     return;


   filesel = gtk_file_selection_new ("Open new logfile");
   gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), PATH_MESSAGES);
   gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
   gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		       "clicked", (GtkSignalFunc) FileSelectOk,
		       filesel);
   gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
			      "clicked", (GtkSignalFunc) FileSelectCancel,
			      GTK_OBJECT (filesel));

   if (!GTK_WIDGET_VISIBLE (filesel))
      gtk_widget_show (filesel);
   else
      gtk_widget_destroy (filesel);

   open_log_visible = TRUE;
}

/* ----------------------------------------------------------------------
   NAME:        log_repaint
   DESCRIPTION: Redraw screen.
   ---------------------------------------------------------------------- */

void
log_repaint (GtkWidget * win, GdkRectangle * area)
{
   static int firsttime = TRUE;
   int ln, y, cm, cd, offset;
   Page *pg;
   LogLine *line;

   if (firsttime)
   {
      if (win == NULL)
	 return;
      firsttime = FALSE;
      canvas = win->window;
      gc = gdk_gc_new (canvas);
      log_line_sep = cfg->fixedb->ascent + 3;
   }
   offset = LOG_TITLEY;

   /* Erase page */
   gdk_window_clear (canvas);

   /* Check that there is at least one log */
   if (curlog == NULL)
      return;

   pg = curlog->currentpg;
   ln = curlog->firstline;
   cm = cd = -1;
   y = offset;

   log_redrawdetail (curlog->pointerpg, curlog->pointerln);

   while ((y - offset) < LOG_CANVAS_H)
   {
      line = &pg->line[ln];
      if ((line->month != cm && line->month > 0) ||
	  (line->date != cd && line->date > 0))
      {
	 DrawMonthHeader (line, y);
	 cm = line->month;
	 cd = line->date;
	 y += 2 * log_line_sep;
      }
      if (ln == curlog->pointerln && pg == curlog->pointerpg)
	 DrawLogCursor (y);

      DrawLogLine (line, y);
      y += log_line_sep;
      ln++;
      if (ln > pg->ll && pg->islastpage == TRUE)
	 break;
      if (ln >= LINES_P_PAGE)
      {
	 if (pg->islastpage == FALSE)
	    pg = pg->next;
	 else
	    break;
	 ln = 0;
      }
   }

}

/* ----------------------------------------------------------------------
   NAME:        log_redrawcursor
   DESCRIPTION: Redraw screen.
   ---------------------------------------------------------------------- */

void
log_redrawcursor (int ol, int nl, Page * np)
{
   int ln, y, cm, cd, offset;
   Page *pg, *op;
   LogLine *line;

   offset = LOG_TITLEY;

   pg = curlog->currentpg;
   op = curlog->pointerpg;
   ln = curlog->firstline;
   cm = cd = -1;
   y = offset;

   while ((y - offset) < LOG_CANVAS_H)
   {
      line = &pg->line[ln];
      if ((line->month != cm && line->month >= 0) ||
	  (line->date != cd && line->date >= 0))
      {
	 cm = line->month;
	 cd = line->date;
	 y += 2 * log_line_sep;
      }
      if ((ln == ol && pg == op) || (ln == nl && pg == np))
      {
	 if (ln == nl)
	    DrawLogCursor (y);
	 else
	    EraseLogCursor (y);

	 DrawLogLine (line, y);
      }
      y += log_line_sep;
      ln++;
      if (ln > pg->ll && pg->islastpage == TRUE)
	 break;
      if (ln >= LINES_P_PAGE)
      {
	 if (pg->islastpage == FALSE)
	    pg = pg->next;
	 else
	    break;
	 ln = 0;
      }
   }

}

/* ----------------------------------------------------------------------
   NAME:        DrawLogLine
   DESCRIPTION: Displays a single log line
   ---------------------------------------------------------------------- */

void
DrawLogLine (LogLine *line, int y)
{
   char tmp[90];

   gdk_gc_set_foreground (gc, &cfg->black);

   if (line->hour >= 0 && line->min >= 0 && line->sec >= 0)
      sprintf (tmp, "%02d:%02d:%02d", line->hour, line->min, line->sec);
   else
      sprintf (tmp, " ");
   gdk_draw_string (canvas, cfg->fixedb, gc, LOG_COL1, y, tmp);
   sprintf (tmp, "%s", line->process);
   if (strlen (tmp) > 13)
      tmp[13] = '\0';
   gdk_draw_string (canvas, cfg->fixed, gc, LOG_COL2, y, tmp);
   strncpy (tmp, line->message, 90);
   if (strlen (tmp) > 67)
      tmp[67] = '\0';
   gdk_draw_string (canvas, cfg->fixed, gc, LOG_COL3, y, tmp);
}


/* ----------------------------------------------------------------------
   NAME:        DrawMonthHeader
   DESCRIPTION: Draw the header for the current month.
   ---------------------------------------------------------------------- */

void
DrawMonthHeader (LogLine * line, int y)
{
   char buf[100];
   int  h, centery, skip;
   GdkColor color[] =
   {cfg->blue1, cfg->blue, cfg->blue3};

   h = cfg->headingb->ascent - cfg->headingb->descent;
   skip = (log_line_sep - cfg->fixed->ascent + cfg->fixed->descent);
   centery = y + log_line_sep - ((2 * log_line_sep - skip - h) >> 1);
   Draw3DBox (canvas, gc, 5, y - log_line_sep + skip , LOG_CANVAS_W - 10, 2*log_line_sep-skip, color);

   gdk_gc_set_foreground (gc, &cfg->black);
   sprintf (buf, "%s %d", month[(int) line->month], line->date);
   gdk_draw_string (canvas, cfg->headingb, gc, LOG_COL1 - 1, centery + 1, buf);
   gdk_gc_set_foreground (gc, &cfg->white);
   gdk_draw_string (canvas, cfg->headingb, gc, LOG_COL1, centery, buf);

}

/* ----------------------------------------------------------------------
   NAME:        DrawLogCursor
   DESCRIPTION: Draw the cursor under the log line.
   ---------------------------------------------------------------------- */

void
DrawLogCursor (int y)
{
   GdkColor color[] =
   {cfg->gray25, cfg->gray50, cfg->gray75};

   Draw3DBox (canvas, gc, 5, y - log_line_sep + 3, LOG_CANVAS_W - 10, log_line_sep, color);
}

/* ----------------------------------------------------------------------
   NAME:        Draw3DBox
   DESCRIPTION: Draw a "3d" box using the three colors povided. 
   color[0]      bottom and right colors
   color[1]      background of box
   color[2]      top and left colors
   ---------------------------------------------------------------------- */

void
Draw3DBox (GdkDrawable *win, GdkGC *gc, int x, int y, int w, int h, GdkColor color[3])
{
   gdk_gc_set_foreground (gc, &color[1]);
   gdk_draw_rectangle (win, gc, TRUE, x, y, w, h);
   h--;
   w--;
   gdk_gc_set_foreground (gc, &color[2]);
   gdk_draw_line (win, gc, x, y + h, x, y);
   gdk_draw_line (win, gc, x, y, x + w, y);
   gdk_gc_set_foreground (gc, &color[0]);
   gdk_draw_line (win, gc, x, y + h, x + w, y + h);
   gdk_draw_line (win, gc, x + w, y + h, x + w, y);
}


/* ----------------------------------------------------------------------
   NAME:                                EraseLogCursor
   DESCRIPTION: Erase the cursor under the log line.
   ---------------------------------------------------------------------- */

void
EraseLogCursor (int y)
{
  gdk_window_clear_area (canvas, 5, y - log_line_sep + 3, LOG_CANVAS_W - 10, log_line_sep);
}


/* ----------------------------------------------------------------------
   NAME:        log_redrawdetail
   DESCRIPTION: Redraw area were the detailed information is drawn.
   ---------------------------------------------------------------------- */

void
log_redrawdetail (Page * pg, int ln)
{
}

/* ----------------------------------------------------------------------
   NAME:        InitPages
   DESCRIPTION: Returns -1 if there was a error otherwise TRUE;
   ---------------------------------------------------------------------- */

int
InitPages ()
{
   int i;

   /* Open default logs.            */
   numlogs = 0;
   for (i = 0; i < DEF_NUM_LOGS; i++)
     {
       curlog = OpenLogFile (deflognames[i]);
       if (curlog == NULL)
	 continue;
       loglist[numlogs] = curlog;
       numlogs++;
     }

   if (numlogs == 0)
      return -1;

   curlognum = numlogs - 1;
   curlog = loglist[curlognum];

   return TRUE;
}



/* ----------------------------------------------------------------------
   NAME:          GetLineAtCursor
   DESCRIPTION:   Given the y-coordinate of the pointer inside the frame
   calculate the line number of the log line under it.
   It returns -1 if the cursor is not over a log line.
   ---------------------------------------------------------------------- */

int
GetLineAtCursor (int y)
{
   Page *pg;
   LogLine *line;
   int sln, ln, cm, cd, i;
   int maxnumlines;

   sln = (y + 13 - LOG_TITLEY) / log_line_sep - 2;
   maxnumlines = LOG_CANVAS_H / log_line_sep;

   /* Check that we are at least bellow the header, */
   if (sln < 0)
      return -1;		/* guess not. */

   ln = curlog->firstline;
   pg = curlog->currentpg;
   line = &pg->line[ln];
   cm = line->month;
   cd = line->date;
   i = 0;
   /* while (i != sln && i < LINES_P_PAGE + SPARE_LINES) */
   while (i != sln && i < maxnumlines)
   {
      i++;
      ln++;
      if (ln > pg->ll && pg->islastpage == TRUE)
	 return -1;
      if (ln >= LINES_P_PAGE)
      {
	 if (pg->islastpage == FALSE)
	    pg = pg->next;
	 else
	    return -1;
	 ln = 0;
      }
      line = &pg->line[ln];
      if ((line->month != cm && line->month >= 0) ||
	  (line->date != cd && line->date >= 0))
      {
	 i += 2;
	 if (sln < i && sln >= i - 2)
	 {
	    return -1;
	 }
	 cm = line->month;
	 cd = line->date;
      }
   }

   return (ln);

}

/* ----------------------------------------------------------------------
   NAME:          GetPageAtCursor
   DESCRIPTION:   Given the y-coordinate of the pointer inside the frame
   calculate the page of the log line under it.
   ---------------------------------------------------------------------- */
Page *
GetPageAtCursor (int y)
{
   LogLine *line;
   Page *pg;
   int sln, ln, cm, cd, i;
   int maxnumlines;

   sln = (y + 13 - LOG_TITLEY) / log_line_sep - 2;
   maxnumlines = LOG_CANVAS_H / log_line_sep;

   ln = curlog->firstline;
   pg = curlog->currentpg;
   line = &pg->line[ln];
   cm = line->month;
   cd = line->date;
   i = 0;

   /* Check that we are at least bellow the header, */
   if (sln < 0)
      return pg;		/* guess not. */

   /* while (i != sln && i < LINES_P_PAGE + SPARE_LINES) */
   while (i != sln && i < maxnumlines)
   {
      i++;
      ln++;
      if (ln > pg->ll && pg->islastpage == TRUE)
	 return pg;
      if (ln >= LINES_P_PAGE)
      {
	 if (pg->islastpage == FALSE)
	    pg = pg->next;
	 else
	    return pg;
	 ln = 0;
      }
      line = &pg->line[ln];
      if ((line->month != cm && line->month >= 0) ||
	  (line->date != cd && line->date >= 0))
      {
	 i += 2;
	 if (sln < i && sln >= i - 2)
	    return pg;
	 cm = line->month;
	 cd = line->date;
      }
   }

   return (pg);

}

