/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include "gtkhost.h"
#include <gtk/gtklabel.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtktty/gtkled.h>


#define CHILD_SPACING     1
#define DEFAULT_LEFT_POS  4
#define DEFAULT_TOP_POS   4
#define DEFAULT_SPACING   7


enum {
  PRESSED,
  RELEASED,
  CLICKED,
  ENTER,
  LEAVE,
  LAST_SIGNAL
};
enum {
  ARG_0,
  ARG_LABEL
};
  


static void gtk_host_class_init     (GtkHostClass   *klass);
static void gtk_host_init           (GtkHost        *button);
static void gtk_host_set_arg        (GtkHost        *button,
				       GtkArg           *arg,
				       guint		 arg_id);
static void gtk_host_map            (GtkWidget        *widget);
static void gtk_host_unmap          (GtkWidget        *widget);
static void gtk_host_realize        (GtkWidget        *widget);
static void gtk_host_size_request   (GtkWidget        *widget,
				       GtkRequisition   *requisition);
static void gtk_host_size_allocate  (GtkWidget        *widget,
				       GtkAllocation    *allocation);
static void gtk_host_paint          (GtkWidget        *widget,
				       GdkRectangle     *area);
static void gtk_host_draw           (GtkWidget        *widget,
				       GdkRectangle     *area);
static void gtk_host_draw_focus     (GtkWidget        *widget);
static void gtk_host_draw_default   (GtkWidget        *widget);
static gint gtk_host_expose         (GtkWidget        *widget,
				       GdkEventExpose   *event);
static gint gtk_host_press          (GtkWidget        *widget,
				       GdkEventButton *event);
static gint gtk_host_release        (GtkWidget        *widget,
				       GdkEventButton   *event);
static gint gtk_host_enter_notify   (GtkWidget        *widget,
				       GdkEventCrossing *event);
static gint gtk_host_leave_notify   (GtkWidget        *widget,
				       GdkEventCrossing *event);
static gint gtk_host_focus_in       (GtkWidget        *widget,
				       GdkEventFocus    *event);
static gint gtk_host_focus_out      (GtkWidget        *widget,
				       GdkEventFocus    *event);
static void gtk_host_add            (GtkContainer     *container,
				       GtkWidget        *widget);
static void gtk_host_remove         (GtkContainer     *container,
				       GtkWidget        *widget);
static void gtk_host_foreach        (GtkContainer     *container,
				       GtkCallback       callback,
				       gpointer          callback_data);
static void gtk_real_host_pressed   (GtkHost        *button);
static void gtk_real_host_released  (GtkHost        *button);
static void gtk_real_host_enter     (GtkHost        *button);
static void gtk_real_host_leave     (GtkHost        *button);


static GtkContainerClass *parent_class;
static guint host_signals[LAST_SIGNAL] = { 0 };


guint
gtk_host_get_type ()
{
  static guint host_type = 0;

  if (!host_type)
    {
      GtkTypeInfo host_info =
      {
	"GtkHost",
	sizeof (GtkHost),
	sizeof (GtkHostClass),
	(GtkClassInitFunc) gtk_host_class_init,
	(GtkObjectInitFunc) gtk_host_init,
	(GtkArgSetFunc) gtk_host_set_arg,
	(GtkArgGetFunc) NULL,
      };

      host_type = gtk_type_unique (gtk_container_get_type (), &host_info);
    }

  return host_type;
}

static void
gtk_host_class_init (GtkHostClass *klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;
  container_class = (GtkContainerClass*) klass;

  parent_class = gtk_type_class (gtk_container_get_type ());

  gtk_object_add_arg_type ("GtkHost::label", GTK_TYPE_STRING, 
			   GTK_ARG_WRITABLE, ARG_LABEL);

  host_signals[PRESSED] =
    gtk_signal_new ("pressed",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkHostClass, pressed),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  host_signals[RELEASED] =
    gtk_signal_new ("released",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkHostClass, released),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  host_signals[CLICKED] =
    gtk_signal_new ("clicked",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkHostClass, clicked),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  host_signals[ENTER] =
    gtk_signal_new ("enter",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkHostClass, enter),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  host_signals[LEAVE] =
    gtk_signal_new ("leave",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (GtkHostClass, leave),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, host_signals, LAST_SIGNAL);

  widget_class->activate_signal = host_signals[CLICKED];
  widget_class->map = gtk_host_map;
  widget_class->unmap = gtk_host_unmap;
  widget_class->realize = gtk_host_realize;
  widget_class->draw = gtk_host_draw;
  widget_class->draw_focus = gtk_host_draw_focus;
  widget_class->draw_default = gtk_host_draw_default;
  widget_class->size_request = gtk_host_size_request;
  widget_class->size_allocate = gtk_host_size_allocate;
  widget_class->expose_event = gtk_host_expose;
  widget_class->button_press_event = gtk_host_press;
  widget_class->button_release_event = gtk_host_release;
  widget_class->enter_notify_event = gtk_host_enter_notify;
  widget_class->leave_notify_event = gtk_host_leave_notify;
  widget_class->focus_in_event = gtk_host_focus_in;
  widget_class->focus_out_event = gtk_host_focus_out;

  container_class->add = gtk_host_add;
  container_class->remove = gtk_host_remove;
  container_class->foreach = gtk_host_foreach;

  klass->pressed = gtk_real_host_pressed;
  klass->released = gtk_real_host_released;
  klass->clicked = NULL;
  klass->enter = gtk_real_host_enter;
  klass->leave = gtk_real_host_leave;
}

static void
gtk_host_init (GtkHost *host)
{
  GTK_WIDGET_SET_FLAGS (host, GTK_CAN_FOCUS);

  host->pixmapwid = NULL;
  host->child = NULL;
  host->in_host = FALSE;
  host->host_down = FALSE;
}

static void
gtk_host_set_arg (GtkHost *host,
		    GtkArg    *arg,
		    guint      arg_id)
{
  GtkWidget *label;

  switch (arg_id)
    {
    case ARG_LABEL:
      gtk_container_disable_resize (GTK_CONTAINER (host));
      
      if (host->child)
	{
	  gtk_widget_unparent (host->child);
	  host->child = NULL;
	}
      
      label = gtk_label_new (GTK_VALUE_STRING(*arg));
      gtk_widget_show (label);
      
      gtk_container_add (GTK_CONTAINER (host), label);
      gtk_container_enable_resize (GTK_CONTAINER (host));
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

GtkWidget*
gtk_host_new ()
{
  return GTK_WIDGET (gtk_type_new (gtk_host_get_type ()));
}

GtkWidget*
gtk_host_new_with_label (const gchar *label)
{
  GtkWidget *host;
  GtkHost   *tmp;
  GtkWidget *box;

  host = gtk_host_new ();
  tmp = GTK_HOST (host);
  tmp->label = gtk_label_new (label);
  tmp->box   = gtk_vbox_new (FALSE, 2);
  gtk_misc_set_alignment (GTK_MISC (tmp->label), 0.5, 0.5);
 
  gtk_container_add (GTK_CONTAINER (host), tmp->box);
  gtk_box_pack_end (GTK_BOX (tmp->box), tmp->label, TRUE, TRUE, 1);
  tmp->act_led = gtk_led_new ();
  gtk_box_pack_end (GTK_BOX (tmp->box), tmp->act_led, FALSE, FALSE, 0);
  gtk_widget_show_all (tmp->box);

  return host;
}

void
gtk_host_set_label (GtkWidget *host, const gchar *label)
{
  GtkHost  *tmp;

  tmp = GTK_HOST (host);

  gtk_label_set (GTK_LABEL (tmp->label), label);
}

void
gtk_host_led_toggle (GtkWidget *host)
{
   GtkHost  *tmp;

   tmp = GTK_HOST (host);

   if (gtk_led_is_on (GTK_LED (tmp->act_led)))
     gtk_led_switch (GTK_LED (tmp->act_led), FALSE);
   else
     gtk_led_switch (GTK_LED (tmp->act_led), TRUE);
}


void 
gtk_host_set_pixmap (GtkWidget *host, GtkWidget *pixmap)
{
  GtkHost   *tmp;

  tmp = GTK_HOST (host);
  if (tmp->pixmapwid == NULL)
    {
      tmp->pixmapwid = pixmap;
      gtk_box_pack_start (GTK_BOX (tmp->box), tmp->pixmapwid, TRUE, TRUE, 1);
      gtk_widget_show (tmp->pixmapwid);
    }
  else
    {
      tmp->pixmapwid = pixmap;
    }
}

void
gtk_host_pressed (GtkHost *host)
{
  gtk_signal_emit (GTK_OBJECT (host), host_signals[PRESSED]);
}

void
gtk_host_released (GtkHost *host)
{
  gtk_signal_emit (GTK_OBJECT (host), host_signals[RELEASED]);
}

void
gtk_host_clicked (GtkHost *host)
{
  gtk_signal_emit (GTK_OBJECT (host), host_signals[CLICKED]);
}

void
gtk_host_enter (GtkHost *host)
{
  gtk_signal_emit (GTK_OBJECT (host), host_signals[ENTER]);
}

void
gtk_host_leave (GtkHost *host)
{
  gtk_signal_emit (GTK_OBJECT (host), host_signals[LEAVE]);
}

static void
gtk_host_map (GtkWidget *widget)
{
  GtkHost *host;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_MAPPED);
  gdk_window_show (widget->window);

  host = GTK_HOST (widget);

  if (host->child &&
      GTK_WIDGET_VISIBLE (host->child) &&
      !GTK_WIDGET_MAPPED (host->child))
    gtk_widget_map (host->child);
}

static void
gtk_host_unmap (GtkWidget *widget)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_MAPPED);
  gdk_window_hide (widget->window);
}

static void
gtk_host_realize (GtkWidget *widget)
{
  GtkHost *host;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  host = GTK_HOST (widget);
  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), 
				   &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, host);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

static void
gtk_host_size_request (GtkWidget      *widget,
			 GtkRequisition *requisition)
{
  GtkHost *host;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));
  g_return_if_fail (requisition != NULL);

  host = GTK_HOST (widget);

  requisition->width = (GTK_CONTAINER (widget)->border_width + CHILD_SPACING +
			GTK_WIDGET (widget)->style->klass->xthickness) * 2;
  requisition->height = (GTK_CONTAINER (widget)->border_width + CHILD_SPACING +
			 GTK_WIDGET (widget)->style->klass->ythickness) * 2;

  if (GTK_WIDGET_CAN_DEFAULT (widget))
    {
      requisition->width += (GTK_WIDGET (widget)->style->klass->xthickness * 2 +
			     DEFAULT_SPACING);
      requisition->height += (GTK_WIDGET (widget)->style->klass->ythickness * 2 +
			      DEFAULT_SPACING);
    }

  if (host->child && GTK_WIDGET_VISIBLE (host->child))
    {
      gtk_widget_size_request (host->child, &host->child->requisition);

      requisition->width += host->child->requisition.width;
      requisition->height += host->child->requisition.height;
    }
}

static void
gtk_host_size_allocate (GtkWidget     *widget,
			  GtkAllocation *allocation)
{
  GtkHost *host;
  GtkAllocation child_allocation;
  gint border_width;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;
  border_width = GTK_CONTAINER (widget)->border_width;

  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    widget->allocation.x + border_width,
			    widget->allocation.y + border_width,
			    widget->allocation.width - border_width * 2,
			    widget->allocation.height - border_width * 2);

  host = GTK_HOST (widget);

  if (host->child && GTK_WIDGET_VISIBLE (host->child))
    {
      child_allocation.x = (CHILD_SPACING + GTK_WIDGET (widget)->style->klass->xthickness);
      child_allocation.y = (CHILD_SPACING + GTK_WIDGET (widget)->style->klass->ythickness);

      child_allocation.width = widget->allocation.width - child_allocation.x * 2 -
	                         border_width * 2;
      child_allocation.height = widget->allocation.height - child_allocation.y * 2 -
	                          border_width * 2;

      if (GTK_WIDGET_CAN_DEFAULT (host))
	{
	  child_allocation.x += (GTK_WIDGET (widget)->style->klass->xthickness +
				 DEFAULT_LEFT_POS);
	  child_allocation.y += (GTK_WIDGET (widget)->style->klass->ythickness +
				 DEFAULT_TOP_POS);
	  child_allocation.width -= (GTK_WIDGET (widget)->style->klass->xthickness * 2 +
				     DEFAULT_SPACING);
	  child_allocation.height -= (GTK_WIDGET (widget)->style->klass->xthickness * 2 +
				      DEFAULT_SPACING);
	}

      gtk_widget_size_allocate (host->child, &child_allocation);
    }
}

static void
gtk_host_paint (GtkWidget    *widget,
		  GdkRectangle *area)
{
  GdkRectangle restrict_area;
  GdkRectangle new_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      restrict_area.x = GTK_WIDGET (widget)->style->klass->xthickness;
      restrict_area.y = GTK_WIDGET (widget)->style->klass->ythickness;
      restrict_area.width = (GTK_WIDGET (widget)->allocation.width - restrict_area.x * 2 -
                             GTK_CONTAINER (widget)->border_width * 2);
      restrict_area.height = (GTK_WIDGET (widget)->allocation.height - restrict_area.y * 2 -
                              GTK_CONTAINER (widget)->border_width * 2);

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  restrict_area.x += DEFAULT_LEFT_POS;
	  restrict_area.y += DEFAULT_TOP_POS;
	  restrict_area.width -= DEFAULT_SPACING;
	  restrict_area.height -= DEFAULT_SPACING;
	}

      if (gdk_rectangle_intersect (area, &restrict_area, &new_area))
	{
	  gtk_style_set_background (widget->style, widget->window, 
				    GTK_WIDGET_STATE (widget));
	  gdk_window_clear_area (widget->window,
				 new_area.x, new_area.y,
				 new_area.width, new_area.height);
	}
    }
}

static void
gtk_host_draw (GtkWidget    *widget,
		 GdkRectangle *area)
{
  GtkHost *host;
  GdkRectangle child_area;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      host = GTK_HOST (widget);

      gtk_host_paint (widget, area);

      if (host->child && gtk_widget_intersect (host->child, area, &child_area))
	gtk_widget_draw (host->child, &child_area);

      gtk_widget_draw_default (widget);
      gtk_widget_draw_focus (widget);
    }
}

static void
gtk_host_draw_focus (GtkWidget *widget)
{
  GtkHost *host;
  GtkShadowType shadow_type;
  gint width, height;
  gint x, y;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      host = GTK_HOST (widget);

      x = 0;
      y = 0;
      width = widget->allocation.width - GTK_CONTAINER 
	(widget)->border_width * 2;
      height = widget->allocation.height - GTK_CONTAINER 
	(widget)->border_width * 2;

      if (GTK_WIDGET_CAN_DEFAULT (widget))
	{
	  x += widget->style->klass->xthickness;
	  y += widget->style->klass->ythickness;
	  width -= 2 * x + DEFAULT_SPACING;
	  height -= 2 * y + DEFAULT_SPACING;
	  x += DEFAULT_LEFT_POS;
	  y += DEFAULT_TOP_POS;
	}

      if (GTK_WIDGET_HAS_FOCUS (widget))
	{
	  x += 1;
	  y += 1;
	  width -= 2;
	  height -= 2;
	}
      else
	{
	  if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_WIDGET_STATE 
						    (widget)], FALSE,
				x + 1, y + 1, width - 4, height - 4);
	  else
	    gdk_draw_rectangle (widget->window,
				widget->style->bg_gc[GTK_WIDGET_STATE 
						    (widget)], FALSE,
				x + 2, y + 2, width - 5, height - 5);
	}

      if (GTK_WIDGET_STATE (widget) == GTK_STATE_ACTIVE)
	shadow_type = GTK_SHADOW_IN;
      else
	shadow_type = GTK_SHADOW_OUT;

      gtk_draw_shadow (widget->style, widget->window,
		       GTK_WIDGET_STATE (widget), shadow_type,
		       x, y, width, height);

      if (GTK_WIDGET_HAS_FOCUS (widget))
	{
	  x -= 1;
	  y -= 1;
	  width += 2;
	  height += 2;

	  gdk_draw_rectangle (widget->window,
			      widget->style->black_gc, FALSE,
			      x, y, width - 1, height - 1);
	}
    }
}

static void
gtk_host_draw_default (GtkWidget *widget)
{
  gint width, height;
  gint x, y;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_HOST (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      x = 0;
      y = 0;
      width = widget->allocation.width - GTK_CONTAINER (widget)->border_width * 2;
      height = widget->allocation.height - GTK_CONTAINER (widget)->border_width * 2;

      if (GTK_WIDGET_HAS_DEFAULT (widget))
	{
	  gtk_draw_shadow (widget->style, widget->window,
			   GTK_STATE_NORMAL, GTK_SHADOW_IN,
			   x, y, width, height);
	}
      else
	{
	  gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_NORMAL],
			      FALSE, x, y, width - 1, height - 1);
	  gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_NORMAL],
			      FALSE, x + 1, y + 1, width - 3, height - 3);
	}
    }
}

static gint
gtk_host_expose (GtkWidget      *widget,
		   GdkEventExpose *event)
{
  GtkHost *host;
  GdkEventExpose child_event;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      host = GTK_HOST (widget);

      gtk_host_paint (widget, &event->area);

      child_event = *event;
      if (host->child && GTK_WIDGET_NO_WINDOW (host->child) &&
	  gtk_widget_intersect (host->child, &event->area, &child_event.area))
	gtk_widget_event (host->child, (GdkEvent*) &child_event);

      gtk_widget_draw_default (widget);
      gtk_widget_draw_focus (widget);
    }

  return FALSE;
}

static gint
gtk_host_press (GtkWidget      *widget, GdkEventButton *event)
{
  GtkHost *host;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->type == GDK_BUTTON_PRESS)
    {
      host = GTK_HOST (widget);

      if (GTK_WIDGET_CAN_DEFAULT (widget) && (event->button == 1))
	gtk_widget_grab_default (widget);
      if (!GTK_WIDGET_HAS_FOCUS (widget))
	gtk_widget_grab_focus (widget);

      if (event->button == 1)
	{
	  gtk_grab_add (GTK_WIDGET (host));
	  gtk_host_pressed (host);
	}
    }

  return TRUE;
}

static gint
gtk_host_release (GtkWidget      *widget,
			   GdkEventButton *event)
{
  GtkHost *host;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->button == 1)
    {
      host = GTK_HOST (widget);
      gtk_grab_remove (GTK_WIDGET (host));
      gtk_host_released (host);
    }

  return TRUE;
}

static gint
gtk_host_enter_notify (GtkWidget        *widget,
			 GdkEventCrossing *event)
{
  GtkHost *host;
  GtkWidget *event_widget;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  host = GTK_HOST (widget);
  event_widget = gtk_get_event_widget ((GdkEvent*) event);

  if ((event_widget == widget) &&
      (event->detail != GDK_NOTIFY_INFERIOR))
    {
      host->in_host = TRUE;
      gtk_host_enter (host);
    }

  return FALSE;
}

static gint
gtk_host_leave_notify (GtkWidget        *widget,
			 GdkEventCrossing *event)
{
  GtkHost *host;
  GtkWidget *event_widget;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  host = GTK_HOST (widget);
  event_widget = gtk_get_event_widget ((GdkEvent*) event);

  if ((event_widget == widget) &&
      (event->detail != GDK_NOTIFY_INFERIOR))
    {
      host->in_host = FALSE;
      gtk_host_leave (host);
    }

  return FALSE;
}

static gint
gtk_host_focus_in (GtkWidget     *widget,
		     GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

  return FALSE;
}

static gint
gtk_host_focus_out (GtkWidget     *widget,
		      GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_HOST (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

  return FALSE;
}

static void
gtk_host_add (GtkContainer *container,
		GtkWidget    *widget)
{
  GtkHost *host;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_HOST (container));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (gtk_widget_basic (widget));

  host = GTK_HOST (container);

  if (!host->child)
    {
      gtk_widget_set_parent (widget, GTK_WIDGET (container));

      if (GTK_WIDGET_VISIBLE (widget->parent))
	{
	  if (GTK_WIDGET_REALIZED (widget->parent) &&
	      !GTK_WIDGET_REALIZED (widget))
	    gtk_widget_realize (widget);
	  
	  if (GTK_WIDGET_MAPPED (widget->parent) &&
	      !GTK_WIDGET_MAPPED (widget))
	    gtk_widget_map (widget);
	}
      
      host->child = widget;

      if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_VISIBLE (container))
	gtk_widget_queue_resize (widget);
    }
}

static void
gtk_host_remove (GtkContainer *container,
		   GtkWidget    *widget)
{
  GtkHost *host;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_HOST (container));
  g_return_if_fail (widget != NULL);

  host = GTK_HOST (container);

  if (host->child == widget)
    {
      gtk_widget_unparent (widget);

      host->child = NULL;

      if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_VISIBLE (container))
	gtk_widget_queue_resize (GTK_WIDGET (container));
    }
}

static void
gtk_host_foreach (GtkContainer *container,
		    GtkCallback   callback,
		    gpointer      callback_data)
{
  GtkHost *host;

  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_HOST (container));
  g_return_if_fail (callback != NULL);

  host = GTK_HOST (container);

  if (host->child)
    (* callback) (host->child, callback_data);
}

static void
gtk_real_host_pressed (GtkHost *host)
{
  GtkStateType new_state;

  g_return_if_fail (host != NULL);
  g_return_if_fail (GTK_IS_HOST (host));

  host->host_down = TRUE;

  new_state = (host->in_host ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL);

  if (GTK_WIDGET_STATE (host) != new_state)
    {
      gtk_widget_set_state (GTK_WIDGET (host), new_state);
      gtk_widget_queue_draw (GTK_WIDGET (host));
    }
}

static void
gtk_real_host_released (GtkHost *host)
{
  GtkStateType new_state;

  g_return_if_fail (host != NULL);
  g_return_if_fail (GTK_IS_HOST (host));

  if (host->host_down)
    {
      host->host_down = FALSE;

      if (host->in_host)
	gtk_host_clicked (host);

      new_state = (host->in_host ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL);

      if (GTK_WIDGET_STATE (host) != new_state)
	{
	  gtk_widget_set_state (GTK_WIDGET (host), new_state);
	  gtk_widget_queue_draw (GTK_WIDGET (host));
	}
    }
}

static void
gtk_real_host_enter (GtkHost *host)
{
  GtkStateType new_state;

  g_return_if_fail (host != NULL);
  g_return_if_fail (GTK_IS_HOST (host));

  new_state = (host->host_down ? GTK_STATE_ACTIVE : GTK_STATE_PRELIGHT);

  if (GTK_WIDGET_STATE (host) != new_state)
    {
      gtk_widget_set_state (GTK_WIDGET (host), new_state);
      gtk_widget_queue_draw (GTK_WIDGET (host));
    }
}

static void
gtk_real_host_leave (GtkHost *host)
{
  g_return_if_fail (host != NULL);
  g_return_if_fail (GTK_IS_HOST (host));

  if (GTK_WIDGET_STATE (host) != GTK_STATE_NORMAL)
    {
      gtk_widget_set_state (GTK_WIDGET (host), GTK_STATE_NORMAL);
      gtk_widget_queue_draw (GTK_WIDGET (host));
    }
}

/* EOF */

