/*
 * $Id: gui.c,v 1.13 1998/08/01 20:04:27 gregm Exp $
 * GXSNMP - An snmp managment application
 * Copyright (C) 1998 Gregory McLean
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * 'lazy mans' gui functions.
 */
#include <gtk/gtk.h>
#include "main.h"
#include "gui.h"

/* Unfortunately, the definition of pixmap_cache clashes with gtk_imlib.
   Raster, we love you - renamed definition to my_pixmap_cache. -- Jochen */

struct my_pixmap_cache {
  GdkPixmap  *pixmap;
  GdkBitmap  *mask;
};

int color_map[][3] = {
  {   0,   0,   0 },         /* black */
  {   0,   0, 255 },         /* Blue */
  {   0, 200,   0 },         /* Green */
  { 200,   0,   0 },         /* Red */
  { 255,   0, 255 },         /* Violet */
  { 255, 255,   0 }          /* Yellow */
};

static GHashTable *pixmap_hash;
extern gxsnmp *app_info;

GdkPixmap *book_open;
GdkPixmap *book_closed;
GdkPixmap *book_open_mask;
GdkPixmap *book_closed_mask;

static char * book_open_xpm[] = {
"16 16 4 1",
"       c None s None",
".      c black",
"X      c #808080",
"o      c white",
"                ",
"  ..            ",
" .Xo.    ...    ",
" .Xoo. ..oo.    ",
" .Xooo.Xooo...  ",
" .Xooo.oooo.X.  ",
" .Xooo.Xooo.X.  ",
" .Xooo.oooo.X.  ",
" .Xooo.Xooo.X.  ",
" .Xooo.oooo.X.  ",
"  .Xoo.Xoo..X.  ",
"   .Xo.o..ooX.  ",
"    .X..XXXXX.  ",
"    ..X.......  ",
"     ..         ",
"                "};

static char * book_closed_xpm[] = {
"16 16 6 1",
"       c None s None",
".      c black",
"X      c red",
"o      c yellow",
"O      c #808080",
"#      c white",
"                ",
"       ..       ",
"     ..XX.      ",
"   ..XXXXX.     ",
" ..XXXXXXXX.    ",
".ooXXXXXXXXX.   ",
"..ooXXXXXXXXX.  ",
".X.ooXXXXXXXXX. ",
".XX.ooXXXXXX..  ",
" .XX.ooXXX..#O  ",
"  .XX.oo..##OO. ",
"   .XX..##OO..  ",
"    .X.#OO..    ",
"     ..O..      ",
"      ..        ",
"                "};

static void
destroy_hash_element (void *key, void *val, void *data)
{
  struct my_pixmap_cache  *pix = val;

  gdk_pixmap_unref (pix->pixmap);
  gdk_pixmap_unref (pix->mask);
  g_free (val);
  g_free (key);
}

void
destroy_pixmap_cache (void)
{
  if (!pixmap_hash)
    return;

  g_hash_table_foreach (pixmap_hash, destroy_hash_element, 0);
  g_hash_table_destroy (pixmap_hash);
  pixmap_hash = NULL;
}

static void
check_hash (void)
{
  if (!pixmap_hash)
    pixmap_hash = g_hash_table_new (g_str_hash, g_str_equal);
}

GtkWidget *
new_window (char *title, GtkWindowType type, GtkWindowPosition pos, int border)
{
  GtkWidget    *window;

  window = gtk_window_new (type);
  gtk_window_set_title (GTK_WINDOW (window), title);
  gtk_window_position (GTK_WINDOW (window), pos);
  gtk_container_border_width (GTK_CONTAINER (window), border);
  return window;
}

/*
 * Build a new gdk pixmap
 *
 * TODO -- do some checks and see if we can read the file, if we can't
 *         read the file then return an inlined 'not found' pixmap.
 */
GtkWidget *
new_pixmap (char      *filename,
	    GdkWindow *window,
	    GdkColor  *background)
{
  struct my_pixmap_cache *pix;

  check_hash ();
  pix = g_hash_table_lookup (pixmap_hash, filename);
  if (!pix)
    {
      pix = g_new (struct my_pixmap_cache, 1);
      pix->pixmap = gdk_pixmap_create_from_xpm (window, &pix->mask,
				       background, filename);
      g_hash_table_insert (pixmap_hash, g_strdup (filename), pix);
    }
  return (gtk_pixmap_new (pix->pixmap, pix->mask));
}

void
init_pixmap_d (GtkWidget *widget)
{
  GdkColor    transparent;

  book_open = gdk_pixmap_create_from_xpm_d (widget->window,
					    &book_open_mask,
					    &transparent,
					    book_open_xpm);
  book_closed = gdk_pixmap_create_from_xpm_d (widget->window,
					      &book_closed_mask,
					      &transparent,
					      book_closed_xpm);
}

GtkWidget *
new_host_widget (hosts *hentry)
{
  GtkWidget  *button;
  GtkWidget  *label;
  GtkWidget  *pixmapwid;
  GtkWidget  *box;
  GdkColor   color;
  char       *kludge;

  button = gtk_host_new_with_label (hentry->hl_disp);
  gtk_widget_show (button);

  kludge = gnome_unconditional_pixmap_file ("desktop.xpm");
  pixmapwid = new_pixmap (kludge, app_info->window->window,
			  &color);
  gtk_widget_set_name (button, "host_widget");
  gtk_widget_show (pixmapwid);
  gtk_host_set_pixmap (button, pixmapwid);

  return button;
}

GdkColor
get_color (int which)
{
  GdkColor   color;
  gint       n;

  color.red  = color_map[which][RED_IDX] | (color_map[which][RED_IDX] << 8);
  color.green= color_map[which][GRN_IDX] | (color_map[which][GRN_IDX] << 8);
  color.blue = color_map[which][BLU_IDX] | (color_map[which][BLU_IDX] << 8);
  color.pixel = 0;   /* required */
  n = 0;
  gdk_color_context_get_pixels (app_info->cc,
				&color.red, 
				&color.green, 
				&color.blue,
				1,
				&color.pixel,
				&n);
  return color;
}

/*
 * notebook page switch function.
 * Basic one to handle changing the pixmaps
 */
void
page_switch_cb (GtkWidget *widget, GtkNotebookPage *page, gint page_num)
{
  GtkNotebookPage *oldpage;
  GtkWidget       *pixwid;


  oldpage = GTK_NOTEBOOK (widget)->cur_page;
  if (page == oldpage)
    return;
  pixwid = ((GtkBoxChild*)(GTK_BOX (page->tab_label)->children->data))->widget;
  gtk_pixmap_set (GTK_PIXMAP (pixwid), book_open, book_open_mask);
  if (oldpage)
    {
      pixwid = ((GtkBoxChild*)(GTK_BOX
			       (oldpage->tab_label)->children->data))->widget;
      gtk_pixmap_set (GTK_PIXMAP (pixwid), book_closed, book_closed_mask);
    }
}

/*
 * Notice dialog. No critical messages, just something the user needs to
 * know.
 */
void
notice_dlg (char *msg, ...)
{
  va_list ap;
  gchar   buf[1024];

  va_start (ap, msg);
  vsnprintf (buf, sizeof(buf), msg, ap);
  va_end (ap);

  gtk_widget_show (gnome_message_box_new (buf, "info",
					  GNOME_STOCK_BUTTON_OK, NULL));
}

/*
 * Do them nifty looking notebook tabs, yeah its a short function but
 * I got tired of writing the same 10-15 lines in every panel.
 */
GtkWidget *
notebook_tab (const char *label)
{
  GtkWidget    *tab_box;
  GtkWidget    *pixmap;
  GtkWidget    *tab_label;

  tab_box = gtk_hbox_new (FALSE, 0);
  pixmap = gtk_pixmap_new (book_closed, book_closed_mask);
  gtk_box_pack_start (GTK_BOX (tab_box), pixmap, FALSE, TRUE, 0);
  gtk_misc_set_padding (GTK_MISC (pixmap), 3, 1);
  tab_label = gtk_label_new (label);
  gtk_misc_set_padding (GTK_MISC (label), 10, 0);
  gtk_box_pack_start (GTK_BOX (tab_box), tab_label, FALSE, TRUE, 0);
  gtk_widget_show_all (tab_box);
  return tab_box;
}

/* EOF */




