#include <stdio.h>
#include <errno.h>

#include <gtk/gtk.h>

#include <gnome.h>

#include "raft_config.h"
#include "raft_interface.h"
#include "raft_cmd.h"

static struct 
{
    GtkWidget *label_entry;
    GtkWidget *tooltips_entry;
    GtkWidget *prog_entry;
    GtkWidget *xpm_entry;
    
    /* the current button we're working on */
    GtkWidget *current_button;
} config_interface;

extern Raft_launcher *raft_launcher;
extern Raft_info raft_info;

static void raft_config_notebook_launcher_page(GtkWidget *notebook);
static void raft_config_notebook_attrib_page(GtkWidget *notebook);
static void raft_config_button_add(GtkWidget *widget, gpointer data);
static void raft_config_button_remove(GtkWidget *widget, gpointer data);
static void raft_config_button_apply(GtkWidget *widget, gpointer data);
static void raft_config_button_reset(GtkWidget *widget, gpointer data);
static void raft_config_save_to_file(GtkWidget *widget, gpointer data);
static gchar *raft_config_null2strlen (gchar *string);
static void raft_config_strlen2null (gchar **string);

static GtkWidget *config_window;


static gint raft_config_delete(GtkWidget *widget, gpointer data)
{
    /* return true so it really does destroy. */
    return(TRUE);
}

static void raft_config_destroy(GtkWidget *widget, gpointer data)
{
    config_window = NULL;
    
    /* turn off config mode as the config window is now closed. */
    raft_info.config_mode = FALSE;
}


/* this is the configuration interface */

void raft_config_interface_show(void)
{
    GtkWidget *notebook;
    GtkWidget *box1;
    GtkWidget *button;
    GtkWidget *box2;
    
    /* make sure one doesn't already exist. */
    if (config_window != NULL)
	    return;
    
    /* make sure the launcher is expanded. */
    raft_interface_show_launcher ();
    
    /* set us in config mode.. this just catches the callbacks from the launcher buttons
     * so they know they should report instead of starting a program. */
    raft_info.config_mode = TRUE;
    
    config_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    
    gtk_signal_connect(GTK_OBJECT(config_window), "delete_event",
		       GTK_SIGNAL_FUNC (raft_config_delete), NULL);

    gtk_signal_connect(GTK_OBJECT(config_window), "destroy",
		       GTK_SIGNAL_FUNC (raft_config_destroy), NULL);

    gtk_window_set_title (GTK_WINDOW(config_window), "raft config");

    gtk_container_border_width (GTK_CONTAINER(config_window), 5);

    box1 = gtk_vbox_new (FALSE, 1);
    gtk_widget_show(box1);
    
    gtk_container_add(GTK_CONTAINER (config_window), box1);
    
    /* create a new notebook */
    notebook = gtk_notebook_new ();
        
    gtk_box_pack_start(GTK_BOX (box1), notebook, FALSE, FALSE, 3);
    
    /* create the launcher button configuration notebook page */
    raft_config_notebook_launcher_page(notebook);
    
    /* create the launcher attributes configuration notebook page */
    raft_config_notebook_attrib_page(notebook);

    /* and a close button */
    box2 = gtk_hbox_new (FALSE, 1);
    gtk_box_pack_start(GTK_BOX (box1), box2, FALSE, FALSE, 0);
    gtk_widget_show (box2);
    
    button = gtk_button_new_with_label("Close");
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer) config_window);
    gtk_box_pack_start(GTK_BOX (box2), button, TRUE, TRUE, 6);
    gtk_widget_show(button);
    
    button = gtk_button_new_with_label("Save");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(raft_config_save_to_file), NULL);
    gtk_box_pack_start(GTK_BOX (box2), button, TRUE, TRUE, 6);
    gtk_widget_show (button);

    gtk_widget_show (notebook);
    gtk_widget_show (config_window);
}

/* utility function that creates a text entry with a label.
 * It packs it all into the box given as the first argument, and
 * returns a pointer to the entry. */

static GtkWidget *raft_config_text_entry_new(GtkWidget *box1, gchar *label_text)
{
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *entry;
    
    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start(GTK_BOX(box1), hbox, TRUE, FALSE, 6);
    gtk_widget_show(hbox);
    
    label = gtk_label_new(label_text);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 6);
    gtk_widget_show(label);
    
    entry = gtk_entry_new();
    gtk_widget_set_usize(entry, 300, 25);
    gtk_box_pack_end(GTK_BOX(hbox), entry, FALSE, FALSE, 6);
    gtk_entry_set_text(GTK_ENTRY(entry), "");
    gtk_widget_show(entry);

    return(entry);
}

/* generate the interface to config the button bar */
static void raft_config_notebook_launcher_page(GtkWidget *notebook)
{
    GtkWidget *frame;
    GtkWidget *label;
    GtkWidget *box1;
    GtkWidget *box2;
    GtkWidget *button;
    GtkWidget *sep;
    
    label = gtk_label_new("Button Configuration");
    gtk_widget_show(label);
    
    frame = gtk_frame_new (NULL);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    gtk_container_border_width(GTK_CONTAINER (frame), 1);
    gtk_widget_show (frame);
    
    /* The main vbox */
    box1 = gtk_vbox_new (FALSE, 3);
    gtk_container_add(GTK_CONTAINER (frame), box1);
    gtk_widget_show(box1);

    label = gtk_label_new("Click on a Launcher button to configure.");
    gtk_box_pack_start(GTK_BOX(box1), label, TRUE, TRUE, 6);
    gtk_widget_show(label);

    sep = gtk_hseparator_new ();
    gtk_box_pack_start(GTK_BOX(box1), sep, FALSE, TRUE, 0);
    gtk_widget_show(sep);
    
    /* these functions pack and show the entry for us. */
    config_interface.label_entry = raft_config_text_entry_new(box1, "Button Label: ");
    
    /* The program name text entry. Probly want a browse button for this too.. */
    config_interface.prog_entry = raft_config_text_entry_new(box1, "Executable: ");
    
    config_interface.tooltips_entry = raft_config_text_entry_new(box1, "Tooltip text: ");

    /* The xpm filename text entry.  This will probably be changed.. or none-existant
     * as the xpm will be shown on the button in real-time.  Just a button here
     * to browse for an xpm.. or I may leave the entry here, but still be able to browse. */
    config_interface.xpm_entry = raft_config_text_entry_new(box1, "Icon Filename: ");

    sep = gtk_hseparator_new ();
    gtk_box_pack_start(GTK_BOX(box1), sep, FALSE, TRUE, 0);
    gtk_widget_show(sep);
    
    box2 = gtk_hbox_new (TRUE, 0);
    gtk_box_pack_start(GTK_BOX(box1), box2, FALSE, FALSE, 3);
    gtk_widget_show(box2);
    
    button = gtk_button_new_with_label ("New Button");
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(raft_config_button_add), NULL);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show(button);
    
    button = gtk_button_new_with_label ("Remove");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(raft_config_button_remove), NULL);
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_show(button);

    button = gtk_button_new_with_label ("Apply Changes");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(raft_config_button_apply), NULL);
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
    gtk_widget_grab_default(button);
    gtk_widget_show(button);
    
    /*
    button = gtk_button_new_with_label ("Reset");
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       GTK_SIGNAL_FUNC(raft_config_button_reset), NULL);
    gtk_box_pack_start(GTK_BOX(box2), button, TRUE, TRUE, 2);
    gtk_widget_show(button);
    */
}

/* generate the interface to config the application */
static void raft_config_notebook_attrib_page(GtkWidget *notebook)
{
    GtkWidget *frame;
    GtkWidget *label;
    GtkWidget *box1;
    
    label = gtk_label_new("Application Configuration");
    gtk_widget_show(label);

    frame = gtk_frame_new (NULL);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), frame, label);
    gtk_widget_show (frame);

    box1 = gtk_vbox_new (FALSE, 3);
    gtk_container_add(GTK_CONTAINER (frame), box1);
    gtk_widget_show(box1);

    label = gtk_label_new("Sorry.. not yet available.");
    gtk_box_pack_start(GTK_BOX(box1), label, TRUE, TRUE, 6);
    gtk_widget_show(label);
}

/* find what button was pressed. */
void raft_config_register_button(GtkWidget *widget)
{
    Raft_launcher *rl;

    rl = raft_launcher;
    
    while (rl->next != NULL) {
	rl = rl->next;
	
	if (rl->button == widget) {

	    config_interface.current_button = widget;

	    gtk_entry_set_text(GTK_ENTRY(config_interface.label_entry), 
			       raft_config_null2strlen(rl->label_text));

	    gtk_entry_set_text(GTK_ENTRY(config_interface.prog_entry), 
			       raft_config_null2strlen(rl->command));

	    gtk_entry_set_text(GTK_ENTRY(config_interface.tooltips_entry), 
			       raft_config_null2strlen(rl->tooltips_text));

	    gtk_entry_set_text(GTK_ENTRY(config_interface.xpm_entry), 
			       raft_config_null2strlen(rl->xpm_file));
	    
	}
    }
}

static void raft_config_button_add(GtkWidget *widget, gpointer data)
{
    Raft_launcher *rl;

    rl = raft_launcher;
    
    while (rl->next != NULL)
	    rl = rl->next;
        
    rl->next = g_malloc0 (sizeof(Raft_launcher));
    
    rl=rl->next;
    
    rl->button = raft_interface_button_new(rl);
    
    gtk_signal_connect(GTK_OBJECT(rl->button), "clicked",
		       GTK_SIGNAL_FUNC(raft_cmd_execute), rl->command);
    
    gtk_box_pack_start(GTK_BOX(raft_info.main_box), rl->button, TRUE, TRUE, 1);

    gtk_widget_show (rl->button);
    
    rl->label_text = g_strdup("New Button");
    
    raft_config_register_button(rl->button);

    /* Yucky hack to make it work right :) */
    g_free(rl->label_text);
    rl->label_text = NULL;
    /* make our changes show right away. */
    raft_config_button_apply(NULL, NULL);
}

static void raft_config_button_remove(GtkWidget *widget, gpointer data)
{
    Raft_launcher *rl;
    Raft_launcher *dead;
    
    rl = raft_launcher;
    
    while (rl->next != NULL) {
	if(rl->next->button == config_interface.current_button) {
	    
	    gtk_widget_destroy(rl->next->button);

	    /* free the to-be-killed link */
	    g_free(rl->next->command);
	    g_free(rl->next->label_text);
	    g_free(rl->next->tooltips_text);
	    g_free(rl->next->xpm_file);

	    /* free it and relink the list */
	    dead = rl->next;
	    rl->next = rl->next->next;
	    g_free(dead);
	    
	    break;
	}
	rl = rl->next;
    }
}

static void raft_config_button_apply(GtkWidget *widget, gpointer data)
{
    Raft_launcher *rl;
    gchar *tmp;
        
    rl = raft_launcher;
    
    while (rl->next != NULL) {
	rl = rl->next;
	if (rl->button == config_interface.current_button) {
	    
	    /* get the new pixmap and install (if necessary). */
	    tmp = gtk_entry_get_text(GTK_ENTRY(config_interface.xpm_entry));
	    if ((rl->xpm_file == NULL) || (strcmp(rl->xpm_file, tmp))) {
		g_free(rl->xpm_file);
		if (strlen(tmp) == 0)
			rl->xpm_file = NULL;
		else 
			rl->xpm_file = g_strdup(tmp);
		raft_interface_launcher_pixmap_new(rl);
	    }
	    
	    /* check the label */
	    tmp = gtk_entry_get_text(GTK_ENTRY(config_interface.label_entry));
	    if ( (rl->label_text == NULL) || (strcmp(rl->label_text, tmp)) ) {
		g_free(rl->label_text);
		if (strlen(tmp) == 0)
			rl->label_text = NULL;
		else 
			rl->label_text = g_strdup(tmp);
		raft_interface_launcher_label_new(rl);
	    }
	    
	    /* redo the tooltips */
	    tmp = gtk_entry_get_text(GTK_ENTRY(config_interface.tooltips_entry));
	    if ( (rl->tooltips_text == NULL) || (strcmp(rl->tooltips_text, tmp)) ) {
		g_free(rl->tooltips_text);
		if (strlen(tmp) == 0)
			rl->tooltips_text = NULL;
		else 
			rl->tooltips_text = g_strdup(tmp);
		raft_interface_launcher_tooltip_new(rl);
	    }

	    /* check the command and install */
	    tmp = gtk_entry_get_text(GTK_ENTRY(config_interface.prog_entry));
	    g_free(rl->command);
	    if (strlen(tmp) == 0) {
		rl->command = NULL;
	    } else {
		rl->command = g_strdup(tmp);
		gtk_signal_connect(GTK_OBJECT(rl->button), "clicked",
				   GTK_SIGNAL_FUNC(raft_cmd_execute), rl->command);
	    }
	}
    }
}

static void raft_config_button_reset(GtkWidget *widget, gpointer data)
{
    Raft_launcher *rl;
    
    rl = raft_launcher;
    
    while (rl->next != NULL) {
	rl = rl->next;
	if (rl->button == config_interface.current_button)
	    raft_config_register_button(rl->button);
    }
}

static void raft_config_save_to_file(GtkWidget *widget, gpointer data)
{
    FILE *fp;
    Raft_launcher *rl;
    gchar *file;
    
    file = gnome_file_from_home_dir("raft.conf");
    fp = fopen(file, "w");
    
    if (fp == NULL) {
	g_warning("Unable to write file %s: %s", file, g_strerror(errno));
	g_free(file);
	return;
    }
    g_free(file);
    
    rl = raft_launcher;
    
    while (rl->next != NULL) {
	rl = rl->next;
	
	fprintf(fp, "%s:%s:%s:%s\n", raft_config_null2strlen(rl->xpm_file),
		raft_config_null2strlen(rl->command), raft_config_null2strlen(rl->label_text),
		raft_config_null2strlen(rl->tooltips_text));
    }
    fclose(fp);
}

/* change string to NULL if it's 0 length */

static void raft_config_strlen2null (gchar **string)
{
    if (strlen(*string) == 0)
	    string = NULL;
}

/* if the stringlength is 0, return a string of 0 length instead ("") */

static gchar *raft_config_null2strlen (gchar *string)
{
    static gchar *strlen0 = "";
    
    if (string == NULL)
	    return (strlen0);
    else
	    return (string);
}

