/*
 * GNOME panel menu module.
 * (C) 1997 The Free Software Foundation
 *
 * Author: Miguel de Icaza
 *
 * Exported interfaces to the GNOME Panel
 */

#include <stdio.h>
#include <libintl.h>
#include <sys/stat.h>
#include <dlfcn.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include "gnome.h"
#include "../applet.h"
#define _(String) gettext(String)

struct item_config {
	char *exec;
	char *icon_base;
	char *docpath;
	char *info;
	int  terminal;
	char *type;

	/* These are computed from icon_base */
	char *small_icon;
	char *transparent_icon;
};

#define free_if_empty(x) { if (x) free (x); }

PanelCallback callback;

char *
query (void)
{
	return "Menu";
}

void
activate_app_def (GtkWidget *widget, void *data)
{
	struct item_config *item = data;
	char *command;

	/* This is the quick hack version of this */
	command = g_copy_strings ("(true;", item->exec, ") &", NULL);
	system (command);
	free (command);
}

void
setup_menuitem (GtkWidget *menuitem, GtkWidget *pixmap, char *thisfile)
{
	GtkWidget *label, *hbox, *align;
	char *p = strstr (thisfile, ".desktop");

	if (p)
		*p = 0;

	label = gtk_label_new (thisfile);
	gtk_misc_set_alignment (GTK_MISC(label), 0.0, 0.5);
	gtk_widget_show (label);
	
	hbox = gtk_hbox_new (FALSE, 0);
	align = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
	gtk_container_border_width (GTK_CONTAINER (align), 1);

	gtk_box_pack_start (GTK_BOX (hbox), align, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 4);
	gtk_container_add (GTK_CONTAINER (menuitem), hbox);

	if (pixmap){
		gtk_container_add (GTK_CONTAINER (align), pixmap);
		gtk_widget_set_usize (align, 22, 16);
	} else
		gtk_widget_set_usize (align, 22, 16);
	
	gtk_widget_show (align);
	gtk_widget_show (hbox);
	gtk_widget_show (menuitem);
}

void
free_app_def (GtkWidget *widget, void *data)
{
	struct item_config *item = data;

	free_if_empty (item->exec);
	free_if_empty (item->icon_base);
	free_if_empty (item->docpath);
	free_if_empty (item->info);
	free_if_empty (item->type);
	free_if_empty (item->small_icon);
	free_if_empty (item->transparent_icon);
	free (item);
}

int
is_program_in_path (char *program)
{
	static char *path;
	static char **paths;
	char **p;
	char *f;
	
	if (!path){
		char *p;
		int i, pc = 1;

		path = strdup (getenv ("PATH"));
		for (p = path; *p; p++)
			if (*p == ':')
				pc++;

		paths = (char **) g_malloc (sizeof (char *) * (pc+1));

		for (p = path, i = 0; i < pc; i++){
			paths [i] = strtok (p, ":");
			p = NULL;
		}
		paths [pc] = NULL;
	}
	p = paths;
	while (*p){
		f = g_concat_dir_and_file (*p, program);
		if (g_file_exists (f)){
			free (f);
			return 1;
		}
		free (f);
		p++;
	}
	return 0;
}
	      
struct item_config *
load_app_def (char *file)
{
	struct item_config *newitem;
	char *prefix = g_copy_strings ("=", file, "=/Desktop Entry/", NULL);
	char *exec_file, *try_file, *dot;

	gnome_config_set_prefix (prefix);
	free (prefix);

	exec_file = gnome_config_get_string ("Exec");
	if (!exec_file){
		gnome_config_drop_prefix ();
		return 0;
	}
	try_file = gnome_config_get_string ("TryExec");
	if (try_file){
		if (!is_program_in_path (try_file)){
			free (try_file);
			free (exec_file);
			gnome_config_drop_prefix ();
			return 0;
		}
	}
	free (try_file);
	newitem = g_malloc (sizeof (struct item_config));
	newitem->exec      = exec_file;
	newitem->icon_base = gnome_config_get_string ("Icon");
	newitem->docpath   = gnome_config_get_string ("DocPath");
	newitem->info      = gnome_config_get_string ("Info");
	newitem->terminal  = gnome_config_get_int    ("Terminal");
	newitem->type      = gnome_config_get_string ("Type");

	if (newitem->icon_base && *newitem->icon_base){
		dot = strstr (newitem->icon_base, ".xpm");

		if (dot){
			*dot = 0;

			newitem->small_icon = g_copy_strings (newitem->icon_base,
					      "-small.xpm", NULL);
			newitem->transparent_icon = g_copy_strings (newitem->icon_base,
					      "-transparent.xpm", NULL);
			*dot = '.';
		}

		/* Sigh, now we need to make them local to the gnome install */
		if (*newitem->icon_base != '/'){
			char *s = newitem->small_icon;
			char *t = newitem->transparent_icon;

			newitem->small_icon = gnome_pixmap_file (s);
			newitem->transparent_icon = gnome_pixmap_file (t);
			free (s);
			free (t);
		}
	} else {
		newitem->small_icon = newitem->transparent_icon = 0;
	}
	gnome_config_drop_prefix ();
	return newitem;
}

GtkWidget *
create_menu_at (GtkWidget *window, char *menudir)
{	
	struct item_config *item_info;
	GtkWidget *menu;
	struct dirent *dent;
	struct stat s;
	char *filename;
	DIR *dir;
	int  items = 0;
	
	dir = opendir (menudir);
	if (dir == NULL)
		return NULL;

	menu = gtk_menu_new ();
	
	while ((dent = readdir (dir)) != NULL){
		GtkWidget *menuitem, *sub, *pixmap;
		char *thisfile = dent->d_name, *pixmap_name;

		/* Skip over . and .. */
		if ((thisfile [0] == '.' && thisfile [1] == 0) ||
		    (thisfile [0] == '.' && thisfile [1] == '.' && thisfile [2] == 0))
			continue;

		filename = g_concat_dir_and_file (menudir, thisfile);
		if (stat (filename, &s) == -1){
			free (filename);
			continue;
		}

		sub = 0;
		item_info = 0;
		if (S_ISDIR (s.st_mode)){
			sub = create_menu_at (window, filename);
			if (!sub){
				free (filename);
				continue;
			}
			/* just for now */
			pixmap_name = 0;
		} else {
			if (strstr (filename, ".desktop") == 0){
				free (filename);
				continue;
			}
			item_info = load_app_def (filename);
			if (!item_info){
				free (filename);
				continue;
			}
			pixmap_name = item_info->small_icon;

		}
		items++;
		menuitem = gtk_menu_item_new ();
		if (sub)
			gtk_menu_item_set_submenu (GTK_MENU_ITEM(menuitem), sub);

		pixmap = 0;
		if (pixmap_name && g_file_exists (pixmap_name)){
			pixmap = gnome_create_pixmap_widget (window, menuitem, item_info->small_icon);
			if (pixmap)
				gtk_widget_show (pixmap);
		}
		setup_menuitem (menuitem, pixmap, thisfile);

		gtk_menu_append (GTK_MENU (menu), menuitem);
		gtk_signal_connect (GTK_OBJECT (menuitem), "destroy", (GtkSignalFunc) free_app_def, item_info);
		gtk_signal_connect (GTK_OBJECT (menuitem), "activate", (GtkSignalFunc) activate_app_def, item_info);
		
		free (filename);
	}
	closedir (dir);

	if (items == 0){
		gtk_widget_destroy (menu);
		return 0;
	}
	return menu;
}

void
menu_position (GtkMenu *menu, gint *x, gint *y, gpointer data)
{
	GtkWidget *widget = (GtkWidget *) data;
	int wx, wy;
	
	gdk_window_get_origin (widget->window, &wx, &wy);
	
	*x = wx;
	*y = wy - GTK_WIDGET (menu)->allocation.height;
}

void
activate_menu (GtkWidget *widget, void *closure)
{
        GtkWidget *menu = closure ;

	gtk_menu_popup (GTK_MENU (menu), 0, 0, menu_position, widget, 1, 0);
}

void
panel_configure (GtkWidget *widget, void *data)
{
	printf ("Panel configuration\n");
}

void
panel_reload (GtkWidget *widget, void *data)
{
}

GtkWidget *
create_panel_submenu (void)
{
	GtkWidget *menu, *menuitem;

	menu = gtk_menu_new ();
	
	menuitem = gtk_menu_item_new ();
	setup_menuitem (menuitem, 0, _("Configure"));
	gtk_menu_append (GTK_MENU (menu), menuitem);
        gtk_signal_connect (GTK_OBJECT (menuitem), "activate", (GtkSignalFunc) panel_configure, 0);

	menuitem = gtk_menu_item_new ();
	setup_menuitem (menuitem, 0, _("Reload configuration"));
	gtk_menu_append (GTK_MENU (menu), menuitem);
        gtk_signal_connect (GTK_OBJECT (menuitem), "activate", (GtkSignalFunc) panel_reload, 0);

	return menu;
}

void
panel_lock (GtkWidget *widget, void *data)
{
	system ("gnome-lock");
}

void
panel_logout (GtkWidget *widget, void *data)
{
	PanelCommand cmd;

	cmd.cmd = PANEL_CMD_QUIT;
}

void
add_menu_separator (GtkWidget *menu)
{
	GtkWidget *menuitem;
	
	menuitem = gtk_menu_item_new ();
	gtk_widget_show (menuitem);
	gtk_container_add (GTK_CONTAINER (menu), menuitem);
}

void
add_special_entries (GtkWidget *menu)
{
	GtkWidget *menuitem;
	
	/* Panel entry */
	menuitem = gtk_menu_item_new ();

	add_menu_separator (menu);

	setup_menuitem (menuitem, 0, _("Panel"));
	gtk_menu_append (GTK_MENU (menu), menuitem);
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem), create_panel_submenu ());

	add_menu_separator (menu);
	
	menuitem = gtk_menu_item_new ();
	setup_menuitem (menuitem, 0, _("Lock screen"));
	gtk_menu_append (GTK_MENU (menu), menuitem);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate", (GtkSignalFunc) panel_lock, 0);

	menuitem = gtk_menu_item_new ();
	setup_menuitem (menuitem, 0, _("Log out"));
	gtk_menu_append (GTK_MENU (menu), menuitem);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate", (GtkSignalFunc) panel_logout, 0);
}

GtkWidget *
create_panel_menu (GtkWidget *window, char *menudir, int main_menu)
{
	GtkWidget *vbox, *button, *pixmap, *button_pixmap, *menu;
	static char *button_pixmap_name;
	char *pixmap_name;

	if (main_menu)
		pixmap_name = gnome_unconditional_pixmap_file ("panel-menu-main.xpm");
	else
		pixmap_name = gnome_unconditional_pixmap_file ("panel-folder.xpm");
		
	if (!button_pixmap_name)
		button_pixmap_name = gnome_pixmap_file ("arrowup.xpm");

	vbox   = gtk_vbox_new (FALSE, 0);
	button = gtk_button_new ();
	gtk_widget_show (vbox);
	gtk_widget_show (button);

	button_pixmap = gnome_create_pixmap_widget (window, button, button_pixmap_name);
	pixmap        = gnome_create_pixmap_widget (window, button, pixmap_name);

	gtk_container_add (GTK_CONTAINER (button), button_pixmap);
	gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), pixmap, FALSE, FALSE, 0);

	gtk_widget_show (button_pixmap);
	gtk_widget_show (button);
	gtk_widget_show (pixmap);
	gtk_widget_show (vbox);

	menu = create_menu_at (window, menudir);
	if (main_menu)
		add_special_entries (menu);
	gtk_signal_connect (GTK_OBJECT (button), "clicked", (GtkSignalFunc) activate_menu, menu);

	free (pixmap_name);
	return vbox;
}

GtkWidget *
create_menu_widget (GtkWidget *window, char *arguments, char *menudir)
{
	GtkWidget *menu;
	int main_menu;
	
	main_menu = (strcmp (arguments, ".") == 0);
	menu = create_panel_menu (window, menudir, main_menu);
	return menu;
}

void
init (PanelCallback cback, Panel *panel, char *params, int xpos, int ypos)
{
	GtkWidget *menu_component;
	char *menu_base = gnome_unconditional_datadir_file ("apps");
	char *this_menu = g_concat_dir_and_file (menu_base, params);
	PanelCommand cmd;

	if (!getenv ("PATH"))
		return;

	if (!g_file_exists (this_menu)){
		free (menu_base);
		free (this_menu);
		return;
	}

	callback = cback;
	menu_component = create_menu_widget (panel->window, params, this_menu);

	cmd.cmd = PANEL_CMD_REGISTER_TOY;
	cmd.params.register_toy.applet = menu_component;
	cmd.params.register_toy.xpos   = xpos;
	cmd.params.register_toy.ypos   = ypos;
	cmd.params.register_toy.flags  = 0;

	(*callback) (panel, &cmd);
}

#if 0
main (int argc, char *argv [])
{
	GtkWidget *window;
	GtkWidget *thing;
	void *lib_handle;
	char *f;
	void *(*init_ptr)(GtkWidget *, char *);

	gtk_init (&argc, &argv);
	gnome_init (&argc, &argv);

	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	thing = init (window, ".");
	if (!thing)
		return printf ("Module was not initialized\n");
	
	gtk_container_add (GTK_CONTAINER (window), (GtkWidget *) thing);
	gtk_widget_set_usize (window, 48, 48);
	gtk_window_set_policy (window, 0, 0, 1);
	gtk_widget_show (window);
	gtk_widget_realize (window);
	gtk_main ();
	
	return 0;
}
#endif
