/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */

/*
 *  Medusa
 *
 *  Copyright (C) 2001 Eazel, Inc.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Rebecca Schulman <rebecka@eazel.com>
 *
 *  medusa-system-state.c -- API to turn off medusa's daily running, turn it back on,
 *  and check if medusa has been blocked on a system.
 */

#include <fcntl.h>
#include <gtk/gtkmain.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <config.h>

#include "medusa-lock.h"
#include "medusa-system-state.h"
#include "medusa-stdio-extensions.h"
#include "medusa-utils.h"

#define USERS_FOR_MEDUSA_FILE "enabled_users_file"
#define USERS_FOR_MEDUSA_PATH MEDUSA_SHAREDSTATEDIR "/" USERS_FOR_MEDUSA_FILE
#define USERS_FOR_MEDUSA_LOCK_PATH "/tmp/" USERS_FOR_MEDUSA_FILE
#define MEDUSA_SYSTEM_CONFIGURATION_FILE "medusa.conf"
#define MEDUSA_SYSTEM_CONFIGURATION_PATH MEDUSA_SYSCONFDIR "/" MEDUSA_SYSTEM_CONFIGURATION_FILE
#define MEDUSA_SYSTEM_CONFIGURATION_LOCK_PATH "/tmp/" MEDUSA_SYSTEM_CONFIGURATION_FILE
#define SYSTEM_STATE_POLL_INTERVAL 3000

 
typedef struct UpdateClosure {
        MedusaSystemStateFunc update;
        MedusaSystemState system_states;
        gpointer update_argument;
        int update_function_id;
        gboolean execute_only_once;
} UpdateClosure;

typedef struct SystemStatePoll {
        MedusaSystemState current_state;
        GList *updates;
        int check_for_state_changes_timeout_id;
} SystemStatePoll;



static SystemStatePoll *system_state_poll;
static int current_function_id;

static char **
get_users_with_medusa_already_on_from_system_file (void)
{
        char *file_contents;
        char **file_lines, **usernames;
        int i, username_count;

        if (medusa_read_whole_file (USERS_FOR_MEDUSA_PATH, &file_contents) < 0) {
                return NULL;
        }
        
        file_lines = g_strsplit (file_contents, "\n", 0);
        g_free (file_contents);

        username_count = 0;
        for (i = 0; file_lines[i] != NULL; i++) {
                if (file_lines[i][0] != '#') {
                        username_count++;
                }
        }
        
        usernames = g_new0 (char *, username_count + 1);

        username_count = 0;
        for (i = 0; file_lines[i] != NULL; i++) {
                if (file_lines[i][0] != '#') {
                        usernames[username_count++] = g_strdup (file_lines[i]);
                }
        }
                
        usernames[username_count] = NULL;

        g_strfreev (file_lines);
        
        return usernames;
}

static void
write_name_and_newline_to_system_file (const char *username, 
                                       FILE *fp)
{
        g_assert (username != NULL);
        g_assert (fp != NULL);

        fwrite (username, sizeof (char), strlen (username), fp);
        fwrite ("\n", sizeof (char), 1, fp);
}

static void
write_explanatory_comment_to_configuration_file (FILE *fp)
{
        const char *explanation_for_the_enabled_users_file = 
                "# This file contains a list of users who've elected to "
                "enable medusa on this system.\n"
                "# If this list is empty, medusa is not enabled.\n";

        g_assert (fp != NULL);
        
        fwrite (explanation_for_the_enabled_users_file,
                sizeof (char),
                strlen (explanation_for_the_enabled_users_file),
                fp);
}

static void
write_users_with_medusa_enabled_to_system_file (gboolean current_user_wants_medusa_enabled)
{
        char *username;
        char **users_with_medusa_enabled;
        FILE *fp;
        int i;
        
        username = g_get_user_name ();
        if (username == NULL) {
                return;
        }

        users_with_medusa_enabled = get_users_with_medusa_already_on_from_system_file ();


        fp = fopen (USERS_FOR_MEDUSA_PATH, "w");
        if (fp == NULL) {
                return;
        }
        
        write_explanatory_comment_to_configuration_file (fp);
        
        for (i = 0; users_with_medusa_enabled != NULL && users_with_medusa_enabled[i] != NULL; i++) {
                if (users_with_medusa_enabled[i][0] == '#' ||
                    strcmp (users_with_medusa_enabled[i], username) == 0) {
                        continue;
                }
                write_name_and_newline_to_system_file (users_with_medusa_enabled[i], fp);
        }

        if (current_user_wants_medusa_enabled) {
                write_name_and_newline_to_system_file (username, fp);
        }
        
        fclose (fp);
        g_strfreev (users_with_medusa_enabled);
 
        return;
}


gboolean
medusa_enable_medusa_services (gboolean turn_services_on)
{
        MedusaWriteLock *write_lock;
        
        if (medusa_system_services_are_blocked ()) {
                return FALSE;
        }
        
        /* There is no file with enabled users */
        if (access (USERS_FOR_MEDUSA_PATH, F_OK) == -1) {
                if (creat (USERS_FOR_MEDUSA_PATH, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH) == -1) {
                        return FALSE;
                }
        }
        
        write_lock = medusa_write_lock_unprivileged_get_with_timeout (USERS_FOR_MEDUSA_LOCK_PATH, 1000);
        if (write_lock == NULL) {
                return FALSE;
        }

        write_users_with_medusa_enabled_to_system_file (turn_services_on);
        medusa_write_lock_release (write_lock);
        
        return TRUE;
}

static char **
get_lock_and_read_enabled_users_file (void)
{
       MedusaReadLock *read_lock;
       char **enabled_users;
       
       
        read_lock = medusa_read_lock_unprivileged_get_with_timeout (USERS_FOR_MEDUSA_LOCK_PATH, 1000);
        if (read_lock == NULL) {
                return NULL;
        }
        
        enabled_users = get_users_with_medusa_already_on_from_system_file ();
        medusa_read_lock_release (read_lock);
        
        return enabled_users;
}

gboolean    
medusa_system_services_are_enabled (void)
{
        char **enabled_users;
        gboolean one_user_has_enabled_medusa;

        if (medusa_system_services_are_blocked ()) {
                return FALSE;
        }

        enabled_users = get_lock_and_read_enabled_users_file ();

        /* Should we check if it's a real username here, just to be sure? */
        one_user_has_enabled_medusa = enabled_users != NULL &&
                enabled_users[0] != NULL;

        g_strfreev (enabled_users);

        return one_user_has_enabled_medusa;
}

gboolean 
medusa_system_services_have_been_enabled_by_user (const char *username)
{
        char **enabled_users;
        int i;
        gboolean user_has_turned_medusa_on;
        
        g_return_val_if_fail (username != NULL, FALSE);

        enabled_users = get_lock_and_read_enabled_users_file ();
        if (enabled_users == NULL) {
                return FALSE;
        }

        user_has_turned_medusa_on = FALSE;
        
        for (i = 0; enabled_users[i] != NULL; i++) {
                if (strcmp (enabled_users[i], username) == 0) {
                        user_has_turned_medusa_on = TRUE;
                }
        }
        
        g_strfreev (enabled_users);

        return user_has_turned_medusa_on;
}

gboolean    
medusa_system_services_are_blocked (void)
{
        char *configuration_file;
        char **configuration_lines;
        const char *blocked_value;
        gboolean medusa_is_blocked;
        MedusaReadLock *read_lock;
        int i;
        
        
        read_lock = medusa_read_lock_unprivileged_get_with_timeout (MEDUSA_SYSTEM_CONFIGURATION_LOCK_PATH, 1000);
        if (read_lock == NULL) {
                return FALSE;
        }

        if (medusa_read_whole_file (MEDUSA_SYSTEM_CONFIGURATION_PATH, &configuration_file) < 0) {
                return FALSE;
        }
        medusa_read_lock_release (read_lock);

        configuration_lines = g_strsplit (configuration_file, "\n", 0);
        g_free (configuration_file);

        medusa_is_blocked = FALSE;
        for (i = 0; configuration_lines[i] != NULL; i++) {
                if (strncasecmp (configuration_lines[i], "blocked=", strlen ("blocked=")) == 0) {
                        blocked_value = configuration_lines[i] + strlen ("blocked=");
                        if (strncasecmp (blocked_value, "yes", strlen ("yes")) == 0 ||
                            strncasecmp (blocked_value, "true", strlen ("true")) == 0) {
                                medusa_is_blocked = TRUE;
                        }
                }
        }

        g_strfreev (configuration_lines);

        return medusa_is_blocked;
}

static void
call_callback_if_checking_for_new_state (gpointer list_data,
                                         gpointer callback_data)
{
        UpdateClosure *update_closure;
        MedusaSystemState system_states;

        g_assert (list_data != NULL);

        
        update_closure = (UpdateClosure *) list_data;
        system_states = GPOINTER_TO_INT (callback_data);
        
        if (update_closure->system_states & system_states) {
                update_closure->update (update_closure->update_argument);
        }
}

static gboolean
is_a_onetime_callback_that_was_run (gpointer list_data,
                                    gpointer callback_data)
{
        UpdateClosure *update_closure;
        MedusaSystemState system_states;

        g_assert (list_data != NULL);
        
        update_closure = (UpdateClosure *) list_data;
        system_states = GPOINTER_TO_INT (callback_data);

        return update_closure->system_states & update_closure->execute_only_once;
}

static MedusaSystemState 
get_system_state (void)
{
        if (medusa_system_services_are_blocked ()) {
                return MEDUSA_SYSTEM_STATE_BLOCKED;
        }
        else if (medusa_system_services_are_enabled ()) {
                return MEDUSA_SYSTEM_STATE_ENABLED;
        }

        return MEDUSA_SYSTEM_STATE_DISABLED;
}

static gboolean
check_for_system_state_changes (gpointer data)
{
        MedusaSystemState new_system_state;

        new_system_state = get_system_state ();

        if (new_system_state != system_state_poll->current_state) {
                /* Call the callbacks that want this update. */
                g_list_foreach (system_state_poll->updates,
                                call_callback_if_checking_for_new_state, GINT_TO_POINTER (new_system_state));
                system_state_poll->updates = medusa_g_list_remove_deep_custom (system_state_poll->updates,
                                                                               is_a_onetime_callback_that_was_run, 
                                                                               g_free, 
                                                                               GINT_TO_POINTER (new_system_state));
                system_state_poll->current_state = new_system_state;
        }
        
        return TRUE;
}

static int 
medusa_setup_callback_for_system_state_change (MedusaSystemState states_that_should_cause_an_update,
                                               MedusaSystemStateFunc callback,
                                               gpointer callback_data,
                                               gboolean execute_callback_only_once)
{
        UpdateClosure *callback_closure;
        
        g_return_val_if_fail (states_that_should_cause_an_update != 0, 0);

        if (system_state_poll == NULL) {
                system_state_poll = g_new0 (SystemStatePoll, 1);
                system_state_poll->current_state = get_system_state ();
                system_state_poll->check_for_state_changes_timeout_id = gtk_timeout_add (SYSTEM_STATE_POLL_INTERVAL,
                                                                                         check_for_system_state_changes,
                                                                                         system_state_poll);
        }        

        callback_closure = g_new0 (UpdateClosure, 1);
        callback_closure->update = callback;
        callback_closure->update_argument = callback_data;
        callback_closure->update_function_id = ++current_function_id;
        callback_closure->execute_only_once = execute_callback_only_once;
        callback_closure->system_states = states_that_should_cause_an_update;
        
        system_state_poll->updates = 
                g_list_prepend (system_state_poll->updates,
                                callback_closure);
        
        return callback_closure->update_function_id;
}

int
medusa_execute_when_system_state_changes (MedusaSystemState states_that_should_cause_an_update,
                                          MedusaSystemStateFunc callback,
                                          gpointer callback_data)
{
        return medusa_setup_callback_for_system_state_change (states_that_should_cause_an_update,
                                                              callback,
                                                              callback_data,
                                                              FALSE);
}

int
medusa_execute_once_when_system_state_changes (MedusaSystemState states_that_should_cause_an_update,
                                               MedusaSystemStateFunc callback,
                                               gpointer callback_data)
{
        return medusa_setup_callback_for_system_state_change (states_that_should_cause_an_update,
                                                              callback,
                                                              callback_data,
                                                              TRUE);
}


static gboolean
has_update_id (gpointer list_data,
               gpointer callback_data)
{
        UpdateClosure *update_closure;
        int function_id_to_match;
               
        g_assert (list_data != NULL);

        update_closure = (UpdateClosure *) list_data;
        function_id_to_match = GPOINTER_TO_INT (callback_data);
        
        return (update_closure->update_function_id == function_id_to_match);
}

static void
system_state_poll_destroy_and_set_to_null (void)
{
        g_assert (system_state_poll->updates == NULL);

        gtk_timeout_remove (system_state_poll->check_for_state_changes_timeout_id);
        g_free (system_state_poll);
        system_state_poll = NULL;
}


void
medusa_remove_state_changed_function (int update_function_id)
{
        if (system_state_poll == NULL) {
                return;
        }

        system_state_poll->updates = medusa_g_list_remove_deep_custom (system_state_poll->updates,
                                                                       has_update_id,
                                                                       g_free,
                                                                       GINT_TO_POINTER (update_function_id));
        if (system_state_poll->updates == NULL) {
                system_state_poll_destroy_and_set_to_null ();
        }
}


