/* GAIL - The GNOME Accessibility Implementation Library
 * Copyright 2001 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "gailwindow.h"
#include "gailwindowfactory.h"
#include "gailtoplevel.h"

static void gail_window_class_init (GailWindowClass *klass);

static void                  gail_window_real_initialize (AtkObject    *obj,
                                                          gpointer     data);

static G_CONST_RETURN gchar* gail_window_get_name       (AtkObject     *accessible);

static AtkObject*            gail_window_get_parent     (AtkObject     *accessible);
static gint                  gail_window_get_index_in_parent (AtkObject *accessible);
static gboolean              gail_window_real_focus_gtk (GtkWidget     *widget,
                                                         GdkEventFocus *event);

static AtkStateSet*          gail_window_ref_state_set  (AtkObject     *accessible);

static gboolean              gail_window_window_state_event_gtk
                                                        (GtkWidget           *widget,
                                                         GdkEventWindowState *event);

static gpointer parent_class = NULL;

GType
gail_window_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo tinfo =
      {
        sizeof (GailWindowClass),
        (GBaseInitFunc) NULL, /* base init */
        (GBaseFinalizeFunc) NULL, /* base finalize */
        (GClassInitFunc) gail_window_class_init, /* class init */
        (GClassFinalizeFunc) NULL, /* class finalize */
        NULL, /* class data */
        sizeof (GailWindow), /* instance size */
        0, /* nb preallocs */
        (GInstanceInitFunc) NULL, /* instance init */
        NULL /* value table */
      };
  
      type = g_type_register_static (GAIL_TYPE_CONTAINER,
                                     "GailWindow", &tinfo, 0);
    }

  return type;
}

static void
gail_window_class_init (GailWindowClass *klass)
{
  GailWidgetClass *widget_class;
  AtkObjectClass  *class = ATK_OBJECT_CLASS (klass);

  widget_class = (GailWidgetClass*)klass;
  widget_class->focus_gtk = gail_window_real_focus_gtk;

  parent_class = g_type_class_peek_parent (klass);

  class->get_name = gail_window_get_name;
  class->get_parent = gail_window_get_parent;
  class->get_index_in_parent = gail_window_get_index_in_parent;
  class->ref_state_set = gail_window_ref_state_set;
  class->initialize = gail_window_real_initialize;
}

AtkObject*
gail_window_new (GtkWidget *widget)
{
  GObject *object;
  AtkObject *accessible;

  g_return_val_if_fail (widget != NULL, NULL);
  /*
   * A GailWindow can be created for a GtkHandleBox or a GtkWindow
   */
  if (!GTK_IS_WINDOW (widget) &&
      !GTK_IS_HANDLE_BOX (widget))
    g_return_val_if_fail (FALSE, NULL);

  object = g_object_new (GAIL_TYPE_WINDOW, NULL);

  accessible = ATK_OBJECT (object);
  atk_object_initialize (accessible, widget);


  if (GTK_IS_FILE_SELECTION (widget))
    accessible->role = ATK_ROLE_FILE_CHOOSER;
  else if (GTK_IS_DIALOG (widget))
    /*
     * GtkDialog
     * GtkColorSelectionDialog
     * GtkFontSelectionDialog
     */
    accessible->role = ATK_ROLE_DIALOG;
  else if (GTK_IS_WINDOW (widget))
    accessible->role = ATK_ROLE_FRAME;
  else if (GTK_IS_HANDLE_BOX (widget))
    accessible->role = ATK_ROLE_UNKNOWN;
  else
    accessible->role = ATK_ROLE_INVALID;

  return accessible;
}

static void
gail_window_real_initialize (AtkObject *obj,
                             gpointer  data)
{
  ATK_OBJECT_CLASS (parent_class)->initialize (obj, data);

  g_signal_connect (data,
                    "window_state_event",
                    G_CALLBACK (gail_window_window_state_event_gtk),
                    NULL);
}

static G_CONST_RETURN gchar*
gail_window_get_name (AtkObject *accessible)
{
  if (accessible->name)
    return accessible->name;

  else
    {
      /*
       * Get the window title if it exists
       */
      GtkWidget* widget = GTK_ACCESSIBLE (accessible)->widget; 

      if (widget == NULL)
        /*
         * State is defunct
         */
        return NULL;

      g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

      if (GTK_IS_WINDOW (widget))
        {
          GtkWindow *window = GTK_WINDOW (widget);
          G_CONST_RETURN gchar* name;
 
          name = gtk_window_get_title (window);
          if (name)
            return name;
        }
      return ATK_OBJECT_CLASS (parent_class)->get_name (accessible);
    }
}

static AtkObject*
gail_window_get_parent (AtkObject *accessible)
{
  return atk_get_root ();
}

static gint
gail_window_get_index_in_parent (AtkObject *accessible)
{
  GtkWidget* widget = GTK_ACCESSIBLE (accessible)->widget; 
  AtkObject* atk_obj = atk_get_root ();
  GailToplevel* toplevel = GAIL_TOPLEVEL (atk_obj);
  guint window_count = -1;

  if (widget == NULL)
    /*
     * State is defunct
     */
    return -1;

  g_return_val_if_fail (GTK_IS_WIDGET (widget), -1);

  if (GTK_IS_WINDOW (widget))
    {
      GtkWindow *window = GTK_WINDOW (widget);

      window_count = g_list_index (toplevel->window_list, window);
    }
  return window_count;
}

static gboolean
gail_window_real_focus_gtk (GtkWidget     *widget,
                            GdkEventFocus *event)
{
  AtkObject* obj;

  obj = gtk_widget_get_accessible (widget);
  atk_object_notify_state_change (obj, ATK_STATE_ACTIVE, event->in);

  return FALSE;
}

static AtkStateSet*
gail_window_ref_state_set (AtkObject *accessible)
{
  AtkStateSet *state_set;
  GtkWidget *widget;
  GtkWindow *window;
  GdkWindowState state;
  GValue value = { 0, };

  state_set = ATK_OBJECT_CLASS (parent_class)->ref_state_set (accessible);
  widget = GTK_ACCESSIBLE (accessible)->widget;
 
  if (widget == NULL)
    return state_set;

  window = GTK_WINDOW (widget);

  if (window->has_focus)
    atk_state_set_add_state (state_set, ATK_STATE_ACTIVE);

  if (widget->window)
    {
      state = gdk_window_get_state (widget->window);
      if (state & GDK_WINDOW_STATE_ICONIFIED)
        atk_state_set_add_state (state_set, ATK_STATE_ICONIFIED);
    } 
  if (gtk_window_get_modal (window))
    atk_state_set_add_state (state_set, ATK_STATE_MODAL);

  g_value_init (&value, G_TYPE_BOOLEAN);
  g_object_get_property (G_OBJECT (window), "resizable", &value);
  if (g_value_get_boolean (&value))
    atk_state_set_add_state (state_set, ATK_STATE_RESIZABLE);
 
  return state_set;
}

static gboolean
gail_window_window_state_event_gtk (GtkWidget           *widget,
                                    GdkEventWindowState *event)
{
  AtkObject* obj;

  obj = gtk_widget_get_accessible (widget);
  atk_object_notify_state_change (obj, ATK_STATE_ICONIFIED,
                         event->new_window_state & GDK_WINDOW_STATE_ICONIFIED);
  return FALSE;
}
