/*
 *
 * gtk-config-gui.c :
 *
 * Copyright (C) 2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <tvgm@ximian.com>
 *
 */

#include <mtm/mtm-config-gui.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <libgnome/libgnome.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <utime.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <dirent.h>

#include "gtk-config-gui.h"

/* A good deal of this code lifted from doorman */

typedef struct 
{
	int out_fd;
	gchar *gtkrc_path;
	pid_t demo_child;
	MtmExt *ext;
	gboolean realized;
} ConfigPreview;

static void update_display (ConfigPreview *preview);
static void preview_gtk_theme (ConfigPreview *preview, gchar *theme);

static gchar**
generate_file_list (void)
{
	GArray *files;
	gchar **ret;
	/* FIXME */
	gchar *dirnames[] = { "/usr/share/themes", "/.themes", NULL};
	int i;

	dirnames[1] = g_strconcat (g_get_home_dir (), dirnames[1], NULL);
	
	files = g_array_new (TRUE, TRUE, sizeof (gchar*));
	
	for (i = 0; dirnames[i] != NULL; i++)
	{
		DIR *dir;
		struct dirent *de;
		dir = opendir (dirnames[i]);
		if (!dir)
			continue;

		while ((de = readdir (dir)))
		{
			gchar *tmp;
			gchar *filename;
			struct stat buf;
			
			if (de->d_name[0] == '.')
				continue;
			
			filename = g_strconcat (dirnames[i], "/", de->d_name, NULL);
			tmp = g_strconcat (filename, "/gtk/gtkrc", NULL);
			if (stat (tmp, &buf) == 0) 
				g_array_append_val (files, filename);
			else
				g_free (filename);

			g_free (tmp);
		}
		
		closedir (dir);
	}


	g_free (dirnames[1]);
	ret = files->data;
	g_array_free (files, FALSE);

	return ret;
}

static void
ext_changed_cb (MtmConfigGui *gui, MtmExt *ext, ConfigPreview *preview)
{
	g_return_if_fail (MTM_IS_EXT (ext));
	if (!preview->realized)
		preview->ext = ext;
	else
		preview_gtk_theme (preview, ext->file);
}
		
static void
destroy_cb (GtkWidget *widget, ConfigPreview *preview)
{
	close (preview->out_fd);
	unlink (preview->gtkrc_path);
	g_free (preview->gtkrc_path);
	g_free (preview);
}

static void
realize_cb (GtkWidget *demo_socket, ConfigPreview *preview)
{
	guint32 id;
  gchar *id_name;
  gchar *fd_name;
  int fds[2];
  int fd;
	pid_t demo_child;

	g_assert (GTK_WIDGET_REALIZED (demo_socket));

	preview->realized = TRUE;

	id = GDK_WINDOW_XWINDOW (demo_socket->window);

        preview->gtkrc_path = g_strdup_printf (
                "%s/mtm-gtkrc-XXXXXX",
                getenv ("TMPDIR") ? getenv ("TMPDIR") : "/tmp");
        fd = mkstemp (preview->gtkrc_path);
        close (fd);

        pipe (fds);
        fcntl (fds[0], F_SETFL, O_NONBLOCK);
        fcntl (fds[1], F_SETFL, O_NONBLOCK);

        signal (SIGPIPE, SIG_DFL);
        signal (SIGCHLD, SIG_DFL);

        demo_child = fork ();

        if (demo_child == -1) {
                /* umm... */
                exit (-1);
        }

        if (demo_child) {
                close (fds[0]);
                preview->out_fd = fds[1];
		preview->demo_child = demo_child;
		gdk_flush ();
		if (preview->ext)
		{
			preview_gtk_theme (preview, preview->ext->file);	
			preview->ext = NULL;
		}
                return;
        }

        close (fds[1]);

        fd_name = g_strdup_printf ("%d", fds[0]);
        id_name = g_strdup_printf ("0x%x", id);

        close (STDOUT_FILENO);
        close (STDERR_FILENO);
        close (STDIN_FILENO);

        execl ("./themehelper", "themehelper", id_name, preview->gtkrc_path, fd_name,
               NULL);
        execl (BINDIR "/themehelper", "themehelper", id_name, preview->gtkrc_path,
               fd_name, NULL);
}

MtmConfigGui*
gtk_config_gui_new (MtmGuiHandler *handler)
{
        GtkWidget *demo_socket = NULL;
	GtkWidget *frame, *vbox;
	MtmConfigGui *gui;
	ConfigPreview *preview;
	gchar **files;

        demo_socket = gtk_socket_new ();
        gtk_widget_show (demo_socket);

	gui = mtm_config_gui_new ();
	
	files = generate_file_list ();
	mtm_config_gui_set_file_list (gui, files);
	g_strfreev (files);
			
	preview = g_new0 (ConfigPreview, 1);
	preview->realized = FALSE;
	preview->ext = NULL;
	gtk_signal_connect_after (GTK_OBJECT (gui), "destroy", destroy_cb, preview);
	gtk_signal_connect (GTK_OBJECT (gui), "set_ext", ext_changed_cb, preview);
	gtk_signal_connect (GTK_OBJECT (gui), "ext_modified", ext_changed_cb, preview);

	vbox = gtk_vbox_new (FALSE, 0);
	frame = gtk_frame_new (_("Preview"));
        gtk_container_add (GTK_CONTAINER (frame), demo_socket);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
	mtm_config_gui_set_config_area (gui, vbox); 
	gtk_widget_show_all (vbox);

	gtk_signal_connect_after (GTK_OBJECT (demo_socket), "realize", realize_cb, preview);

	return gui;
}

static void
update_display (ConfigPreview *preview)
{
        static time_t last_written_time = 0;
        time_t current_time = time (NULL);
        struct utimbuf buf;

        if (last_written_time >= current_time) {
                current_time = last_written_time + 1;
                buf.actime = current_time;
                buf.modtime = current_time;
                utime (preview->gtkrc_path, &buf);
        }

        last_written_time = current_time;

        write (preview->out_fd, "R ", strlen ("R "));
}

static void
preview_gtk_theme (ConfigPreview *preview, gchar *theme)
{
        int fd;
        gchar *line;

	g_return_if_fail (theme != NULL);

        fd = open (preview->gtkrc_path, O_WRONLY | O_TRUNC);
        line = g_strdup_printf ("# -- THEME AUTO-WRITTEN DO NOT EDIT\n"
                                "include \"%s/gtk/gtkrc\"\n\n"
                                "include \"%s/.gtkrc.mine\"\n\n"
                                "# -- THEME AUTO-WRITTEN DO NOT EDIT\n",
                                theme, g_get_home_dir ());
        write (fd, line, strlen (line));
        close (fd);

        g_free (line);

        update_display (preview);
}

#if 0
static void
reset_gtk_theme (void)
{
        gchar *line;

        line = g_strdup_printf ("%s/.gtkrc", g_get_home_dir ());

        if (!exists (line)) {
                g_free (line);
                line = g_strdup (GTK_THEMEDIR "/Default/gtk/gtkrc");
        }

        cp (line, gtkrc_path);

        g_free (line);

        update_display ();
}
#endif
