/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-

   eel-background-style.c: Style subclass used by EelBackground.
 
   Copyright (C) 2001 Bent Spoon Software
  
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
  
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
  
   Author: Darin Adler <darin@bentspoon.com>
*/

#include <config.h>
#include "eel-background-style.h"

#include "eel-gdk-extensions.h"
#include <gtk/gtktreeview.h>
#include <libgnome/gnome-macros.h>

GNOME_CLASS_BOILERPLATE (EelBackgroundStyle, eel_background_style,
			 GtkStyle, GTK_TYPE_STYLE)

struct EelBackgroundStyleDetails {
	int dummy;
};

static void
eel_background_style_instance_init (EelBackgroundStyle *style)
{
	style->details = g_new0 (EelBackgroundStyleDetails, 1);
}

static void
eel_background_style_finalize (GObject *object)
{
	EelBackgroundStyle *style;

	style = EEL_BACKGROUND_STYLE (object);

	g_free (style->details);

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
eel_gdk_window_update_sizes (GdkWindow *window, int *width, int *height)
{
	g_return_if_fail (window != NULL);
	g_return_if_fail (width != NULL);
	g_return_if_fail (height != NULL);

	if (*width == -1 && *height == -1) {
		gdk_window_get_size (window, width, height);
	} else if (*width == -1) {
		gdk_window_get_size (window, width, NULL);
	} else if (*height == -1) {
		gdk_window_get_size (window, NULL, height);
	}
}

static void
eel_background_style_draw_flat_box (GtkStyle *style,
				    GdkWindow *window,
				    GtkStateType state_type,
				    GtkShadowType shadow_type,
				    GdkRectangle *area,
				    GtkWidget *widget,
				    const char *detail,
				    int x,
				    int y,
				    int width,
				    int height)
{
	gboolean call_parent;
	EelBackground *background;
	GdkGC *gc;

	call_parent = TRUE;

	background = NULL;
	if (state_type == GTK_STATE_NORMAL) {
		background = eel_get_widget_background (widget);
		if (background != NULL && !GTK_IS_TREE_VIEW (widget)) {
			call_parent = FALSE;
		}
	}

	if (call_parent) {
		GNOME_CALL_PARENT (GTK_STYLE_CLASS, draw_flat_box,
				   (style, window, state_type, shadow_type, area, widget,
				    detail, x, y, width, height));
		return;
	}

    	gc = style->bg_gc[state_type];

	if (area)
		gdk_gc_set_clip_rectangle (gc, area);

	eel_gdk_window_update_sizes (window, &width, &height);	

	if (area)
		eel_background_draw_to_drawable (background, window, gc,
						 area->x, area->y,
						 area->width, area->height,
						 widget->allocation.width, widget->allocation.height);
	else
		eel_background_draw_to_drawable (background, window, gc,
						 0, 0,
						 width, height,
						 widget->allocation.width, widget->allocation.height);

	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}


EelBackgroundStyle *
eel_background_style_new (GtkStyle *base_style,
			  EelBackground *background)
{
	EelBackgroundStyle *style;
	char *background_color, *start_color_spec;
	GdkColor color;
	
	g_return_val_if_fail (GTK_IS_STYLE (base_style), NULL);
	g_return_val_if_fail (EEL_IS_BACKGROUND (background), NULL);
	
	style = g_object_new (EEL_TYPE_BACKGROUND_STYLE, NULL);

	GTK_STYLE_CLASS (g_type_class_peek (GTK_TYPE_STYLE))->copy (GTK_STYLE (style), base_style);

	background_color = eel_background_get_color (background);
	start_color_spec = eel_gradient_get_start_color_spec (background_color);
	g_free (background_color);
	eel_gdk_color_parse_with_white_default (start_color_spec, &color);
	g_free (start_color_spec);
	
	GTK_STYLE (style)->bg[GTK_STATE_NORMAL] = color;
	GTK_STYLE (style)->base[GTK_STATE_NORMAL] = color;
	GTK_STYLE (style)->bg[GTK_STATE_ACTIVE] = color;
	GTK_STYLE (style)->base[GTK_STATE_ACTIVE] = color;

	return style;
}

static void
eel_background_style_class_init (EelBackgroundStyleClass *class)
{
	G_OBJECT_CLASS (class)->finalize = eel_background_style_finalize;
	GTK_STYLE_CLASS (class)->draw_flat_box = eel_background_style_draw_flat_box;
}
