/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * display.c
 *
 * Copyright (C) 2001 Iain Holmes
 * Authors: Iain Holmes  <iain@ximian.com>
 */

#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include "gnome-cd.h"
#include "cdrom.h"
#include "display.h"

#define X_OFFSET 2
#define Y_OFFSET 2

static GtkDrawingAreaClass *parent_class = NULL;

#define LEFT 0
#define RIGHT 1
#define TOP 2
#define BOTTOM 3

#define TOPLEFT 0
#define TOPRIGHT 1
#define BOTTOMLEFT 2
#define BOTTOMRIGHT 3

typedef struct _CDImage {
	GdkPixbuf *pixbuf;
	GdkPixbuf *scaled;
	GdkRectangle rect;
} CDImage;

struct _CDDisplayPrivate {
	GdkColor red;
	GdkColor blue;

	GnomeCDText *layout[CD_DISPLAY_END];
	int max_width;
	int height;
	int need_height;

	CDImage *corners[4];
	CDImage *straights[4];
	CDImage *middle;

	CDImage *cd, *track, *loop, *once;

	GnomeCDRomMode playmode, loopmode;
};

enum {
	PLAYMODE_CHANGED,
	LOOPMODE_CHANGED,
	LAST_SIGNAL
};

static gulong display_signals[LAST_SIGNAL] = { 0, };

static char *default_text[CD_DISPLAY_END] = {
	"0:00",
	" ",
	N_("Unknown artist"),
	N_("Unknown disc")
};

static void
free_cd_text (GnomeCDText *text)
{
	g_free (text->text);
	g_object_unref (text->layout);

	g_free (text);
}

static void
scale_image (CDImage *image)
{
	if (image->scaled != NULL) {
		g_object_unref (image->scaled);
	}

	image->scaled = gdk_pixbuf_scale_simple (image->pixbuf,
						 image->rect.width, 16,
						 GDK_INTERP_BILINEAR);
}

static void
size_allocate (GtkWidget *drawing_area,
	       GtkAllocation *allocation)
{
	CDDisplay *disp;
	CDDisplayPrivate *priv;
	PangoContext *context;
	PangoDirection base_dir;
	int i, mod;
	
	disp = CD_DISPLAY (drawing_area);
	priv = disp->priv;

	context = pango_layout_get_context (priv->layout[0]->layout);
	base_dir = pango_context_get_base_dir (context);

	for (i = 0; i < CD_DISPLAY_END; i++) {
		PangoRectangle rect;

		pango_layout_set_alignment (priv->layout[i]->layout,
					    base_dir == PANGO_DIRECTION_LTR ? PANGO_ALIGN_LEFT : PANGO_ALIGN_RIGHT);
		pango_layout_set_width (priv->layout[i]->layout, allocation->width * 1000);
		pango_layout_get_extents (priv->layout[i]->layout, NULL, &rect);
		priv->layout[i]->height = rect.height / 1000;
	}

	/* Resize and position pixbufs */
	priv->corners[TOPRIGHT]->rect.x = allocation->width - 16;
	priv->corners[BOTTOMLEFT]->rect.y = allocation->height - 16;
	priv->corners[BOTTOMRIGHT]->rect.x = allocation->width - 16;
	priv->corners[BOTTOMRIGHT]->rect.y = allocation->height - 16;

	priv->straights[BOTTOM]->rect.y = allocation->height - 16;
	priv->straights[RIGHT]->rect.x = allocation->width - 16;

	priv->straights[TOP]->rect.width = allocation->width - 32;
	priv->straights[BOTTOM]->rect.width = allocation->width - 32;
	scale_image (priv->straights[TOP]);
	scale_image (priv->straights[BOTTOM]);

	priv->straights[LEFT]->rect.height = allocation->height - 32;
	priv->straights[RIGHT]->rect.height = allocation->height - 32;

	priv->middle->rect.width = allocation->width - 32;
	priv->middle->rect.height = allocation->height - 32;
	scale_image (priv->middle);
	
	GTK_WIDGET_CLASS (parent_class)->size_allocate (drawing_area, allocation);
}

static void
size_request (GtkWidget *widget,
	      GtkRequisition *requisition)
{
	CDDisplay *disp;
	CDDisplayPrivate *priv;
	int i, height = 0, width = 0, mod;
	
	disp = CD_DISPLAY (widget);
	priv = disp->priv;

	GTK_WIDGET_CLASS (parent_class)->size_request (widget, requisition);
	
	for (i = 0; i < CD_DISPLAY_END; i++) {
		PangoRectangle rect;

		if (i == CD_DISPLAY_LINE_INFO) {
			priv->track->rect.y = height;
			priv->cd->rect.y = height;
			priv->once->rect.y = height;
			priv->loop->rect.y = height;

			height += priv->track->rect.height;
			width = MAX (width, priv->track->rect.width * 2);
		} else {
			pango_layout_get_extents (priv->layout[i]->layout, NULL, &rect);
			height += (rect.height / 1000);
			width = MAX (width, rect.width / 1000);
		}
	}

	requisition->width = width + (2 * X_OFFSET);

	requisition->height = height + (2 * Y_OFFSET);
}

static void
draw_pixbuf (GdkPixbuf *pixbuf,
	     GdkDrawable *drawable,
	     GdkGC *gc,
	     int src_x, int src_y,
	     int dest_x, int dest_y,
	     int width, int height)
{
	gdk_pixbuf_render_to_drawable_alpha (pixbuf,
					     drawable,
					     src_x, src_y,
					     dest_x, dest_y,
					     width, height,
					     GDK_PIXBUF_ALPHA_FULL,
					     128,
					     GDK_RGB_DITHER_NORMAL,
					     0, 0);
}
	     
static gboolean
expose_event (GtkWidget *drawing_area,
	      GdkEventExpose *event)
{
	CDDisplay *disp;
	CDDisplayPrivate *priv;
	PangoContext *context;
	PangoDirection base_dir;
	GdkRectangle *area;
	int height = 0;
	int i;

	disp = CD_DISPLAY (drawing_area);
	priv = disp->priv;

	area = &event->area;

	/* Check corners and draw them */
	for (i = 0; i < 4; i++) {
		GdkRectangle inter;
		
		if (gdk_rectangle_intersect (area, &priv->corners[i]->rect, &inter) == TRUE) {
			draw_pixbuf (priv->corners[i]->pixbuf,
				     drawing_area->window,
				     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
				     inter.x - priv->corners[i]->rect.x,
				     inter.y - priv->corners[i]->rect.y,
				     inter.x, inter.y,
				     inter.width, inter.height);
		}
	}

	for (i = TOP; i <= BOTTOM; i++) {
		GdkRectangle inter;

		if (gdk_rectangle_intersect (&priv->straights[i]->rect, area, &inter) == TRUE) {
			draw_pixbuf (priv->straights[i]->scaled,
				     drawing_area->window,
				     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
				     inter.x - priv->straights[i]->rect.x,
				     inter.y - priv->straights[i]->rect.y,
				     inter.x, inter.y,
				     inter.width, inter.height);
		}
	}

	for (i = LEFT; i <= RIGHT; i++) {
		GdkRectangle inter;
		if (gdk_rectangle_intersect (area, &priv->straights[i]->rect, &inter) == TRUE) {
			int repeats, extra_s, extra_end, d, j;

			d = inter.y / 16;
			extra_s = inter.y - (d * 16);
			
			if (extra_s > 0) {
				draw_pixbuf (priv->straights[i]->pixbuf,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->straights[i]->rect.x,
					     16 - extra_s,
					     inter.x, inter.y,
					     inter.width, extra_s);
			}

			repeats = (inter.height - extra_s) / 16;
			extra_end = (inter.height - extra_s) % 16;
			
			for (j = 0; j < repeats; j++) {
				draw_pixbuf (priv->straights[i]->pixbuf,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->straights[i]->rect.x,
					     0,
					     inter.x,
					     inter.y + (j * 16) + extra_s,
					     inter.width, 16);
			}

			if (extra_end > 0) {
				draw_pixbuf (priv->straights[i]->pixbuf,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->straights[i]->rect.x,
					     0,
					     inter.x,
					     inter.y + (repeats * 16) + extra_s,
					     inter.width, extra_end);
			}
		}
	}
							       
	/* Do the middle - combination of the above */
	{
		GdkRectangle inter;
		if (gdk_rectangle_intersect (area, &priv->middle->rect, &inter) == TRUE) {
			int repeats, extra_s, extra_end, j, d;

			d = inter.y / 16;
			extra_s = inter.y - (d * 16);
			
			if (extra_s > 0) {
				draw_pixbuf (priv->middle->scaled,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->middle->rect.x,
					     16 - extra_s,
					     inter.x, inter.y,
					     inter.width, extra_s);
			}

			repeats = (inter.height - extra_s) / 16;
			extra_end = (inter.height - extra_s) % 16;
			
			for (j = 0; j < repeats; j++) {
				draw_pixbuf (priv->middle->scaled,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->middle->rect.x,
					     0,
					     inter.x,
					     inter.y + (j * 16) + extra_s,
					     inter.width, 16);
			}

			if (extra_end > 0) {
				draw_pixbuf (priv->middle->scaled,
					     drawing_area->window,
					     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
					     inter.x - priv->middle->rect.x,
					     0,
					     inter.x,
					     inter.y + (repeats * 16) + extra_s,
					     inter.width, extra_end);
			}
		}
	}

	/* Do the info line */
	{
		GdkRectangle inter;
		CDImage *im;

		/* Track / CD? */
		im = (priv->playmode == GNOME_CDROM_SINGLE_TRACK ? priv->track : priv->cd);
		if (gdk_rectangle_intersect (area, &im->rect, &inter) == TRUE) {
			draw_pixbuf (im->pixbuf,
				     drawing_area->window,
				     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
				     inter.x - im->rect.x,
				     inter.y - im->rect.y,
				     inter.x, inter.y,
				     inter.width, inter.height);
		}

		im = (priv->loopmode == GNOME_CDROM_PLAY_ONCE ? priv->once : priv->loop);
		if (gdk_rectangle_intersect (area, &im->rect, &inter) == TRUE) {
			draw_pixbuf (im->pixbuf,
				     drawing_area->window,
				     drawing_area->style->bg_gc[GTK_STATE_NORMAL],
				     inter.x - im->rect.x,
				     inter.y - im->rect.y,
				     inter.x, inter.y,
				     inter.width, inter.height);
		}
	}
	
	context = pango_layout_get_context (priv->layout[0]->layout);
	base_dir = pango_context_get_base_dir (context);

	for (i = 0; i < CD_DISPLAY_END &&
		     height < area->y + area->height + Y_OFFSET; i++) {

		if (height + priv->layout[i]->height >= Y_OFFSET + area->y) {
			pango_layout_set_alignment (priv->layout[i]->layout,
						    base_dir == PANGO_DIRECTION_LTR ? PANGO_ALIGN_LEFT : PANGO_ALIGN_RIGHT);

			gdk_draw_layout_with_colors (drawing_area->window,
						     drawing_area->style->black_gc,
						     X_OFFSET, 
						     Y_OFFSET + height,
						     priv->layout[i]->layout,
						     &priv->red, NULL);
		}
		height += priv->layout[i]->height;
	}

	return TRUE;
}

static void
finalize (GObject *object)
{
	CDDisplay *disp;
	CDDisplayPrivate *priv;
	int i;

	disp = CD_DISPLAY (object);
	priv = disp->priv;

	if (priv == NULL) {
		return;
	}

	/* FIXME: Free the CDImages here */
	
	for (i = 0; i < CD_DISPLAY_END; i++) {
		free_cd_text (priv->layout[i]);
	}

	g_free (priv);

	disp->priv = NULL;

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
realize (GtkWidget *widget)
{
	CDDisplay *disp;
	GdkColormap *cmap;

	disp = CD_DISPLAY (widget);

	cmap = gtk_widget_get_colormap (widget);
	disp->priv->red.red = 65535;
	disp->priv->red.green = 0;
	disp->priv->red.blue = 0;
	disp->priv->red.pixel = 0;
	gdk_color_alloc (cmap, &disp->priv->red);

	disp->priv->blue.red = 0;
	disp->priv->blue.green = 0;
	disp->priv->blue.blue = 65535;
	disp->priv->blue.pixel = 0;
	gdk_color_alloc (cmap, &disp->priv->blue);

	gtk_widget_add_events (widget, GDK_BUTTON_PRESS_MASK |
			       GDK_KEY_PRESS_MASK);
	GTK_WIDGET_CLASS (parent_class)->realize (widget);
}

static int
button_press_event (GtkWidget *widget,
		    GdkEventButton *ev)
{
	CDDisplay *disp = CD_DISPLAY (widget);
	CDDisplayPrivate *priv = disp->priv;

	if (ev->button != 1) {
		return FALSE;
	}
	
	/* Check the play mode */
	if (ev->x >= priv->track->rect.x &&
	    ev->x <= (priv->track->rect.x + priv->track->rect.width) &&
	    ev->y >= priv->track->rect.y &&
	    ev->y <= (priv->track->rect.y + priv->track->rect.height)) {
		priv->playmode = (priv->playmode == GNOME_CDROM_SINGLE_TRACK ?
				  GNOME_CDROM_WHOLE_CD : GNOME_CDROM_SINGLE_TRACK);
		g_signal_emit (G_OBJECT (widget), display_signals[PLAYMODE_CHANGED],
			       0, priv->playmode);
		gtk_widget_queue_draw_area (widget, priv->track->rect.x,
					    priv->track->rect.y,
					    priv->track->rect.width,
					    priv->track->rect.height);
		return TRUE;
	}

	/* Check the loop mode */
	if (ev->x >= priv->once->rect.x &&
	    ev->x <= (priv->once->rect.x + priv->once->rect.width) &&
	    ev->y >= priv->once->rect.y &&
	    ev->y <= (priv->once->rect.y + priv->once->rect.height)) {
		priv->loopmode = (priv->loopmode == GNOME_CDROM_PLAY_ONCE ?
				  GNOME_CDROM_LOOP : GNOME_CDROM_PLAY_ONCE);
		g_signal_emit (G_OBJECT (widget), display_signals[LOOPMODE_CHANGED],
			       0, priv->loopmode);
		gtk_widget_queue_draw_area (widget, priv->track->rect.x,
					    priv->track->rect.y,
					    priv->track->rect.width,
					    priv->track->rect.height);
		return TRUE;
	}

	return FALSE;
}
		     
static void
class_init (CDDisplayClass *klass)
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = G_OBJECT_CLASS (klass);
	widget_class = GTK_WIDGET_CLASS (klass);

	object_class->finalize = finalize;

	widget_class->size_allocate = size_allocate;
	widget_class->size_request = size_request;
	widget_class->expose_event = expose_event;
  	widget_class->realize = realize;
	widget_class->button_press_event = button_press_event;
	
	/* Signals */
	display_signals[PLAYMODE_CHANGED] = g_signal_new ("playmode-changed",
							  G_TYPE_FROM_CLASS (klass),
							  G_SIGNAL_RUN_FIRST | G_SIGNAL_NO_RECURSE,
							  G_STRUCT_OFFSET (CDDisplayClass, playmode_changed),
							  NULL, NULL,
							  g_cclosure_marshal_VOID__INT,
							  G_TYPE_NONE,
							  1, G_TYPE_INT);
	display_signals[LOOPMODE_CHANGED] = g_signal_new ("loopmode-changed",
							  G_TYPE_FROM_CLASS (klass),
							  G_SIGNAL_RUN_FIRST | G_SIGNAL_NO_RECURSE,
							  G_STRUCT_OFFSET (CDDisplayClass, loopmode_changed),
							  NULL, NULL,
							  g_cclosure_marshal_VOID__INT,
							  G_TYPE_NONE,
							  1, G_TYPE_INT);
	
	parent_class = g_type_class_peek_parent (klass);
}

static CDImage *
cd_image_new (const char *filename,
	      int x,
	      int y)
{
	CDImage *image;
	char *fullname;
	
	image = g_new (CDImage, 1);

	fullname = gnome_pixmap_file (filename);
	if (fullname == NULL) {
		g_warning ("Error loading %s", filename);
		g_free (image);
		return NULL;
	}
	
	image->pixbuf = gdk_pixbuf_new_from_file (fullname, NULL);
	g_free (fullname);
	
	if (image->pixbuf == NULL) {
		g_warning ("Error loading %s", filename);
		g_free (image);
		return NULL;
	}

	image->scaled = NULL;
	image->rect.x = x;
	image->rect.y = y;
	image->rect.width = gdk_pixbuf_get_width (image->pixbuf);
	image->rect.height = gdk_pixbuf_get_height (image->pixbuf);

	return image;
}

static void
init (CDDisplay *disp)
{
	int i;
	CDDisplayPrivate *priv;

	disp->priv = g_new0 (CDDisplayPrivate, 1);
	priv = disp->priv;

	GTK_WIDGET_UNSET_FLAGS (disp, GTK_NO_WINDOW);

	for (i = 0; i < CD_DISPLAY_END; i++) {
		PangoRectangle rect;
		
		priv->layout[i] = g_new (GnomeCDText, 1);
		
		priv->layout[i]->text = g_strdup (default_text[i]);
		priv->layout[i]->length = strlen (default_text[i]);
		priv->layout[i]->layout = gtk_widget_create_pango_layout (GTK_WIDGET (disp), priv->layout[i]->text);
		pango_layout_set_text (priv->layout[i]->layout,
				       priv->layout[i]->text,
				       priv->layout[i]->length);
		
		pango_layout_get_extents (priv->layout[i]->layout, NULL, &rect);
		priv->layout[i]->height = rect.height / 1000;
		
		priv->max_width = MAX (priv->max_width, rect.width / 1000);
		priv->height += priv->layout[i]->height;
	}

	priv->height += (Y_OFFSET * 2);
	priv->max_width += (X_OFFSET * 2);
	gtk_widget_queue_resize (GTK_WIDGET (disp));

	priv->playmode = GNOME_CDROM_WHOLE_CD;
	priv->loopmode = GNOME_CDROM_PLAY_ONCE;
	
	/* Load pixbufs */
	priv->corners[TOPLEFT] = cd_image_new ("gnome-cd/lcd-theme/top-left.png", 0, 0);
	priv->corners[TOPRIGHT] = cd_image_new ("gnome-cd/lcd-theme/top-right.png", 48, 0);
	priv->corners[BOTTOMLEFT] = cd_image_new ("gnome-cd/lcd-theme/bottom-left.png", 0, 48);
	priv->corners[BOTTOMRIGHT] = cd_image_new ("gnome-cd/lcd-theme/bottom-right.png", 48, 48);

	priv->straights[LEFT] = cd_image_new ("gnome-cd/lcd-theme/middle-left.png", 0, 16);
	priv->straights[RIGHT] = cd_image_new ("gnome-cd/lcd-theme/middle-right.png", 48, 16);
	priv->straights[TOP] = cd_image_new ("gnome-cd/lcd-theme/top.png", 16, 0);
	priv->straights[BOTTOM] = cd_image_new ("gnome-cd/lcd-theme/bottom.png", 16, 48);

	priv->middle = cd_image_new ("gnome-cd/lcd-theme/middle.png", 16, 16);

	/* Don't know where these are to be placed yet */
	priv->track = cd_image_new ("gnome-cd/track.png", 16, 0);
	priv->cd = cd_image_new ("gnome-cd/disc.png", 16, 0);
	priv->loop = cd_image_new ("gnome-cd/repeat.png", 36, 0);
	priv->once = cd_image_new ("gnome-cd/once.png", 36, 0);
}

GType
cd_display_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		GTypeInfo info = {
			sizeof (CDDisplayClass), NULL, NULL,
			(GClassInitFunc) class_init, NULL, NULL,
			sizeof (CDDisplay), 0, (GInstanceInitFunc) init,
		};

		type = g_type_register_static (GTK_TYPE_DRAWING_AREA, "CDDisplay", &info, 0);
	}

	return type;
}

CDDisplay *
cd_display_new (void)
{
	CDDisplay *disp;

	disp = g_object_new (cd_display_get_type (), NULL);
	return CD_DISPLAY (disp);
}

const char *
cd_display_get_line (CDDisplay *disp,
		     int line)
{
	CDDisplayPrivate *priv;
	GnomeCDText *text;

	priv = disp->priv;
	text = priv->layout[line];

	return text->text;
}

void
cd_display_set_line (CDDisplay *disp,
		     CDDisplayLine line,
		     const char *new_str)
{
	CDDisplayPrivate *priv;
	GnomeCDText *text;
	PangoRectangle rect;
	int height;

	g_return_if_fail (disp != NULL);
	g_return_if_fail (new_str != NULL);

	priv = disp->priv;
	
	text = priv->layout[line];
	height = priv->height - text->height;
	
	g_free (text->text);
	text->text = g_strdup (new_str);
	text->length = strlen (new_str);
	pango_layout_set_text (text->layout, text->text, text->length);
	pango_layout_get_extents (text->layout, NULL, &rect);
	text->height = rect.height / 1000;
	
	priv->height = height + text->height;
	priv->max_width = MAX (priv->max_width, rect.width / 1000);

	gtk_widget_queue_resize (GTK_WIDGET (disp));
}

void
cd_display_clear (CDDisplay *disp)
{
	CDDisplayPrivate *priv;
	CDDisplayLine line;
	int height, max_width = 0;

	g_return_if_fail (disp != NULL);

	priv = disp->priv;
	for (line = CD_DISPLAY_LINE_TIME; line < CD_DISPLAY_END; line++) {
		GnomeCDText *text;
		PangoRectangle rect;

		text = priv->layout[line];
		height = priv->height - text->height;

		g_free (text->text);
		text->text = g_strdup (" ");
		text->length = 1;
		pango_layout_set_text (text->layout, text->text, 1);
		pango_layout_get_extents (text->layout, NULL, &rect);
		text->height = rect.height / 1000;

		priv->height = height + text->height;
		max_width = MAX (max_width, rect.width / 1000);
	}

	priv->max_width = max_width;
	gtk_widget_queue_resize (GTK_WIDGET (disp));
}
