/*
 * Copyright (C) 2005 Ronald S. Bultje <rbultje@ronald.bitfreak.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/video/video.h>

#include "mixer.h"
#include "mixerpad.h"

GST_DEBUG_CATEGORY_EXTERN (imagemixer_debug);
#define GST_CAT_DEFAULT imagemixer_debug

static GstStaticPadTemplate src_templ = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC, GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ AYUV, Y444, I420, YUY2 }")));
static GstStaticPadTemplate sink_templ = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK, GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ AYUV, Y444, I420, YUY2 }")));
static GstStaticPadTemplate subpicture_sink_templ =
GST_STATIC_PAD_TEMPLATE ("subpicture_sink_%d", GST_PAD_SINK, GST_PAD_REQUEST,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ AYUV, Y444, I420, YUY2 }")));

static GstCaps *gst_image_mixer_fixate (GstPad * pad, const GstCaps * caps);
static GstCaps *gst_image_mixer_get_caps (GstPad * pad);
static GstPadLinkReturn
gst_image_mixer_set_caps (GstPad * pad, const GstCaps * caps);
static GstPad *gst_image_mixer_request_pad (GstElement * element,
    GstPadTemplate * templ, const gchar * name);
static void gst_image_mixer_release_pad (GstElement * element, GstPad * pad);

static gboolean gst_image_mixer_src_event (GstPad * pad, GstEvent * event);
static void gst_image_mixer_loop (GstElement * element);
static GstElementStateReturn
gst_image_mixer_change_state (GstElement * element);

/*
 * Boilerplate code.
 */

GST_BOILERPLATE (GstImageMixer, gst_image_mixer, GstElement, GST_TYPE_ELEMENT);

static void
gst_image_mixer_base_init (gpointer data)
{
  GstImageMixerClass *klass = data;
  GstElementClass *eklass = GST_ELEMENT_CLASS (klass);
  static GstElementDetails imagemixer_details =
      GST_ELEMENT_DETAILS ("Image mixer",
      "Filter/Editor/Video",
      "Generic image mixer",
      "Ronald S. Bultje <rbultje@ronald.bitfreak.net>");

  gst_element_class_set_details (eklass, &imagemixer_details);
  gst_element_class_add_pad_template (eklass,
      gst_static_pad_template_get (&src_templ));
  gst_element_class_add_pad_template (eklass,
      gst_static_pad_template_get (&sink_templ));
  gst_element_class_add_pad_template (eklass,
      gst_static_pad_template_get (&subpicture_sink_templ));
}

static void
gst_image_mixer_class_init (GstImageMixerClass * klass)
{
  GstElementClass *eklass = GST_ELEMENT_CLASS (klass);

  eklass->change_state = gst_image_mixer_change_state;
  eklass->request_new_pad = gst_image_mixer_request_pad;
  eklass->release_pad = gst_image_mixer_release_pad;
}

static void
gst_image_mixer_init (GstImageMixer * mixer)
{
  mixer->sink =
      gst_pad_new_from_template (gst_static_pad_template_get (&sink_templ),
      "sink");
  gst_pad_set_link_function (mixer->sink, gst_image_mixer_set_caps);
  gst_pad_set_getcaps_function (mixer->sink, gst_image_mixer_get_caps);
  gst_element_add_pad (GST_ELEMENT (mixer), mixer->sink);

  mixer->src =
      gst_pad_new_from_template (gst_static_pad_template_get (&src_templ),
      "src");
  gst_pad_set_link_function (mixer->src, gst_image_mixer_set_caps);
  gst_pad_set_getcaps_function (mixer->src, gst_image_mixer_get_caps);
  gst_pad_set_event_function (mixer->src, gst_image_mixer_src_event);
  gst_element_add_pad (GST_ELEMENT (mixer), mixer->src);

  mixer->subpads = NULL;
  mixer->fmt.width = 0;
  mixer->fmt.height = 0;
  mixer->fmt.fourcc = 0;
  mixer->npad = 0;
  mixer->last_buf = NULL;
  mixer->next_data = NULL;

  gst_element_set_loop_function (GST_ELEMENT (mixer), gst_image_mixer_loop);

  GST_FLAG_SET (mixer, GST_ELEMENT_EVENT_AWARE);
}

/*
 * Request pad management.
 */

static GstPad *
gst_image_mixer_request_pad (GstElement * element,
    GstPadTemplate * templ, const gchar * req_name)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (element);
  GstElementClass *eklass = GST_ELEMENT_GET_CLASS (element);
  gchar *name;
  GstPad *pad;

  g_return_val_if_fail (templ ==
      gst_element_class_get_pad_template (eklass, "subpicture_sink_%d"), NULL);

  name = g_strdup_printf ("subpicture_sink_%d", mixer->npad);
  pad =
      gst_pad_custom_new_from_template (GST_TYPE_IMAGE_MIXER_PAD, templ, name);
  g_free (name);

  GST_IMAGE_MIXER_PAD (pad)->z_order = mixer->npad;
  mixer->npad++;
  mixer->subpads = g_list_append (mixer->subpads, pad);

  gst_pad_set_getcaps_function (pad, gst_image_mixer_get_caps);
  gst_pad_set_link_function (pad, gst_image_mixer_set_caps);
  gst_pad_set_fixate_function (pad, gst_image_mixer_fixate);
  gst_element_add_pad (element, pad);

  return pad;
}

static void
gst_image_mixer_release_pad (GstElement * element, GstPad * pad)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (element);

  mixer->subpads = g_list_remove (mixer->subpads, pad);
  gst_element_remove_pad (element, pad);
}

/*
 * Caps negotiation. 
 */

static GstCaps *
gst_image_mixer_fixate (GstPad * pad, const GstCaps * caps)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (gst_pad_get_parent (pad));
  GstCaps *copy;
  GstStructure *s;

  if (!mixer->fmt.width)
    return NULL;

  /* fixate to our own width */
  copy = gst_caps_copy (caps);
  s = gst_caps_get_structure (copy, 0);
  if (gst_caps_structure_fixate_field_nearest_int (s, "width",
          mixer->fmt.width))
    return copy;
  gst_caps_free (copy);

  return NULL;
}

static GstCaps *
gst_image_mixer_get_caps (GstPad * pad)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (gst_pad_get_parent (pad));

  if (pad == mixer->sink || pad == mixer->src) {
    /* main src/sink pad */
    GstCaps *allowed, *res;
    GstPad *otherpad = (mixer->src == pad) ? mixer->sink : mixer->src;

    allowed = gst_pad_get_allowed_caps (otherpad);
    res = gst_caps_intersect (allowed,
        gst_pad_template_get_caps (gst_pad_get_pad_template (pad)));
    gst_caps_free (allowed);

    return res;
  } else {
    /* sub-picture caps */
    return
        gst_caps_copy (gst_pad_template_get_caps (gst_pad_get_pad_template
            (pad)));
  }
}

static GstPadLinkReturn
gst_image_mixer_set_caps (GstPad * pad, const GstCaps * caps)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (gst_pad_get_parent (pad));
  GstStructure *s = gst_caps_get_structure (caps, 0);

  if (pad == mixer->src || pad == mixer->sink) {
    GstPadLinkReturn res;
    GstPad *otherpad = (pad == mixer->src) ? mixer->sink : mixer->src;

    res = gst_pad_try_set_caps (otherpad, caps);
    if (GST_PAD_LINK_FAILED (res))
      return res;

    gst_structure_get_int (s, "width", &mixer->fmt.width);
    gst_structure_get_int (s, "height", &mixer->fmt.height);
    gst_structure_get_fourcc (s, "format", &mixer->fmt.fourcc);
  } else {
    GstImageMixerPad *ipad = GST_IMAGE_MIXER_PAD (pad);

    gst_structure_get_int (s, "width", &ipad->fmt.width);
    gst_structure_get_int (s, "height", &ipad->fmt.height);
    gst_structure_get_fourcc (s, "format", &ipad->fmt.fourcc);
  }

  return GST_PAD_LINK_OK;
}

/*
 * Actual data handling.
 */

static void
gst_image_mixer_read_a (guint8 * sdata,
    GstImageMixerFormat * fmt, gint x, gint y,
    gint h_ss, gint v_ss, gint in_alpha, gint * palpha)
{
  gint alpha;

  switch (fmt->fourcc) {
    case GST_MAKE_FOURCC ('A', 'Y', 'U', 'V'):
      alpha = sdata[4 * (x + y * fmt->width)];
      if (h_ss > 1) {
        alpha += sdata[4 * (1 + x + y * fmt->width)];
      }
      if (v_ss > 1) {
        alpha += sdata[4 * (x + (1 + y) * fmt->width)];
        if (h_ss > 1) {
          alpha += sdata[4 * (1 + x + (1 + y) * fmt->width)];
        }
      }
      *palpha = in_alpha * alpha / (h_ss * v_ss * 255);
      break;
    default:
      break;
  }
}

static gint
gst_image_mixer_read_y (guint8 * sdata,
    GstImageMixerFormat * fmt, gint x, gint y)
{
  gint s_off;

  switch (fmt->fourcc) {
    case GST_MAKE_FOURCC ('A', 'Y', 'U', 'V'):
      s_off = 4 * (x + y * fmt->width) + 1;
      break;
    case GST_MAKE_FOURCC ('I', '4', '2', '0'):
      s_off = x + y * fmt->width;
      break;
    case GST_MAKE_FOURCC ('Y', '4', '4', '4'):
      s_off = 3 * (x + y * fmt->width);
      break;
    case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
      s_off = 2 * (x + y * fmt->width);
      break;
    default:
      g_return_val_if_reached (0);
  }

  return sdata[s_off];
}

static void
gst_image_mixer_read_u_v (guint8 * sdata,
    GstImageMixerFormat * fmt, gint x, gint y,
    gint h_ss, gint v_ss, gint * pu, gint * pv)
{
  gint u, v;

  switch (fmt->fourcc) {
    case GST_MAKE_FOURCC ('A', 'Y', 'U', 'V'):
      u = sdata[4 * (x + y * fmt->width) + 2];
      v = sdata[4 * (x + y * fmt->width) + 3];
      if (h_ss > 1) {
        u += sdata[4 * (1 + x + y * fmt->width) + 2];
        v += sdata[4 * (1 + x + y * fmt->width) + 3];
      }
      if (v_ss > 1) {
        u += sdata[4 * (x + (1 + y) * fmt->width) + 2];
        v += sdata[4 * (x + (1 + y) * fmt->width) + 3];
        if (h_ss > 1) {
          u += sdata[4 * (1 + x + (1 + y) * fmt->width) + 2];
          v += sdata[4 * (1 + x + (1 + y) * fmt->width) + 3];
        }
      }
      u /= (h_ss * v_ss);
      v /= (h_ss * v_ss);
      break;
    case GST_MAKE_FOURCC ('I', '4', '2', '0'):
      u = sdata[fmt->width * fmt->height + (y * fmt->width / 2 + x) / 2];
      v = sdata[(fmt->width * fmt->height * 5 / 4) +
          (y * fmt->width / 2 + x) / 2];
      break;
    case GST_MAKE_FOURCC ('Y', '4', '4', '4'):
      u = sdata[3 * (x + y * fmt->width) + 1];
      v = sdata[3 * (x + y * fmt->width) + 2];
      if (h_ss > 1) {
        u += sdata[3 * (1 + x + y * fmt->width) + 1];
        v += sdata[3 * (1 + x + y * fmt->width) + 2];
      }
      if (v_ss > 1) {
        u += sdata[3 * (x + (1 + y) * fmt->width) + 1];
        v += sdata[3 * (x + (1 + y) * fmt->width) + 2];
        if (h_ss > 1) {
          u += sdata[3 * (1 + x + (1 + y) * fmt->width) + 1];
          v += sdata[3 * (1 + x + (1 + y) * fmt->width) + 2];
        }
      }
      u /= (h_ss * v_ss);
      v /= (h_ss * v_ss);
      break;
    case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
      u = sdata[2 * (x + y * fmt->width) + 1];
      v = sdata[2 * (x + y * fmt->width) + 3];
      if (v_ss > 1) {
        u += sdata[2 * (x + (1 + y) * fmt->width) + 1];
        v += sdata[2 * (x + (1 + y) * fmt->width) + 3];
        u /= 2;
        v /= 2;
      }
      break;
    default:
      g_return_if_reached ();
  }

  *pu = u;
  *pv = v;
}

#define min(a, b) (a > b ? b : a)
#define max(a, b) (a > b ? a : b)

static void
gst_image_mixer_do_mix_i420 (GstImageMixer * mixer,
    GstImageMixerFormat * fmt, guint8 * ydata, guint8 * sdata,
    gint x_off, gint y_off, gfloat pad_alpha)
{
  guint8 *udata = &ydata[mixer->fmt.width * mixer->fmt.height],
      *vdata = &udata[mixer->fmt.width * mixer->fmt.height / 4];
  gint x, y, py, pu, pv, in_alpha = pad_alpha * 255, alpha = in_alpha,
      maxy = min (y_off + fmt->height, mixer->fmt.height),
      maxx = min (x_off + fmt->width, mixer->fmt.width),
      miny = max (y_off, 0), minx = max (x_off, 0);

  /* Y plane */
  for (y = miny; y < maxy; y++) {
    for (x = minx; x < maxx; x++) {
      gst_image_mixer_read_a (sdata, fmt, x - x_off, y - y_off,
          1, 1, in_alpha, &alpha);
      if (alpha > 0) {
        py = gst_image_mixer_read_y (sdata, fmt, x - x_off, y - y_off);
        ydata[y * mixer->fmt.width + x] = (alpha * py +
            (255 - alpha) * ydata[y * mixer->fmt.width + x]) >> 8;
      }
    }
  }

  /* U/V plane */
  for (y = miny + 1; y < maxy; y += 2) {
    for (x = minx + 1; x < maxx; x += 2) {
      gst_image_mixer_read_a (sdata, fmt, x - 1 - x_off, y - 1 - y_off,
          2, 2, in_alpha, &alpha);
      if (alpha > 0) {
        gst_image_mixer_read_u_v (sdata, fmt, x - 1 - x_off, y - 1 - y_off,
            2, 2, &pu, &pv);
        udata[(y * mixer->fmt.width / 2 + x) / 2] = (alpha * pu +
            (255 - alpha) * udata[(y * mixer->fmt.width / 2 + x) / 2]) >> 8;
        vdata[(y * mixer->fmt.width / 2 + x) / 2] = (alpha * pv +
            (255 - alpha) * vdata[(y * mixer->fmt.width / 2 + x) / 2]) >> 8;
      }
    }
  }
}

static void
gst_image_mixer_do_mix_yuy2 (GstImageMixer * mixer,
    GstImageMixerFormat * fmt, guint8 * ddata, guint8 * sdata,
    gint x_off, gint y_off, gfloat pad_alpha)
{
  gint x, y, py, pu, pv, in_alpha = pad_alpha * 255, alpha = in_alpha,
      maxy = min (y_off + fmt->height, mixer->fmt.height),
      maxx = min (x_off + fmt->width, mixer->fmt.width),
      miny = max (y_off, 0), minx = max (x_off, 0);

  for (y = miny; y < maxy; y++) {
    for (x = minx; x < maxx; x += 2) {
      /* Y */
      gst_image_mixer_read_a (sdata, fmt, x - x_off, y - y_off,
          1, 1, in_alpha, &alpha);
      if (alpha > 0) {
        py = gst_image_mixer_read_y (sdata, fmt, x - x_off, y - y_off);
        ddata[2 * (y * mixer->fmt.width + x)] = (alpha * py +
            (255 - alpha) * ddata[2 * (y * mixer->fmt.width + x)]) >> 8;
      }
      gst_image_mixer_read_a (sdata, fmt, 1 + x - x_off, y - y_off,
          1, 1, in_alpha, &alpha);
      if (alpha > 0) {
        py = gst_image_mixer_read_y (sdata, fmt, 1 + x - x_off, y - y_off);
        ddata[2 * (y * mixer->fmt.width + x + 1)] = (alpha * py +
            (255 - alpha) * ddata[2 * (y * mixer->fmt.width + x + 1)]) >> 8;
      }

      /* U/V */
      gst_image_mixer_read_a (sdata, fmt, x - x_off, y - y_off,
          2, 1, in_alpha, &alpha);
      if (alpha > 0) {
        gst_image_mixer_read_u_v (sdata, fmt, x - x_off, y - y_off,
            2, 1, &pu, &pv);
        ddata[2 * (y * mixer->fmt.width + x) + 1] = (alpha * pu +
            (255 - alpha) * ddata[2 * (y * mixer->fmt.width + x) + 1]) >> 8;
        ddata[2 * (y * mixer->fmt.width + x) + 3] = (alpha * pv +
            (255 - alpha) * ddata[2 * (y * mixer->fmt.width + x) + 3]) >> 8;
      }
    }
  }
}

static void
gst_image_mixer_do_mix_x444 (GstImageMixer * mixer,
    GstImageMixerFormat * fmt, guint8 * ddata, guint8 * sdata,
    gint x_off, gint y_off, gfloat pad_alpha, gint bpp)
{
  gint x, y, py, pu, pv, h_a = bpp - 3,
      in_alpha = pad_alpha * 255, alpha = in_alpha,
      maxy = min (y_off + fmt->height, mixer->fmt.height),
      maxx = min (x_off + fmt->width, mixer->fmt.width),
      miny = max (y_off, 0), minx = max (x_off, 0);

  /* all together */
  for (y = miny; y < maxy; y++) {
    for (x = minx; x < maxx; x++) {
      gst_image_mixer_read_a (sdata, fmt, x - x_off, y - y_off,
          1, 1, in_alpha, &alpha);
      if (alpha > 0) {
        py = gst_image_mixer_read_y (sdata, fmt, x - x_off, y - y_off);
        gst_image_mixer_read_u_v (sdata, fmt, x - x_off, y - y_off,
            1, 1, &pu, &pv);

        ddata[bpp * (y * mixer->fmt.width + x) + h_a] = (alpha * py +
            (255 - alpha) * ddata[bpp * (y * mixer->fmt.width + x) + h_a]) >> 8;
        ddata[bpp * (y * mixer->fmt.width + x) + 1 + h_a] = (alpha * pu +
            (255 - alpha) * ddata[bpp * (y * mixer->fmt.width + x) + 1 +
                h_a]) >> 8;
        ddata[bpp * (y * mixer->fmt.width + x) + 2 + h_a] =
            (alpha * pv + (255 - alpha) * ddata[bpp * (y * mixer->fmt.width +
                    x) + 2 + h_a]) >> 8;
      }
    }
  }
}

static inline void
gst_image_mixer_do_mix_y444 (GstImageMixer * mixer,
    GstImageMixerFormat * fmt, guint8 * ddata, guint8 * sdata,
    gint x_off, gint y_off, gfloat pad_alpha)
{
  gst_image_mixer_do_mix_x444 (mixer, fmt, ddata, sdata,
      x_off, y_off, pad_alpha, 3);
}

static inline void
gst_image_mixer_do_mix_ayuv (GstImageMixer * mixer,
    GstImageMixerFormat * fmt, guint8 * ddata, guint8 * sdata,
    gint x_off, gint y_off, gfloat pad_alpha)
{
  gst_image_mixer_do_mix_x444 (mixer, fmt, ddata, sdata,
      x_off, y_off, pad_alpha, 4);
}

#undef min
#undef max

static void
gst_image_mixer_do_mix (GstImageMixer * mixer,
    GstImageMixerPad * ipad, GstBuffer * dest, GstBuffer * src)
{
  gint x_off, y_off;
  GstImageMixerFormat *fmt;
  gfloat alpha;

  if (ipad) {
    switch (ipad->v_align) {
      case GST_IMAGE_MIXER_PAD_V_ALIGN_TOP:
        y_off = ipad->v_offset;
        break;
      case GST_IMAGE_MIXER_PAD_V_ALIGN_MIDDLE:
        y_off = (mixer->fmt.height - ipad->fmt.height) / 2;
        break;
      case GST_IMAGE_MIXER_PAD_V_ALIGN_BOTTOM:
        y_off = mixer->fmt.height - (ipad->fmt.height + ipad->v_offset);
        break;
      default:
        g_return_if_reached ();
        return;                 /* make compiler happy */
    }

    switch (ipad->h_align) {
      case GST_IMAGE_MIXER_PAD_H_ALIGN_LEFT:
        x_off = ipad->h_offset;
        break;
      case GST_IMAGE_MIXER_PAD_H_ALIGN_CENTER:
        x_off = (mixer->fmt.width - ipad->fmt.width) / 2;
        break;
      case GST_IMAGE_MIXER_PAD_H_ALIGN_RIGHT:
        x_off = mixer->fmt.width - (ipad->fmt.width + ipad->h_offset);
        break;
      default:
        g_return_if_reached ();
        return;                 /* make compiler happy */
    }

    fmt = &ipad->fmt;
    alpha = ipad->alpha;
  } else {
    fmt = &mixer->fmt;
    x_off = y_off = 0;
    alpha = 1.0;
  }

  GST_DEBUG_OBJECT (mixer, "Blending subpicture of %dx%d ("
      GST_FOURCC_FORMAT ") at position x=%d, y=%d of %dx%d ("
      GST_FOURCC_FORMAT ")", fmt->width, fmt->height,
      GST_FOURCC_ARGS (fmt->fourcc), x_off, y_off,
      mixer->fmt.width, mixer->fmt.height, GST_FOURCC_ARGS (mixer->fmt.fourcc));

  /* actual blending goes here */
  switch (mixer->fmt.fourcc) {
    case GST_MAKE_FOURCC ('I', '4', '2', '0'):
      gst_image_mixer_do_mix_i420 (mixer, fmt,
          GST_BUFFER_DATA (dest), GST_BUFFER_DATA (src), x_off, y_off, alpha);
      break;
    case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
      gst_image_mixer_do_mix_yuy2 (mixer, fmt,
          GST_BUFFER_DATA (dest), GST_BUFFER_DATA (src), x_off, y_off, alpha);
      break;
    case GST_MAKE_FOURCC ('A', 'Y', 'U', 'V'):
      gst_image_mixer_do_mix_ayuv (mixer, fmt,
          GST_BUFFER_DATA (dest), GST_BUFFER_DATA (src), x_off, y_off, alpha);
      break;
    case GST_MAKE_FOURCC ('Y', '4', '4', '4'):
      gst_image_mixer_do_mix_y444 (mixer, fmt,
          GST_BUFFER_DATA (dest), GST_BUFFER_DATA (src), x_off, y_off, alpha);
      break;
    default:
      g_return_if_reached ();
  }
}

static gboolean
is_past_end (GstData * data, guint64 ts)
{
  guint64 t, d;

  if (GST_IS_EVENT (data)) {
    switch (GST_EVENT_TYPE (data)) {
      case GST_EVENT_EOS:
        return FALSE;
      case GST_EVENT_INTERRUPT:
        return TRUE;
      case GST_EVENT_FILLER:
        if (!GST_CLOCK_TIME_IS_VALID (t = GST_EVENT_TIMESTAMP (data)))
          return TRUE;
        d = gst_event_filler_get_duration (GST_EVENT (data));
        break;
      default:
        g_assert_not_reached ();
        return FALSE;           /* make compiler happy */
    }
  } else {
    t = GST_BUFFER_TIMESTAMP (data);
    d = GST_BUFFER_DURATION (data);
  }

  if (!GST_CLOCK_TIME_IS_VALID (d))
    return FALSE;

  return (t + d <= ts);
}

static void
gst_image_mixer_blend_subpicture (GstImageMixerPad * ipad,
    GstBuffer ** _data_buf, GstBuffer * time_buf)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (GST_OBJECT_PARENT (ipad));
  GstBuffer *data_buf = *_data_buf;

  GST_DEBUG_OBJECT (mixer, "Handling subpicture pad %s",
      GST_OBJECT_NAME (ipad));

  /* get the picture to blend */
  if (ipad->current_data) {
    if (is_past_end (ipad->current_data, GST_BUFFER_TIMESTAMP (time_buf))) {
      gst_data_unref (ipad->current_data);
      ipad->current_data = NULL;
    }
  }

  /* This state machine was originally in pango/textoverlay. It handles
   * stuff such as fillers, EOS and buffers (timestamps, durations, etc.)
   * all well. */
  while (!ipad->current_data) {
    GstData *in_data;

    GST_DEBUG_OBJECT (mixer, "Pulling data from pad %s",
        GST_OBJECT_NAME (ipad));

    in_data = gst_pad_pull (GST_PAD (ipad));
    if (GST_IS_EVENT (in_data) &&
        !(GST_EVENT_TYPE (in_data) == GST_EVENT_FILLER &&
            GST_CLOCK_TIME_IS_VALID (GST_EVENT_TIMESTAMP (in_data)))) {
      if (GST_EVENT_TYPE (in_data) == GST_EVENT_INTERRUPT ||
          GST_EVENT_TYPE (in_data) == GST_EVENT_EOS ||
          GST_EVENT_TYPE (in_data) == GST_EVENT_FILLER) {
        GST_DEBUG_OBJECT (mixer, "Received empty filler/EOS/interrupt event");
        ipad->current_data = in_data;
      } else {
        GST_DEBUG_OBJECT (mixer, "Received unknown event - dropping");
        gst_data_unref (in_data);
      }
    } else {
      if (is_past_end (in_data, GST_BUFFER_TIMESTAMP (time_buf))) {
        GST_DEBUG_OBJECT (mixer, "Received outdated %s of time %"
            GST_TIME_FORMAT " and duration %" GST_TIME_FORMAT,
            GST_IS_EVENT (in_data) ? "filler event" : "buffer",
            GST_TIME_ARGS (GST_IS_BUFFER (in_data) ?
                GST_BUFFER_TIMESTAMP (in_data) :
                GST_EVENT_TIMESTAMP (in_data)),
            GST_TIME_ARGS (GST_IS_BUFFER (in_data) ?
                GST_BUFFER_DURATION (in_data) :
                gst_event_filler_get_duration (GST_EVENT (in_data))));

        gst_data_unref (in_data);
      } else {
        GST_DEBUG_OBJECT (mixer, "Using %s of time %" GST_TIME_FORMAT
            " and duration %" GST_TIME_FORMAT,
            GST_IS_EVENT (in_data) ? "event" : "buffer",
            GST_TIME_ARGS (GST_IS_BUFFER (in_data) ?
                GST_BUFFER_TIMESTAMP (in_data) :
                GST_EVENT_TIMESTAMP (in_data)),
            GST_TIME_ARGS (GST_IS_BUFFER (in_data) ?
                GST_BUFFER_DURATION (in_data) :
                gst_event_filler_get_duration (GST_EVENT (in_data))));

        ipad->current_data = in_data;
      }
    }
  }

  /* now blend (or skip) */
  if (GST_IS_BUFFER (ipad->current_data) &&
      (GST_BUFFER_TIMESTAMP (ipad->current_data) <=
          GST_BUFFER_TIMESTAMP (time_buf) ||
          !GST_BUFFER_TIMESTAMP_IS_VALID (ipad->current_data))) {
    if (!data_buf) {
      guint bpp = 0;

      switch (mixer->fmt.fourcc) {
        case GST_MAKE_FOURCC ('A', 'Y', 'U', 'V'):
          bpp = 32;
          break;
        case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
          bpp = 16;
          break;
        case GST_MAKE_FOURCC ('I', '4', '2', '0'):
          bpp = 12;
          break;
        case GST_MAKE_FOURCC ('Y', '4', '4', '4'):
          bpp = 24;
          break;
      }
      data_buf = gst_buffer_new_and_alloc (mixer->fmt.width *
          mixer->fmt.height * bpp >> 3);
    } else {
      data_buf = gst_buffer_copy_on_write (data_buf);
    }
    *_data_buf = data_buf;

    /* actually merge */
    GST_DEBUG_OBJECT (mixer, "Drawing subpicture");
    gst_image_mixer_do_mix (mixer, ipad, data_buf,
        GST_BUFFER (ipad->current_data));
  } else {
    GST_DEBUG_OBJECT (mixer, "Not drawing subpicture");
  }
}

static gboolean
gst_image_mixer_src_event (GstPad * pad, GstEvent * event)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (gst_pad_get_parent (pad));
  gboolean res;

  if (!GST_PAD_PEER (mixer->sink)) {
    gst_event_unref (event);
    return FALSE;
  }

  gst_event_ref (event);
  res = gst_pad_send_event (GST_PAD_PEER (mixer->sink), event);

  /* on seek, flush current data on pads */
  if (res && GST_EVENT_TYPE (event) == GST_EVENT_SEEK) {
    GstPad *tpad;
    GstImageMixerPad *ipad;
    const GList *item;

    GST_DEBUG_OBJECT (mixer, "Forwarding seek event to all subpic pads");
    for (item = mixer->subpads; item != NULL; item = item->next) {
      tpad = item->data;
      ipad = item->data;
      if (GST_PAD_PEER (tpad)) {
        gst_event_ref (event);
        if (gst_pad_send_event (GST_PAD_PEER (tpad), event)) {
          if (ipad->current_data) {
            gst_data_unref (ipad->current_data);
            ipad->current_data = NULL;
          }
        }
      }
    }

    /* also clean own cached data */
    if (mixer->last_buf) {
      gst_buffer_unref (mixer->last_buf);
      mixer->last_buf = NULL;
    }
    if (mixer->next_data) {
      gst_data_unref (mixer->next_data);
      mixer->next_data = NULL;
    }
  }

  gst_event_unref (event);

  return res;
}

static gboolean
gst_image_mixer_handle_sink_event (GstImageMixer * mixer, GstEvent * event)
{
  const GList *item;

  GST_DEBUG_OBJECT (mixer, "Handling event of type %d", GST_EVENT_TYPE (event));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_INTERRUPT:
      gst_event_unref (event);
      return FALSE;
    case GST_EVENT_EOS:
      /* EOS subpicture pads */
      GST_DEBUG_OBJECT (mixer, "Received EOS - EOS'ing subpicture pads");

      for (item = mixer->subpads; item != NULL; item = item->next) {
        GstPad *pad = item->data;
        GstImageMixerPad *ipad = item->data;

        GST_DEBUG_OBJECT (mixer, "EOS'ing pad %s", GST_OBJECT_NAME (pad));

        if (ipad->current_data) {
          gst_data_unref (ipad->current_data);
          ipad->current_data = NULL;
        }

        do {
          GstData *in_data;

          GST_DEBUG_OBJECT (mixer, "pulling data");
          in_data = gst_pad_pull (pad);
          if (GST_IS_EVENT (in_data) &&
              GST_EVENT_TYPE (in_data) == GST_EVENT_EOS) {
            GST_DEBUG_OBJECT (mixer, "Received EOS");
            gst_data_unref (in_data);
            break;
          }
          gst_data_unref (in_data);
        } while (1);
      }
      gst_pad_event_default (mixer->sink, event);
      return FALSE;
    default:
      gst_pad_event_default (mixer->sink, event);
      return TRUE;
  }
}

static gint
cb_compare (GstImageMixerPad * a, GstImageMixerPad * b)
{
  return a->z_order - b->z_order;
}

static void
gst_image_mixer_loop (GstElement * element)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (element);
  GstBuffer *out_buf = NULL, *in_buf = NULL;
  const GList *item;
  GstImageMixerPad *pad;

  do {
    GstData *in_data;

    if (mixer->next_data) {
      in_data = mixer->next_data;
      mixer->next_data = NULL;
    } else {
      in_data = gst_pad_pull (mixer->sink);
    }
    GST_DEBUG_OBJECT (mixer, "Pulling data from main pad");

    if (GST_IS_EVENT (in_data) &&
        !(GST_EVENT_TYPE (in_data) == GST_EVENT_FILLER &&
            GST_CLOCK_TIME_IS_VALID (GST_EVENT_TIMESTAMP (in_data)))) {
      if (!gst_image_mixer_handle_sink_event (mixer, GST_EVENT (in_data)))
        return;
    } else if (mixer->last_buf != NULL && GST_IS_EVENT (in_data)) {
      in_buf = gst_buffer_create_sub (mixer->last_buf, 0,
          GST_BUFFER_SIZE (mixer->last_buf));
      GST_BUFFER_TIMESTAMP (in_buf) = GST_EVENT_TIMESTAMP (in_data);
      GST_BUFFER_DURATION (in_buf) =
          gst_event_filler_get_duration (GST_EVENT (in_data));
      gst_data_unref (in_data);
    } else if (GST_IS_EVENT (in_data)) {
      gst_data_unref (in_data);
    } else {
      in_buf = GST_BUFFER (in_data);
    }
  } while (!in_buf);
  g_assert (mixer->next_data == NULL);
  mixer->next_data = gst_pad_pull (mixer->sink);

  /* Cache this buffer if next is event, because it could be a filler. We
   * don't look at event type because event handling has to be done in
   * order so we can't do anything with it anyway. */
  if (mixer->last_buf) {
    gst_buffer_unref (mixer->last_buf);
    mixer->last_buf = NULL;
  }
  if (GST_IS_EVENT (mixer->next_data)) {
    mixer->last_buf = gst_buffer_ref (in_buf);
  }

  GST_DEBUG_OBJECT (mixer, "Processing buffer %p of time %" GST_TIME_FORMAT
      " and duration %" GST_TIME_FORMAT, in_buf,
      GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (in_buf)),
      GST_TIME_ARGS (GST_BUFFER_DURATION (in_buf)));

  /* Sort pads by Z-order and process in that order */
  mixer->subpads = g_list_sort (mixer->subpads, (GCompareFunc) cb_compare);

  /* negative Z-order behind main picture */
  for (item = mixer->subpads; item != NULL; item = item->next) {
    pad = item->data;
    if (pad->z_order >= 0)
      break;
    gst_image_mixer_blend_subpicture (pad, &out_buf, in_buf);
  }

  /* main picture comes next */
  if (!out_buf) {
    GST_LOG_OBJECT (mixer, "Using input picture as base");
    out_buf = in_buf;
  } else {
    GST_LOG_OBJECT (mixer, "Merging main picture");
    gst_image_mixer_do_mix (mixer, NULL, out_buf, in_buf);
    gst_buffer_stamp (out_buf, in_buf);
    gst_buffer_unref (in_buf);
  }

  /* positive Z-order comes over main picture */
  for (; item != NULL; item = item->next) {
    pad = item->data;
    gst_image_mixer_blend_subpicture (pad, &out_buf, out_buf);
  }

  GST_DEBUG_OBJECT (mixer, "Processing finished, now pushing buffer %p"
      " at time %" GST_TIME_FORMAT, in_buf,
      GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (out_buf)));

  /* done, now push */
  gst_pad_push (mixer->src, GST_DATA (out_buf));
}

/*
 * Clean up on paused->ready.
 */

static GstElementStateReturn
gst_image_mixer_change_state (GstElement * element)
{
  GstImageMixer *mixer = GST_IMAGE_MIXER (element);
  const GList *item;

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_PAUSED_TO_READY:
      /* clean up pending data in request pads */
      for (item = mixer->subpads; item != NULL; item = item->next) {
        GstImageMixerPad *ipad = item->data;

        if (ipad->current_data) {
          gst_data_unref (ipad->current_data);
          ipad->current_data = NULL;
        }
      }
      if (mixer->last_buf) {
        gst_buffer_unref (mixer->last_buf);
        mixer->last_buf = NULL;
      }
      if (mixer->next_data) {
        gst_data_unref (mixer->next_data);
        mixer->next_data = NULL;
      }
      break;
    default:
      break;
  }

  return GST_CALL_PARENT_WITH_DEFAULT (GST_ELEMENT_CLASS,
      change_state, (element), GST_STATE_SUCCESS);
}
