/* GStreamer
 * Copyright (C) <2005> Luca Ognibene <luogni@tin.it>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ximageutil.h"

static gboolean error_caught = FALSE;

static int
ximageutil_handle_xerror (Display * display, XErrorEvent * xevent)
{
  char error_msg[1024];

  XGetErrorText (display, xevent->error_code, error_msg, 1024);
  GST_DEBUG ("ximageutil failed to use XShm calls. error: %s", error_msg);
  error_caught = TRUE;
  return 0;
}

/* This function checks that it is actually really possible to create an image
   using XShm */
gboolean
ximageutil_check_xshm_calls (GstXContext * xcontext)
{
#ifndef HAVE_XSHM
  return FALSE;
#else
  GstXImage *ximage = NULL;
  int (*handler) (Display *, XErrorEvent *);
  gboolean result = FALSE;

  g_return_val_if_fail (xcontext != NULL, FALSE);

  ximage = g_new0 (GstXImage, 1);
  g_return_val_if_fail (ximage != NULL, FALSE);

  /* Setting an error handler to catch failure */
  error_caught = FALSE;
  handler = XSetErrorHandler (ximageutil_handle_xerror);

  /* Trying to create a 1x1 picture */
  GST_DEBUG ("XShmCreateImage of 1x1");

  ximage->ximage = XShmCreateImage (xcontext->disp, xcontext->visual,
      xcontext->depth, ZPixmap, NULL, &ximage->SHMInfo, 1, 1);
  if (!ximage->ximage) {
    GST_WARNING ("could not XShmCreateImage a 1x1 image");
    goto beach;
  }
  ximage->size = ximage->ximage->height * ximage->ximage->bytes_per_line;

  ximage->SHMInfo.shmid = shmget (IPC_PRIVATE, ximage->size, IPC_CREAT | 0777);
  if (ximage->SHMInfo.shmid == -1) {
    GST_WARNING ("could not get shared memory of %d bytes", ximage->size);
    goto beach;
  }

  ximage->SHMInfo.shmaddr = shmat (ximage->SHMInfo.shmid, NULL, 0);
  if (ximage->SHMInfo.shmaddr == ((void *) -1)) {
    GST_WARNING ("Failed to shmat: %s", g_strerror (errno));
    goto beach;
  }

  ximage->ximage->data = ximage->SHMInfo.shmaddr;
  ximage->SHMInfo.readOnly = FALSE;

  if (XShmAttach (xcontext->disp, &ximage->SHMInfo) == 0) {
    GST_WARNING ("Failed to XShmAttach");
    goto beach;
  }

  XSync (xcontext->disp, 0);

  XShmDetach (xcontext->disp, &ximage->SHMInfo);
  XSync (xcontext->disp, FALSE);

  shmdt (ximage->SHMInfo.shmaddr);
  shmctl (ximage->SHMInfo.shmid, IPC_RMID, NULL);

  /* To be sure, reset the SHMInfo entry */
  ximage->SHMInfo.shmaddr = ((void *) -1);

  /* store whether we succeeded in result and reset error_caught */
  result = !error_caught;
  error_caught = FALSE;

beach:
  XSetErrorHandler (handler);
  if (ximage->ximage)
    XFree (ximage->ximage);
  g_free (ximage);
  XSync (xcontext->disp, FALSE);
  return result;
#endif /* HAVE_XSHM */
}

/* This function handles GstXImage creation depending on XShm availability */
GstXImage *
ximageutil_ximage_new (GstElement * parent, GstXContext * xcontext,
    gint width, gint height)
{
  GstXImage *ximage = NULL;
  gboolean succeeded = FALSE;

  g_return_val_if_fail (GST_IS_ELEMENT (parent), NULL);
  GST_DEBUG_OBJECT (parent, "creating %dx%d", width, height);

  ximage = g_new0 (GstXImage, 1);

  ximage->width = width;
  ximage->height = height;
  ximage->parent = parent;

#ifdef HAVE_XSHM
  if (xcontext->use_xshm) {
    ximage->ximage = XShmCreateImage (xcontext->disp,
        xcontext->visual,
        xcontext->depth,
        ZPixmap, NULL, &ximage->SHMInfo, ximage->width, ximage->height);
    if (!ximage->ximage) {
      GST_ELEMENT_ERROR (parent, RESOURCE, WRITE, (NULL),
          ("could not XShmCreateImage a %dx%d image"));
      goto beach;
    }

    /* we have to use the returned bytes_per_line for our shm size */
    ximage->size = ximage->ximage->bytes_per_line * ximage->ximage->height;
    GST_DEBUG_OBJECT (parent, "XShm image size is %d, width %d, stride %d",
        ximage->size, ximage->width, ximage->ximage->bytes_per_line);

    ximage->SHMInfo.shmid = shmget (IPC_PRIVATE, ximage->size,
        IPC_CREAT | 0777);
    if (ximage->SHMInfo.shmid == -1) {
      GST_ELEMENT_ERROR (parent, RESOURCE, WRITE, (NULL),
          ("could not get shared memory of %d bytes", ximage->size));
      goto beach;
    }

    ximage->SHMInfo.shmaddr = shmat (ximage->SHMInfo.shmid, NULL, 0);
    if (ximage->SHMInfo.shmaddr == ((void *) -1)) {
      GST_ELEMENT_ERROR (parent, RESOURCE, WRITE, (NULL),
          ("Failed to shmat: %s", g_strerror (errno)));
      goto beach;
    }

    ximage->ximage->data = ximage->SHMInfo.shmaddr;
    ximage->SHMInfo.readOnly = FALSE;

    if (XShmAttach (xcontext->disp, &ximage->SHMInfo) == 0) {
      GST_ELEMENT_ERROR (parent, RESOURCE, WRITE, (NULL),
          ("Failed to XShmAttach"));
      goto beach;
    }

    XSync (xcontext->disp, FALSE);
  } else
#endif /* HAVE_XSHM */
  {
    ximage->ximage = XCreateImage (xcontext->disp,
        xcontext->visual,
        xcontext->depth,
        ZPixmap, 0, NULL, ximage->width, ximage->height, xcontext->bpp, 0);
    if (!ximage->ximage) {
      GST_ELEMENT_ERROR (parent, RESOURCE, WRITE, (NULL),
          ("could not XCreateImage a %dx%d image"));
      goto beach;
    }

    /* we have to use the returned bytes_per_line for our image size */
    ximage->size = ximage->ximage->bytes_per_line * ximage->ximage->height;
    ximage->ximage->data = g_malloc (ximage->size);

    XSync (xcontext->disp, FALSE);
  }
  succeeded = TRUE;

beach:
  if (!succeeded) {
    ximageutil_ximage_destroy (xcontext, ximage);
    ximage = NULL;
  }

  return ximage;
}

/* This function destroys a GstXImage handling XShm availability */
void
ximageutil_ximage_destroy (GstXContext * xcontext, GstXImage * ximage)
{
  g_return_if_fail (ximage != NULL);

#ifdef HAVE_XSHM
  if (xcontext->use_xshm) {
    if (ximage->SHMInfo.shmaddr != ((void *) -1)) {
      XShmDetach (xcontext->disp, &ximage->SHMInfo);
      XSync (xcontext->disp, 0);
      shmdt (ximage->SHMInfo.shmaddr);
    }
    if (ximage->SHMInfo.shmid > 0)
      shmctl (ximage->SHMInfo.shmid, IPC_RMID, NULL);
    if (ximage->ximage)
      XDestroyImage (ximage->ximage);

  } else
#endif /* HAVE_XSHM */
  {
    if (ximage->ximage) {
      XDestroyImage (ximage->ximage);
    }
  }

  XSync (xcontext->disp, FALSE);

  g_free (ximage);
}

/* This function gets the X Display and global info about it. Everything is
   stored in our object and will be cleaned when the object is disposed. Note
   here that caps for supported format are generated without any window or
   image creation */
GstXContext *
ximageutil_xcontext_get (GstElement * parent, gchar * display_name)
{
  GstXContext *xcontext = NULL;
  XPixmapFormatValues *px_formats = NULL;
  gint nb_formats = 0, i;

  xcontext = g_new0 (GstXContext, 1);

  xcontext->disp = XOpenDisplay (display_name);

  if (!xcontext->disp) {
    g_free (xcontext);
    return NULL;
  }

  xcontext->screen = DefaultScreenOfDisplay (xcontext->disp);
  xcontext->screen_num = DefaultScreen (xcontext->disp);
  xcontext->visual = DefaultVisual (xcontext->disp, xcontext->screen_num);
  xcontext->root = DefaultRootWindow (xcontext->disp);
  xcontext->white = XWhitePixel (xcontext->disp, xcontext->screen_num);
  xcontext->black = XBlackPixel (xcontext->disp, xcontext->screen_num);
  xcontext->depth = DefaultDepthOfScreen (xcontext->screen);

  xcontext->width = DisplayWidth (xcontext->disp, xcontext->screen_num);
  xcontext->height = DisplayHeight (xcontext->disp, xcontext->screen_num);
  xcontext->widthmm = DisplayWidthMM (xcontext->disp, xcontext->screen_num);
  xcontext->heightmm = DisplayHeightMM (xcontext->disp, xcontext->screen_num);

  xcontext->caps = NULL;

  GST_DEBUG_OBJECT (parent, "X reports %dx%d pixels and %d mm x %d mm",
      xcontext->width, xcontext->height, xcontext->widthmm, xcontext->heightmm);

  ximageutil_calculate_pixel_aspect_ratio (xcontext);

  /* We get supported pixmap formats at supported depth */
  px_formats = XListPixmapFormats (xcontext->disp, &nb_formats);

  if (!px_formats) {
    XCloseDisplay (xcontext->disp);
    g_free (xcontext);
    return NULL;
  }

  /* We get bpp value corresponding to our running depth */
  for (i = 0; i < nb_formats; i++) {
    if (px_formats[i].depth == xcontext->depth)
      xcontext->bpp = px_formats[i].bits_per_pixel;
  }

  XFree (px_formats);

  xcontext->endianness =
      (ImageByteOrder (xcontext->disp) ==
      LSBFirst) ? G_LITTLE_ENDIAN : G_BIG_ENDIAN;

#ifdef HAVE_XSHM
  /* Search for XShm extension support */
  if (XShmQueryExtension (xcontext->disp) &&
      ximageutil_check_xshm_calls (xcontext)) {
    xcontext->use_xshm = TRUE;
    GST_DEBUG ("ximageutil is using XShm extension");
  } else {
    xcontext->use_xshm = FALSE;
    GST_DEBUG ("ximageutil is not using XShm extension");
  }
#endif /* HAVE_XSHM */

  /* our caps system handles 24/32bpp RGB as big-endian. */
  if ((xcontext->bpp == 24 || xcontext->bpp == 32) &&
      xcontext->endianness == G_LITTLE_ENDIAN) {
    xcontext->endianness = G_BIG_ENDIAN;
    xcontext->visual->red_mask = GUINT32_TO_BE (xcontext->visual->red_mask);
    xcontext->visual->green_mask = GUINT32_TO_BE (xcontext->visual->green_mask);
    xcontext->visual->blue_mask = GUINT32_TO_BE (xcontext->visual->blue_mask);
    if (xcontext->bpp == 24) {
      xcontext->visual->red_mask >>= 8;
      xcontext->visual->green_mask >>= 8;
      xcontext->visual->blue_mask >>= 8;
    }
  }

  return xcontext;
}

/* This function cleans the X context. Closing the Display and unrefing the
   caps for supported formats. */
void
ximageutil_xcontext_clear (GstXContext * xcontext)
{

  g_return_if_fail (xcontext != NULL);

  if (xcontext->caps != NULL)
    gst_caps_free (xcontext->caps);
  g_free (xcontext->par);

  XCloseDisplay (xcontext->disp);

  g_free (xcontext);
}

/* This function calculates the pixel aspect ratio based on the properties
 * in the xcontext structure and stores it there. */
void
ximageutil_calculate_pixel_aspect_ratio (GstXContext * xcontext)
{
  gint par[][2] = {
    {1, 1},                     /* regular screen */
    {16, 15},                   /* PAL TV */
    {11, 10},                   /* 525 line Rec.601 video */
    {54, 59}                    /* 625 line Rec.601 video */
  };
  gint i;
  gint index;
  gdouble ratio;
  gdouble delta;

#define DELTA(idx) (ABS (ratio - ((gdouble) par[idx][0] / par[idx][1])))

  /* first calculate the "real" ratio based on the X values;
   * which is the "physical" w/h divided by the w/h in pixels of the display */
  ratio = (gdouble) (xcontext->widthmm * xcontext->height)
      / (xcontext->heightmm * xcontext->width);

  /* DirectFB's X in 720x576 reports the physical dimensions wrong, so
   * override here */
  if (xcontext->width == 720 && xcontext->height == 576) {
    ratio = 4.0 * 576 / (3.0 * 720);
  }
  GST_DEBUG ("calculated pixel aspect ratio: %f", ratio);

  /* now find the one from par[][2] with the lowest delta to the real one */
  delta = DELTA (0);
  index = 0;

  for (i = 1; i < sizeof (par) / (sizeof (gint) * 2); ++i) {
    gdouble this_delta = DELTA (i);

    if (this_delta < delta) {
      index = i;
      delta = this_delta;
    }
  }

  GST_DEBUG ("Decided on index %d (%d/%d)", index,
      par[index][0], par[index][1]);

  g_free (xcontext->par);
  xcontext->par = g_new0 (GValue, 1);
  g_value_init (xcontext->par, GST_TYPE_FRACTION);
  gst_value_set_fraction (xcontext->par, par[index][0], par[index][1]);
  GST_DEBUG ("set xcontext PAR to %d/%d",
      gst_value_get_fraction_numerator (xcontext->par),
      gst_value_get_fraction_denominator (xcontext->par));
}
