/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-filter-multipage.c: Filter for printing several pages onto single output page
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Chris Lahey <clahey@helixcode.com>
 *
 *  Copyright (C) 1999-2003 Ximian Inc.
 */

#include <config.h>
#include "gnome-print-filter-multipage.h"

#include <string.h>
#include <math.h>

#include <gmodule.h>

#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_bpath.h>

#include <libgnomeprint/gnome-print-private.h>
#include <libgnomeprint/gp-gc-private.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gnome-print-i18n.h>
#include <libgnomeprint/gnome-print-filter.h>

struct _GnomePrintFilterMultipage {
	GnomePrintFilter parent;

	GList *affines; /* Of type double[6] */
	GList *subpage;
};

struct _GnomePrintFilterMultipageClass {
	GnomePrintFilterClass parent_class;
};

#define FILTER_NAME        N_("multipage")
#define FILTER_DESCRIPTION N_("Filter for printing several pages onto single output page")

static GnomePrintFilterClass *parent_class = NULL;

enum {
	PROP_0,
	PROP_NAME,
	PROP_DESCRIPTION,
	PROP_AFFINES
};

static void
gnome_print_filter_multipage_get_property (GObject *object, guint n, GValue *v,
		GParamSpec *pspec)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (object);

	switch (n) {
	case PROP_AFFINES:
		{
			guint i, n = mp->affines ? g_list_length (mp->affines) : 0;
			GValueArray *va;
			GValue vd = {0,};

			if (!n) break;
			va = g_value_array_new (n);
			g_value_init (&vd, G_TYPE_DOUBLE);
			for (i = 0; i < n; i++) {
				gdouble *a = g_list_nth_data (mp->affines, i);
				guint j;

				for (j = 0; j < 6; j++) {
					g_value_set_double (&vd, a[j]);
					g_value_array_append (va, &vd);
				}
			}
			g_value_unset (&vd);
			g_value_set_boxed (v, va);
			g_value_array_free (va);
		}
		break;
	case PROP_NAME:
		g_value_set_string (v, _(FILTER_NAME));
		break;
	case PROP_DESCRIPTION:
		g_value_set_string (v, _(FILTER_DESCRIPTION));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, n, pspec);
	}
}

static void
gnome_print_filter_multipage_free_affines (GnomePrintFilterMultipage *mp)
{
	g_return_if_fail (GNOME_IS_PRINT_FILTER_MULTIPAGE (mp));

	while (mp->affines) {
		g_free (mp->affines->data);
		mp->affines = g_list_remove (mp->affines, mp->affines->data);
	}
}

static void
gnome_print_filter_multipage_set_property (GObject *object, guint n,
		const GValue *v, GParamSpec *pspec)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (object);

	switch (n) {
	case PROP_AFFINES:
		{
			GValueArray *a = g_value_get_boxed (v);

			gnome_print_filter_multipage_free_affines (mp);
			if (a && a->n_values) {
				guint i, j;

				for (i = 0; 6 * i < a->n_values; i++) {
					gdouble *affine;

					affine = g_new (gdouble, 6);
					for (j = 0; (j < 6) && (6 * i + j < a->n_values); j++)
						affine[j] = g_value_get_double (
								g_value_array_get_nth (a, 6 * i + j));
					mp->affines = g_list_append (mp->affines, affine);
				}
				mp->subpage = mp->affines;
			}
		}
		gnome_print_filter_changed (GNOME_PRINT_FILTER (mp));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, n, pspec);
	}
}

static void
gnome_print_filter_multipage_finalize (GObject *object)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (object);

	gnome_print_filter_multipage_free_affines (mp);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

#define CR(res) { gint r = (res); if (r < 0) return r; }

static gint
gnome_print_filter_multipage_clip (GnomePrintFilter *filter,
		const ArtBpath *bpath, ArtWindRule rule)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);
	ArtBpath *p;
	gint ret;

	if (!mp->affines)
		return parent_class->clip (filter, bpath, rule);

	p = art_bpath_affine_transform (bpath, mp->subpage->data);

	ret = parent_class->clip (filter, p, rule);
	art_free (p);
	return ret;
}

static gint
gnome_print_filter_multipage_fill (GnomePrintFilter *filter,
		const ArtBpath *bpath, ArtWindRule rule)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);
	ArtBpath *p;
	gint ret;
	GnomePrintContext *pc = NULL;

	if (!mp->affines)
		return parent_class->fill (filter, bpath, rule);

	g_object_get (G_OBJECT (filter), "context", &pc, NULL);
	p = art_bpath_affine_transform (bpath, mp->subpage->data);

	gnome_print_setrgbcolor (pc, gp_gc_get_red (pc->gc), gp_gc_get_green (pc->gc), gp_gc_get_blue (pc->gc));
	gnome_print_setopacity (pc, gp_gc_get_opacity (pc->gc));

	ret = parent_class->fill (filter, p, rule);
	art_free (p);
	return ret;
}

static gint
gnome_print_filter_multipage_stroke (GnomePrintFilter *filter, const ArtBpath *bpath)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);
	const ArtVpathDash *dash;
	ArtBpath *p;
	gint ret;
	GnomePrintContext *pc;

	if (!mp->affines)
		return parent_class->stroke (filter, bpath);

	g_object_get (G_OBJECT (filter), "context", &pc, NULL);
	dash = gp_gc_get_dash (pc->gc);
	p = art_bpath_affine_transform (bpath, mp->subpage->data);

	gnome_print_setrgbcolor   (pc, gp_gc_get_red (pc->gc), gp_gc_get_green (pc->gc), gp_gc_get_blue (pc->gc));
	gnome_print_setopacity    (pc, gp_gc_get_opacity    (pc->gc));
	gnome_print_setlinewidth  (pc, gp_gc_get_linewidth  (pc->gc));
	gnome_print_setmiterlimit (pc, gp_gc_get_miterlimit (pc->gc));
	gnome_print_setlinejoin   (pc, gp_gc_get_linejoin   (pc->gc));
	gnome_print_setlinecap    (pc, gp_gc_get_linecap    (pc->gc));
	gnome_print_setdash       (pc, dash->n_dash, dash->dash, dash->offset);

	ret = parent_class->stroke (filter, p);
	art_free (p);
	return ret;
}

static gint
gnome_print_filter_multipage_image (GnomePrintFilter *filter,
		const gdouble *affine, const guchar *px,
		gint w, gint h, gint rowstride, gint ch)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);
	gdouble a[6];

	if (!mp->affines)
		return parent_class->image (filter, a, px, w, h, rowstride, ch);

	art_affine_multiply (a, affine, mp->subpage->data);
	return parent_class->image (filter, a, px, w, h, rowstride, ch);
}

static gint
gnome_print_filter_multipage_glyphlist (GnomePrintFilter *filter, 
		const gdouble *affine, GnomeGlyphList *gl)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);
	gdouble a[6];

	if (!mp->affines)
		return parent_class->glyphlist (filter, a, gl);

	art_affine_multiply (a, affine, mp->subpage->data);
	return parent_class->glyphlist (filter, a, gl);
	return GNOME_PRINT_OK;
}

static void
gnome_print_filter_multipage_flush (GnomePrintFilter *filter)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);

	if (!mp->affines) {
		parent_class->flush (filter);
		return;
	}

	if (mp->subpage != mp->affines) {
		/* We have not filled whole page yet */
		mp->subpage = mp->affines;
		parent_class->showpage (filter);
	}
}

static gint
gnome_print_filter_multipage_beginpage (GnomePrintFilter *filter,
		GnomePrintContext *pc, const guchar *name)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);

	if (!mp->affines)
		return parent_class->beginpage (filter, pc, name);

	if (mp->subpage == mp->affines)
		CR (parent_class->beginpage (filter, pc, name));

	/* Start new local page gsave and current affine matrix */
	CR (parent_class->gsave (filter));

	return GNOME_PRINT_OK;
}

static gint
gnome_print_filter_multipage_showpage (GnomePrintFilter *filter)
{
	GnomePrintFilterMultipage *mp = GNOME_PRINT_FILTER_MULTIPAGE (filter);

	if (!mp->affines)
		return parent_class->showpage (filter);

	CR (parent_class->grestore (filter));

	mp->subpage = mp->subpage->next;
	if (mp->subpage == NULL) {

		/*
		 * Finished global page, start from the
		 * beginning and show it 
		 */
		mp->subpage = mp->affines;
		CR (parent_class->showpage (filter));
	}

	return GNOME_PRINT_OK;
}

static void
gnome_print_filter_multipage_class_init (GnomePrintFilterMultipageClass *klass)
{
	GObjectClass *object_class;
	GnomePrintFilterClass *f_class;

	f_class = GNOME_PRINT_FILTER_CLASS (klass);
	f_class->beginpage = gnome_print_filter_multipage_beginpage;
	f_class->showpage  = gnome_print_filter_multipage_showpage;
	f_class->clip      = gnome_print_filter_multipage_clip;
	f_class->fill      = gnome_print_filter_multipage_fill;
	f_class->stroke    = gnome_print_filter_multipage_stroke;
	f_class->image     = gnome_print_filter_multipage_image;
	f_class->glyphlist = gnome_print_filter_multipage_glyphlist;
	f_class->flush     = gnome_print_filter_multipage_flush;

	object_class = (GObjectClass *) klass;
	object_class->finalize     = gnome_print_filter_multipage_finalize;
	object_class->get_property = gnome_print_filter_multipage_get_property;
	object_class->set_property = gnome_print_filter_multipage_set_property;

	g_object_class_override_property (object_class, PROP_NAME, "name");
	g_object_class_override_property (object_class, PROP_DESCRIPTION, "description");
	g_object_class_install_property (object_class, PROP_AFFINES,
		g_param_spec_value_array ("affines", _("Affines"), _("Affines"),
			g_param_spec_double ("affine", _("Affine"), _("Affine"),
				-G_MAXDOUBLE, G_MAXDOUBLE, 0., G_PARAM_READWRITE), G_PARAM_READWRITE));

	parent_class = g_type_class_peek_parent (klass);
}

GType
gnome_print_filter_multipage_get_type (void)
{
	static GType type = 0;
	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePrintFilterMultipageClass),
			NULL, NULL,
			(GClassInitFunc) gnome_print_filter_multipage_class_init,
			NULL, NULL,
			sizeof (GnomePrintFilterMultipage),
			0, NULL
		};
		type = g_type_register_static (GNOME_TYPE_PRINT_FILTER, "GnomePrintFilterMultipage", &info, 0);
	}
	return type;
}

G_MODULE_EXPORT GType gnome_print__filter_get_type (void);

G_MODULE_EXPORT GType
gnome_print__filter_get_type (void)
{
	return GNOME_TYPE_PRINT_FILTER_MULTIPAGE;
}
