/*
 * GNOME-MAG Magnification service for GNOME
 *
 * Copyright 2006 Carlos Eduardo Rodrigues Digenes
 * Copyright 2004 Sun Microsystems Inc. (damage-client.c)
 * Copyright 2001 Sun Microsystems Inc. (magnifier.c)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "magnifier.h"
#include "magnifier-private.h"
#include "gmag-events.h"

#include <stdlib.h>

#include <X11/Xlib.h>
#ifdef HAVE_XFIXES
#include <X11/extensions/Xfixes.h>
#ifdef HAVE_DAMAGE
#include <X11/extensions/Xdamage.h>
#ifdef HAVE_COMPOSITE
#include <X11/extensions/Xrender.h>
#include <X11/extensions/Xcomposite.h>
#endif /* HAVE_COMPOSITE */
#endif /* HAVE_DAMAGE */
#endif /* HAVE_XFIXES */

#include <glib.h>

#include <gdk/gdkx.h>
#include <gtk/gtk.h>

static Display       *dpy_conn = NULL;
static guint          dpy_gsource = 0;
static Window         root_window, mag_window;

static gboolean       use_damage, use_composite;

gint                  fixes_event_base = 0, fixes_error_base;
#ifdef HAVE_XFIXES
#ifdef HAVE_DAMAGE
static gint           damage_event_base, damage_error_base;
static Damage         root_window_damage;
static XserverRegion  gmag_events_tmp_region;
#ifdef HAVE_COMPOSITE
static GQueue        *mag_windows_list;
static Damage         off_screen_damage;
static Picture        off_screen_picture;
static XserverRegion  off_screen_region;
static XserverRegion  tmp_region, new_region, old_region, exp_region;
#endif /* HAVE_COMPOSITE */
#endif /* HAVE_DAMAGE */
#endif /* HAVE_XFIXES */

#define EVENTS_DEBUG
#undef  EVENTS_DEBUG

#ifdef  EVENTS_DEBUG

#ifdef HAVE_DAMAGE
#define DAMAGE_DEBUG
#undef  DAMAGE_DEBUG
#endif /* HAVE_DAMAGE */

#ifdef HAVE_COMPOSITE
#define COMPOSITE_DEBUG
#undef  COMPOSITE_DEBUG
#endif /* HAVE_COMPOSITE */

#ifdef HAVE_XFIXES
#define CURSOR_DEBUG
#undef  CURSOR_DEBUG
#define XFIXES_DEBUG
#undef  XFIXES_DEBUG
#endif /* HAVE_XFIXES */

#endif /* EVENTS_DEBUG */

#ifdef HAVE_XFIXES
#ifdef HAVE_COMPOSITE

/*
 * GCompareFunc used in g_queue_find_custom to find windows in
 * mag_windows_list.
 */
static gint
gmag_events_g_compare_func (GmagWinPtr pgmag_win, Window xwin)
{
	if (pgmag_win->xwin == xwin)
		return 0;

	return 1;
}

/*
 * Calculates the clip for all windows in mag_windows_list.
 */
static void
gmag_events_calculate_windows_clip ()
{
	GList         *elem = NULL;
	XserverRegion clipSum;

	clipSum = XFixesCreateRegion (dpy_conn, 0, 0);
	elem = g_queue_peek_tail_link (mag_windows_list);
	if (!elem) {
		XFixesDestroyRegion (dpy_conn, clipSum);
		return;
	}
	do {
		GmagWinPtr pgmag_win = (GmagWinPtr) elem->data;
		if (pgmag_win->pic)
			if (pgmag_win->attr.map_state == IsViewable) {
				XFixesCopyRegion (
					dpy_conn,
					pgmag_win->clip,
					XFixesCreateRegionFromWindow (
						dpy_conn,
						pgmag_win->xwin,
						WindowRegionBounding));
				XFixesTranslateRegion (
					dpy_conn,
					pgmag_win->clip, pgmag_win->attr.x,
					pgmag_win->attr.y);
				XFixesCopyRegion (
					dpy_conn,
					pgmag_win->win_region,
					pgmag_win->clip);
				XFixesSubtractRegion (
					dpy_conn,
					pgmag_win->clip, pgmag_win->clip,
					clipSum);
				XFixesUnionRegion (
					dpy_conn,
					clipSum, clipSum,
					pgmag_win->win_region);
			}
	} while ((elem = g_list_previous (elem)));
	XFixesDestroyRegion (dpy_conn, clipSum);
}

/*
 * Calculates the clip of a single window in mag_windows_list.
 */
static void
gmag_events_calculate_window_clip (GmagWinPtr pgmag_win_newclip)
{
	GList         *elem = NULL;
	XserverRegion clipSum;

	clipSum = XFixesCreateRegion (dpy_conn, 0, 0);
	elem = g_queue_peek_tail_link (mag_windows_list);
	if (!elem) {
		XFixesDestroyRegion (dpy_conn, clipSum);
		return;
	}
	do {
		GmagWinPtr pgmag_win = (GmagWinPtr) elem->data;
		if (pgmag_win->xwin == pgmag_win_newclip->xwin) {
			/* the window that must have a new clip was founded */
			XFixesCopyRegion (
				dpy_conn,
				pgmag_win->clip,
				XFixesCreateRegionFromWindow (
					dpy_conn,
					pgmag_win->xwin,
					WindowRegionBounding));
			XFixesTranslateRegion (dpy_conn,
					       pgmag_win->clip,
					       pgmag_win->attr.x,
					       pgmag_win->attr.y);
			XFixesCopyRegion (dpy_conn,
					  pgmag_win->win_region,
					  pgmag_win->clip);
			XFixesSubtractRegion (dpy_conn,
					      pgmag_win->clip, pgmag_win->clip,
					      clipSum);
			break;
		}
		if (pgmag_win->pic)
			if (pgmag_win->attr.map_state == IsViewable) {
				XFixesUnionRegion (
					dpy_conn,
					clipSum, clipSum,
					pgmag_win->win_region);
			}
	} while ((elem = g_list_previous (elem)));
	XFixesDestroyRegion (dpy_conn, clipSum);
}

/*
 * Paint a window. If region is None, the window clip region is painted, else
 * the intersection of the window clip region and region is painted.
 */
static void
gmag_events_paint_window (GmagWinPtr pgmag_win, XserverRegion region)
{
	static XserverRegion final_clip = None;

	if (!pgmag_win->damaged && !region)
		return;
	if (!pgmag_win->pic)
		return;
	if (pgmag_win->attr.map_state != IsViewable)
		return;

	if (!final_clip)
		final_clip = XFixesCreateRegion (
			dpy_conn, 0, 0);

	XFixesSetRegion (dpy_conn, final_clip, 0, 0);

	if (region) {
		XFixesIntersectRegion (dpy_conn,
				       final_clip, region, pgmag_win->clip);
		XFixesSetPictureClipRegion (dpy_conn,
					    pgmag_win->pic,
					    -(pgmag_win->attr.x),
					    -(pgmag_win->attr.y), final_clip);
	} else
		XFixesSetPictureClipRegion (dpy_conn,
					    pgmag_win->pic,
					    -(pgmag_win->attr.x),
					    -(pgmag_win->attr.y),
					    pgmag_win->clip);
	XRenderComposite (dpy_conn, PictOpSrc,
			  pgmag_win->pic, None, off_screen_picture,
			  0, 0, 0, 0, pgmag_win->attr.x, pgmag_win->attr.y,
			  pgmag_win->attr.width, pgmag_win->attr.height);
}

/*
 * Paint all the windows in mag_windows_list with the specified
 * exposedRegion.
 */
static void
gmag_events_paint_windows (XserverRegion exposedRegion)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

	elem = g_queue_peek_head_link (mag_windows_list);

	while (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		gmag_events_paint_window (pgmag_win, exposedRegion);
		elem = g_list_next (elem);
	}
}

/*
 * Sometimes XGetWindowAttributes fail (when the window is destroied), so we
 * put default values in it to not have problems in other parts of the program.
 * I think that only some ones need to be setted, but this was copied from
 * Compiz, so...
 */
static void
gmag_events_set_default_window_attributes (XWindowAttributes *wa)
{
	wa->x		          = 0;
	wa->y		          = 0;
	wa->width		  = 1;
	wa->height		  = 1;
	wa->border_width	  = 0;
	wa->depth		  = 0;
	wa->visual		  = NULL;
	wa->root		  = None;
	wa->class		  = InputOnly;
	wa->bit_gravity	          = NorthWestGravity;
	wa->win_gravity	          = NorthWestGravity;
	wa->backing_store	  = NotUseful;
	wa->backing_planes	  = 0;
	wa->backing_pixel	  = 0;
	wa->save_under	          = FALSE;
	wa->colormap	          = None;
	wa->map_installed	  = FALSE;
	wa->map_state	          = IsUnviewable;
	wa->all_event_masks	  = 0;
	wa->your_event_mask	  = 0;
	wa->do_not_propagate_mask = 0;
	wa->override_redirect     = TRUE;
	wa->screen		  = NULL;
}

/*
 * Creates the necessary information of a redirected window and add it to
 * mag_windows_list.
 */
static void
gmag_events_add_window (Window xwin)
{
	GmagWinPtr                new;
	XRenderPictureAttributes  pic_attr;
	XRenderPictFormat        *format;

	new = (GmagWinPtr) malloc (sizeof (GmagWin));
	if (!new)
		g_error ("can't allocate GmagWin (struct _GmagWin)");

	if (!XGetWindowAttributes (dpy_conn, xwin,
				   &new->attr))
		gmag_events_set_default_window_attributes (&new->attr);

	new->xwin = xwin;

	if (new->attr.class == InputOnly) {
		new->pic = None;
		new->damage = None;
		new->damaged = FALSE;
		new->damaged_region = None;
	} else {
		format = XRenderFindVisualFormat (
			dpy_conn, new->attr.visual);
		pic_attr.subwindow_mode = IncludeInferiors;
		new->pic = XRenderCreatePicture (
			dpy_conn, xwin, format,
			CPSubwindowMode, &pic_attr);
		new->damage = XDamageCreate (dpy_conn, xwin,
					     XDamageReportDeltaRectangles);
		new->damaged = TRUE;
		new->damaged_region = XFixesCreateRegion (dpy_conn, 0, 0);
		new->clip = XFixesCreateRegion (dpy_conn, 0, 0);
		new->win_region = XFixesCreateRegion (dpy_conn, 0, 0);
	}
	
	g_queue_push_tail (mag_windows_list, new);
}

/*
 * Create the mag_windows_list querying the xserver for the actual
 * windows and adding them to the windows list.
 */
static void
gmag_events_create_windows_list ()
{
	Window  root_return, parent_return, *children;
	guint   nchildren;
	gint    i;

	if (!mag_windows_list)
		mag_windows_list = g_queue_new ();

	XGrabServer (dpy_conn);
	XSelectInput (dpy_conn, root_window,
		      SubstructureNotifyMask);
	XQueryTree (dpy_conn, root_window,
		    &root_return, &parent_return, &children, &nchildren);
	for (i = 0; i < nchildren; i++)
		gmag_events_add_window (children[i]);
	XFree (children);
	XUngrabServer (dpy_conn);
}

/*
 * Destroy the window resources and remove it from the
 * mag_windows_list.
 */
static void
gmag_events_remove_window (Window xwin)
{
	GList     *elem = NULL;
	GmagWinPtr pgmag_win;

	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) xwin,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		g_queue_remove (mag_windows_list, pgmag_win);
		XFixesDestroyRegion (dpy_conn,
				     pgmag_win->clip);
		XFixesDestroyRegion (dpy_conn,
				     pgmag_win->win_region);
		free (pgmag_win);
	}
}

/*
 * Add a window damaged region, making a union with the actual damaged region,
 * to the window in mag_windows_list.
 */
static void
gmag_events_add_win_damaged_region (Window xwin, XserverRegion region)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) xwin,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		XFixesTranslateRegion (dpy_conn, region,
				       pgmag_win->attr.x, pgmag_win->attr.y);
		XFixesUnionRegion (dpy_conn,
				   pgmag_win->damaged_region,
				   pgmag_win->damaged_region, region);
		pgmag_win->damaged = TRUE;
	}
}

/*
 * Paint all the windows that have some damage.
 */
static void
gmag_events_paint_damaged_windows ()
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

	elem = g_queue_peek_head_link (mag_windows_list);
	while (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		if (pgmag_win->damaged) {
			gmag_events_paint_window (pgmag_win,
						  pgmag_win->damaged_region);
			XFixesSetRegion (dpy_conn,
					 pgmag_win->damaged_region, 0, 0);
			pgmag_win->damaged = FALSE;
		}

		elem = g_list_next (elem);
	}
}

static void
gmag_events_circulate_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received CirculateNotify event: 0x%x\n",
		(guint) ev->xcirculate.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xcirculate.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xcirculate.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		g_queue_remove (mag_windows_list, pgmag_win);
		if (ev->xcirculate.place == PlaceOnTop) {
			g_queue_push_tail (mag_windows_list,
					   pgmag_win);
			if (pgmag_win->attr.map_state == IsViewable) {
				XFixesSubtractRegion (
					dpy_conn,
					tmp_region, pgmag_win->win_region,
					pgmag_win->clip);
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region, tmp_region);
			}
		} else {
			g_queue_push_head (mag_windows_list,
					   pgmag_win);
			if (pgmag_win->attr.map_state == IsViewable)
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region,
					pgmag_win->clip);
		}
	}
}

static void
gmag_events_configure_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;
	
#ifdef COMPOSITE_DEBUG
	printf ("Received ConfigureNotify event: 0x%x\n",
		(guint) ev->xconfigure.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xconfigure.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xconfigure.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		if ((pgmag_win->attr.x != ev->xconfigure.x) ||
		    (pgmag_win->attr.y != ev->xconfigure.y) ||
		    (pgmag_win->attr.width != ev->xconfigure.width) ||
		    (pgmag_win->attr.height != ev->xconfigure.height) ||
		    (pgmag_win->attr.border_width !=
		     ev->xconfigure.border_width)) {
			/* If an attribute of the window has changed we could
			 * have an exposed area that is not reported due to the
			 * overlay window. So we subtract the new region, from
			 * the old one, and we have the value of the exposed
			 * region that must be repainted.
			 */
			pgmag_win->attr.x = ev->xconfigure.x;
			pgmag_win->attr.y = ev->xconfigure.y;
			pgmag_win->attr.width = ev->xconfigure.width;
			pgmag_win->attr.height = ev->xconfigure.height;
			pgmag_win->attr.border_width =
				ev->xconfigure.border_width;
			      
			if (pgmag_win->attr.map_state == IsViewable) {
				XFixesCopyRegion (
					dpy_conn,
					old_region, pgmag_win->clip);
				gmag_events_calculate_window_clip (pgmag_win);
				XFixesCopyRegion (
					dpy_conn,
					new_region, pgmag_win->clip);
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region, old_region);
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region, new_region);
			}
		}
		if (!ev->xconfigure.above) {
			g_queue_remove (mag_windows_list, pgmag_win);
			g_queue_push_head (mag_windows_list,
					   pgmag_win);
			if (pgmag_win->attr.map_state == IsViewable) {
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region,
					pgmag_win->win_region);
			}
		} else {
			elem = g_queue_find_custom (
				mag_windows_list,
				(gconstpointer) ev->xconfigure.above,
				(GCompareFunc) gmag_events_g_compare_func);
			if (elem) {
				g_queue_remove (mag_windows_list,
						pgmag_win);
				g_queue_insert_after (mag_windows_list,
						      elem, pgmag_win);
				if (pgmag_win->attr.map_state == IsViewable) {
					XFixesUnionRegion (
						dpy_conn,
						exp_region, exp_region,
						pgmag_win->win_region);
				}
			}
		}
	}
}

static void
gmag_events_create_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received CreateNotify event: 0x%x\n",
		(guint) ev->xcreatewindow.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xcreatewindow.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	gmag_events_add_window (ev->xcreatewindow.window);
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xcreatewindow.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		if (pgmag_win->attr.map_state == IsViewable) {
			gmag_events_calculate_window_clip (pgmag_win);
			XFixesUnionRegion (dpy_conn,
					   exp_region, exp_region,
					   pgmag_win->clip);
		}
	}
}

static void
gmag_events_destroy_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received DestroyNotify event: 0x%x\n",
		(guint) ev->xdestroywindow.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xdestroywindow.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xdestroywindow.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		if (pgmag_win->attr.map_state == IsViewable)
			XFixesUnionRegion (dpy_conn,
					   exp_region, exp_region,
					   pgmag_win->clip);
		gmag_events_remove_window (ev->xdestroywindow.window);
	}
}

static void
gmag_events_map_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received MapNotify event: 0x%x\n",
		(guint) ev->xmap.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xmap.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xmap.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		pgmag_win->attr.map_state = IsViewable;
		gmag_events_calculate_window_clip (pgmag_win);
		XFixesUnionRegion (dpy_conn, exp_region,
				   exp_region, pgmag_win->clip);
	}
}

static void
gmag_events_unmap_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received UnmapNotify event: 0x%x\n",
		(guint) ev->xunmap.window);
#endif /* COMPOSITE_DEBUG */
	if (ev->xunmap.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	elem = g_queue_find_custom (mag_windows_list,
				    (gconstpointer) ev->xunmap.window,
				    (GCompareFunc) gmag_events_g_compare_func);
	if (elem) {
		pgmag_win = (GmagWinPtr) elem->data;
		pgmag_win->attr.map_state = IsUnmapped;
		XFixesUnionRegion (dpy_conn, exp_region,
				   exp_region, pgmag_win->clip);
	}
}

static void
gmag_events_reparent_notify_handler (XEvent *ev)
{
	GList      *elem;
	GmagWinPtr  pgmag_win;

#ifdef COMPOSITE_DEBUG
	printf ("Received ReparentNotify event: 0x%x (Window), 0x%x (Parent)\n", (guint) ev->xreparent.window, (guint) ev->xreparent.parent);
#endif /* COMPOSITE_DEBUG */
	if  (ev->xreparent.window == mag_window) {
#ifdef HAVE_OVERLAY
#ifdef COMPOSITE_DEBUG
		printf ("Overlay window = 0x%x\n",
			(guint) gmag_events_overlay_window);
#endif /* COMPOSITE_DEBUG */
#endif /* HAVE_OVERLAY */
		return;
	}
	if (ev->xreparent.parent != root_window) {
		gmag_events_remove_window (ev->xreparent.window);
	} else {
		gmag_events_add_window (ev->xreparent.window);
		elem = g_queue_find_custom (
			mag_windows_list,
			(gconstpointer) ev->xreparent.window,
			(GCompareFunc) gmag_events_g_compare_func);
		if (elem) {
			pgmag_win = (GmagWinPtr) elem->data;
			if (pgmag_win->attr.map_state == IsViewable) {
			        gmag_events_calculate_window_clip (pgmag_win);
				XFixesUnionRegion (
					dpy_conn,
					exp_region, exp_region,
					pgmag_win->clip);
			}
		}
	}
}

#endif /* HAVE_COMPOSITE */

#ifdef HAVE_DAMAGE

static void
gmag_events_damage_notify_handler (XEvent *ev)
{
	XDamageNotifyEvent *dev = (XDamageNotifyEvent *) ev;
#ifdef DAMAGE_DEBUG
	g_message ("Damage area %3d, %3d x %3d, %3d",
		   (int) dev->area.x, (int) dev->area.x + dev->area.width,
		   (int) dev->area.y, (int) dev->area.y + dev->area.height);
	g_message ("Damage geometry %3d, %3d x %3d, %3d",
		   (int) dev->geometry.x,
		   (int) dev->geometry.x + dev->geometry.width,
		   (int) dev->geometry.y,
		   (int) dev->geometry.y + dev->geometry.height);
#endif /* DAMAGE_DEBUG */

#ifdef HAVE_COMPOSITE
	if (use_composite) {
		if (dev->damage == off_screen_damage) {
#ifdef DAMAGE_DEBUG
			g_message ("off_screen_damage damaged");
#endif /* DAMAGE_DEBUG */
			XDamageSubtract (dpy_conn, dev->damage, None,
					 gmag_events_tmp_region);
			XFixesUnionRegion (dpy_conn,
					   off_screen_region,
					   off_screen_region,
					   gmag_events_tmp_region);
		} else {
#ifdef DAMAGE_DEBUG
			g_message ("Window with damage: 0x%x", dev->drawable);
#endif /* DAMAGE_DEBUG */
			XDamageSubtract (dpy_conn, dev->damage, None,
					 gmag_events_tmp_region);
			gmag_events_add_win_damaged_region (
				dev->drawable, gmag_events_tmp_region);
		}
	}
#endif /* HAVE_COMPOSITE */
}

#endif /* HAVE_DAMAGE */

static void
gmag_events_cursor_convert_to_rgba (Magnifier *magnifier,
				    XFixesCursorImage *cursor_image)
{
	int i, count = cursor_image->width * cursor_image->height;
	for (i = 0; i < count; ++i) {
		guint32 pixval = GUINT_TO_LE (cursor_image->pixels[i]);
		cursor_image->pixels[i] = pixval;
	}
}

static void
gmag_events_free_cursor_pixels (guchar *pixels, gpointer data)
{
    /* XFree (data); FIXME why doesn't this work properly? */
}

#endif /* HAVE_XFIXES */

GdkPixbuf *
gmag_events_get_source_pixbuf (Magnifier *magnifier)
{
#ifdef HAVE_XFIXES
	XFixesCursorImage *cursor_image = XFixesGetCursorImage (
		dpy_conn);
        GdkPixbuf *cursor_pixbuf = NULL;
	gchar s[6];
	if (cursor_image)
	{
	        gmag_events_cursor_convert_to_rgba (magnifier, cursor_image);
		cursor_pixbuf = gdk_pixbuf_new_from_data (
			(guchar *) cursor_image->pixels, GDK_COLORSPACE_RGB,
			TRUE, 8, cursor_image->width, cursor_image->height,
			cursor_image->width * 4,
			gmag_events_free_cursor_pixels, cursor_image);
		gdk_pixbuf_set_option (cursor_pixbuf, "x_hot", 
				       g_ascii_dtostr (
					       s, 6,
					       (gdouble) cursor_image->xhot));
		gdk_pixbuf_set_option (cursor_pixbuf, "y_hot", 
				       g_ascii_dtostr (
					       s, 6,
					       (gdouble) cursor_image->yhot));
	}
	return cursor_pixbuf;
#else
	return NULL;
#endif /* HAVE_XFIXES */
}

gboolean
gmag_events_source_has_damage_extension (Magnifier *magnifier)
{
#ifdef HAVE_DAMAGE
	gint event_base, error_base;
	Display *dpy;
	g_assert (magnifier);
	dpy = GDK_DISPLAY_XDISPLAY (magnifier->source_display);
	if (g_getenv ("MAGNIFIER_IGNORE_DAMAGE"))
	        return FALSE;
	if (XDamageQueryExtension (dpy, &event_base, &error_base))
		return TRUE;
#endif /* HAVE_DAMAGE */
	return FALSE;
}

static gboolean
gmag_events_handler (GIOChannel *source, GIOCondition condition, gpointer data)
{
#ifdef HAVE_XFIXES
	XEvent                   ev;
	XFixesCursorNotifyEvent *cev = NULL;
	gboolean                 cursor_changed = FALSE;
	Magnifier               *magnifier = (Magnifier *) data;
	XRectangle              *rectlist;
#ifdef HAVE_COMPOSITE
	gboolean                 calc_clip = FALSE;
#endif /* HAVE_COMPOSITE */

#ifdef HAVE_OVERLAY
	if (magnifier->priv->overlay)
		mag_window = GDK_WINDOW_XID (magnifier->priv->overlay);
#else
	if (magnifier->priv->w && magnifier->priv->w->window)
		mag_window = GDK_WINDOW_XID (magnifier->priv->w->window);
#endif /* HAVE_OVERLAY */

	do
	{
		XNextEvent(dpy_conn, &ev);

#ifdef HAVE_COMPOSITE
		if (use_composite) {
			switch (ev.type) {
			case CirculateNotify:
				gmag_events_circulate_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case ConfigureNotify:
				gmag_events_configure_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case CreateNotify:
				gmag_events_create_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case DestroyNotify:
				gmag_events_destroy_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case MapNotify:
				gmag_events_map_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case UnmapNotify:
				gmag_events_unmap_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			case ReparentNotify:
				gmag_events_reparent_notify_handler (&ev);
				calc_clip = TRUE;
				break;
			}
		}
#endif /* HAVE_COMPOSITE */

#ifdef HAVE_DAMAGE
		if (use_damage) {
			if (ev.type == damage_event_base + XDamageNotify) {
				gmag_events_damage_notify_handler (&ev);
			}
		}
#endif /* HAVE_DAMAGE */

#ifdef HAVE_XFIXES
		if (ev.type == fixes_event_base + XFixesCursorNotify) {
			cursor_changed = TRUE;
			cev = (XFixesCursorNotifyEvent *) &ev;
		}
#endif /* HAVE_XFIXES */

	} while (XPending (dpy_conn));

#ifndef HAVE_OVERLAY
	if (use_composite && mag_window) {
		XRaiseWindow (dpy_conn, mag_window);
	}
#endif /* HAVE_OVERLAY */

#ifdef HAVE_DAMAGE
	if (!use_composite) {
		XDamageSubtract (dpy_conn, root_window_damage, None,
				 gmag_events_tmp_region);
	}

	if (use_damage) {
		if (magnifier) {
			int i, howmany;
			/* TODO: maintain this list on the client instead, to
			 * avoid the roundtrip below */
#ifdef HAVE_COMPOSITE
			if (use_composite) {
				rectlist = XFixesFetchRegion (
					dpy_conn,
					off_screen_region,
					&howmany);
			} else {
#endif /* HAVE_COMPOSITE */
				rectlist = XFixesFetchRegion (
					dpy_conn, gmag_events_tmp_region,
					&howmany);
#ifdef HAVE_COMPOSITE
			}
#endif /* HAVE_COMPOSITE */
			if (rectlist == NULL) /* no reply from fetch */
				return TRUE;
			for (i=0; i < howmany; ++i) {
				magnifier_notify_damage (magnifier,
							 &rectlist[i]);
			}
			XFree (rectlist);
		}
	}
#endif /* HAVE_DAMAGE */

#ifdef HAVE_COMPOSITE
	if (use_composite) {
		if (calc_clip) {
			gmag_events_calculate_windows_clip ();
			gmag_events_paint_windows (exp_region);
		}
		gmag_events_paint_damaged_windows ();
	}
#endif /* HAVE_COMPOSITE */

#ifdef HAVE_XFIXES
	if (cursor_changed) {
		if (magnifier->priv->use_source_cursor) {
			GdkPixbuf *cursor_pixbuf =
				gmag_events_get_source_pixbuf (magnifier);
			magnifier_set_cursor_from_pixbuf (magnifier,
							  cursor_pixbuf);
			if (cursor_pixbuf) g_object_unref (cursor_pixbuf);
		} else {
			magnifier_set_cursor_pixmap_by_name (magnifier, cev ? gdk_x11_get_xatom_name (cev->cursor_name) : "default", TRUE);
		}
	  
		magnifier_transform_cursor (magnifier);
#ifdef CURSOR_DEBUG
		if (cev)
			g_message ("cursor changed: subtype=%d, " \
				   "cursor_serial=%lu, name=[%x] %s\n",
				   (int) cev->subtype, cev->cursor_serial,
				   (int) cev->cursor_name,
				   gdk_x11_get_xatom_name (cev->cursor_name));
#endif /* CURSOR_DEBUG */
		cursor_changed = FALSE;
	}
#endif /* HAVE_XFIXES */

#ifdef HAVE_COMPOSITE
	if (use_composite) {
		XFixesSetRegion (dpy_conn, tmp_region, 0, 0);
		XFixesSetRegion (dpy_conn, new_region, 0, 0);
		XFixesSetRegion (dpy_conn, old_region, 0, 0);
		XFixesSetRegion (dpy_conn, exp_region, 0, 0);
		XFixesSetRegion (dpy_conn, off_screen_region, 0, 0);
	}
#endif /* HAVE_COMPOSITE */

	XFlush (dpy_conn);
#else
	return FALSE;
#endif /* HAVE_XFIXES */
	return TRUE;
}

static gboolean
gmag_events_use_damage ()
{
#ifdef HAVE_DAMAGE
	gint major, event, error;
	if (XQueryExtension (dpy_conn, "DAMAGE", &major, &event, &error) &&
	    !g_getenv ("MAGNIFIER_IGNORE_DAMAGE"))
		return TRUE;
	return FALSE;
#else
	return FALSE;
#endif /* HAVE_DAMAGE */
}

static gboolean
gmag_events_use_composite ()
{
	if (!gmag_events_use_damage ()) {
		return FALSE;
	}
#ifdef HAVE_COMPOSITE
	gint major, event, error;
	if (XQueryExtension (dpy_conn, "Composite", &major, &event, &error) &&
	    !g_getenv ("MAGNIFIER_IGNORE_COMPOSITE"))
		return TRUE;
	return FALSE;
#else
	return FALSE;
#endif /* HAVE_COMPOSITE */
}

void
gmag_events_client_init (Magnifier *magnifier)
{
	GIOChannel               *ioc;
	gint                      fd;
	gint                      event_base, error_base;
#ifdef HAVE_COMPOSITE
	XRenderPictureAttributes  pic_attr;
	XRenderPictFormat        *format;
	GdkDisplay               *gdk_display_connection;
	GdkScreen                *gdkscr;
	gint                      scr = 0, root_w, root_h;
#endif /* HAVE_COMPOSITE */

	if (dpy_conn) {
		/* remove the old watch */
		if (dpy_gsource) 
			g_source_remove (dpy_gsource);
		XCloseDisplay (dpy_conn);
	}

	if (magnifier) {
		/* we need our own connection here to keep from gumming up the
		 * works */
		dpy_conn = XOpenDisplay (magnifier->source_display_name); 
		root_window = GDK_WINDOW_XWINDOW (magnifier->priv->root);
	} else {
		dpy_conn = XOpenDisplay (NULL);
		root_window = RootWindow (dpy_conn, DefaultScreen (dpy_conn));
		g_message ("warning - using DefaultScreen for X connection.");
	}

#ifdef EVENTS_DEBUG
	XSynchronize (dpy_conn, True);
#endif /* EVENTS_DEBUG */

	fd = ConnectionNumber (dpy_conn);
	ioc = g_io_channel_unix_new (fd);
	dpy_gsource = g_io_add_watch (ioc,
				      G_IO_IN | G_IO_HUP | G_IO_PRI | G_IO_ERR,
				      gmag_events_handler, magnifier);
	g_io_channel_unref (ioc); 

#ifdef HAVE_XFIXES

	use_damage = gmag_events_use_damage ();
	use_composite = gmag_events_use_composite ();

	if (!XFixesQueryExtension (dpy_conn, &fixes_event_base,
				   &fixes_error_base)) {
		g_warning ("XFixes extension not currently active.\n");
	} else {
		XFixesSelectCursorInput (dpy_conn, root_window,
					 XFixesDisplayCursorNotifyMask);
		g_message ("added event source to xfixes cursor-notify " \
			   "connection");
	}

#ifdef HAVE_DAMAGE
	if (!XDamageQueryExtension (dpy_conn, &damage_event_base,
				    &damage_error_base)) {
		g_warning ("Damage extension not currently active.\n");
	} else if (g_getenv ("MAGNIFIER_IGNORE_DAMAGE")) {
		g_warning ("Damage extension being ignored at user request.");
	} else {
		gmag_events_tmp_region = XFixesCreateRegion (dpy_conn, 0, 0);
		if (!use_composite) {
			root_window_damage = XDamageCreate (
				dpy_conn, root_window,
				XDamageReportDeltaRectangles);
			/* I don't know why, but without this XDamageSubtract
			 * call below the damage events aren't hanled normally.
			 * They start to be handled normally, without the call
			 * below, only after you move your mouse.
			 */
			XDamageSubtract (dpy_conn, root_window_damage, None,
					 None);
		}
		g_message ("added event source to damage connection");
	}
#else
	g_warning ("this copy of gnome-mag was built without damage " \
		   "extension support.\n");
#endif /* HAVE_DAMAGE */

#ifdef HAVE_COMPOSITE
	if (!XCompositeQueryExtension (dpy_conn, &event_base, &error_base)) {
		g_warning ("Composite extension not currently active.\n");
	} else if (g_getenv ("MAGNIFIER_IGNORE_COMPOSITE")) {
		g_warning ("Composite extension being ignored at user " \
			   "request.");
	} else if (!use_damage) {
		g_setenv ("MAGNIFIER_IGNORE_COMPOSITE", "1", TRUE);
		g_warning ("Composite extension being ignored due Damage " \
			   "is not actived.");
	} else {
#ifndef HAVE_OVERLAY
		g_warning ("update composite to version 0.3 or higher to " \
			   "have overlay window support.\n");
#endif /* HAVE_OVERLAY */

		gdk_drawable_get_size (magnifier->priv->root, &root_w,
				       &root_h);
		magnifier->priv->source_drawable = gdk_pixmap_new (
			magnifier->priv->root, root_w, root_h, -1);
		/* GTK+ uses it's own connection with X, so we must flush that
		 * to not receive a BadDrawable when creating a picture of this
		 * drawable below. */
		gdk_flush ();

		gdk_display_connection = gdk_drawable_get_display (
			magnifier->priv->root);
		gdkscr = gdk_display_get_default_screen (
			gdk_display_connection);
	    
		scr = GDK_SCREEN_XNUMBER (gdkscr);

		XCompositeRedirectSubwindows (dpy_conn, root_window,
					      CompositeRedirectAutomatic);
		off_screen_region = XFixesCreateRegion (
			dpy_conn, 0, 0);
		tmp_region = XFixesCreateRegion (dpy_conn, 0, 0);
		new_region = XFixesCreateRegion (dpy_conn, 0, 0);
		old_region = XFixesCreateRegion (dpy_conn, 0, 0);
		exp_region = XFixesCreateRegion (dpy_conn, 0, 0);
		off_screen_damage = XDamageCreate (
				dpy_conn, 
				GDK_DRAWABLE_XID (
					magnifier->priv->source_drawable),
				XDamageReportDeltaRectangles);

		format = XRenderFindVisualFormat (
			dpy_conn,
			DefaultVisual (dpy_conn, scr));
		pic_attr.subwindow_mode = IncludeInferiors;
		off_screen_picture = XRenderCreatePicture (
			dpy_conn,
			GDK_DRAWABLE_XID (magnifier->priv->source_drawable),
			format, CPSubwindowMode, &pic_attr);

		gmag_events_create_windows_list (gdk_display_connection,
						 gdkscr);
		gmag_events_calculate_windows_clip ();
		g_message ("added event source to composite connection");
	}
#else
	g_warning ("this copy of gnome-mag was built without composite " \
		   "extension support.\n");
#endif /* HAVE_COMPOSITE */

#else
	g_warning ("this copy of gnome-mag was built without xfixes " \
		   "extension support.\n");	
#endif /* HAVE_XFIXES */
	XFlush (dpy_conn);
}
