
/* Ekiga -- A VoIP and Video-Conferencing application
 * Copyright (C) 2000-2007 Damien Sandras
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 *
 * Ekiga is licensed under the GPL license and as a special exception,
 * you have permission to link or otherwise combine this program with the
 * programs OPAL, OpenH323 and PWLIB, and distribute the combination,
 * without applying the requirements of the GNU GPL to the OPAL, OpenH323
 * and PWLIB programs, as long as you do follow the requirements of the
 * GNU GPL for all the rest of the software thus combined.
 */


/*
 *                         evolution-book.cpp  -  description
 *                         ------------------------------------------
 *   begin                : written in 2007 by Julien Puydt
 *   copyright            : (c) 2007 by Julien Puydt
 *   description          : implementation of an evolution addressbook
 *
 */

#include <iostream>
#include <string>

#include "config.h"

#include "form-request-simple.h"

#include "evolution-book.h"
#include "evolution-contact.h"

/* this structure is so that when we give ourself to a C callback,
 * the object lives long enough
 */
struct Wrapper
{
  Wrapper (Evolution::Book &_book): book(_book) {}

  Evolution::Book &book;
};

static void
on_view_contacts_added_c (EBook * /*ebook*/,
			  GList *contacts,
			  gpointer data)
{
  ((Evolution::Book *)data)->on_view_contacts_added (contacts);
}

void
Evolution::Book::on_view_contacts_added (GList *econtacts)
{
  EContact *econtact = NULL;
  int nbr = 0;
  gchar* c_status = NULL;

  for (; econtacts != NULL; econtacts = g_list_next (econtacts)) {

    econtact = E_CONTACT (econtacts->data);

    if (e_contact_get_const (econtact, E_CONTACT_FULL_NAME) != NULL) {

      Evolution::Contact *contact =  new Evolution::Contact (services, book, econtact);

      add_contact (*contact);
      nbr++;
    }
  }

  c_status = g_strdup_printf (ngettext ("%d user found", "%d users found", nbr),
			      nbr);
  status = c_status;
  g_free (c_status);

  updated.emit ();
}

static void
on_view_contacts_removed_c (EBook */*ebook*/,
			    GList *ids,
			    gpointer data)
{
  ((Evolution::Book *)data)->on_view_contacts_removed (ids);
}

void
Evolution::Book::on_view_contacts_removed (GList *ids)
{
  for (; ids != NULL; ids = g_list_next (ids))
    for (iterator iter = begin ();
	 iter != end ();
	 iter++)
      if (iter->get_id () == (gchar *)ids->data) {

	remove_contact (*iter);
	break; // will do the loop on ids, but stop using iter which is invalid
      }
}

static void
on_view_contacts_changed_c (EBook */*ebook*/,
			    GList *econtacts,
			    gpointer data)
{
  ((Evolution::Book *)data)->on_view_contacts_changed (econtacts);
}

void
Evolution::Book::on_view_contacts_changed (GList *econtacts)
{
  EContact *econtact = NULL;

  for (; econtacts != NULL; econtacts = g_list_next (econtacts)) {

    econtact = E_CONTACT (econtacts->data);

    for (iterator iter = begin ();
	 iter != end ();
	 iter++)

      if (iter->get_id()
	  == (const gchar *)e_contact_get_const (econtact, E_CONTACT_UID))
	iter->update_econtact (econtact);
  }
}

static void
on_book_view_obtained_c (EBook */*book*/,
			 EBookStatus status,
			 EBookView *view,
			 gpointer data)
{
  ((Wrapper *)data)->book.on_book_view_obtained (status, view);
}

void
Evolution::Book::on_book_view_obtained (EBookStatus _status,
					EBookView *_view)
{
  if (_status == E_BOOK_ERROR_OK) {

    if (view != NULL)
      g_object_unref (view);

    view = _view;

    /* no need for wrappers here : the view will only die with us */
    g_object_ref (view);

    g_signal_connect (view, "contacts-added",
		      G_CALLBACK (on_view_contacts_added_c), this);

    g_signal_connect (view, "contacts-removed",
		      G_CALLBACK (on_view_contacts_removed_c), this);

    g_signal_connect (view, "contacts-changed",
		      G_CALLBACK (on_view_contacts_changed_c), this);

    e_book_view_start (view);
  } else
    removed.emit ();
}

static void
on_book_opened_c (EBook */*book*/,
		  EBookStatus _status,
		  gpointer data)
{
  ((Wrapper *)data)->book.on_book_opened (_status);
}

void
Evolution::Book::on_book_opened (EBookStatus _status)
{
  EBookQuery *query = NULL;
  self = new Wrapper (*this);

  if (_status == E_BOOK_ERROR_OK) {

    if (search_filter.empty ())
      query = e_book_query_field_exists (E_CONTACT_FULL_NAME);
    else
      query = e_book_query_field_test (E_CONTACT_FULL_NAME, 
                                       E_BOOK_QUERY_CONTAINS, 
                                       search_filter.c_str ());

    (void)e_book_async_get_book_view (book, query, NULL, 100,
				      on_book_view_obtained_c, self);

    e_book_query_unref (query);

  } 
  else {

    book = NULL;
    removed.emit ();
  }
}

Evolution::Book::Book (Ekiga::ServiceCore &_services,
		       EBook *_book)
  : services(_services), book(_book), view(NULL)
{
  self = new Wrapper (*this);

  g_object_ref (book);

  refresh ();
}

Evolution::Book::~Book ()
{
  if (book != NULL)
    g_object_unref (book);
  if (view != NULL)
    g_object_unref (view);

  delete self;
}

const std::string
Evolution::Book::get_name () const
{
  ESource *source = NULL;
  std::string result;

  source = e_book_get_source (book);
  if (source && E_IS_SOURCE (source))
    result = e_source_peek_name (source);

  return result;
}

EBook *
Evolution::Book::get_ebook () const
{
  return book;
}

bool
Evolution::Book::populate_menu (Ekiga::MenuBuilder &builder)
{
  builder.add_action ("refresh", _("_Refresh"),
		      sigc::mem_fun (this, &Evolution::Book::refresh));
  builder.add_separator ();
  builder.add_action ("new", _("New _Contact"),
		      sigc::mem_fun (this,
				     &Evolution::Book::new_contact_action));
  return true;
}

void
Evolution::Book::set_search_filter (std::string _search_filter)
{
  search_filter = _search_filter;
  refresh ();
}

const std::string
Evolution::Book::get_status () const
{
  return status;
}

const std::string
Evolution::Book::get_type () const
{
  return "local";
}

void
Evolution::Book::refresh ()
{
  /* we flush */
  iterator iter = begin ();
  while (iter != end ()) {

    remove_contact (*iter);
    iter = begin ();
  }

  /* we go */
  if (e_book_is_opened (book)) 
    on_book_opened_c (book, E_BOOK_ERROR_OK, self);
  else 
    e_book_async_open (book, TRUE,
                       on_book_opened_c, self);
}

void
Evolution::Book::new_contact_action ()
{
  Ekiga::FormRequestSimple request;

  request.title (_("New contact"));

  request.instructions (_("Please update the following fields:"));

  request.text ("name", _("_Name:"), "");
  request.text ("video", _("VoIP _URI:"), "");
  request.text ("home", _("_Home phone:"), "");
  request.text ("work", _("_Office phone:"), "");
  request.text ("cell", _("_Cell phone:"), "");
  request.text ("pager", _("_Pager:"), "");

  request.submitted.connect (sigc::mem_fun (this,
					    &Evolution::Book::on_new_contact_form_submitted));

  if (!questions.handle_request (&request)) {

    // FIXME: better error reporting
#ifdef __GNUC__
    std::cout << "Unhandled form request in "
	      << __PRETTY_FUNCTION__ << std::endl;
#endif
  }
}

void
Evolution::Book::set_econtact_attribute_value (EContact *econtact,
					       const std::string subtype,
					       const std::string value) const
{
  EVCardAttribute *attribute = NULL;
  EVCardAttributeParam *param = NULL;

  attribute = e_vcard_attribute_new ("", EVC_TEL);
  param = e_vcard_attribute_param_new (EVC_TYPE);
  e_vcard_attribute_param_add_value (param, subtype.c_str ());
  e_vcard_attribute_add_param (attribute, param);
  e_vcard_attribute_add_value (attribute, value.c_str ());
  e_vcard_add_attribute (E_VCARD (econtact), attribute);
}

void
Evolution::Book::on_new_contact_form_submitted (Ekiga::Form &result)
{
  try {
    EContact *econtact = NULL;


    /* first check we have everything before using */
    std::string name = result.text ("name");
    std::string home = result.text ("home");
    std::string cell = result.text ("cell");
    std::string work = result.text ("work");
    std::string pager = result.text ("pager");
    std::string video = result.text ("video");

    econtact = e_contact_new ();
    e_contact_set (econtact, E_CONTACT_FULL_NAME, (gpointer)name.c_str ());
    if ( !home.empty ())
      set_econtact_attribute_value (econtact, "HOME", home);
    if ( !cell.empty ())
      set_econtact_attribute_value (econtact, "CELL", cell);
    if ( !work.empty ())
      set_econtact_attribute_value (econtact, "WORK", work);
    if ( !pager.empty ())
      set_econtact_attribute_value (econtact, "PAGER", pager);
    if ( !video.empty ())
      set_econtact_attribute_value (econtact, "VIDEO", video);

    e_book_add_contact (book, econtact, NULL);
    g_object_unref (econtact);

  } catch (Ekiga::Form::not_found) {
#ifdef __GNUC__
    std::cerr << "Invalid result form submitted to "
	      << __PRETTY_FUNCTION__
	      << std::endl;
#endif
  }
}
