/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-tb-button.h"
#include "gul-gui.h"
#include "gul-string.h"
#include "galeon-reload-button.h"
#include <gtk/gtkstock.h>
#include <string.h>
#include "galeon-shell.h"

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
#define DEBUG_MSG(x) g_print x
//#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GaleonReloadButtonPrivate 
{
	GulTbButton *widget;
	gboolean sensitive;
};

/**
 * Private functions, only availble from this file
 */
static void		galeon_reload_button_class_init			(GaleonReloadButtonClass *klass);
static void		galeon_reload_button_init			(GaleonReloadButton *tb);
static void		galeon_reload_button_finalize_impl		(GObject *o);
static GtkWidget *	galeon_reload_button_get_widget_impl		(GulTbItem *i);
static GdkPixbuf *	galeon_reload_button_get_icon_impl		(GulTbItem *i);
static gchar *		galeon_reload_button_get_name_human_impl	(GulTbItem *i);
static gchar *		galeon_reload_button_to_string_impl		(GulTbItem *i);
static GulTbItem *	galeon_reload_button_clone_impl			(GulTbItem *i);
static void		galeon_reload_button_parse_properties_impl	(GulTbItem *i, const gchar *props);
static void		galeon_reload_button_clicked_cb			(GtkWidget *w, GaleonReloadButton *b);


static gpointer gul_tb_item_class;

/**
 * TbiZoom object
 */

MAKE_GET_TYPE (galeon_reload_button, "GaleonReloadButton", GaleonReloadButton,
	       galeon_reload_button_class_init, 
	       galeon_reload_button_init, GALEON_TYPE_TBI);

static void
galeon_reload_button_class_init (GaleonReloadButtonClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_reload_button_finalize_impl;
	
	GUL_TB_ITEM_CLASS (klass)->get_widget = galeon_reload_button_get_widget_impl;
	GUL_TB_ITEM_CLASS (klass)->get_icon = galeon_reload_button_get_icon_impl;
	GUL_TB_ITEM_CLASS (klass)->get_name_human = galeon_reload_button_get_name_human_impl;
	GUL_TB_ITEM_CLASS (klass)->to_string = galeon_reload_button_to_string_impl;
	GUL_TB_ITEM_CLASS (klass)->clone = galeon_reload_button_clone_impl;
	GUL_TB_ITEM_CLASS (klass)->parse_properties = galeon_reload_button_parse_properties_impl;

	gul_tb_item_class = g_type_class_peek_parent (klass);
}

static void 
galeon_reload_button_init (GaleonReloadButton *tbi)
{
	GaleonReloadButtonPrivate *p = g_new0 (GaleonReloadButtonPrivate, 1);
	tbi->priv = p;

	p->sensitive = TRUE;
}

GaleonReloadButton *
galeon_reload_button_new (void)
{
	GaleonReloadButton *ret = g_object_new (GALEON_TYPE_RELOAD_BUTTON, NULL);
	return ret;
}

static void
galeon_reload_button_finalize_impl (GObject *o)
{
	GaleonReloadButton *it = GALEON_RELOAD_BUTTON (o);
	GaleonReloadButtonPrivate *p = it->priv;

	if (p->widget)
	{
		g_object_unref (p->widget);
	}

	g_free (p);
	
	DEBUG_MSG (("GaleonReloadButton finalized\n"));
	
	G_OBJECT_CLASS (gul_tb_item_class)->finalize (o);
}

static void
galeon_reload_button_setup_widget (GaleonReloadButton *b)
{
	GaleonReloadButtonPrivate *p = b->priv;

	if (!p->widget)
	{
		galeon_reload_button_get_widget_impl (GUL_TB_ITEM (b));
	}
	g_assert (GUL_IS_TB_BUTTON (p->widget));

	gul_tb_button_set_label (p->widget, GTK_STOCK_REFRESH);
	gul_tb_button_set_tooltip_text (p->widget, _("Display the latest content of the current page"));
	gul_tb_button_set_priority (p->widget, FALSE);
	gul_tb_button_set_show_arrow (p->widget, FALSE);
	gul_tb_button_set_sensitivity (p->widget, p->sensitive);
}

static GaleonEmbed *
galeon_reload_button_get_embed (GaleonReloadButton *b)
{
	GaleonWindow *window;
	GaleonEmbed *embed;

	window = galeon_tbi_get_window (GALEON_TBI (b));
	g_return_val_if_fail (window != NULL, NULL);

	embed = galeon_window_get_active_embed (window);
	g_return_val_if_fail (embed != NULL, NULL);

	return embed;
}

static void
galeon_reload_button_reload_all_tabs_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	GList *tabs;
	GaleonWindow *window;
	GList *li;

	window = galeon_tbi_get_window (GALEON_TBI (b));
	g_return_if_fail (window != NULL);

	tabs = galeon_window_get_tabs (window);
	for (li = tabs; li; li = li->next)
	{
		GaleonEmbed *embed = galeon_tab_get_embed (li->data);
		galeon_embed_reload (embed, EMBED_RELOAD_NORMAL);
	}
	
	g_list_free (tabs);
}

static void
galeon_reload_button_reload_all_windows_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	const GList *windows;
	const GList *lj;
	Session *session;

	session = galeon_shell_get_session (galeon_shell);
	windows = session_get_windows (session);

	for (lj = windows; lj; lj = lj->next)
	{
		GaleonWindow *window = lj->data;
		GList *tabs;
		GList *li;
		
		tabs = galeon_window_get_tabs (window);
		for (li = tabs; li; li = li->next)
		{
			GaleonEmbed *embed = galeon_tab_get_embed (li->data);
			galeon_embed_reload (embed, EMBED_RELOAD_NORMAL);
		}
		
		g_list_free (tabs);
	}
}

static void
galeon_reload_button_normal_reload_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	GaleonEmbed *embed = galeon_reload_button_get_embed (b);
	galeon_embed_reload (embed, EMBED_RELOAD_NORMAL);
}

static void
galeon_reload_button_bypass_cache_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	GaleonEmbed *embed = galeon_reload_button_get_embed (b);
	galeon_embed_reload (embed, EMBED_RELOAD_BYPASSCACHE);
}

static void
galeon_reload_button_bypass_proxy_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	GaleonEmbed *embed = galeon_reload_button_get_embed (b);
	galeon_embed_reload (embed, EMBED_RELOAD_BYPASSPROXY);
}

static void
galeon_reload_button_bypass_cache_and_proxy_activate_cb (GtkMenuItem *mi, GaleonReloadButton *b)
{
	GaleonEmbed *embed = galeon_reload_button_get_embed (b);
	galeon_embed_reload (embed, EMBED_RELOAD_BYPASSCACHE | EMBED_RELOAD_BYPASSPROXY);
}

static gboolean
galeon_reload_button_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					    GaleonReloadButton *b)
{
	if (event->button == 1
	    && ((event->state & GDK_Shift_L)
		|| (event->state & GDK_Shift_R)))
	{
		GaleonEmbed *embed = galeon_reload_button_get_embed (b);
		galeon_embed_reload (embed, EMBED_RELOAD_BYPASSCACHE | EMBED_RELOAD_BYPASSPROXY);
		return TRUE;
	}
	
	return FALSE;
}

static GtkWidget *
galeon_reload_button_get_widget_impl (GulTbItem *i)
{
	GaleonReloadButton *iz = GALEON_RELOAD_BUTTON (i);
	GaleonReloadButtonPrivate *p = iz->priv;

	if (!p->widget)
	{
		GtkMenuShell *ms;
		
		p->widget = gul_tb_button_new ();
		g_object_ref (p->widget);
		gtk_object_sink (GTK_OBJECT (p->widget));

		gul_tb_button_set_use_stock (p->widget, TRUE);
		gul_tb_button_set_enable_menu (p->widget, TRUE);
		
		galeon_reload_button_setup_widget (iz);

		gtk_widget_show (GTK_WIDGET (p->widget));
		
		g_signal_connect (gul_tb_button_get_button (p->widget), "clicked",
				  G_CALLBACK (galeon_reload_button_clicked_cb), i);
		g_signal_connect (gul_tb_button_get_button (p->widget), "button-press-event",
				  G_CALLBACK (galeon_reload_button_button_press_event_cb), i);

		ms = gul_tb_button_get_menu (p->widget);

		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Normal _Reload"), 
					     G_CALLBACK (galeon_reload_button_normal_reload_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Bypass _Cache"), 
					     G_CALLBACK (galeon_reload_button_bypass_cache_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Bypass _Proxy"), 
					     G_CALLBACK (galeon_reload_button_bypass_proxy_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Bypass Cache _And Proxy"), 
					     G_CALLBACK (galeon_reload_button_bypass_cache_and_proxy_activate_cb), i);
		gul_gui_append_separator (GTK_WIDGET (ms));
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Reload All Tabs in This _Window"), 
					     G_CALLBACK (galeon_reload_button_reload_all_tabs_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Reload All Tabs in A_ll Windows"), 
					     G_CALLBACK (galeon_reload_button_reload_all_windows_activate_cb), i);
		
	}

	return GTK_WIDGET (p->widget);
}

static GdkPixbuf *
galeon_reload_button_get_icon_impl (GulTbItem *i)
{
	static GdkPixbuf *pb_reload = NULL;

	if (!pb_reload)
	{
		/* what's the easier way? */
		GtkWidget *b = gtk_button_new ();
		pb_reload = gtk_widget_render_icon (b,
						GTK_STOCK_REFRESH,
						GTK_ICON_SIZE_SMALL_TOOLBAR,
						NULL);
		gtk_widget_destroy (b);
	}

	return g_object_ref (pb_reload);
}

static gchar *
galeon_reload_button_get_name_human_impl (GulTbItem *i)
{
	return g_strdup (_("Reload"));
}

static gchar *
galeon_reload_button_to_string_impl (GulTbItem *i)
{
	/* if it had any properties, the string should include them */

	return g_strdup_printf ("%s=reload_button", i->id);
}

static GulTbItem *
galeon_reload_button_clone_impl (GulTbItem *i)
{
	GulTbItem *ret = GUL_TB_ITEM (galeon_reload_button_new ());

	gul_tb_item_set_id (ret, i->id);

	return ret;
}

static void
galeon_reload_button_parse_properties_impl (GulTbItem *it, const gchar *props)
{
	/* we have no properties */
}

void
galeon_reload_button_set_sensitive (GaleonReloadButton *b, gboolean s)
{
	GaleonReloadButtonPrivate *p = b->priv;
	p->sensitive = s;
	if (p->widget)
	{
		gul_tb_button_set_sensitivity (p->widget, s);
	}
	else
	{
		galeon_reload_button_get_widget_impl (GUL_TB_ITEM (b));
	}
}

static void
galeon_reload_button_clicked_cb (GtkWidget *w, GaleonReloadButton *b)
{
	GaleonEmbed *embed = galeon_reload_button_get_embed (b);
	galeon_embed_reload (embed, EMBED_RELOAD_NORMAL);
}
