/* gok-utf8-word-complete.c
*
* Copyright 2001,2002 Sun Microsystems, Inc.,
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Library General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Library General Public License for more details.
*
* You should have received a copy of the GNU Library General Public
* License along with this library; if not, write to the
* Free Software Foundation, Inc., 59 Temple Place - Suite 330,
* Boston, MA 02111-1307, USA.
*
*/

#include <gnome.h>
#include "gok-log.h"
#include "gok-utf8-word-complete.h"

/* private functions */

/* implementations of GokWordCompleteClass virtual methods */

typedef enum 
{
	WORD_CASE_LOWER,
	WORD_CASE_INITIAL_CAPS,
	WORD_CASE_ALL_CAPS,
	WORD_CASE_TITLE
} WordPredictionCaseType;

typedef struct
{
	gchar *string;
	gint   priority;
} WordPrediction;

/* The filename of the dictionary file */
static gchar *dictionary_full_path;

/* declarations of implementation methods */
static gchar** utf8_wordcomplete_predict_string (GokWordComplete *complete, const gchar* pWord, gint num_predictions);
static gboolean utf8_wordcomplete_open (GokWordComplete *complete, const gchar *directory);

/* private intermal methods */
static GSList *utf8_wordcomplete_find (const gchar *word);
static gint utf8_sort_by_priority (gconstpointer a, gconstpointer b);

/* 
 * This macro initializes GokUtf8WordComplete with the GType system 
 *   and defines ..._class_init and ..._instance_init functions 
 */
GNOME_CLASS_BOILERPLATE (GokUTF8WordComplete, gok_utf8_wordcomplete,
			 GokWordComplete, GOK_TYPE_WORDCOMPLETE)

static void
gok_utf8_wordcomplete_instance_init (GokUTF8WordComplete *complete)
{
}

static void
gok_utf8_wordcomplete_class_init (GokUTF8WordCompleteClass *klass)
{
	GokWordCompleteClass *word_complete_class = (GokWordCompleteClass *) klass;
	word_complete_class->open = utf8_wordcomplete_open;
	word_complete_class->predict_string = utf8_wordcomplete_predict_string;
}

static gboolean
utf8_wordcomplete_open (GokWordComplete *complete, const gchar *directory)
{
	FILE *file;

	gok_log ("gok_utf8_wordcomplete_open");
	
	gok_wordcomplete_reset (complete); /* reset, anyhow */

	/* open the dictionary file */
	dictionary_full_path = g_build_filename (directory, "dictionary.txt", NULL);
	file = fopen (dictionary_full_path, "rt");
	if (file == NULL)
	{
		/* can't open dictionary file! */
		gok_log_x ("Can't open dictionary file in utf8_wordcomplete_open !\n");
		return FALSE;
	}
	return TRUE;
}

static gchar**
utf8_wordcomplete_predict_string (GokWordComplete *complete, const gchar* word, gint num_predictions)
{
	gchar **word_predict_list = NULL;
	gchar *normalized, *tmp;
	gint  i, n, m = num_predictions;
	GSList *predictions;
	gpointer data;
	
	/* validate the given values */
	if (word == NULL || num_predictions < 1)
	{
	  return NULL;
	}
	
	word_predict_list = g_new0 (gchar *, num_predictions + 1); /* NULL-terminated */

	/* convert the given string to a normalized, case-insensitive form */
	tmp = g_utf8_normalize (word, -1, G_NORMALIZE_ALL);
	normalized = g_utf8_casefold (tmp, -1);
	g_free (tmp);

	/* find matching words */
	predictions = utf8_wordcomplete_find (normalized);

	/* reorder according to priority */
	predictions = g_slist_sort (predictions, utf8_sort_by_priority); 

	/* convert to array of strings */
	n = -1;
	do
	{
		WordPrediction *prediction;
		++n;
		prediction = g_slist_nth_data (predictions, n);
		word_predict_list [n] = prediction ? prediction->string : NULL;
	} while (word_predict_list [n] && n < num_predictions);

	word_predict_list [num_predictions] = NULL; /* in case predictions-len >= num_predictions */

	/* free the extra storage; returned strings will be freed later */
	data = g_slist_nth_data (predictions, num_predictions);
	while (data) {
		WordPrediction *prediction = (WordPrediction *) data;
		m = num_predictions;
		g_free (prediction->string);
		g_free (prediction);
		++m;
		data = g_slist_nth_data (predictions, m);
	}
	g_slist_free (predictions);

	/* were there any predictions? */
	if (n == 0)
	{
		g_free (word_predict_list);
 	        return NULL; /* indicates no predictions made */
	}

	/* post-process the predictions to match the case of the input */
/*	apply_case (word, word_predict_list, num_predictions); */ /* TODO!!! */

	return word_predict_list; /* predictions made! */
}

static WordPrediction*
prediction_new (const gchar *string, gint priority)
{
	WordPrediction *prediction = g_new0 (WordPrediction, 1);
	prediction->string = strdup (string);
	prediction->priority = priority;
	return prediction;
}

static GSList *
utf8_wordcomplete_find (const gchar *word)
{
	GSList *found = NULL;
	WordPrediction *prediction;
	found = g_slist_append (found, prediction_new ("foo", 1));
	found = g_slist_append (found, prediction_new ("bar", 1));
	found = g_slist_append (found, prediction_new ("kilroy", 2));
	found = g_slist_append (found, prediction_new ("was", 2));
	found = g_slist_append (found, prediction_new ("here", 2));
	return found;
}

static gint
utf8_sort_by_priority (gconstpointer a, gconstpointer b)
{
        WordPrediction *p_a = (WordPrediction *) a;
	WordPrediction *p_b = (WordPrediction *) b;
	if (a && b)
		return p_b->priority - p_a->priority;
	else
		return (a) ? -1 : ((b) ? 1 : 0);
}
