/*
 * Copyright (C) 2003 Sun Microsystems, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Authors:
 *	Mark McLoughlin <mark@skynet.ie>
 */

#include <config.h>

#include "netstatus-icon.h"

#include <gtk/gtk.h>
#include <libgnome/gnome-i18n.h>

#include "netstatus-util.h"
#include "netstatus-enums.h"
#include "netstatus-fallback-pixbuf.h"

struct _NetstatusIconPrivate
{
  GtkWidget      *image;

  NetstatusIface *iface;
  NetstatusState  state;

  GdkPixbuf      *icons [NETSTATUS_STATE_LAST];
  GdkPixbuf      *scaled_icons [NETSTATUS_STATE_LAST];

  GtkOrientation  orientation;
  int             size;

  GtkTooltips    *tooltips;

  gulong          state_changed_id;
  gulong          name_changed_id;

  guint           tooltips_enabled : 1;
};

enum {
  INVOKED,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_IFACE,
  PROP_ORIENTATION,
  PROP_TOOLTIPS_ENABLED
};

static GObjectClass *parent_class;
static guint icon_signals [LAST_SIGNAL] = { 0 };


static GdkPixbuf *
netstatus_icon_get_default_pixbuf (NetstatusIcon *icon)
{
  static GdkPixbuf *fallback_pixbuf = NULL;

  if (!fallback_pixbuf)
    {
      fallback_pixbuf = gdk_pixbuf_new_from_inline (-1,
						    fallback_icon_data,
						    FALSE,
						    NULL);
      g_object_add_weak_pointer (G_OBJECT (fallback_pixbuf),
				 (gpointer) &fallback_pixbuf);

      return fallback_pixbuf;
    }

  return g_object_ref (fallback_pixbuf);
}

static void
netstatus_icon_init_pixbuf (NetstatusIcon  *icon,
			    GdkPixbuf     **pixbuf,
			    const char     *filename)
{
  GError *error;

  error = NULL;
  *pixbuf = gdk_pixbuf_new_from_file (filename, &error);
  if (error)
    {
      netstatus_adopt_error (error, NETSTATUS_ERROR_ICONS);
      netstatus_iface_set_error (icon->priv->iface, error);
      g_error_free (error);
      
      *pixbuf = netstatus_icon_get_default_pixbuf (icon);
    }

  g_assert (*pixbuf != NULL);
}

static void
netstatus_icon_init_pixbufs (NetstatusIcon *icon)
{
  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_DISCONNECTED],
			      NETSTATUS_ICONDIR "/gnome-netstatus-disconn.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_DISCONNECTED] = NULL;

  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_IDLE],
			      NETSTATUS_ICONDIR "/gnome-netstatus-idle.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_IDLE] = NULL;

  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_TX],
			      NETSTATUS_ICONDIR "/gnome-netstatus-tx.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_TX] = NULL;

  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_RX],
			      NETSTATUS_ICONDIR "/gnome-netstatus-rx.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_RX] = NULL;

  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_TX_RX],
			      NETSTATUS_ICONDIR "/gnome-netstatus-txrx.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_TX_RX] = NULL;

  netstatus_icon_init_pixbuf (icon,
			      &icon->priv->icons [NETSTATUS_STATE_ERROR],
			      NETSTATUS_ICONDIR "/gnome-netstatus-error.png");
  icon->priv->scaled_icons [NETSTATUS_STATE_ERROR] = NULL;
}

static void
netstatus_icon_update_image (NetstatusIcon *icon)
{
  GdkPixbuf *pixbuf;

  if (!icon->priv->icons [icon->priv->state])
    netstatus_icon_init_pixbufs (icon);

  pixbuf = icon->priv->scaled_icons [icon->priv->state];
  if (!pixbuf)
    pixbuf = icon->priv->icons [icon->priv->state];

  gtk_image_set_from_pixbuf (GTK_IMAGE (icon->priv->image), pixbuf);
}

static void
netstatus_icon_name_changed (NetstatusIface *iface,
			     GParamSpec     *pspec,
			     NetstatusIcon  *icon)
{
  const char *iface_name;
  const char *tip;
  char       *freeme = NULL;

  iface_name = netstatus_iface_get_name (icon->priv->iface);
  if (iface_name)
    {
      freeme = g_strdup_printf (_("Network Connection: %s"), iface_name);
      tip = freeme;
    }
  else
    {
      tip = _("Network Connection");
    }

  gtk_tooltips_set_tip (icon->priv->tooltips, GTK_WIDGET (icon), tip, NULL);

  g_free (freeme);
}

static void
netstatus_icon_state_changed (NetstatusIface *iface,
			      GParamSpec     *pspec,
			      NetstatusIcon  *icon)
{
  NetstatusState state;

  g_return_if_fail (NETSTATUS_IS_ICON (icon));

  state = netstatus_iface_get_state (iface);

  if (icon->priv->state != state)
    {
      icon->priv->state = state;

      netstatus_icon_update_image (icon);
    }
}

static void
netstatus_icon_destroy (GtkObject *widget)
{
  NetstatusIcon *icon = (NetstatusIcon *) widget;

  if (icon->priv->state_changed_id)
    {
      g_assert (icon->priv->iface != NULL);
      g_assert (icon->priv->name_changed_id != 0);
      g_signal_handler_disconnect (icon->priv->iface,
				   icon->priv->state_changed_id);
      g_signal_handler_disconnect (icon->priv->iface,
				   icon->priv->name_changed_id);
    }
  icon->priv->state_changed_id = 0;
  icon->priv->name_changed_id = 0;

  if (icon->priv->tooltips)
    g_object_unref (icon->priv->tooltips);
  icon->priv->tooltips = NULL;
  
  icon->priv->image = NULL;

  GTK_OBJECT_CLASS (parent_class)->destroy (widget);
}

static void
netstatus_icon_scale_icons (NetstatusIcon *icon,
			    int            width,
			    int            height)
{
  int i;

  g_return_if_fail (width  > 0);
  g_return_if_fail (height > 0);

  if (!icon->priv->icons [0])
    netstatus_icon_init_pixbufs (icon);

  for (i = 0; i < NETSTATUS_STATE_LAST; i++)
    {
      int orig_width;
      int orig_height;

      if (icon->priv->scaled_icons [i])
	g_object_unref (icon->priv->scaled_icons [i]);
      icon->priv->scaled_icons [i] = NULL;

      orig_width  = gdk_pixbuf_get_width  (icon->priv->icons [i]);
      orig_height = gdk_pixbuf_get_height (icon->priv->icons [i]);

      if (width >= orig_width && height >= orig_height)
	{
	  icon->priv->scaled_icons [i] =
	    g_object_ref (icon->priv->icons [i]);
	}
      else
	{
	  double scale;

	  scale = MIN (((double) width)  / orig_width,
		       ((double) height) / orig_height);
	  
	  icon->priv->scaled_icons [i] =
	    gdk_pixbuf_scale_simple (icon->priv->icons [i],
				     orig_width  * scale,
				     orig_height * scale,
				     GDK_INTERP_BILINEAR);
	}
    }

  netstatus_icon_update_image (icon);
}

static void
netstatus_icon_size_allocate (GtkWidget     *widget,
			      GtkAllocation *allocation)
{
  NetstatusIcon *icon = (NetstatusIcon *) widget;
  int            size;

  if (icon->priv->orientation == GTK_ORIENTATION_HORIZONTAL)
    size = allocation->height;
  else
    size = allocation->width;

  if (icon->priv->size != size)
    {
      icon->priv->size = size;

      netstatus_icon_scale_icons (icon, size, size);
    }

  GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
}


static gboolean
netstatus_icon_button_press_event (GtkWidget      *widget,
				   GdkEventButton *event)
{
  if (event->button == 1)
    {
      netstatus_icon_invoke (NETSTATUS_ICON (widget));
      return TRUE;
    }

  return FALSE;
}


static void
netstatus_icon_set_property (GObject      *object,
			     guint         prop_id,
			     const GValue *value,
			     GParamSpec   *pspec)
{
  NetstatusIcon *icon;

  g_return_if_fail (NETSTATUS_IS_ICON (object));

  icon = NETSTATUS_ICON (object);

  switch (prop_id)
    {
    case PROP_IFACE:
      netstatus_icon_set_iface (icon, g_value_get_object (value));
      break;
    case PROP_ORIENTATION:
      netstatus_icon_set_orientation (icon, g_value_get_enum (value));
      break;
    case PROP_TOOLTIPS_ENABLED:
      netstatus_icon_set_tooltips_enabled (icon, g_value_get_boolean (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
netstatus_icon_get_property (GObject    *object,
			     guint       prop_id,
			     GValue     *value,
			     GParamSpec *pspec)
{
  NetstatusIcon *icon;

  g_return_if_fail (NETSTATUS_IS_ICON (object));

  icon = NETSTATUS_ICON (object);

  switch (prop_id)
    {
    case PROP_IFACE:
      g_value_set_object (value, icon->priv->iface);
      break;
    case PROP_ORIENTATION:
      g_value_set_enum (value, icon->priv->orientation);
      break;
    case PROP_TOOLTIPS_ENABLED:
      g_value_set_boolean (value, icon->priv->tooltips_enabled);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
netstatus_icon_finalize (GObject *object)
{
  NetstatusIcon *icon = (NetstatusIcon *) object;
  int            i;

  if (icon->priv->iface)
    g_object_unref (icon->priv->iface);
  icon->priv->iface = NULL;

  for (i = 0; i < NETSTATUS_STATE_LAST; i++) 
    {
      if (icon->priv->icons [i])
	g_object_unref (icon->priv->icons [i]);
      icon->priv->icons [i] = NULL;

      if (icon->priv->scaled_icons [i])
	g_object_unref (icon->priv->scaled_icons [i]);
      icon->priv->scaled_icons [i] = NULL;
    }

  g_free (icon->priv);
  icon->priv = NULL;

  parent_class->finalize (object);
}

static void
netstatus_icon_class_init (NetstatusIconClass *klass)
{
  GObjectClass   *gobject_class   = (GObjectClass   *) klass;
  GtkObjectClass *gtkobject_class = (GtkObjectClass *) klass;
  GtkWidgetClass *widget_class    = (GtkWidgetClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->set_property = netstatus_icon_set_property;
  gobject_class->get_property = netstatus_icon_get_property;
  gobject_class->finalize     = netstatus_icon_finalize;

  gtkobject_class->destroy = netstatus_icon_destroy;

  widget_class->size_allocate      = netstatus_icon_size_allocate;
  widget_class->button_press_event = netstatus_icon_button_press_event;

  g_object_class_install_property (gobject_class,
				   PROP_IFACE,
				   g_param_spec_object ("iface",
							_("Interface"),
							_("The current interface the icon is monitoring."),
							NETSTATUS_TYPE_IFACE,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
  
  g_object_class_install_property (gobject_class,
                                   PROP_ORIENTATION,
                                   g_param_spec_enum ("orientation",
                                                      _("Orientation"),
                                                      _("The orientation of the tray."),
                                                      GTK_TYPE_ORIENTATION,
                                                      GTK_ORIENTATION_HORIZONTAL,
                                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (gobject_class,
                                   PROP_TOOLTIPS_ENABLED,
                                   g_param_spec_boolean ("tooltips-enabled",
							 _("Tooltips Enabled"),
							 _("Whether or not the icon's tooltips are enabled."),
							 TRUE,
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  icon_signals [INVOKED] =
    g_signal_new ("invoked",
		  G_OBJECT_CLASS_TYPE (gobject_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (NetstatusIconClass, invoked),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
}

static void
netstatus_icon_instance_init (NetstatusIcon      *icon,
			      NetstatusIconClass *klass)
{
  icon->priv = g_new0 (NetstatusIconPrivate, 1);

  icon->priv->iface            = NULL;
  icon->priv->state            = NETSTATUS_STATE_DISCONNECTED;
  icon->priv->orientation      = GTK_ORIENTATION_HORIZONTAL;
  icon->priv->size             = 0;
  icon->priv->state_changed_id = 0;

  icon->priv->image = gtk_image_new ();
  gtk_container_add (GTK_CONTAINER (icon), icon->priv->image);
  gtk_widget_show (icon->priv->image);

  icon->priv->tooltips = gtk_tooltips_new ();
  g_object_ref (icon->priv->tooltips);
  gtk_object_sink (GTK_OBJECT (icon->priv->tooltips));

  gtk_widget_add_events (GTK_WIDGET (icon),
			 GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
}

GType
netstatus_icon_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (NetstatusIconClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) netstatus_icon_class_init,
	  NULL,
	  NULL,
	  sizeof (NetstatusIcon),
	  0,
	  (GInstanceInitFunc) netstatus_icon_instance_init,
	  NULL
	};

      type = g_type_register_static (GTK_TYPE_EVENT_BOX, "NetstatusIcon", &info, 0);
    }

  return type;
}

GtkWidget *
netstatus_icon_new (NetstatusIface *iface)
{
  return g_object_new (NETSTATUS_TYPE_ICON,
		       "iface", iface,
		       NULL);
}

void
netstatus_icon_invoke (NetstatusIcon *icon)
{
  g_return_if_fail (NETSTATUS_IS_ICON (icon));

  if (netstatus_iface_get_state (icon->priv->iface) != NETSTATUS_STATE_ERROR)
    {
      g_signal_emit (icon, icon_signals [INVOKED], 0);
    }
  else
    {
      const GError *error;
      GtkWidget    *dialog;

      error = netstatus_iface_get_error (icon->priv->iface);
      g_assert (error != NULL);

      dialog = gtk_message_dialog_new (NULL, 0,
				       GTK_MESSAGE_ERROR,
				       GTK_BUTTONS_CLOSE,
				       _("Please contact your system administrator to resolve the following problem:\n\n%s"),
				       error->message);

      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
    }
}

void
netstatus_icon_set_iface (NetstatusIcon  *icon,
			  NetstatusIface *iface)
{

  g_return_if_fail (NETSTATUS_IS_ICON (icon));

  if (icon->priv->iface != iface)
    {
      NetstatusIface *old_iface;

      old_iface = icon->priv->iface;

      if (icon->priv->state_changed_id)
	{
	  g_assert (icon->priv->iface != NULL);
	  g_assert (icon->priv->name_changed_id != 0);
	  g_signal_handler_disconnect (icon->priv->iface,
				       icon->priv->state_changed_id);
	  g_signal_handler_disconnect (icon->priv->iface,
				       icon->priv->name_changed_id);
	}

      if (iface)
	g_object_ref (iface);
      icon->priv->iface = iface;

      if (old_iface)
	g_object_unref (old_iface);

      icon->priv->state_changed_id = g_signal_connect (icon->priv->iface, "notify::state",
						       G_CALLBACK (netstatus_icon_state_changed), icon);
      icon->priv->name_changed_id  = g_signal_connect (icon->priv->iface, "notify::name",
						       G_CALLBACK (netstatus_icon_name_changed), icon);

      netstatus_icon_state_changed (icon->priv->iface, NULL, icon);
      netstatus_icon_name_changed  (icon->priv->iface, NULL, icon);

      g_object_notify (G_OBJECT (icon), "iface");
    }
}

NetstatusIface *
netstatus_icon_get_iface (NetstatusIcon *icon)
{
  g_return_val_if_fail (NETSTATUS_IS_ICON (icon), 0);

  return icon->priv->iface;
}

void
netstatus_icon_set_orientation (NetstatusIcon  *icon,
				GtkOrientation  orientation)
{
  g_return_if_fail (NETSTATUS_IS_ICON (icon));

  if (icon->priv->orientation != orientation)
    {
      icon->priv->orientation = orientation;

      g_object_notify (G_OBJECT (icon), "orientation");
    }
}

GtkOrientation
netstatus_icon_get_orientation (NetstatusIcon *icon)
{
  g_return_val_if_fail (NETSTATUS_IS_ICON (icon), GTK_ORIENTATION_HORIZONTAL);

  return icon->priv->orientation;
}

void
netstatus_icon_set_tooltips_enabled (NetstatusIcon *icon,
				     gboolean       enabled)
{
  g_return_if_fail (NETSTATUS_IS_ICON (icon));

  enabled = enabled != FALSE;

  if (icon->priv->tooltips_enabled != enabled)
    {
      icon->priv->tooltips_enabled = enabled;

      if (enabled)
	gtk_tooltips_enable (icon->priv->tooltips);
      else
	gtk_tooltips_disable (icon->priv->tooltips);

      g_object_notify (G_OBJECT (icon), "tooltips-enabled");
    }
}

gboolean
netstatus_icon_get_tooltips_enabled (NetstatusIcon *icon)
{
  g_return_val_if_fail (NETSTATUS_ICON (icon), TRUE);

  return icon->priv->tooltips_enabled;
}
