/* GStreamer
 * Copyright (C) <2004> David A. Schleef <ds@schleef.org>
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/video/video.h>

#include <string.h>
#include <stdlib.h>

#include <librfb/rfb.h>


#define GST_TYPE_RFBSRC \
  (gst_rfbsrc_get_type())
#define GST_RFBSRC(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_RFBSRC,GstRfbsrc))
#define GST_RFBSRC_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_RFBSRC,GstRfbsrc))
#define GST_IS_RFBSRC(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_RFBSRC))
#define GST_IS_RFBSRC_CLASS(obj) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_RFBSRC))

typedef struct _GstRfbsrc GstRfbsrc;
typedef struct _GstRfbsrcClass GstRfbsrcClass;

struct _GstRfbsrc {
  GstElement element;

  GstPad *srcpad;

  RfbDecoder *decoder;

  guint8 *frame;
  gboolean go;
  gboolean inter;
};

struct _GstRfbsrcClass {
  GstElementClass parent_class;
};

GType gst_rfbsrc_get_type(void) G_GNUC_CONST;



static GstElementDetails rfbsrc_details = GST_ELEMENT_DETAILS (
  "Video test source",
  "Source/Video",
  "Creates a test video stream",
  "David A. Schleef <ds@schleef.org>"
);

/* GstRfbsrc signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  ARG_0,
  /* FILL ME */
};

static void gst_rfbsrc_base_init (gpointer g_class);
static void gst_rfbsrc_class_init (GstRfbsrcClass * klass);
static void gst_rfbsrc_init (GstRfbsrc * rfbsrc);
static GstElementStateReturn gst_rfbsrc_change_state (GstElement * element);

static void gst_rfbsrc_set_property (GObject * object, guint prop_id,
					   const GValue * value, GParamSpec * pspec);
static void gst_rfbsrc_get_property (GObject * object, guint prop_id, GValue * value,
					   GParamSpec * pspec);

static GstData *gst_rfbsrc_get (GstPad * pad);

static const GstQueryType *
		gst_rfbsrc_get_query_types (GstPad      *pad);
static gboolean gst_rfbsrc_src_query (GstPad      *pad,
					    GstQueryType type,
					    GstFormat   *format,
					    gint64      *value);
static void gst_rfbsrc_paint_rect (RfbDecoder *decoder, int x, int y, int w, int h, guint8 *data);
static gboolean gst_rfbsrc_handle_src_event (GstPad *pad, GstEvent *event);

static GstElementClass *parent_class = NULL;


static GstStaticPadTemplate gst_rfbsrc_src_template =
GST_STATIC_PAD_TEMPLATE (
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_STATIC_CAPS (
    "video/x-raw-rgb, "
    "bpp = (int) 32, "
    "depth = (int) 24, "
    "endianness = (int) BIG_ENDIAN, "
    "red_mask = (int) 0x0000ff00, "
    "green_mask = (int) 0x00ff0000, "
    "blue_mask = (int) 0xff000000, "
    "width = [ 16, 4096 ], "
    "height = [ 16, 4096 ], "
    "framerate = [ 1.0, 60.0] ")
);

GType
gst_rfbsrc_get_type (void)
{
  static GType rfbsrc_type = 0;

  if (!rfbsrc_type) {
    static const GTypeInfo rfbsrc_info = {
      sizeof (GstRfbsrcClass),
      gst_rfbsrc_base_init,
      NULL,
      (GClassInitFunc) gst_rfbsrc_class_init,
      NULL,
      NULL,
      sizeof (GstRfbsrc),
      0,
      (GInstanceInitFunc) gst_rfbsrc_init,
    };

    rfbsrc_type =
      g_type_register_static (GST_TYPE_ELEMENT, "GstRfbsrc", &rfbsrc_info, 0);
  }
  return rfbsrc_type;
}

static void
gst_rfbsrc_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (element_class, &rfbsrc_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rfbsrc_src_template));
}

static void
gst_rfbsrc_class_init (GstRfbsrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;

#if 0
  g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_WIDTH,
      g_param_spec_int ("width", "width", "Default width",
        1, G_MAXINT, 320, G_PARAM_READWRITE));
#endif

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_rfbsrc_set_property;
  gobject_class->get_property = gst_rfbsrc_get_property;

  gstelement_class->change_state = gst_rfbsrc_change_state;
}

#if 0
static void
gst_rfbsrc_set_clock (GstElement *element, GstClock *clock)
{
  GstRfbsrc *v;

  v = GST_RFBSRC (element);

  gst_object_replace ((GstObject **)&v->clock, (GstObject *)clock);
}
#endif

static GstElementStateReturn
gst_rfbsrc_change_state (GstElement * element)
{
  GstRfbsrc *rfbsrc;

  rfbsrc = GST_RFBSRC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      rfbsrc->decoder = rfb_decoder_new ();
      rfb_decoder_connect_tcp (rfbsrc->decoder, "127.0.0.1", 5901);
      rfbsrc->decoder->paint_rect = gst_rfbsrc_paint_rect;
      rfbsrc->decoder->decoder_private = rfbsrc;
      break;
    case GST_STATE_READY_TO_PAUSED:
      break;
    case GST_STATE_PAUSED_TO_PLAYING:
      break;
    case GST_STATE_PLAYING_TO_PAUSED:
      break;
    case GST_STATE_PAUSED_TO_READY:
      //rfbsrc->timestamp_offset = 0;
      //rfbsrc->n_frames = 0;
      break;
    case GST_STATE_READY_TO_NULL:
      break;
  }

  return parent_class->change_state (element);
}

static void
gst_rfbsrc_init (GstRfbsrc * rfbsrc)
{
  GST_DEBUG ("gst_rfbsrc_init");

  rfbsrc->srcpad = gst_pad_new_from_template (
      gst_static_pad_template_get (&gst_rfbsrc_src_template), "src");
  gst_pad_use_explicit_caps (rfbsrc->srcpad);
  gst_element_add_pad (GST_ELEMENT (rfbsrc), rfbsrc->srcpad);
  gst_pad_set_get_function (rfbsrc->srcpad, gst_rfbsrc_get);
  gst_pad_set_query_function (rfbsrc->srcpad, gst_rfbsrc_src_query);
  gst_pad_set_query_type_function (rfbsrc->srcpad,
				   gst_rfbsrc_get_query_types);
  gst_pad_set_event_function (rfbsrc->srcpad, gst_rfbsrc_handle_src_event);
}


static const GstQueryType *
gst_rfbsrc_get_query_types (GstPad *pad)
{
  static const GstQueryType query_types[] = {
    GST_QUERY_POSITION,
    0,
  };

  return query_types;
} 

static gboolean
gst_rfbsrc_src_query (GstPad      *pad,
			    GstQueryType type,
			    GstFormat   *format,
			    gint64      *value)
{
  gboolean res = FALSE;
  //GstRfbsrc *rfbsrc = GST_RFBSRC (gst_pad_get_parent (pad));
	        
  switch (type) {
    case GST_QUERY_POSITION:
      switch (*format) {
        case GST_FORMAT_TIME:
          //*value = rfbsrc->n_frames * GST_SECOND / (double) rfbsrc->rate;
          res = TRUE;
          break;
        case GST_FORMAT_DEFAULT: /* frames */
          //*value = rfbsrc->n_frames;
          res = TRUE;
          break;
        default:
          break;
      }
      break;
    default:
      break;
  }

  return res;
}

static gboolean
gst_rfbsrc_handle_src_event (GstPad *pad, GstEvent *event)
{
  GstRfbsrc *rfbsrc;
  double x,y;
  int button;

  rfbsrc = GST_RFBSRC (gst_pad_get_parent (pad));

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_NAVIGATION:
      gst_structure_get_double (event->event_data.structure.structure,
          "pointer_x", &x);
      gst_structure_get_double (event->event_data.structure.structure,
          "pointer_y", &y);

      button = 0;
      rfb_decoder_send_pointer_event (rfbsrc->decoder, button, (int)x, (int)y);
      break;
    default:
      break;
  }

  return TRUE;
}

static void
gst_rfbsrc_paint_rect (RfbDecoder *decoder, int x, int y, int w, int h,
    guint8 *data)
{
  int i,j;
  guint8 *frame;
  guint8 color;
  GstRfbsrc *rfbsrc;
  int width;
  int offset;

g_print ("painting %d,%d (%dx%d)\n", x, y, w, h);
  rfbsrc = GST_RFBSRC (decoder->decoder_private);

  frame = rfbsrc->frame;
  width = decoder->width;
  for(j=0;j<h;j++){
    for(i=0;i<w;i++){
      color = data[j*w + i];

#define RGB332_R(x)  ((((x)&0x07) * 0x124)>>3)
#define RGB332_G(x)  ((((x)&0x38) * 0x124)>>6)
#define RGB332_B(x)  ((((x)&0xc0) * 0x149)>>8)
      offset = ((j+x)*width + (i+x))*4;
      frame[offset + 0] = RGB332_B(color);
      frame[offset + 1] = RGB332_G(color);
      frame[offset + 2] = RGB332_R(color);
      frame[offset + 3] = 0;
    }
  }

  rfbsrc->go = FALSE;
}

static GstData *
gst_rfbsrc_get (GstPad * pad)
{
  GstRfbsrc *rfbsrc;
  gulong newsize;
  GstBuffer *buf;
  RfbDecoder *decoder;

  GST_DEBUG ("gst_rfbsrc_get");

  g_return_val_if_fail (pad != NULL, NULL);
  g_return_val_if_fail (GST_IS_PAD (pad), NULL);

  rfbsrc = GST_RFBSRC (gst_pad_get_parent (pad));
  decoder = rfbsrc->decoder;

  if (!decoder->inited) {
    GstCaps *caps;

    while (!decoder->inited) {
      rfb_decoder_iterate (decoder);
    }
    caps = gst_caps_new_simple ("video/x-raw-rgb",
        "bpp", G_TYPE_INT, 32,
        "depth", G_TYPE_INT, 24,
        "endianness", G_TYPE_INT, G_BIG_ENDIAN,
        "red_mask", G_TYPE_INT, 0x0000ff00,
        "green_mask", G_TYPE_INT, 0x00ff0000,
        "blue_mask", G_TYPE_INT, 0xff000000,
        "width", G_TYPE_INT, decoder->width,
        "height", G_TYPE_INT, decoder->height,
        "framerate", G_TYPE_DOUBLE, 10.0,
        NULL);
    gst_pad_set_explicit_caps (rfbsrc->srcpad, caps);

    if (rfbsrc->frame) g_free (rfbsrc->frame);
    rfbsrc->frame = g_malloc (decoder->width * decoder->height * 4);

g_print("red_mask = %08x\n", decoder->red_max << decoder->red_shift);
g_print("green_mask = %08x\n", decoder->green_max << decoder->green_shift);
g_print("blue_mask = %08x\n", decoder->blue_max << decoder->blue_shift);
    rfbsrc->inter = FALSE;
  }

  rfb_decoder_send_update_request (decoder, rfbsrc->inter, 0, 0, decoder->width,
      decoder->height);
  //rfbsrc->inter = TRUE;

  rfbsrc->go = TRUE;
  while (rfbsrc->go) {
    rfb_decoder_iterate (decoder);
    g_print("iterate...\n");
  }

  newsize = decoder->width * decoder->height * 4;
  g_return_val_if_fail (newsize > 0, NULL);

  GST_DEBUG ("size=%ld %dx%d", newsize, decoder->width, decoder->height);

  buf = gst_buffer_new_and_alloc (newsize);
  g_return_val_if_fail (GST_BUFFER_DATA (buf) != NULL, NULL);

  memcpy (GST_BUFFER_DATA (buf), rfbsrc->frame, newsize);

  return GST_DATA (buf);
}

static void
gst_rfbsrc_set_property (GObject * object, guint prop_id, const GValue * value,
			       GParamSpec * pspec)
{
  GstRfbsrc *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_RFBSRC (object));
  src = GST_RFBSRC (object);

  GST_DEBUG ("gst_rfbsrc_set_property");
  switch (prop_id) {
    default:
      break;
  }
}

static void
gst_rfbsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec)
{
  GstRfbsrc *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_RFBSRC (object));
  src = GST_RFBSRC (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "rfbsrc", GST_RANK_NONE, GST_TYPE_RFBSRC);
}

GST_PLUGIN_DEFINE (
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "rfbsrc",
  "Connects to a VNC server and decodes RFB stream",
  plugin_init,
  VERSION,
  GST_LICENSE,
  GST_PACKAGE,
  GST_ORIGIN
)
