/* $Id: gdict-pref.c,v 1.25 2004/12/14 18:56:47 vnoel Exp $ */

/*
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Mike Hughes <mfh@psilord.com>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict preferences
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <panel-applet-gconf.h>
#include "dict.h"
#include "gdict-app.h"
#include "gdict-pref.h"
#include "gdict-applet.h"

#define GCONF_PATH "/apps/gnome_dictionary/%s"

static GConfClient *gdict_client = NULL;

gboolean
gdict_get_gconf_bool (GDictWindow *gdict, gchar *key)
{
        gboolean bool;
        GDictApplet *applet = gdict->applet;
        if (applet) {
                bool = panel_applet_gconf_get_bool (applet->parent_applet, key, NULL);
        } else {
                gchar *full_key;
                full_key = g_strdup_printf(GCONF_PATH, key);
                bool = gconf_client_get_bool (gdict_get_gconf_client (), full_key, NULL);
                g_free (full_key);
        }
        return bool;
}
   
void
gdict_set_gconf_bool (GDictWindow *gdict, gchar *key, gboolean bool)
{
        GDictApplet *applet = gdict->applet;
        if (applet) {
                panel_applet_gconf_set_bool (applet->parent_applet, key, bool, NULL);
        } else {
                gchar *full_key;
                full_key = g_strdup_printf(GCONF_PATH, key);
                gconf_client_set_bool (gdict_get_gconf_client (), full_key, bool, NULL);
                g_free (full_key);
        }
}

int
gdict_get_gconf_int (GDictWindow *gdict, gchar *key)
{
        int value;
        GDictApplet *applet = gdict->applet;
        if (applet) {
                value = panel_applet_gconf_get_int (applet->parent_applet, key, NULL);
        } else {
                gchar *full_key;
                full_key = g_strdup_printf(GCONF_PATH, key);
                value = gconf_client_get_int (gdict_get_gconf_client (), full_key, NULL);
                g_free (full_key);
        }
        return value;
}

void
gdict_set_gconf_int (GDictWindow *gdict, gchar *key, int value)
{
        GDictApplet *applet = gdict->applet;
        if (applet) {
                panel_applet_gconf_set_int (applet->parent_applet, key, value, NULL);
        } else {
                gchar *full_key;
                full_key = g_strdup_printf (GCONF_PATH, key);
                gconf_client_set_int (gdict_get_gconf_client (), full_key, value, NULL);
                g_free (full_key);
        }
}

gchar *
gdict_get_gconf_string (GDictWindow *gdict, gchar *key)
{
        GDictApplet *applet = gdict->applet;
        gchar *string;

        if (applet) {
                string = panel_applet_gconf_get_string (applet->parent_applet, key, NULL); 
        } else {
                gchar *full_key;
                full_key = g_strdup_printf(GCONF_PATH, key);
                string = gconf_client_get_string (gdict_get_gconf_client (), full_key, NULL);
                g_free (full_key);
        }
        return string;
}

void
gdict_set_gconf_string (GDictWindow *gdict, gchar *key, gchar *string)
{
        GDictApplet *applet = gdict->applet;
        
        if (applet) {
                panel_applet_gconf_set_string (applet->parent_applet, key, string, NULL);
        } else {
                gchar *full_key;
                full_key = g_strdup_printf (GCONF_PATH, key);
                gconf_client_set_string (gdict_get_gconf_client (), full_key, string, NULL);
                g_free (full_key);
        }
}

void
gdict_add_gconf_notify (GDictWindow *gdict, gchar *key, GConfClientNotifyFunc cb, gpointer data)
{
        GDictApplet *applet = gdict->applet; 
        gchar *full_key;
 
        if (applet)
                full_key = panel_applet_gconf_get_full_key (applet->parent_applet, key);
        else
                full_key = g_strdup_printf (GCONF_PATH, key);
            
        gconf_client_notify_add (gdict_get_gconf_client (), full_key, cb, data, NULL, NULL);
        g_free (full_key);
}

gboolean
gdict_is_gconf_key_writable (GDictWindow *gdict, gchar *key)
{
        GDictApplet *applet = gdict->applet;
        gboolean is_writable;
        gchar *full_key;

        if (applet)
                full_key = panel_applet_gconf_get_full_key (applet->parent_applet, key);
        else
                full_key = g_strdup_printf (GCONF_PATH, key);

        is_writable = gconf_client_key_is_writable (gdict_get_gconf_client (), full_key, NULL);
        return is_writable;
}

GConfClient *
gdict_get_gconf_client (void)
{
    if (!gdict_client)
        gdict_client = gconf_client_get_default ();

    return gdict_client;
}

static void
database_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
    GDictWindow *gdict = data;
    GConfValue *value = gconf_entry_get_value (entry);
 
    g_free (gdict->pref->database);   
    gdict->pref->database = g_strdup (gconf_value_get_string (value));
}

static void
strat_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
    GDictWindow *gdict = data;
    GConfValue *value = gconf_entry_get_value (entry);
    
    g_free (gdict->pref->dfl_strat);   
    gdict->pref->dfl_strat = g_strdup (gconf_value_get_string (value));
}

static void
smart_changed_cb (GConfClient *client, guint id, GConfEntry *entry, gpointer data)
{
    GDictWindow *gdict = data;
    GConfValue *value = gconf_entry_get_value (entry);
    
    gdict->pref->smart = gconf_value_get_bool (value);
}

void
gdict_window_store_geometry (GDictWindow *gdict)
{
	int x, y;
	gtk_window_get_position (GTK_WINDOW (gdict), &x, &y);
	gdict->x = x;
	gdict->y = y;
}

/* gdict_pref_load
 *
 * Loads configuration from config file
 */

static void 
gdict_pref_load (GDictWindow *gdict, GDictPref *pref)
{
    GError *error = NULL;

    gconf_client_add_dir(gdict_get_gconf_client (), "/apps/gnome-dictionary", GCONF_CLIENT_PRELOAD_NONE, NULL);
    
    if (pref->server != NULL)
        g_free(pref->server);
    
    /* FIXME: notification for the server stuff is really tricky
    ** Leaving it out for now */
    pref->server = gdict_get_gconf_string (gdict, "server");
    
    pref->port = gdict_get_gconf_int (gdict, "port");
    
    pref->smart = gdict_get_gconf_bool (gdict, "smart");

    gdict_add_gconf_notify (gdict, "smart", smart_changed_cb, gdict);

    pref->database = gdict_get_gconf_string (gdict, "database");
    gdict_add_gconf_notify (gdict, "database", database_changed_cb, gdict);
    
    pref->dfl_strat = gdict_get_gconf_string (gdict, "strategy");
    gdict_add_gconf_notify (gdict, "strategy", strat_changed_cb, gdict);
    
    /* If things go bad and gconf doesn't return values, we shouldn't die so just return the 
       default schemas */
    if (!pref->server)
	pref->server = g_strdup ("dict.org");
    if (!pref->port)
	pref->port = 2628;
    if (!pref->database)
	pref->database = "!";
    if (!pref->dfl_strat)
	pref->dfl_strat = "lev";
}

GDictPref *
gdict_pref_new (gpointer gdict)
{
        GDictPref *pref;

        pref = g_new0 (GDictPref, 1);
        gdict_pref_load (gdict, pref);
        pref->smart = TRUE;

        return pref;
}
