/* arch-tag: 0743f5ae-bbb1-4d30-aca2-750d80033982 */

/*  eXperience GTK engine: image.c
 *  
 *  Copyright (C) 2004-2005  Benjamin Berg <benjamin@sipsolutions.net>
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include "experience.h"
#include "image.h"
#include "utils.h"
#include "filter.h"
#include "image_loading.h"

#define CHECK_AND_SET_OPTION(image, option, message) { \
	if (image->options_set & option) { \
			g_printerr ("Error in image #%i in group %s: %s\n", image->drawable.number, image->drawable.group_name, message); \
			return; \
	} else image->options_set |= option; \
}

void
experience_image_set_file (eXperienceImage * image, gchar * filename)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_FILE, "Tried to set the filename of an image more than once!");
	
	image->file = filename;
}

void
experience_image_set_width (eXperienceImage * image, eXperienceImagePercent width)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_WIDTH, "Tried to set the width of an image more than once!");
	
	image->width = width;
}

void
experience_image_set_height (eXperienceImage * image, eXperienceImagePercent height)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_HEIGHT, "Tried to set the height of an image more than once!");
	
	image->height = height;
}

void
experience_image_set_pos (eXperienceImage * image, eXperiencePosPercent pos, eXperienceOrientation orientation)
{
	g_assert (image != NULL);
	
	if (orientation & ORIENTATION_HORIZONTAL) {
		CHECK_AND_SET_OPTION (image, OPTION_XPOS, "Tried to set the positioning more than once!");
		
		image->xpos = pos;
	}
	
	if (orientation & ORIENTATION_VERTICAL) {
		CHECK_AND_SET_OPTION (image, OPTION_YPOS, "Tried to set the positioning more than once!");
		
		image->ypos = pos;
	}
}

void
experience_image_set_border (eXperienceImage * image, guint left, guint right, guint top, guint bottom)
{
	g_assert ( image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_BORDER, "Tried to set the border more than once!");
	
	image->border.top    = top;
	image->border.bottom = bottom;
	image->border.left   = left;
	image->border.right  = right;
}

void
experience_image_set_repeat (eXperienceImage * image, guint left, guint right, guint top, guint bottom)
{
	g_assert ( image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_REPEAT, "Tried to set the repeat more than once!");
	
	image->repeat.top    = top;
	image->repeat.bottom = bottom;
	image->repeat.left   = left;
	image->repeat.right  = right;
}

void
experience_image_set_inner_padding (eXperienceImage * image, guint left, guint right, guint top, guint bottom)
{
	g_assert ( image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_INNER_PADDING, "Tried to set inner_padding more than once!");
	
	image->inner_padding.top    = top;
	image->inner_padding.bottom = bottom;
	image->inner_padding.left   = left;
	image->inner_padding.right  = right;
}

void
experience_image_set_rounding (eXperienceImage * image, eXperienceRoundingMethod rounding)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_ROUNDING, "Tried to set the rounding method more than once!");
	
	image->rounding = rounding;
}

void
experience_image_set_draw_components (eXperienceImage * image, eXperienceComponents draw_components)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_DRAW_COMPONENTS, "Tried to set draw_components more than once!");
	
	image->draw_components = draw_components;
}

void
experience_image_set_interp_type (eXperienceImage * image, GdkInterpType interp_type)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_INTERP_TYPE, "Tried to set interpolation_type more than once!");
	
	image->interp_type = interp_type;
}

void
experience_image_set_draw_entire_only (eXperienceImage * image, gboolean draw_entire_only)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_DRAW_ENTIRE_ONLY, "Tried to set draw_entire_only more than once!");
	
	image->draw_entire_only = draw_entire_only;
}

void
experience_image_set_dont_clip (eXperienceImage * image, gboolean dont_clip)
{
	g_assert (image != NULL);
	
	CHECK_AND_SET_OPTION (image, OPTION_DONT_CLIP, "Tried to set dont_clip more than once!");
	
	image->dont_clip = dont_clip;
}


/*###################################*/
/* class defs */

static eXperienceDrawable *
create (void)
{
	eXperienceImage * new_image = g_new0 (eXperienceImage, 1);
	
	new_image->draw_components = COMPONENT_ALL;
	
	new_image->rounding = ROUND_NORMAL;
	
	new_image->interp_type = GDK_INTERP_BILINEAR;
	
	new_image->width. widget = 1;
	new_image->height.widget = 1;
	
	new_image->repeat.left   = 1;
	new_image->repeat.right  = 1;
	new_image->repeat.top    = 1;
	new_image->repeat.bottom = 1;
	
	experience_filter_init (&new_image->drawable.filter, FILTER_ALL);
	
	return (eXperienceDrawable*) new_image;
}

static void
destroy (eXperienceDrawable * drawable)
{
	eXperienceImage * image = (eXperienceImage*) drawable;
	g_assert (drawable != NULL);
	g_assert (drawable->class == experience_image_class);
	
	if (image->pixbuf != NULL)
		g_object_remove_weak_pointer ((GObject*) image->pixbuf, (gpointer*) &image->pixbuf);
	
	g_free (image->file);
	g_free (image);
}

static void
inherit_from_drawable (eXperienceDrawable * drawable, eXperienceDrawable * from)
{
	eXperienceImage * image = (eXperienceImage*) drawable;
	eXperienceImage * from_image = (eXperienceImage*) from;
	
	g_assert (drawable != NULL);
	g_assert (from  != NULL);
	g_assert (drawable->class == experience_image_class);
	
	if (image->file == NULL) image->file = g_strdup(from_image->file);
	
	if (!(image->options_set & OPTION_BORDER))
		image->border = from_image->border;
	
	if (!(image->options_set & OPTION_INNER_PADDING))
		image->inner_padding = from_image->inner_padding;
	
	if (!(image->options_set & OPTION_REPEAT))
		image->repeat = from_image->repeat;
	
	if (!(image->options_set & OPTION_XPOS))
		image->xpos = from_image->xpos;
	
	if (!(image->options_set & OPTION_YPOS))
		image->ypos = from_image->ypos;
	
	if (!(image->options_set & OPTION_WIDTH))
		image->width = from_image->width;
	
	if (!(image->options_set & OPTION_HEIGHT))
		image->height = from_image->height;
	
	if (!(image->options_set & OPTION_DRAW_COMPONENTS))
		image->draw_components = from_image->draw_components;
	
	if (!(image->options_set & OPTION_ROUNDING))
		image->rounding = from_image->rounding;
	
	if (!(image->options_set & OPTION_INTERP_TYPE))
		image->interp_type = from_image->interp_type;
		
	if (!(image->options_set & OPTION_DRAW_ENTIRE_ONLY))
		image->draw_entire_only = from_image->draw_entire_only;
	
	if (!(image->options_set & OPTION_DONT_CLIP))
		image->dont_clip = from_image->dont_clip;
	
	image->options_set |= from_image->options_set;
}

static void
apply_group_settings (eXperienceDrawable * drawable, eXperienceGroup * group)
{
	eXperienceImage * image = (eXperienceImage*) drawable;
	eXperienceBorder border_tmp;
	eXperiencePadding inner_padding_tmp;
	eXperiencePosPercent pos_percent_tmp;
	eXperienceImagePercent image_percent_tmp;
	
	g_assert (drawable != NULL);
	g_assert (drawable->class == experience_image_class);
	
	if (group->filter.mirror & ORIENTATION_HORIZONTAL) {
		border_tmp = image->border;
		
		image->border.right  = border_tmp.left;
		image->border.left   = border_tmp.right;

		border_tmp = image->border;
		
		image->inner_padding.right  = inner_padding_tmp.left;
		image->inner_padding.left   = inner_padding_tmp.right;
		
		border_tmp = image->repeat;
		
		image->repeat.right  = border_tmp.left;
		image->repeat.left   = border_tmp.right;
		
		image->xpos.pixel = -image->xpos.pixel;
		image->xpos.widget = -image->xpos.widget;
	}
	
	if (group->filter.mirror & ORIENTATION_VERTICAL) {
		border_tmp = image->border;
		
		image->border.top    = border_tmp.bottom;
		image->border.bottom = border_tmp.top;

		inner_padding_tmp = image->inner_padding;
		
		image->inner_padding.top    = inner_padding_tmp.bottom;
		image->inner_padding.bottom = inner_padding_tmp.top;
		
		border_tmp = image->repeat;
		
		image->repeat.top    = border_tmp.bottom;
		image->repeat.bottom = border_tmp.top;
		
		image->ypos.pixel = -image->ypos.pixel;
		image->ypos.widget = -image->ypos.widget;
	}
	
	switch (group->filter.rotation) {
		case ROTATE_CW:
			border_tmp = image->border;
			
			image->border.top    = border_tmp.left;
			image->border.right  = border_tmp.top;
			image->border.bottom = border_tmp.right;
			image->border.left   = border_tmp.bottom;
			
			inner_padding_tmp = image->inner_padding;
			
			image->inner_padding.top    = inner_padding_tmp.left;
			image->inner_padding.right  = inner_padding_tmp.top;
			image->inner_padding.bottom = inner_padding_tmp.right;
			image->inner_padding.left   = inner_padding_tmp.bottom;
			
			border_tmp = image->repeat;
			
			image->repeat.top    = border_tmp.left;
			image->repeat.right  = border_tmp.top;
			image->repeat.bottom = border_tmp.right;
			image->repeat.left   = border_tmp.bottom;
			
			pos_percent_tmp      = image->xpos;
			image->xpos.pixel    = -image->ypos.pixel;
			image->xpos.widget   = -image->ypos.widget;
			image->ypos          = pos_percent_tmp;
			
			image_percent_tmp    = image->width;
			image->width         = image->height;
			image->height        = image_percent_tmp;
			break;
		case ROTATE_CCW:
			border_tmp = image->border;
			
			image->border.top    = border_tmp.right;
			image->border.right  = border_tmp.bottom;
			image->border.bottom = border_tmp.left;
			image->border.left   = border_tmp.top;
			
			inner_padding_tmp = image->inner_padding;
			
			image->inner_padding.top    = inner_padding_tmp.right;
			image->inner_padding.right  = inner_padding_tmp.bottom;
			image->inner_padding.bottom = inner_padding_tmp.left;
			image->inner_padding.left   = inner_padding_tmp.top;
			
			border_tmp = image->repeat;
			
			image->repeat.top    = border_tmp.right;
			image->repeat.right  = border_tmp.bottom;
			image->repeat.bottom = border_tmp.left;
			image->repeat.left   = border_tmp.top;			
			
			pos_percent_tmp      = image->xpos;
			image->xpos          = image->ypos;
			image->ypos.pixel    = -pos_percent_tmp.pixel;
			image->ypos.widget   = -pos_percent_tmp.widget;
			
			image_percent_tmp    = image->width;
			image->width         = image->height;
			image->height        = image_percent_tmp;
			break;
		case ROTATE_AROUND:
			border_tmp = image->border;
			
			image->border.top    = border_tmp.bottom;
			image->border.right  = border_tmp.left;
			image->border.bottom = border_tmp.top;
			image->border.left   = border_tmp.right;
			
			inner_padding_tmp = image->inner_padding;
			
			image->inner_padding.top    = inner_padding_tmp.bottom;
			image->inner_padding.right  = inner_padding_tmp.left;
			image->inner_padding.bottom = inner_padding_tmp.top;
			image->inner_padding.left   = inner_padding_tmp.right;
			
			border_tmp = image->repeat;
			
			image->repeat.top    = border_tmp.bottom;
			image->repeat.right  = border_tmp.left;
			image->repeat.bottom = border_tmp.top;
			image->repeat.left   = border_tmp.right;
			
			image->xpos.pixel    = -image->xpos.pixel;
			image->xpos.widget   = -image->xpos.widget;
			image->ypos.pixel    = -image->ypos.pixel;
			image->ypos.widget   = -image->ypos.widget;
			break;
		case ROTATE_NONE:
			break;
	}
}

enum {
	NORTH,
	SOUTH,
	EAST,
	WEST,
	NORTH_EAST,
	NORTH_WEST,
	SOUTH_EAST,
	SOUTH_WEST,
	CENTER,
} eXperienceAreas;

static GdkPixbuf *
scale_image_part (GdkPixbuf * src, GdkRectangle * src_area, guint width, guint height, GdkInterpType interp_type)
{
	GdkPixbuf * sub_pixbuf;
	GdkPixbuf * result;
	
	sub_pixbuf = gdk_pixbuf_new_subpixbuf (src, src_area->x, src_area->y, src_area->width, src_area->height);
	
	result = experience_gdk_pixbuf_scale_or_ref (sub_pixbuf, width, height, interp_type);
	
	g_object_unref (sub_pixbuf);
	return result;
}

const eXperienceComponents convert[9] = {
	COMPONENT_NORTH,
	COMPONENT_SOUTH,
	COMPONENT_EAST,
	COMPONENT_WEST,
	COMPONENT_NORTH_EAST,
	COMPONENT_NORTH_WEST,
	COMPONENT_SOUTH_EAST,
	COMPONENT_SOUTH_WEST,
	COMPONENT_CENTER,
};

static gboolean
draw (eXperienceDrawable * drawable, GdkPixbuf * dest, GdkRectangle * dest_area, GtkStyle * style)
{
	eXperienceCacheImage cache_image;
	eXperienceImage * image = (eXperienceImage*) drawable;
	gint width, height;
	gint px_width, px_height;
	eXperienceBorder px_border;
	GdkPixbuf * scaled_pixbuf[9];
	GdkRectangle src_area[9];
	GdkRectangle clip_area, tmp_area;
	gint scaled_width[9], scaled_height[9];
	gint xpos, ypos;
	gint image_xpos = 0, image_ypos = 0, image_ypos_row_start, ypos_row_start;
	gint i;
		
	g_assert (drawable   != NULL);
	g_assert (dest_area  != NULL);
	g_assert (style != NULL);
	g_assert (drawable->class == experience_image_class);
	
	/* Nothing is supposed to be drawn. So return. */
	
	for (i = 0; i <= CENTER; i++) {
		scaled_pixbuf[i] = NULL;
	}
	
	if (image->pixbuf == NULL) {
		/* try load image ... */
		cache_image.file   = image->file;
		cache_image.filter = drawable->filter;
		
		image->pixbuf = experience_get_image_pixbuf (&cache_image, style);
		
		if (image->pixbuf == NULL) {
			return FALSE;
		}
		
		g_object_add_weak_pointer ((GObject*) image->pixbuf, (gpointer*) &image->pixbuf);
	}
	
	width  = image->width. widget * dest_area->width  + image->width. image * gdk_pixbuf_get_width  (image->pixbuf) + image->width. pixel;
	height = image->height.widget * dest_area->height + image->height.image * gdk_pixbuf_get_height (image->pixbuf) + image->height.pixel;
	
	px_width  = width  - (image->inner_padding.left + image->inner_padding.right);
	px_height = height - (image->inner_padding.top  + image->inner_padding.bottom);
	
	if ((px_width <= 0) | (px_height <= 0)) {
		return TRUE;
	}
	
	clip_area.x = 0;
	clip_area.y = 0;
	clip_area.width  = gdk_pixbuf_get_width  (dest);
	clip_area.height = gdk_pixbuf_get_height (dest);
	
	if (!image->dont_clip)
		gdk_rectangle_intersect (dest_area, &clip_area, &clip_area);
	
	/* make image border smaller if it doesn't fit on the image. */
	if (image->border.left + image->border.right >= gdk_pixbuf_get_width (image->pixbuf)) {
		g_printerr ("Image border (horizontal) of image #%i in group \"%s\" is too big!\n", drawable->number, drawable->group_name);
		image->border.left  =  gdk_pixbuf_get_width (image->pixbuf) / 2;
		image->border.right = (gdk_pixbuf_get_width (image->pixbuf) - 1) / 2;
	}
	if (image->border.top + image->border.bottom >= gdk_pixbuf_get_height (image->pixbuf)) {
		g_printerr ("Image border (vertical) of image #%i in group \"%s\" is too big!\n", drawable->number, drawable->group_name);
		image->border.top    =  gdk_pixbuf_get_height (image->pixbuf) / 2;
		image->border.bottom = (gdk_pixbuf_get_height (image->pixbuf) - 1) / 2;
	}
	
	px_border = image->border;
/*	g_print ("drawing: %s, %i, %s\n", drawable->group_name, drawable->number, image->file);*/
	/* make border smaller if it is to big. */
	if ((px_border.left + px_border.right) >= px_width) {
		px_border.left  = px_width / 2;
		px_border.right = (px_width - 1) / 2;
	}
	if ((px_border.top + px_border.bottom) >= px_height) {
		px_border.top    = px_height / 2;
		px_border.bottom = (px_height - 1) / 2;
	}
	
	/*############*/
	src_area[NORTH_WEST].x = 0;
	src_area[NORTH_WEST].y = 0;
	src_area[NORTH_WEST].width  = image->border.left;
	src_area[NORTH_WEST].height = image->border.top;
	scaled_width[NORTH_WEST]  = px_border.left;
	scaled_height[NORTH_WEST] = px_border.top;
	
	src_area[NORTH].x = image->border.left;
	src_area[NORTH].y = 0;
	src_area[NORTH].width  = gdk_pixbuf_get_width (image->pixbuf) - image->border.left - image->border.right;
	src_area[NORTH].height = image->border.top;
	scaled_width[NORTH]  = px_width - px_border.left - px_border.right;
	scaled_height[NORTH] = px_border.top;

	src_area[NORTH_EAST].x = gdk_pixbuf_get_width (image->pixbuf) - image->border.right;
	src_area[NORTH_EAST].y = 0;
	src_area[NORTH_EAST].width  = image->border.right;
	src_area[NORTH_EAST].height = image->border.top;
	scaled_width[NORTH_EAST]  = px_border.right;
	scaled_height[NORTH_EAST] = px_border.top;
	
	/*--*/
	
	src_area[WEST].x = 0;
	src_area[WEST].y = image->border.top;
	src_area[WEST].width  = image->border.left;
	src_area[WEST].height = gdk_pixbuf_get_height (image->pixbuf) - image->border.top - image->border.bottom;
	scaled_width[WEST]  = px_border.left;
	scaled_height[WEST] = px_height - px_border.top - px_border.bottom;
	
	src_area[CENTER].x = image->border.left;
	src_area[CENTER].y = image->border.top;
	src_area[CENTER].width  = gdk_pixbuf_get_width (image->pixbuf) - image->border.left - image->border.right;
	src_area[CENTER].height = gdk_pixbuf_get_height (image->pixbuf) - image->border.top - image->border.bottom;
	scaled_width[CENTER]  = px_width - px_border.left - px_border.right;
	scaled_height[CENTER] = px_height - px_border.top - px_border.bottom;

	src_area[EAST].x = gdk_pixbuf_get_width (image->pixbuf) - image->border.right;
	src_area[EAST].y = image->border.top;
	src_area[EAST].width  = image->border.right;
	src_area[EAST].height = gdk_pixbuf_get_height (image->pixbuf) - image->border.top - image->border.bottom;
	scaled_width[EAST]  = px_border.right;
	scaled_height[EAST] = px_height - px_border.top - px_border.bottom;
	
	/*--*/
	
	src_area[SOUTH_WEST].x = 0;
	src_area[SOUTH_WEST].y = gdk_pixbuf_get_height (image->pixbuf) - image->border.bottom;
	src_area[SOUTH_WEST].width  = image->border.left;
	src_area[SOUTH_WEST].height = image->border.bottom;
	scaled_width[SOUTH_WEST]  = px_border.left;
	scaled_height[SOUTH_WEST] = px_border.bottom;
	
	src_area[SOUTH].x = image->border.left;
	src_area[SOUTH].y = gdk_pixbuf_get_height (image->pixbuf) - image->border.bottom;
	src_area[SOUTH].width  = gdk_pixbuf_get_width (image->pixbuf) - image->border.left - image->border.right;
	src_area[SOUTH].height = image->border.bottom;
	scaled_width[SOUTH]  = px_width - px_border.left - px_border.right;
	scaled_height[SOUTH] = px_border.bottom;

	src_area[SOUTH_EAST].x = gdk_pixbuf_get_width (image->pixbuf) - image->border.right;
	src_area[SOUTH_EAST].y = gdk_pixbuf_get_height (image->pixbuf) - image->border.bottom;
	src_area[SOUTH_EAST].width  = image->border.right;
	src_area[SOUTH_EAST].height = image->border.bottom;
	scaled_width[SOUTH_EAST]  = px_border.right;
	scaled_height[SOUTH_EAST] = px_border.bottom;
	
	/*#############*/
	
	xpos = experience_round(image->rounding, ((image->xpos.widget + 1.0) * (gfloat) dest_area->width  / 2.0) - ((image->xpos.widget + 1.0) * (gfloat) width  / 2.0));
	ypos = experience_round(image->rounding, ((image->ypos.widget + 1.0) * (gfloat) dest_area->height / 2.0) - ((image->ypos.widget + 1.0) * (gfloat) height / 2.0));
	
	xpos += image->xpos.pixel;
	ypos += image->ypos.pixel;
	
	xpos += dest_area->x;
	ypos += dest_area->y;
	
	image_xpos = -1;
	image_ypos = -1;
	
	while (((xpos + (gint) width) > clip_area.x) && ((-image_xpos < (gint) image->repeat.left) || (image->repeat.left == 0))) {
		xpos -= width;
		image_xpos -= 1;
	}
	while (((ypos + (gint) height) > clip_area.y) && ((-image_ypos < (gint) image->repeat.top) || (image->repeat.top == 0))) {
		ypos -= height;
		image_ypos -= 1;
	}
	/* add one to position ... */
	image_xpos += 1;
	image_ypos += 1;
	
	/* now I am in the top left corner. */
	image_ypos_row_start = image_ypos;
	ypos_row_start = ypos;
	while ((xpos <= (gint) (clip_area.x + clip_area.width)) && ((image_xpos < (gint) image->repeat.right) || (image->repeat.right == 0))) {
		image_ypos = image_ypos_row_start;
		ypos = ypos_row_start;
		while ((ypos <= (gint) (clip_area.y + clip_area.height)) && ((image_ypos < (gint) image->repeat.bottom) || (image->repeat.bottom == 0))) {
			/* draw experiences */
			#define DRAW_EXPERIENCE(area, the_xpos, the_ypos) { \
				if (image->draw_components & convert[area]) { \
					if ((src_area[area].width > 0) && (src_area[area].height > 0)) { \
						tmp_area.x = the_xpos; \
						tmp_area.y = the_ypos; \
						tmp_area.width  = scaled_width [area]; \
						tmp_area.height = scaled_height[area]; \
						gdk_rectangle_intersect (&clip_area, &tmp_area, &tmp_area); \
						/* only draw if size is bigger than 0. */ \
						if ((tmp_area.width > 0) && (tmp_area.height > 0)) {\
							/* this needs to be drawn! */ \
							if (scaled_pixbuf[area] == NULL) { \
								scaled_pixbuf[area] = scale_image_part (image->pixbuf, &src_area[area], scaled_width[area], scaled_height[area], image->interp_type); \
							} \
							gdk_pixbuf_composite (scaled_pixbuf[area], dest, tmp_area.x, tmp_area.y, \
							                      tmp_area.width, tmp_area.height, \
							                      the_xpos, the_ypos, 1, 1, GDK_INTERP_NEAREST, 255); \
						} \
					} \
				} \
			}
			if ((!image->draw_entire_only) || !((xpos < dest_area->x) || (ypos < dest_area->y)
			    || (xpos + width  > dest_area->x + dest_area->width)
			    || (ypos + height > dest_area->y + dest_area->height))) {
				DRAW_EXPERIENCE (NORTH_WEST, xpos + image->inner_padding.left,
				                         ypos + image->inner_padding.top);
				DRAW_EXPERIENCE (NORTH,      xpos + image->inner_padding.left + (gint) px_border.left,
				                         ypos + image->inner_padding.top);
				DRAW_EXPERIENCE (NORTH_EAST, xpos + image->inner_padding.left + px_width - (gint) px_border.right,
				                         ypos + image->inner_padding.top);
				
				DRAW_EXPERIENCE (WEST,   xpos + image->inner_padding.left,
				                     ypos + image->inner_padding.top  + (gint) px_border.top);
				DRAW_EXPERIENCE (CENTER, xpos + image->inner_padding.left + (gint) px_border.left,
				                     ypos + image->inner_padding.top  + (gint) px_border.top);
				DRAW_EXPERIENCE (EAST,   xpos + image->inner_padding.left + (gint) px_width - (gint) px_border.right,
				                     ypos + image->inner_padding.top  + (gint) px_border.top);
	
				DRAW_EXPERIENCE (SOUTH_WEST, xpos + image->inner_padding.left,
				                         ypos + image->inner_padding.top  + (gint) px_height - (gint) px_border.bottom);
				DRAW_EXPERIENCE (SOUTH,      xpos + image->inner_padding.left + (gint) px_border.left,
				                         ypos + image->inner_padding.top  + (gint) px_height - (gint) px_border.bottom);
				DRAW_EXPERIENCE (SOUTH_EAST, xpos + image->inner_padding.left + (gint) px_width  - (gint) px_border.right,
				                         ypos + image->inner_padding.top  + (gint) px_height - (gint) px_border.bottom);
			}
			
			ypos += height;
			image_ypos += 1;
		}
		xpos += width;
		image_xpos += 1;
	}
	
	for (i = 0; i <= CENTER; i++) {
		if (scaled_pixbuf[i] != NULL) {
			g_object_unref ((GObject*) scaled_pixbuf[i]);
			scaled_pixbuf[i] = NULL;
		}
	}
	
	return TRUE;
}

static eXperienceDrawableClass _experience_image_class;

void
experience_image_init_class (void)
{
	_experience_image_class.object_type = "image";
	_experience_image_class.create  = create;
	_experience_image_class.destroy = destroy;
	_experience_image_class.draw    = draw;
	_experience_image_class.inherit_from_drawable = inherit_from_drawable;
	_experience_image_class.apply_group_settings  = apply_group_settings;
	
	experience_image_class = &_experience_image_class;
}

