/******************************************************************************/
/* smooth_gtk1_rc.c - RC Function for GTK1 Engine                             */
/******************************************************************************/
/* Smooth Theme Engine                                                        */
/* Copyright (C) 2002-2004 Andrew Johnson                                     */
/*                                                                            */
/* This library is free software; you can redistribute it and/or              */
/* modify it under the terms of the GNU Lesser General Public                 */
/* License as published by the Free Software Foundation; either               */
/* version 2.1 of the License, or (at your option) any later version.         */
/*                                                                            */
/* This library is distributed in the hope that it will be useful,            */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of             */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          */
/* Lesser General Public License for more details.                            */
/*                                                                            */
/* You should have received a copy of the GNU Lesser General Public           */
/* License along with this library; if not, write to the Free Software        */
/* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  */
/*                                                                            */
/* Author(s): Andrew Johnson <ajgenius@ajgenius.us>                           */
/******************************************************************************/
/* Portions Based on GTK+                                                     */
/*   Peter Mattis <petm@xcf.berkeley.edu>                                     */
/*   Spencer Kimball <spencer@xcf.berkeley.edu>                               */
/*   Josh MacDonald <jmacd@xcf.berkeley.edu>                                  */
/*                                                                            */
/* Portions Based on the EnGradient Engine                                    */
/*   Andrew Cattau                                                            */
/*                                                                            */
/* Portions Based on the Xenophilia Engine                                    */
/*   Johan Hanson <misagon@bahnhof.se>                                        */
/******************************************************************************/
#include <stdio.h>
#include "smooth_gtk1_engine.h"
#include "smooth_gtk1_drawing.h"
#include "smooth_gtk1_misc.h"
#include "smooth_gtk1_rc.h"

enum
  {
    TOKEN_XTHICKNESS = TOKEN_LAST + 1,
    TOKEN_YTHICKNESS
  };
  
ThemeSymbols gtk1_theme_symbols[] =
{
  { "xthickness",          TOKEN_XTHICKNESS },
  { "ythickness",          TOKEN_YTHICKNESS },
};

guint n_gtk1_theme_symbols = sizeof(gtk1_theme_symbols) / sizeof(gtk1_theme_symbols[0]);

void smooth_rc_style_init (SmoothRcStyle *style)
{
  gint i;

  style->real_sliders    = DEFAULT_REAL_SLIDERS;
  style->line.style      = SMOOTH_BEVEL_STYLE_DEFAULT;
  style->line.thickness  = DEFAULT_LINETHICKNESS;

  memset(&style->edge, 0, sizeof(smooth_edge_style));

  style->edge.style           = DEFAULT_EDGESTYLE;
  style->edge.use_line        = FALSE;
  style->edge.line.style      = SMOOTH_BEVEL_STYLE_DEFAULT;
  style->edge.line.thickness  = DEFAULT_LINETHICKNESS;

  style->fill.style      = SMOOTH_FILL_STYLE_DEFAULT;
  style->fill.default_dither_depth_set = FALSE;
  style->fill.default_dither_depth = 8;
  style->fill.gradient_direction[FALSE] = DEFAULT_VGRADIENTDIRECTION;
  style->fill.gradient_direction[TRUE] = DEFAULT_HGRADIENTDIRECTION;
  style->fill.quadratic_gradient = FALSE;
  style->fill.shade1 = 1.3;
  style->fill.shade2 = 0.7;
  
  style->buffered_fill = TRUE;

  for (i=0; i < 5; i++) {
    style->fill.use_color1[i] = FALSE;
    style->fill.use_color2[i] = FALSE;
    style->fill.file_name[i] = NULL;
    style->fill.use_dither_depth[i] = FALSE;

    style->focus.use_foreground[i] = FALSE;
    style->focus.pattern[i] = NULL;
    style->focus.use_line_width[i] = FALSE;
    style->focus.line_width[i] = 1;
  }

  style->focus.interior = FALSE;
  style->focus.width = 1;
  style->focus.pad = 0;

  style->range.slider_width = 7;
  style->range.trough_border = 1;
  style->range.stepper_size = 11;
  style->range.stepper_spacing = 1;

  style->scale.slider_length = 7;

  style->scrollbar.min_slider_length = 13;

  memset(&style->arrow, 0, sizeof(SmoothArrowPart));
  style->arrow.DefaultStyle = g_new(SmoothArrow, 1);
  style->arrow.DefaultStyle->Style     = SMOOTH_ARROW_STYLE_DEFAULT;
  style->arrow.DefaultStyle->Solid     = DEFAULT_SOLIDARROW;
  style->arrow.DefaultStyle->Etched    = DEFAULT_ETCHEDARROW;
  style->arrow.DefaultStyle->Tail      = DEFAULT_TAILARROW;

  part_init(THEME_PART(&style->grip), DEFAULT_GRIPSTYLE);

  style->grip.count      = DEFAULT_GRIPCOUNT;
  style->grip.spacing    = DEFAULT_GRIPSPACING;
  style->grip.overlap    = DEFAULT_GRIPOVERLAP;

  part_init(THEME_PART(&style->check), SMOOTH_CHECKMARK_STYLE_DEFAULT);
  style->check.motif = TRUE;

  part_init(THEME_PART(&style->option), SMOOTH_CHECKMARK_STYLE_DEFAULT_OPTION);
  style->option.motif = TRUE;

  part_init(&style->progress, 0);

  part_init(THEME_PART(&style->trough), 0);
  style->trough.show_value = DEFAULT_TROUGH_SHOW_VALUE;
  
  part_init(THEME_PART(&style->stepper), 0);
  memset(&style->stepper.Arrow, 0, sizeof(SmoothArrowPart));
  style->stepper.Arrow.Inherited = &style->arrow;

  part_init(THEME_PART(&style->button), 0);

  part_init(&style->button.button_default, SMOOTH_BUTTON_DEFAULT_STYLE_DEFAULT);
  style->button.default_triangle = SMOOTH_BUTTON_DEFAULT_STYLE_DEFAULT;
  style->button.embeddable = FALSE;
  style->button.use_button_default = FALSE;
  
  part_init(THEME_PART(&style->tabs), DEFAULT_TABSTYLE);

  part_init(&style->tabs.active_tab, DEFAULT_TABSTYLE);
  style->tabs.use_active_tab	= FALSE;

  style->xthickness       = -1;
  style->ythickness       = -1;

  style->paned_handle_size	  = 10;
  style->buffered_fill = FALSE;
}

guint 
theme_parse_pixmap(GScanner *scanner, 
                   GTokenType wanted_token, 
                   gchar *default_value,
                   GString **retval)
{
	guint token;
	gchar *tmpstr=NULL;

	token = g_scanner_cur_token (scanner);
	if (token != wanted_token)
	{	
		token = g_scanner_get_next_token (scanner);
		if (token != wanted_token)
		{
			return wanted_token;
		}
	}
	
	if (wanted_token != G_TOKEN_EQUAL_SIGN)
	{
		token = g_scanner_get_next_token (scanner);
		if (token != G_TOKEN_EQUAL_SIGN)
			return G_TOKEN_EQUAL_SIGN;
	}
		
	token = g_scanner_get_next_token (scanner);
	if (token == G_TOKEN_STRING)
		tmpstr = gtk_rc_find_pixmap_in_path(scanner, scanner->value.v_string);	

	if (tmpstr == NULL)
		tmpstr = default_value;

	if (tmpstr == NULL)
		*retval =  NULL;
	else  
		*retval =  g_string_new(tmpstr);
	
	return token;
}

static void
smooth_gtk1_style_overides(SmoothRcStyle *smooth_style, SmoothRcProperty * smooth_prop)
{		       
  #define is_class(xx) ((xx) && (!strcmp(xx, smooth_prop->class_name)))
  #define is_prop(xx) ((xx) && (!strcmp(xx, smooth_prop->property_name)))

  if (is_class("GtkRange"))
  {
    if (is_prop("slider-width") || is_prop("slider_width")) 
        smooth_style->range.slider_width = smooth_prop->value.d.long_data;
    if (is_prop("stepper_size") || is_prop("stepper-size")) 
        smooth_style->range.stepper_size = smooth_prop->value.d.long_data;
    if (is_prop("stepper_spacing") || is_prop("stepper-spacing")) 
        smooth_style->range.stepper_spacing = smooth_prop->value.d.long_data;
    if (is_prop("trough_border") || is_prop("trough_border")) 
        smooth_style->range.trough_border = smooth_prop->value.d.long_data;
  } 
  else
    if (is_class("GtkPaned"))
    {
      if (is_prop("handle-size") || is_prop("handle_size")) 
        smooth_style->paned_handle_size = smooth_prop->value.d.long_data;
    }
  else
    if (is_class("GtkScrollbar"))
    {
      if (is_prop("min_slider_length") || is_prop("min-slider-length")) 
        smooth_style->scrollbar.min_slider_length = smooth_prop->value.d.long_data;
    }
  else
    if (is_class("GtkScale"))
    {
      GtkScaleClass *scaleclass = gtk_type_class(gtk_scale_get_type());
      GtkRangeClass *rangeclass = (GtkRangeClass *)scaleclass;
      if (is_prop("slider_width") || is_prop("slider-width")) 
        rangeclass->slider_width = smooth_prop->value.d.long_data;
      if (is_prop("slider_length") || is_prop("slider-length")) 
        smooth_style->scale.slider_length = smooth_prop->value.d.long_data;
    }
  else
    if (is_class("GtkCheckButton"))
    {
      GtkCheckButtonClass *checkbuttonclass = (GtkCheckButtonClass*)gtk_type_class (gtk_check_button_get_type ());
  
      if (is_prop("indicator_size") || is_prop("indicator-size")) 
        checkbuttonclass->indicator_size = smooth_prop->value.d.long_data;
    }
  else
    if (is_class("GtkRadioButton"))
    {
      GtkCheckButtonClass *checkbuttonclass = (GtkCheckButtonClass*)gtk_type_class (gtk_radio_button_get_type ());

      if (is_prop("indicator_size") || is_prop("indicator-size"))  
        checkbuttonclass->indicator_size = smooth_prop->value.d.long_data;
    }
  else 
    if (is_class("GtkWidget"))
    {
      if (is_prop("interior_focus") || is_prop("interior-focus"))  
        smooth_style->focus.interior = smooth_prop->value.d.long_data;

      if (is_prop("focus_line_width") || is_prop("focus-line-width"))  
        smooth_style->focus.width = smooth_prop->value.d.long_data;

      if (is_prop("focus_padding ") || is_prop("focus-padding"))  
        smooth_style->focus.pad = smooth_prop->value.d.long_data;
    }

   
}

static guint
gtk_rc_parse_assignment (GScanner      *scanner,
			 SmoothRcProperty *prop)
{
  gboolean scan_identifier = scanner->config->scan_identifier;
  gboolean scan_symbols = scanner->config->scan_symbols;
  gboolean identifier_2_string = scanner->config->identifier_2_string;
  gboolean char_2_token = scanner->config->char_2_token;
  gboolean scan_identifier_NULL = scanner->config->scan_identifier_NULL;
  gboolean numbers_2_int = scanner->config->numbers_2_int;
  gboolean negate = FALSE;
  guint token;

  /* check that this is an assignment */
  if (g_scanner_get_next_token (scanner) != '=')
    return '=';

  /* adjust scanner mode */
  scanner->config->scan_identifier = TRUE;
  scanner->config->scan_symbols = FALSE;
  scanner->config->identifier_2_string = FALSE;
  scanner->config->char_2_token = TRUE;
  scanner->config->scan_identifier_NULL = FALSE;
  scanner->config->numbers_2_int = TRUE;

  /* parse optional sign */
  if (g_scanner_peek_next_token (scanner) == '-')
    {
      g_scanner_get_next_token (scanner); /* eat sign */
      negate = TRUE;
    }

  /* parse one of LONG, DOUBLE and STRING or, if that fails, create an unparsed compund */
  token = g_scanner_peek_next_token (scanner);
  switch (token)
    {
    case G_TOKEN_INT:
      g_scanner_get_next_token (scanner);
      prop->value.type = GTK_TYPE_LONG;
      prop->value.d.long_data = negate ? -scanner->value.v_int : scanner->value.v_int;
      token = G_TOKEN_NONE;
      break;
    case G_TOKEN_FLOAT:
      g_scanner_get_next_token (scanner);
      prop->value.type = GTK_TYPE_DOUBLE;
      prop->value.d.double_data = negate ? -scanner->value.v_float : scanner->value.v_float;
      token = G_TOKEN_NONE;
      break;
    case G_TOKEN_STRING:
      g_scanner_get_next_token (scanner);
      if (negate)
	token = G_TOKEN_INT;
      else
	{
          prop->value.type = GTK_TYPE_STRING;
          prop->value.d.string_data = scanner->value.v_string;
	  token = G_TOKEN_NONE;
	}
      break;
    case G_TOKEN_LEFT_CURLY :
      while (token != G_TOKEN_RIGHT_CURLY) {
        g_scanner_peek_next_token (scanner);   
        g_scanner_get_next_token (scanner);
        token = g_scanner_peek_next_token (scanner);
      } 
      g_scanner_get_next_token (scanner);
      token = G_TOKEN_NONE;      
      break;      
    default:
      g_scanner_get_next_token (scanner);
      token = G_TOKEN_INT;
      break;
    }

  /* restore scanner mode */
  scanner->config->scan_identifier = scan_identifier;
  scanner->config->scan_symbols = scan_symbols;
  scanner->config->identifier_2_string = identifier_2_string;
  scanner->config->char_2_token = char_2_token;
  scanner->config->scan_identifier_NULL = scan_identifier_NULL;
  scanner->config->numbers_2_int = numbers_2_int;

  return token;
}

guint
smooth_gtkrc_parse(GScanner   *Scanner,
			SmoothRcStyle *Style,
			guint Token)
{
	switch (Token)
	{
		case TOKEN_EDGE:
			Token = theme_parse_edge (Scanner, TOKEN_EDGE, &Style->edge);
		break;

		case TOKEN_FILL :
			Token = theme_parse_fill (Scanner, TOKEN_FILL, &Style->fill);
		break;

		case TOKEN_TABS:
			Token = theme_parse_tab (Scanner, TOKEN_TABS, &Style->tabs);
		break;

		case TOKEN_BUTTON:
			Token = theme_parse_button (Scanner, TOKEN_BUTTON, &Style->button);
		break;

		case TOKEN_REAL_SLIDERS:
			Token = theme_parse_boolean (Scanner, TOKEN_REAL_SLIDERS, DEFAULT_REAL_SLIDERS, &Style->real_sliders);
		break;

		case TOKEN_RESIZE_GRIP:
			Token = theme_parse_boolean (Scanner, TOKEN_RESIZE_GRIP, DEFAULT_RESIZE_GRIP, &Style->resize_grip);
		break;

		case TOKEN_LINE:
			Token = theme_parse_line (Scanner, TOKEN_LINE, &Style->line);
		break;

		case TOKEN_GRIP:
			Token = theme_parse_grip (Scanner, TOKEN_GRIP, &Style->grip);
		break;

		case TOKEN_STEPPER:
			Token = theme_parse_stepper (Scanner, TOKEN_STEPPER, &Style->stepper);
		break;

		case TOKEN_ARROW:
			Token = theme_parse_arrow_part (Scanner, TOKEN_ARROW, &Style->arrow);
		break;	  

		case TOKEN_FOCUS:
			Token = theme_parse_focus (Scanner, TOKEN_FOCUS, &Style->focus);
		break;

		case TOKEN_PROGRESS:
			Token = theme_parse_generic_part (Scanner, TOKEN_PROGRESS, &Style->progress);
		break;

		case TOKEN_TROUGH:
			Token = theme_parse_trough_part (Scanner, TOKEN_TROUGH, &Style->trough);
		break;

		case TOKEN_CHECK:
			Token = theme_parse_check (Scanner, TOKEN_CHECK, &Style->check);
		break;

		case TOKEN_OPTION:
			Token = theme_parse_option (Scanner, TOKEN_OPTION, &Style->option);
		break;

		/* backward compatibility for <=0.5.4 */
		case TOKEN_DEPRECATED_TABSTYLE:
			Token = theme_parse_custom_enum(Scanner, TOKEN_DEPRECATED_TABSTYLE, TranslateTabStyleName, DEFAULT_TABSTYLE, 
							&THEME_PART(&Style->tabs)->style);
		break;

		/* backward compatibility for <=0.5.0 */
		case TOKEN_DEPRECATED_ARROWSTYLE:
		{
			if (!Style->arrow.DefaultStyle)
				Style->arrow.DefaultStyle = g_new0(SmoothArrow, 1);

			Token = theme_parse_custom_enum(Scanner, TOKEN_DEPRECATED_ARROWSTYLE, 
							(SmoothTranslateEnumFunc)TranslateArrowStyleName, 
							SMOOTH_ARROW_STYLE_DEFAULT, 
							&Style->arrow.DefaultStyle->Style);
			Style->arrow.DefaultStyle->HasStyle = SmoothTrue;

			switch (Style->arrow.DefaultStyle->Style)
			{
				case SMOOTH_ARROW_STYLE_DEPRECATED_CLEANICE :
					Style->arrow.DefaultStyle->Style = SMOOTH_ARROW_STYLE_CLEAN;
					Style->arrow.DefaultStyle->Tail = 0;
					Style->arrow.DefaultStyle->HasTail = SmoothTrue;
				break;
	
				case SMOOTH_ARROW_STYLE_DEPRECATED_XFCE :
					Style->arrow.DefaultStyle->Style = SMOOTH_ARROW_STYLE_CLEAN;
					Style->arrow.DefaultStyle->Tail = 1;
					Style->arrow.DefaultStyle->HasTail = SmoothTrue;
				break;

				case SMOOTH_ARROW_STYLE_DEPRECATED_THINICE :
					Style->arrow.DefaultStyle->Style = SMOOTH_ARROW_STYLE_CLEAN;
					Style->arrow.DefaultStyle->Tail = 2;
					Style->arrow.DefaultStyle->HasTail = SmoothTrue;
				break;

				default :
				break;
			}
		}
		break;

		case TOKEN_DEPRECATED_SOLIDARROW:
		{
			if (!Style->arrow.DefaultStyle)
				Style->arrow.DefaultStyle = g_new0(SmoothArrow, 1);

			Token = theme_parse_boolean (Scanner, TOKEN_DEPRECATED_SOLIDARROW, 
							DEFAULT_SOLIDARROW, 
							&Style->arrow.DefaultStyle->Solid);
			Style->arrow.DefaultStyle->HasSolid = SmoothTrue;
		}
		break;

		case TOKEN_DEPRECATED_ETCHEDARROW:
		{
			if (!Style->arrow.DefaultStyle)
				Style->arrow.DefaultStyle = g_new0(SmoothArrow, 1);

			Token = theme_parse_boolean (Scanner, TOKEN_DEPRECATED_ETCHEDARROW, 
							DEFAULT_ETCHEDARROW, &Style->arrow.DefaultStyle->Etched);
			Style->arrow.DefaultStyle->HasEtched = SmoothTrue;
		}
		break;

		/* backward compatibility for <=0.4.0 */
		case TOKEN_DEPRECATED_GRADIENT:
		{
			gboolean use_gradient=FALSE;
	
			Token = theme_parse_boolean (Scanner, TOKEN_DEPRECATED_GRADIENT, TRUE, &use_gradient);
	    
			if (use_gradient) 
			{
				Style->fill.style = SMOOTH_FILL_STYLE_SHADE_GRADIENT;
				Style->fill.quadratic_gradient = TRUE;
				Style->fill.gradient_direction[FALSE] = SMOOTH_GRADIENT_HORIZONTAL;			
				Style->fill.gradient_direction[TRUE] = SMOOTH_GRADIENT_VERTICAL;			
				Style->fill.shade1 = 1.3;
				Style->fill.shade2 = 0.7;
			}
		}  
		break;

		case TOKEN_XTHICKNESS:
			Token = theme_parse_int (Scanner, TOKEN_XTHICKNESS, 2, &Style->xthickness, 0, 100);
		break;	  

		case TOKEN_YTHICKNESS:
			Token = theme_parse_int (Scanner, TOKEN_YTHICKNESS, 2, &Style->ythickness, 0, 100);
		break;	  
		
		case G_TOKEN_IDENTIFIER:
		{
			SmoothRcProperty prop = {NULL, NULL, {1, NULL,}};
	      
			g_scanner_get_next_token (Scanner); /* eat type name */
	      
			prop.class_name = g_strdup(Scanner->value.v_identifier);
	      
			if (g_scanner_get_next_token (Scanner) != ':' ||
				g_scanner_get_next_token (Scanner) != ':')
			{
				Token = ':';
				break;
			}

			g_scanner_get_next_token (Scanner);

			prop.property_name = g_strdup(Scanner->value.v_identifier);
			Token = gtk_rc_parse_assignment (Scanner, &prop);
			if (Token == G_TOKEN_NONE)
				smooth_gtk1_style_overides(Style, &prop);
			if (prop.class_name) g_free(prop.class_name);
			if (prop.property_name) g_free(prop.property_name);
		}
		break;

		default:
			g_scanner_get_next_token (Scanner);
			Token = G_TOKEN_RIGHT_CURLY;
		break;
	}

	return Token;
}
	
void
smooth_gtkrc_style_merge (SmoothRcStyle *dest_data,
                          SmoothRcStyle *src_data)
{
      gint i;

      dest_data->real_sliders = src_data->real_sliders;
      dest_data->line.style = src_data->line.style;
      dest_data->line.thickness = src_data->line.thickness;

      dest_data->edge.use_line        = src_data->edge.use_line;
      dest_data->edge.line.style      = src_data->edge.line.style;
      dest_data->edge.line.thickness  = src_data->edge.line.thickness;

      dest_data->fill.style = src_data->fill.style;
      dest_data->fill.quadratic_gradient = src_data->fill.quadratic_gradient;
      dest_data->fill.gradient_direction[FALSE] = src_data->fill.gradient_direction[FALSE];
      dest_data->fill.gradient_direction[TRUE] = src_data->fill.gradient_direction[TRUE];
      dest_data->fill.shade1 = src_data->fill.shade1;
      dest_data->fill.shade2 = src_data->fill.shade2;
      dest_data->fill.default_dither_depth_set = src_data->fill.default_dither_depth_set;
      dest_data->fill.default_dither_depth = src_data->fill.default_dither_depth;

      for (i=0; i < 5; i++) {
        dest_data->fill.use_color1[i] = src_data->fill.use_color1[i];
        if (src_data->fill.use_color1[i])
          dest_data->fill.color1[i] = src_data->fill.color1[i];

        dest_data->fill.use_color2[i] = src_data->fill.use_color2[i];
        if (src_data->fill.use_color2[i])
          dest_data->fill.color2[i] = src_data->fill.color2[i];

        dest_data->fill.use_dither_depth[i] = src_data->fill.use_dither_depth[i];
        if (src_data->fill.use_dither_depth[i])
	        dest_data->fill.dither_depth[i] = src_data->fill.dither_depth[i];

        if (src_data->fill.file_name[i]) 
	{
		if (!dest_data->fill.file_name[i])
			dest_data->fill.file_name[i] =  g_string_sized_new(src_data->fill.file_name[i]->len);
		
		g_string_assign(dest_data->fill.file_name[i], src_data->fill.file_name[i]->str);
        }
        
        if (src_data->focus.pattern[i]) {
          if (dest_data->focus.pattern[i])
            g_free(dest_data->focus.pattern[i]);

          dest_data->focus.pattern[i] = g_strdup(src_data->focus.pattern[i]);
        }

        dest_data->focus.use_foreground[i] = src_data->focus.use_foreground[i];
        if (src_data->focus.use_foreground[i])
          dest_data->focus.foreground[i] = src_data->focus.foreground[i];

        dest_data->focus.use_line_width[i] = src_data->focus.use_line_width[i];
        if (src_data->focus.use_line_width[i])
          dest_data->focus.line_width[i] = src_data->focus.line_width[i];

      }
      
      dest_data->focus.interior = src_data->focus.interior;
      dest_data->focus.width = src_data->focus.width;
      dest_data->focus.pad = src_data->focus.pad;

      dest_data->range.slider_width = src_data->range.slider_width;
      dest_data->range.trough_border = src_data->range.trough_border;

      dest_data->range.stepper_size = src_data->range.stepper_size;
      dest_data->range.stepper_spacing = src_data->range.stepper_spacing;

      dest_data->scale.slider_length = src_data->scale.slider_length;

      dest_data->scrollbar.min_slider_length = src_data->scrollbar.min_slider_length;

      arrow_merge (&dest_data->arrow, &src_data->arrow);

      part_merge(THEME_PART(&dest_data->grip),THEME_PART(&src_data->grip));
      dest_data->grip.count = src_data->grip.count;
      dest_data->grip.spacing = src_data->grip.spacing;
      dest_data->grip.overlap = src_data->grip.overlap;

      part_merge(THEME_PART(&dest_data->check),THEME_PART(&src_data->check));
      dest_data->check.motif = src_data->check.motif;
      
      part_merge(THEME_PART(&dest_data->option),THEME_PART(&src_data->option));
      dest_data->option.motif = src_data->option.motif;

      part_merge(THEME_PART(&dest_data->trough),THEME_PART(&src_data->trough));
      dest_data->trough.show_value = src_data->trough.show_value;
      
      part_merge(THEME_PART(&dest_data->stepper),THEME_PART(&src_data->stepper));
      arrow_merge (&dest_data->stepper.Arrow, &src_data->stepper.Arrow);
      dest_data->stepper.Arrow.Inherited = &dest_data->arrow;

      part_merge(&dest_data->progress,&src_data->progress);

      part_merge(THEME_PART(&dest_data->button), THEME_PART(&src_data->button));
      dest_data->button.default_triangle = src_data->button.default_triangle;
      dest_data->button.embeddable = src_data->button.embeddable;
      dest_data->button.use_button_default = src_data->button.use_button_default;
      part_merge(&dest_data->button.button_default,&src_data->button.button_default);

      part_merge(THEME_PART(&dest_data->tabs), THEME_PART(&src_data->tabs));
      dest_data->tabs.use_active_tab = src_data->tabs.use_active_tab;
      part_merge(&dest_data->tabs.active_tab,&src_data->tabs.active_tab);

      dest_data->buffered_fill = src_data->buffered_fill;
}
