/**************************************************************************
                          cmapwidget.cpp
                      -------------------
    description          : Map Display Widget
    begin                : Fri Oct 22 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmapwidget.h"

#include <qdir.h>
#include <qfile.h>
#include <qkeycode.h>
#include <qstack.h>
#include <qqueue.h>
#include <qimage.h>
#include <qmsgbox.h>

#include "dialogs/dlgmaptextproperties.h"
#include "dialogs/dlgmaproomproperties.h"
#include "dialogs/dlgmappathproperties.h"
#include "dialogs/dlgmapzoneproperties.h"

#include "cmapstructs_1_3.h"

static unsigned char move_bits[] = {
   0x80, 0x01, 0xc0, 0x03, 0xe0, 0x07, 0x80, 0x01, 0x80, 0x01, 0x84, 0x21,
   0x86, 0x61, 0xff, 0xff, 0xff, 0xff, 0x86, 0x61, 0x84, 0x21, 0x80, 0x01,
   0x80, 0x01, 0xe0, 0x07, 0xc0, 0x03, 0x80, 0x01};


const char mapFileVerMajor=1;
const char mapFileVerMinor=10;

// Sturcts used to save and load the maps

struct zoneTyp
{
	signed int x;
	signed int y;
	signed int numLevels;
	signed int level;
	signed int zone;
};

struct roomHeaderTyp
{
	int numRooms;
};

struct roomTyp
{
	int x;
	int y;
	int red,green,blue;
	int useDefaultCol;
	signed int level;
	int zone;
};

struct speedwalkRoomTyp
{
	int x;
	int y;
	signed int level;
	int zone;
};

struct textHeaderTyp
{
	int numText;
};

struct textTyp
{
	signed int level;
	int x,y;
	int size;
	int weight;
	int italic;
	int red, green,	blue;
	int zone;
};

struct pathHeaderTyp
{
	int numPaths;
};

struct pathTyp
{
	int srcX;
	int srcY;
	signed int srcLevel;
	int srcZone;
	int destX;
	int destY;
	signed destLevel;
	int destZone;	
	direction srcDir;
	direction destDir;
	bool special;
	int bendsCount;
};


struct mainHeaderTyp
{	
	int numZones;
};

CMapToolTip::CMapToolTip( QWidget * parent )
    : QToolTip( parent )
{
    // no explicit initialization needed
}

void CMapToolTip::maybeTip( const QPoint &pos )
{	
	if ( !parentWidget()->inherits( "CMapWidget" ) )
		return;
	
	QString s;
	CMapElement *element;
	int level = ((CMapWidget*)parentWidget())->getCurrentLevel();
	int elementx = (pos.x() / ROOM_SIZE)+1;
	int elementy = (pos.y() / ROOM_SIZE)+1;	
	
	element = ((CMapWidget*)parentWidget())->findElementAt(elementx,elementy,level,((CMapWidget*)parentWidget())->getCurrentZone());
	
	if (element)
	{		
	  if (element->getElementType() == ROOM)
	  {
	    s = ((CMapRoom*)element)->getLabel();
	  }
	  else if (element->getElementType() == ZONE)
	  {
	    s = ((CMapZone*)element)->getName();
	  }
		if (!s.stripWhiteSpace().isEmpty())
		{
			QRect r(element->getLowX(),element->getLowY(),element->getHiX()-element->getLowX(),
			        element->getHiY()-element->getLowY());
			tip( r, s );
		}
	}
}

CMapWidget::CMapWidget(CMapData *mapData,KmudDoc *document,QString mudTitle,CMudProfile* mudPro,QScrollView *view,QWidget *parent ) : QWidget(parent,"mapwidget")
{
	mapperData = mapData;
	setBackgroundMode(NoBackground);
	buffer = NULL;
	overviewBuffer = NULL;
	bNeedsCreate = true;
	xMax = 0;
	yMax = 0;
	bCtrlPressed = false;
	mapView = view;
	pathToolMode = 0;
	pathStartRoom = NULL;
	bMouseDrag = false;
	QBitmap mouseDragCursorShape(16,16, move_bits,TRUE);
	mouseDragCursor = new QCursor( mouseDragCursorShape, mouseDragCursorShape, -1,-1);
		
	currentTool = NULL;
	
	doc = document;
	
	mudProfile = mudPro;

	// Setup default values
	XGridSpace = ROOM_SIZE;
	YGridSpace = ROOM_SIZE;	
	hasGrid = false;

	resize(ROOM_SIZE*6,ROOM_SIZE*6);
	mapName = mudTitle + ".map";
	selected.element = NULL;

	// Create the rooms menu
	room_menu = new QPopupMenu();
	room_menu->insertItem(i18n("Set &Current Position"),ID_MAPPER_ROOM_MOVE_LOC);
	room_menu->insertItem(i18n("Set Room to &Login Point"),ID_MAPPER_ROOM_SET_LOGIN);
	room_menu->insertSeparator();
	room_menu->insertItem(i18n("&Speed walk to room"),ID_MAPPER_ROOM_MOVE_PLAYER);
	room_menu->insertItem(i18n("&Add to speed walk list"),ID_MAPPER_ROOM_ADD_SPEED);
	room_menu->insertSeparator();	
	room_menu->insertItem(i18n("&Delete room"),ID_MAPPER_ROOM_DEL);	
	room_menu->insertSeparator();
	room_menu->insertItem(i18n("&Properties..."),ID_MAPPER_ROOM_PROP);

	path_menu = new QPopupMenu();
	path_menu->setCheckable(true);
	path_menu->insertItem(i18n("&One way"),ID_MAPPER_PATH_ONEWAY);
	path_menu->insertItem(i18n("&Two way"),ID_MAPPER_PATH_TWOWAY);
	path_menu->insertSeparator();
	path_menu->insertItem(i18n("&Add Bend"),ID_MAPPER_PATH_ADDBEND);		
	path_menu->insertItem(i18n("&Remove Segment"),ID_MAPPER_PATH_DELBEND);			
	path_menu->insertItem(i18n("&Edit Bends"),ID_MAPPER_PATH_EDITBEND);				
	path_menu->insertSeparator();
	path_menu->insertItem(i18n("&Delete Path"),ID_MAPPER_PATH_DEL);
	path_menu->insertSeparator();
	path_menu->insertItem(i18n("&Properties"),ID_MAPPER_PATH_PROP);
	
	text_menu = new QPopupMenu();
	text_menu->insertItem(i18n("&Delete Text"),ID_MAPPER_TEXT_DEL);
	text_menu->insertSeparator();	
	text_menu->insertItem(i18n("&Properties"),ID_MAPPER_TEXT_PROP);	
	
	zone_menu = new QPopupMenu();	
	zone_menu->insertItem(i18n("&Open Zone"),ID_MAPPER_ZONE_OPEN);
	zone_menu->insertSeparator();	
	zone_menu->insertItem(i18n("&Delete Zone"),ID_MAPPER_ZONE_DEL);
	zone_menu->insertSeparator();	
	zone_menu->insertItem(i18n("&Properties"),ID_MAPPER_ZONE_PROP);		

	setFocusPolicy(QWidget::StrongFocus);
	setFocus();

	tip = new CMapToolTip(this);

	CONNECT_CMD(room_menu);
	CONNECT_CMD(path_menu);
	CONNECT_CMD(text_menu);		
	CONNECT_CMD(zone_menu);
}

CMapWidget::~CMapWidget()
{
	if (buffer)
		delete buffer;

	if (overviewBuffer)
		delete overviewBuffer;
	
	delete room_menu;
	delete path_menu;
	delete text_menu;
	delete zone_menu;
	delete tip;
	delete mouseDragCursor;
}

void CMapWidget::setSpeedWalkAbortActive(bool set)
{	speedWalkAbortActive = set; }

bool CMapWidget::getSpeedWalkAbortActive(void)
{	return speedWalkAbortActive; }

void CMapWidget::setSpeedWalkLimit(int limit)
{ speedWalkAbortLimit = limit; }

int CMapWidget::getSpeedWalkLimit(void)
{ return speedWalkAbortLimit; }

void CMapWidget::setViewOverview(bool visiable)
{
	bOverview = visiable;
	redraw();
}

/** Used to get the current level number */
int CMapWidget::getCurrentLevel(void)
{
	return currentLevelNum;
}

/** Used to set the default path type */
void CMapWidget::setDefaultPathTwoWay(bool set)
{ defaultTwoWayPath = set; }

/** Used to return the default path type */
bool CMapWidget::getDefaultPathTwoWay(void)
{ return defaultTwoWayPath; }

/** Get the totals for the map */
void CMapWidget::getTotals(int *lvl,int *room, int *path, int *text, int *zone)
{
	*lvl = 0;
	*room = 0;
	*path = 0;
	*text = 0;
	*zone = mapperData->zoneList.count();

	for (CMapZone *zone = mapperData->zoneList.first(); zone!=0; zone = mapperData->zoneList.next())
	{	
		*lvl+=zone->getLevels()->count();
		for (CMapLevelList *map = zone->getLevels()->first(); map!=0; map = zone->getLevels()->next())
		{
			for(CMapRoom *room1 = map->getRoomList()->first(); room1!=0; room1=map->getRoomList()->next())
			{
				*path += room1->pathList.count();
			}
			*text+=map->getTextList()->count();
			*room+=map->getRoomList()->count();
		}	
	}
}

/** Get the login room */
CMapRoom *CMapWidget::getLoginRoom(void)
{
	return loginRoom;
}

/** Draw the grid if it's visable */
void CMapWidget::drawGrid(QPainter* p)
{
	int x=0,y=0;
	signed int x1,x2,y1,y2;
	int maxx = width();
	int maxy = height();

	// Is the grid visable
	if (hasGrid)
	{
		p->setPen(mapperData->gridColour);
	 	// Draw the lines going across
		for (y=0;y<=maxy;y+=XGridSpace)
		{
			x1 = 5;
			y1 = y;
			x2 = maxx;
			y2 = y;

			p->drawLine(x1,y1,x2,y2);
		}

		// Draw the lines going down
		for (x=0;x<=maxx;x+=YGridSpace)
		{
			x1 = x;
			y1 = 5;
			x2 = x;
			y2 = maxy;

			p->drawLine(x1,y1,x2,y2);
		}
	}
}

/** Used to calulate and set the cords of a path */
void CMapWidget::setPathCords(CMapPath *path)
{	
	int x1,y1,x2,y2;

	int room_x = ((path->getSrcRoom()->getX() * ROOM_SIZE) - ROOM_SIZE) + HALF_ROOM_SIZE;
	int room_y = ((path->getSrcRoom()->getY() * ROOM_SIZE) - ROOM_SIZE) + HALF_ROOM_SIZE;

	// Get the start of the path
	directionToCord(path->getSrcDir(),HALF_ROOM_SIZE,&x1,&y1);

	x1+=room_x;
	y1+=room_y;

	// Get the end of the path
	directionToCord(path->getDestDir(),HALF_ROOM_SIZE,&x2,&y2);

	x2 += ((path->getDestRoom()->getX() * ROOM_SIZE) - ROOM_SIZE)  + HALF_ROOM_SIZE;
	y2 += ((path->getDestRoom()->getY() * ROOM_SIZE) - ROOM_SIZE)  + HALF_ROOM_SIZE;

	path->setCords(x1,y1,x2,y2);		
}


/** draw the map widget */
void CMapWidget::paintEvent(QPaintEvent *pe)
{
	// this is the area we actually have to draw
	QRect drawArea(mapView->contentsX(), mapView->contentsY(),
     	           mapView->viewport()->width(), mapView->viewport()->height());

	bNeedsCreate = true; // FIXXME (2000-12-19, kevin): try to find a better way
	if (bNeedsCreate)
	{	
	  // delete the buffer only when we need one with a different size
		if (buffer && (buffer->size() != drawArea.size()))
		{
			delete buffer;
			buffer = NULL;
		}		
	
		if (!buffer)
		{
		  buffer = new QPixmap(drawArea.size());
		}
		
		QPainter p;		
		QPainter p2;
	
		p.begin(buffer);
	
		// Check that there is a map in memory
		if (currentLevelList->count() == 0 )
			return;

		if (bOverview)
		{
			if (!overviewBuffer)
			{
  			overviewBuffer = new QPixmap(180,100);
			}
			p2.begin(overviewBuffer);
		}
			
		// Clear the map
		p.fillRect(0,0,width(),height(),mapperData->backgroundColour);
	
		if (bOverview)
		{
		  p2.fillRect(0,0, 180,100, mapperData->backgroundColour);
		  p2.setWindow(rect());
		  p2.setViewport(overviewBuffer->rect());
		  p2.setPen(black);
		  p2.drawRect(rect());          // border of the map
		  p2.drawRect(drawArea);        // where we are on the map
		}  	
		
		p.translate(-drawArea.x(), -drawArea.y()); // trabslate the cordinates origin
		
		// Draw the grid
		drawGrid(&p);

		for ( CMapElement *element=elementListLower.first(); element != 0; element=elementListLower.next() )
			if (element->getElementType()==PATH)
				((CMapPath *)element)->setDone(false);
				
		for ( CMapElement *element=elementList.first(); element != 0; element=elementList.next() )
			if (element->getElementType()==PATH)
				((CMapPath *)element)->setDone(false);

		for ( CMapElement *element=elementListUpper.first(); element != 0; element=elementListUpper.next() )
			if (element->getElementType()==PATH)
				((CMapPath *)element)->setDone(false);
								
		
		// Paint the map elements of the lower map
		if (bViewLowerMap)
			for ( CMapElement *element=elementListLower.first(); element != 0; element=elementListLower.next() )
			{	
				element->lowerPaint(&p,getCurrentZone());	
    		
    		if (bOverview) // draw the overview if neccessary
		      element->lowerPaint(&p2,getCurrentZone());
				
	}
	
	// Paint the map elements of the current map
	for ( CMapElement *element=elementList.first(); element != 0; element=elementList.next() )
	{
		element->paint(&p,getCurrentZone());	
		
    	if (bOverview) // draw the overview if neccessary
	    	element->paint(&p2,getCurrentZone());
	}	
	
	// Paint the map elements of the higher map
	if (bViewHigherMap)
		for ( CMapElement *element=elementListUpper.first(); element != 0; element=elementListUpper.next() )
		{
			element->higherPaint(&p,getCurrentZone());				
			if (bOverview) // draw the overview if neccessary
	       		element->higherPaint(&p2,getCurrentZone());
		}

    	if (bOverview)
			p2.end();

		p.end();

		bNeedsCreate = false;
	}

	if (bOverview && width()>180 && height()>100)
	{
		int view_x,view_y;
		if (mapView->viewport()->width()>mapView->contentsWidth())
				view_x =mapView->contentsWidth();
		else
			view_x =mapView->viewport()->width();

		if (mapView->viewport()->height()>mapView->contentsHeight())
			view_y =mapView->contentsHeight();
		else					
			view_y =mapView->viewport()->height();		
			
		bitBlt(buffer,view_x-180, view_y-100,overviewBuffer);
		
	}

	bitBlt(this, drawArea.x(), drawArea.y(), buffer);
}

void CMapWidget::createZoomedPixmap(QPixmap *newPixmap, QPixmap pm,int w,int h)
{
	
	QImage img = pm.convertToImage();
	
	newPixmap->convertFromImage(img.smoothScale(w,h),QPixmap::Auto);		
}

/** Turns the grid on or off */
void CMapWidget::setGridVisable(bool visable)
{
	hasGrid = visable;

	redraw();
}

/** Returns the visable state of the grid */
bool CMapWidget::GridVisable(void)
{ return hasGrid; }

void CMapWidget::setElementsColour(QColor loPath,QColor defPath,QColor hiPath,
                                   QColor loRoom,QColor defRoom,QColor hiRoom,
                                   QColor login,QColor sel,QColor defText,QColor specialCol,
                                   QColor edit,QColor current )
{

	mapperData->lowerRoomColour = loRoom;
	mapperData->defaultRoomColour = defRoom;
	mapperData->higherRoomColour = hiRoom;
        	
 	mapperData->lowerPathColour = loPath;
	mapperData->defaultPathColour = defPath;
	mapperData->higherPathColour = hiPath;
	
	mapperData->defaultTextColour = defText;
	mapperData->selectedColour = sel;
	mapperData->loginColour = login;
	mapperData->specialColour = specialCol;
	mapperData->editColour = edit;
	mapperData->currentColour = current;

	for (CMapZone *zone = mapperData->zoneList.first() ; zone != 0 ; zone = mapperData->zoneList.next() )
		for (CMapLevelList *map=zone->getLevels()->first(); map !=0;map = zone->getLevels()->next())
		{
			for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
			{
				room->setLowerColour(loRoom);
				room->setDefaultColour(defRoom);
				room->setHigherColour(hiRoom);
				room->setSelectColour(sel);
				room->setLoginColour(login);
				room->setSpecialColour(specialCol);
				room->setCurrentColour(current);

				for (CMapPath *path=room->pathList.first();path!=0; path=room->pathList.next())
				{
					path->setLowerColour(loPath);
					path->setDefaultColour(defPath);
					path->setHigherColour(hiPath);
					path->setSelectColour(sel);
					path->setEditColour(edit);
				}
			}
			
			for (CMapText *text=map->getTextList()->first();text !=0 ; text = map->getTextList()->next())
			{
				text->setSelectColour(sel);
			}
		}
		

}

CMapRoom *CMapWidget::getCurrentRoom(void)
{	
	return currentRoom;
}

/** This method is used to delete a path on the map */
void CMapWidget::deletePath(CMapPath *path)
{
	CMapRoom *srcRoom = path->getSrcRoom();
	CMapRoom *destRoom = path->getDestRoom();
	CMapPath *opsitePath = path->getOpsitePath();

	if (opsitePath)
	{
		if (opsitePath->getSrcRoom()->getLevel()< currentLevelNum)
			elementListLower.remove(opsitePath);

		if (opsitePath->getSrcRoom()->getLevel()> currentLevelNum)
			elementListUpper.remove(opsitePath);

		if (opsitePath->getSrcRoom()->getLevel() == currentLevelNum)
			elementList.remove(opsitePath);

		opsitePath->getDestRoom()->connectingPaths.remove(opsitePath);
		opsitePath->getSrcRoom()->pathList.setAutoDelete(true);
		opsitePath->getSrcRoom()->pathList.remove(opsitePath);
	}

	if (srcRoom->getLevel()< currentLevelNum)
		elementListLower.remove(path);

	if (srcRoom->getLevel()> currentLevelNum)
		elementListUpper.remove(path);

	if (srcRoom->getLevel() == currentLevelNum)
		elementList.remove(path);

	destRoom->connectingPaths.remove(path);
	srcRoom->pathList.setAutoDelete(true);
	srcRoom->pathList.remove(path);
}

CMapPath *CMapWidget::makePathTwoWay(CMapPath *path)
{
	CMapPath *newPath = (CMapPath *)path->copy();
	newPath->setSrcRoom(path->getDestRoom());
	newPath->setDestRoom(path->getSrcRoom());
	newPath->setSrcDir(path->getDestDir());
	newPath->setDestDir(path->getSrcDir());
	newPath->setOpsitePath(path);
		
	setPathCords(newPath);
	
	path->getDestRoom()->addPath(newPath);
	path->getSrcRoom()->connectingPaths.append(newPath);
	path->setOpsitePath(newPath);

	elementList.append(newPath);
	
	return newPath;
}

void CMapWidget::makePathOneWay(CMapPath *path)
{
	for (CMapPath *path2 = path->getDestRoom()->pathList.first();path2!=0;path2=path->getDestRoom()->pathList.next())
	{			
		if (path2->getDestRoom()==path->getSrcRoom())
		{			
			deletePath(path2);
			path->setOpsitePath(NULL);
			break;
		}
	}			
}

void CMapWidget::deleteText(CMapText *text,int level,CMapZone *fromZone)
{
	if (text->getLevel()< currentLevelNum)
		elementListLower.remove(text);

	if (text->getLevel()> currentLevelNum)
		elementListUpper.remove(text);

	if (text->getLevel() == currentLevelNum)
		elementList.remove(text);

	fromZone->getLevels()->at(level)->getTextList()->setAutoDelete(true);
	fromZone->getLevels()->at(level)->getTextList()->remove(text);

}

void CMapWidget::redraw(void)
{
	bNeedsCreate = true;
	repaint(false);
}

/** This checks the given size and resizes if needed */		
void CMapWidget::checkSize(int x,int y)
{
	int view_x,view_y;
	int newx,newy;

	if (x > xMax) xMax = x;
	if (y > yMax) yMax = y;

	view_x =mapView->viewport()->width() + mapView->verticalScrollBar()->width();	
	view_y =mapView->viewport()->height() + mapView->horizontalScrollBar()->height();

	if (xMax > view_x)
	{
		newx = xMax;
	}
	else
	{
		newx = view_x;
	}

	if (yMax > view_y)
	{
		newy = yMax;
	}
	else
	{
		newy = view_y;
	}

	if (newy != height() || newx !=width())
	{
		bNeedsCreate = true;
		resize(newx + (ROOM_SIZE * 3),newy + (ROOM_SIZE * 3));
	}
}

CMapText *CMapWidget::createText(QString str,int x,int y,QFont font,QColor col,int level,CMapZone *zone,bool doSizeCheck)
{
	CMapText *text = new CMapText(str,x,y,font,currentZone);

	text->setLevel(level);
	text->setSelected(false);
	text->setColour(col);

	text->setSelectColour(mapperData->selectedColour);
	text->setZone(zone);
	zone->getLevels()->at(level)->getTextList()->append(text);

	if (zone==currentZone) elementList.append(text);

	if (doSizeCheck) checkSize(text->getHiX(),text->getHiY());

	return text;
}

/** This method is used to delete a room on the map */
void CMapWidget::deleteRoom(CMapRoom *room,int level,CMapZone *fromZone)
{
	bool currentThisLevel = (currentRoom->getLevel() == currentMap->first()->getLevel() );
	
	// Find new start room
	if (room->getCurrentRoom())
	{
	  currentRoom = currentMap->first();
	
	  // Is there only the current room we wish to delete on this level
	  while(currentRoom != NULL && currentRoom == room)
	  {
	    currentRoom= currentMap->next();
	  }
	
	  if (currentRoom != NULL)
	  {
	    currentRoom->setCurrentRoom(true);
	  }
		else
		{
			// Yes so look for a new room to make the current room
			// First check for rooms on above levels
			if (level+1 < (int)(fromZone->getLevels()->count()))
			{
				for (CMapLevelList *map = fromZone->getLevels()->at(level+1); map!=0 ; map = fromZone->getLevels()->next())
				{				
					if (map->getRoomList()->first() !=NULL)
					{
						currentRoom = map->getRoomList()->first();
						map->getRoomList()->first()->setCurrentRoom(true);
						break;
					}
				}
				
			}
			
			// If none were found look form them on the lower levels
			if ((level-1) >= 0 && !currentRoom)
			{
				for (CMapLevelList *map = fromZone->getLevels()->at(level-1); map!=0 ; map = fromZone->getLevels()->prev())
				{
					if (map->getRoomList()->first() !=NULL)
					{
						currentRoom = map->getRoomList()->first();
						map->getRoomList()->first()->setCurrentRoom(true);
						break;
					}
				}
			}
		}
	}

	// Delete the paths for the room		
	for (CMapPath *path=room->pathList.last(); path!=0; path=room->pathList.last())
		deletePath(path);
	
	// Delete any paths connecting with this room		
	for (CMapPath *path=room->connectingPaths.last(); path!=0; path = room->connectingPaths.last())
		deletePath(path);			

	// Update the speedwalk list if nesscary
	if (mapperData->delSpeedwalkRoom((CMapRoom *)selected.element))
		emit updateSpeedwalkList(NULL);	
	
	// delete the room
	elementList.remove(room);
	currentMap->setAutoDelete(true);
	currentMap->remove(room);
		
	// if the map is now empty create a new room and make the current	
	if (currentThisLevel)
	{
		if (currentMap->count()==0 && (currentRoom==NULL))
		{
			currentRoom = createRoom(3,3,currentLevelNum,currentZone);
			currentMap->first()->setCurrentRoom(true);
		}
	}
}

/** This method is used to delete a zone on the map */
void CMapWidget::deleteZone(CMapZone *zone,int level,CMapZone *fromZone)
{		
	// Delete the paths for the room
	while (zone->getLevels()->count()!=0)
	{
		deleteLevel(zone->getLevels()->count()-1,zone,false);
	}

	// delete the zone
	elementList.remove(zone);

	fromZone->getLevels()->at(level)->getZoneList()->remove(zone);

	mapperData->zoneList.setAutoDelete(true);

	mapperData->zoneList.remove(zone);	
}

/** Used to create a new level of the map */
void CMapWidget::createLevel(direction dir,CMapZone *intoZone)
{
	if (dir==UP)
		intoZone->getLevels()->append( new CMapLevelList() );	
	else
	{
		for (CMapLevelList *map=intoZone->getLevels()->first(); map !=0;map = currentLevelList->next())
		{
			for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
			{
				room->setLevel(room->getLevel()+1);
			}

			for (CMapText *text=map->getTextList()->first();text!=0 ; text = map->getTextList()->next())
			{
				text->setLevel(text->getLevel()+1);
			}
			
			for (CMapZone *zone=map->getZoneList()->first();zone !=0 ; zone = map->getZoneList()->next())
			{
				zone->setLevel(zone->getLevel()+1);
			}
		}
		intoZone->getLevels()->insert( 0, new CMapLevelList() );
	}
}

/** This method is used to create a new zone on the map */
CMapZone *CMapWidget::createZone(int x,int y,signed int level,CMapZone *intoZone,bool doExsitCheck)
{
	CMapZone *zone = NULL;	
	
	if (doExsitCheck)
	{
		if (findElementAt(x,y,level,intoZone))
		{
			return NULL;
		}
	}

	// Create the new room
	zone = new CMapZone("Unamed Zone",x,y,level,intoZone);
	zone->setLowerColour(mapperData->lowerRoomColour);
	zone->setHigherColour(mapperData->higherRoomColour);	
	zone->setSelectColour(mapperData->selectedColour);	

	// Add the zone to the list
	intoZone->getLevels()->at(level)->getZoneList()->append( zone );

	if (currentZone == intoZone)
	{
		if (currentLevelNum == level)
			elementList.append((CMapElement *)zone);

		checkSize(zone->getHiX(),zone->getHiY());	
	}


	mapperData->zoneList.append(zone);

	zone->getLevels()->insert( 0, new CMapLevelList() );	

	return zone;
}

/** This method is used to create a new room on the map */
CMapRoom *CMapWidget::createRoom(int x,int y,signed int level,CMapZone *zone)
{
	if (findElementAt(x,y,level,zone))
		return NULL;	
	// Create the new room
	CMapRoom *room = new CMapRoom(zone);

	room->setX(x);
	room->setY(y);
	room->setLevel(level);
	room->setLowerColour(mapperData->lowerRoomColour);
	room->setDefaultColour(mapperData->defaultRoomColour);
	room->setHigherColour(mapperData->higherRoomColour);	
	room->setSelectColour(mapperData->selectedColour);
	room->setCurrentColour(mapperData->currentColour);
	room->setSpecialColour(mapperData->specialColour);
	room->setLoginColour(mapperData->loginColour);

	// Add the room to the list
	if (level>=(int)(zone->getLevels()->count()) || level<0 )
	{
 		if (level < 0)
		{				
		
			createLevel(DOWN,zone);
			zone->getLevels()->at(level+1)->getRoomList()->append( room );
			room->setLevel(level+1);
		}
		else
		{
			createLevel(UP,zone);
			zone->getLevels()->at(level)->getRoomList()->append( room );
		}
	}
	else
	{
		zone->getLevels()->at(level)->getRoomList()->append( room );
	}		

	if (currentZone == zone )
	{
		if ( level == currentLevelNum)
			elementList.append((CMapElement *)room);

        checkSize(room->getHiX(),room->getHiY());	
	}

	return room;
}

/** This method is used to create a path between to rooms */
CMapPath *CMapWidget::createPath(int srcX,int srcY,signed int srcLevel,CMapZone *srcZone,direction srcDir,int destX,int destY,signed int destLevel,CMapZone *destZone,direction destDir)
{	
	CMapRoom *room=0;
	CMapRoom *srcRoom=0;
	CMapRoom *destRoom=0;
	CMapLevelList *levelElement;

	// Setup and find the src room for the new path	
	levelElement = srcZone->getLevels()->at(srcLevel);

	for ( room=levelElement->getRoomList()->first(); room != 0; room=levelElement->getRoomList()->next() )
	{
		if (room->getX() == srcX && room->getY() == srcY)
		{
			srcRoom = room;
			break;
		}
	}

	// Setup and find the dest room for the path	

	levelElement = destZone->getLevels()->at(destLevel);

	for ( room=levelElement->getRoomList()->first(); room != 0; room=levelElement->getRoomList()->next() )
	{
		if (room->getX() == destX && room->getY() == destY)
		{
			destRoom = room;
			break;
		}
	}
	

	if (destRoom ==NULL || srcRoom == NULL)
	{
		return NULL;
	}

	
	return createPath(srcRoom,destRoom,srcDir,destDir);
}

CMapPath *CMapWidget::createPath (CMapRoom *srcRoom,CMapRoom *destRoom,direction srcDir,direction destDir)
{
	CMapPath *newPath = new CMapPath (srcDir,destDir,destRoom,srcRoom);

	newPath->setLowerColour(mapperData->lowerPathColour);
	newPath->setDefaultColour(mapperData->defaultPathColour);
	newPath->setHigherColour(mapperData->higherPathColour);
	newPath->setSelectColour(mapperData->selectedColour);
	newPath->setEditColour(mapperData->editColour);

	setPathCords(newPath);

	srcRoom->addPath( newPath );
	destRoom->connectingPaths.append(newPath); 	
	elementList.append(newPath);

	for (CMapPath *path =newPath->getDestRoom()->pathList.first(); path!=0; path=newPath->getDestRoom()->pathList.next())
	{
		if (path->getDestRoom()==srcRoom && path->getSrcDir() == destDir && path->getDestDir() == srcDir && path->getSpecialCmd() == newPath->getSpecialCmd())
		{
			newPath->setOpsitePath(path);
			path->setOpsitePath(newPath);
		}
	}
	
	return newPath;
}

/** This method is used to move the map by the given vector */
void CMapWidget::moveMap(signed int x,signed int y)
{	
	// Move the rooms
	for (CMapLevelList *map=currentLevelList->first(); map !=0;map = currentLevelList->next())
	{
		for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
		{
			room->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));					

			for (CMapPath *path=room->pathList.first();path!=0; path=room->pathList.next())
			{
				path->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));			
				path->setLowHiX(path->getLowX()+(x * ROOM_SIZE),path->getHiX()+(x * ROOM_SIZE));
				path->setLowHiY(path->getLowY()+(y * ROOM_SIZE),path->getHiY()+(y * ROOM_SIZE));
			}
		}

		for (CMapZone *zone=map->getZoneList()->first();zone !=0 ; zone = map->getZoneList()->next())
		{
			zone->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));					
		}
		
		for (CMapText *text=map->getTextList()->first();text!=0; text = map->getTextList()->next())
		{
			text->moveBy((x * ROOM_SIZE) ,(y * ROOM_SIZE));						
		}
	}
	
	checkSize(xMax + (x * ROOM_SIZE),yMax + (y * ROOM_SIZE));
}

/** This method is used to save a map */
void CMapWidget::saveMap(void)
{
	QDir localKmudDir;

	// if it's not there... create it
 	localKmudDir.setCurrent(QDir::homeDirPath());
 	localKmudDir.mkdir(".kde",false);
 	localKmudDir.mkdir(".kde/share",false);
 	localKmudDir.mkdir(".kde/share/apps",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud/maps",false);

	// Setup maps path
 	localKmudDir.setPath(QDir::homeDirPath()+"/.kde/share/apps/kmud/maps/");

	// Check to see if we can write to the map direcrtory
 	if (localKmudDir.exists() && localKmudDir.isReadable())
		exportMap(localKmudDir.path() + QString::QString("/") + mapName);
		
}

void CMapWidget::exportMap(QString filename)
{
	
	struct speedwalkRoomTyp out_speedwalk_room;
	struct zoneTyp          out_zone;
	struct roomHeaderTyp    out_room_header;
	struct roomTyp          out_room;
	struct textHeaderTyp    out_text_header;
	struct textTyp          out_text;	
	struct pathHeaderTyp    out_path_header;
	struct pathTyp          out_path;
	struct mainHeaderTyp    out_main_header;

	int path_total,text_total,room_total;

	// Create and open for file to save the map to
	QFile f(filename);
	f.open(IO_WriteOnly);

	// Write out file format version information
	f.writeBlock((char*)&mapFileVerMinor,sizeof(char));
	f.writeBlock((char*)&mapFileVerMajor,sizeof(char));
	
	// Write out the main header for the map	
	path_total = 0;
	text_total = 0;
	room_total = 0;

	// Output zone header
	out_main_header.numZones = mapperData->zoneList.count();
	f.writeBlock((char *)&out_main_header,sizeof(out_main_header));
	
	// Write out the zones at levels
	for ( CMapZone *zone=mapperData->zoneList.first(); zone !=0 ; zone=mapperData->zoneList.next())
	{		
		// Output zone details
		out_zone.x = zone->getX();
		out_zone.y = zone->getY();
		out_zone.numLevels = zone->getLevels()->count();
		out_zone.level = zone->getLevel();

		if (zone->getZone())
		{		
			CMapZone *current = mapperData->zoneList.current();
			out_zone.zone = mapperData->zoneList.findRef(zone->getZone());
			mapperData->zoneList.findRef(current);
		}
		else
		{
			out_zone.zone = -1;
		}
		
		f.writeBlock((char *)&out_zone,sizeof(out_zone));		
		writeStr(&f,zone->getName());
		// Count map elements
		
		for (CMapLevelList *level=zone->getLevels()->first(); level!=0; level=zone->getLevels()->next())
		{
			for ( CMapRoom *room=level->getRoomList()->first(); room != 0; room=level->getRoomList()->next() )
			{
				room_total++;
				path_total +=room->pathList.count();
			}

			text_total += level->getTextList()->count();
		}
		
	}
	
	// Write out the room header
	out_room_header.numRooms = room_total;
	f.writeBlock((char *)&out_room_header,sizeof(out_room_header));
	int zoneCount = 0;		
	for ( CMapZone *zone=mapperData->zoneList.first(); zone !=0 ; zone=mapperData->zoneList.next())		
	{	
		int levelCount = 0;		
		for ( CMapLevelList *map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			
			for ( CMapRoom *room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
			{
				// Write out room
				out_room.x = room->getX();
				out_room.y = room->getY();
				out_room.useDefaultCol = room->getUseDefaultCol();
				out_room.red = room->getColour().red();
				out_room.green = room->getColour().green();
				out_room.blue = room->getColour().blue();
				out_room.zone = zoneCount;
				out_room.level = levelCount;
									
				f.writeBlock((char *)&out_room,sizeof(out_room));
			
				writeStr(&f,room->getLabel());
				writeStr(&f,room->getDescription());
				writeStr(&f,room->getBeforeEnterCommand());			
				writeStr(&f,room->getAfterEnterCommand());
				writeStr(&f,room->getBeforeExitCommand());			
				writeStr(&f,room->getAfterExitCommand());				
			}
			levelCount++;

		}
		zoneCount ++;
	}
	// Write out the path header

	out_path_header.numPaths = path_total;
	f.writeBlock((char *)&out_path_header,sizeof(out_path_header));

	// Loop to add the paths to the bottom of the file for each room
	for ( CMapZone *zone = mapperData->zoneList.first(); zone != 0; zone= mapperData->zoneList.next())
	{	
		for ( CMapLevelList *map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			for ( CMapRoom *room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
			{
				for (CMapPath *path=room->pathList.first(); path != 0; path=room->pathList.next())
				{
					// Write out the detils for each path in the room
					out_path.srcX = room->getX();
					out_path.srcY = room->getY();
					out_path.srcLevel = room->getLevel();
					CMapZone *current = mapperData->zoneList.current();
					out_path.srcZone = mapperData->zoneList.findRef(room->getZone());
					out_path.destX = path->getDestRoom()->getX();
					out_path.destY = path->getDestRoom()->getY();
					out_path.destLevel = path->getDestRoom()->getLevel();					
					out_path.destZone = mapperData->zoneList.findRef(path->getDestRoom()->getZone());
					mapperData->zoneList.findRef(current);					
					out_path.srcDir = path->getSrcDir();
					out_path.destDir = path->getDestDir();
					out_path.special = path->getSpecialExit();				
					out_path.bendsCount =path->getBends()->count();
					
					f.writeBlock((char *)&out_path,sizeof(out_path));
					writeStr(&f,path->getSpecialCmd());
				
					for (QPoint *point = path->getBends()->first();point !=0; point = path->getBends()->next())
					{
						writeInt(&f,point->x());
						writeInt(&f,point->y());					
					}
				}				
			}
			
		}		
        }
	// Write out the text header
	out_text_header.numText = text_total;
	f.writeBlock((char *)&out_text_header,sizeof(out_text_header));

	// Save all the text elements
	for ( CMapZone *zone = mapperData->zoneList.first(); zone != 0; zone= mapperData->zoneList.next())
	{
		for ( CMapLevelList *map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			for ( CMapText *text=map->getTextList()->first(); text != 0; text=map->getTextList()->next() )
			{
				out_text.level = text->getLevel();
				out_text.x = text->getLowX();
				out_text.y = text->getLowY();
				out_text.size = text->getFont().pointSize();
				out_text.weight = text->getFont().weight();
				out_text.italic = text->getFont().italic();
				out_text.red = text->getColour().red();
				out_text.green = text->getColour().green();
				out_text.blue = text->getColour().blue();
				CMapZone *current = mapperData->zoneList.current();				
				out_text.zone = mapperData->zoneList.findRef(text->getZone());

				mapperData->zoneList.findRef(current);

								
				f.writeBlock((char *)&out_text,sizeof(out_text));
				writeStr(&f,text->getText());
				writeStr(&f,text->getFont().family());
			}
		}
	}

	// Save the speed walk list
	out_room_header.numRooms = mapperData->speedwalkList.count();

	f.writeBlock((char *)&out_room_header,sizeof(out_room_header));		

    for (CMapRoom *room = mapperData->speedwalkList.first(); room!=0; room=mapperData->speedwalkList.next())
	{
		// Write out room
		out_speedwalk_room.x = room->getX();
		out_speedwalk_room.y = room->getY();
		out_speedwalk_room.zone = mapperData->zoneList.find(room->getZone());
		out_speedwalk_room.level = room->getLevel();
									
		f.writeBlock((char *)&out_speedwalk_room,sizeof(out_speedwalk_room));
	}

	// Close the map, it has now been saved
	f.close();
}

/** export map to a KCMapFile */
void CMapWidget::exportKCMap(QString filename)
{
	CMapRoom *room;
	CMapLevelList *map;
	CMapPath *path;
	CMapText *text;
	CMapZone *zone;
  int count;
  int level_count;
  int zone_count;
  int speedwalk_count;
  	
	if (QFile::exists(filename))
	{
	  QFile::remove(filename);
	}
	
	QFile file(filename);
	
  KCMapFile kcmfile;
  kcmfile.writeVersion(0, 3);
	
	count = 0;
	level_count = 0;
	zone_count = 0;
	for (zone=mapperData->zoneList.first(); zone != 0; zone=mapperData->zoneList.next())
	{
	  zone_count++;
	  kcmfile.setZoneGroup(zone_count);
	  zone->write(&kcmfile);
	  if (zone->getZone())
	  {
			CMapZone *current = mapperData->zoneList.current();
			kcmfile.writeEntry("parent zone", mapperData->zoneList.findRef(zone->getZone()));
			mapperData->zoneList.findRef(current);
	  }
	  else
	  {
			kcmfile.writeEntry("parent zone", -1);
	  }
	
	  for ( map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
	  {
      level_count++;
		  for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
		  {
		    count++;
		    kcmfile.setRoomGroup(count);
		    room->write(&kcmfile);
		    kcmfile.writeEntry("zone", (zone_count-1));
		  }
	  }
	}
	
	kcmfile.writeCount("zones", zone_count);
	kcmfile.writeCount("levels", level_count);
  kcmfile.writeCount("rooms", count);

	count = 0;
	for (zone=mapperData->zoneList.first(); zone != 0; zone=mapperData->zoneList.next())
	{
	  for ( map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
  	{
	  	for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
		  {
			  for (path=room->pathList.first(); path != 0; path=room->pathList.next())
			  {
			    count++;
	  	    kcmfile.setPathGroup(count);
		      path->write(&kcmfile);
		
					CMapZone *current = mapperData->zoneList.current();
					int srcZone = mapperData->zoneList.findRef(room->getZone());
					int destZone = mapperData->zoneList.findRef(path->getDestRoom()->getZone());
					mapperData->zoneList.findRef(current);					
		      kcmfile.writeEntry("srcZone", srcZone);
		      kcmfile.writeEntry("destZone", destZone);
		    }
		  }
	  }
	}
	kcmfile.writeCount("paths", count);

	// Save all the text elements
  count = 0;
  zone_count = 0;
	for (zone=mapperData->zoneList.first(); zone != 0; zone=mapperData->zoneList.next())
	{
	  zone_count++;
	  for ( map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
	  {
		  for ( text=map->getTextList()->first(); text != 0; text=map->getTextList()->next() )
		  {
			  count++;
	  	  kcmfile.setTextGroup(count);
		    text->write(&kcmfile);
		    kcmfile.writeEntry("zone", (zone_count-1));
		  }
    }
  }
  kcmfile.writeCount("texts", count);

	// Save the speed walk list
  speedwalk_count = 0;
  for (CMapRoom *room = mapperData->speedwalkList.first(); room!=0; room=mapperData->speedwalkList.next())
	{
	  kcmfile.setSpeedwalkGroup(speedwalk_count+1);
		kcmfile.writeEntry("X", room->getX());
		kcmfile.writeEntry("Y",room->getY());
		kcmfile.writeEntry("zone", mapperData->zoneList.find(room->getZone()));
		kcmfile.writeEntry("level", room->getLevel());
		
		speedwalk_count++;
	}

  kcmfile.writeCount("speedwalks", speedwalk_count);
	

  // finished all elements, write to disk now
	kcmfile.writeFile(file);
}


int CMapWidget::importMap(QString filename)
{
	CMapLevelList *levelElement;
	struct speedwalkRoomTyp in_speedwalk_room;
	struct zoneTyp       	in_zone;
	struct roomHeaderTyp 	in_room_header;
	struct roomTyp       	in_room;	
	struct pathHeaderTyp 	in_path_header;
	struct pathTyp       	in_path;	
	struct textHeaderTyp 	in_text_header;
	struct textTyp       	in_text;		
	struct mainHeaderTyp 	in_main_header;	
	CMapRoom *room;
	QString str;				
	QString family;
	char minor,major;
	int startx=-2;
	int starty=-2;
	int startlvl=-2;
	int startzone=0;
	CMapRoom *firstRoom = NULL;
		
	// Create and open for file to save the map to
	QFile f(filename);

	if (!f.open(IO_ReadOnly))
		return -1;

	f.readBlock((char *)&minor,sizeof(char));
	f.readBlock((char *)&major,sizeof(char));		
		
	if (minor==mapFileVerMinor && major==mapFileVerMajor)
	{
	}
	else if(major==1 && minor==3)
	{
	  return importMap_1_3(&f);
	}
	else
	{
		QMessageBox::information (this,i18n("Kmud Mapper"),
                                     i18n("The file is the incorect version.\n"
                                     "or not a valid map file."));

		return -2;
	}

	eraseMap();
	f.readBlock((char *)&in_main_header, sizeof(in_main_header));

	for ( signed int zoneCount=0 ; zoneCount<in_main_header.numZones; zoneCount++)
	{	
		CMapZone *zone;
		f.readBlock((char *)&in_zone,sizeof(in_zone));
		QString zoneName = readStr(&f);

		if (in_zone.zone == -1)
			zone = new CMapZone(zoneName,in_zone.x,in_zone.y,in_zone.level,NULL);		
		else
		{
			zone = new CMapZone(zoneName,in_zone.x,in_zone.y,in_zone.level,mapperData->zoneList.at(in_zone.zone));
			mapperData->zoneList.at(in_zone.zone)->getLevels()->at(in_zone.level)->getZoneList()->append(zone);
		}
			
		zone->setLowerColour(mapperData->lowerRoomColour);
		zone->setHigherColour(mapperData->higherRoomColour);	
		zone->setSelectColour(mapperData->selectedColour);	

		mapperData->zoneList.append(zone);
		
		for ( int lvlCount=0 ; lvlCount<in_zone.numLevels; lvlCount++)
		{
			// Add new level
			levelElement = new CMapLevelList();
			zone->getLevels()->append( levelElement );
		}
	}

	// Read in the room header		
	f.readBlock((char *)&in_room_header,sizeof(in_room_header));
		
	for ( int roomCount=0; roomCount < in_room_header.numRooms; roomCount++)
	{
		// Read the room			
		f.readBlock((char *)&in_room,sizeof(in_room));
			
		CMapZone *zone = mapperData->zoneList.at(in_room.zone);
		levelElement = zone->getLevels()->at(in_room.level);		
			
		// Create the new room
		room = new CMapRoom(zone);
		room->setX(in_room.x);
		room->setY(in_room.y);				
		room->setLevel(in_room.level);
		room->setLowerColour(mapperData->lowerRoomColour);
		room->setDefaultColour(mapperData->defaultRoomColour);
		room->setHigherColour(mapperData->higherRoomColour);			
		room->setUseDefaultCol(in_room.useDefaultCol);
		room->setLoginColour(mapperData->loginColour);
		room->setSelectColour(mapperData->selectedColour);
		room->setCurrentColour(mapperData->currentColour);
		room->setSpecialColour(mapperData->specialColour);

		room->setColour(QColor::QColor(in_room.red,in_room.green,in_room.blue));
			
		room->setLabel(readStr(&f));
		room->setDescription(readStr(&f));
		room->setBeforeEnterCommand(readStr(&f));						
		room->setAfterEnterCommand(readStr(&f));
		room->setBeforeExitCommand(readStr(&f));
		room->setAfterExitCommand(readStr(&f));		
		room->setZone(zone);						

		// Add the room to the list
		levelElement->getRoomList()->append( room );
		elementList.append(room);
			
		if (!firstRoom) firstRoom = room;
	}

	f.readBlock((char *)&in_path_header,sizeof(in_path_header));
	for (int pathCount=0; pathCount<in_path_header.numPaths ; pathCount++)
	{
		f.readBlock((char *)&in_path,sizeof(in_path));
		CMapPath *path =createPath(in_path.srcX,in_path.srcY,in_path.srcLevel,mapperData->zoneList.at(in_path.srcZone),in_path.srcDir,in_path.destX,in_path.destY,in_path.destLevel,mapperData->zoneList.at(in_path.destZone),in_path.destDir);
		path->setSpecialExit(in_path.special);
		path->setSpecialCmd(readStr(&f));		
		
		for (int i = 0;i<in_path.bendsCount; i++)
		{
			int x = readInt(&f);
			int y = readInt(&f);					
			path->getBends()->append(new QPoint(x,y));
		}

	}

	// Write out the text header
	f.readBlock((char *)&in_text_header,sizeof(in_text_header));

	// Save all the text elements
	for ( int textCount = 0; textCount < in_text_header.numText; textCount++)
	{
		f.readBlock((char *)&in_text,sizeof(in_text));

 		str = readStr(&f);

 		QString family = readStr(&f);

		QFont font(family,in_text.size,in_text.weight,in_text.italic);							

		QColor color(in_text.red,in_text.green,in_text.blue);
		if (str)
		{
			createText(str,in_text.x,in_text.y,font,color,in_text.level,mapperData->zoneList.at(in_text.zone),false);
		}
	}

	// Read in the room header		
	f.readBlock((char *)&in_room_header,sizeof(in_room_header));
	for ( int roomCount=0; roomCount < in_room_header.numRooms; roomCount++)
	{
		// Read the room			
		f.readBlock((char *)&in_speedwalk_room,sizeof(in_speedwalk_room));

		CMapRoom *room = findRoomAt(in_speedwalk_room.x,in_speedwalk_room.y,in_speedwalk_room.level,mapperData->zoneList.at(in_speedwalk_room.zone));			
		if (room)
		{
			mapperData->speedwalkList.append(room);
		}
	}

	emit updateSpeedwalkList(NULL);

	// Close the map, it has now been saved
	f.close();

	// Setup the characters starting positions
	CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
	if (prof)
	{
	 	startx = prof->getMapStartX();
	 	starty =  prof->getMapStartY();
	 	startlvl = prof->getMapStartLevel();
	 	startzone = prof->getMapStartZone();
		
		if (startzone+1>(int)(mapperData->zoneList.count()))
		{
			startx = -2;
	 		starty = -2;
	 		startlvl = firstRoom->getLevel();;
	 		startzone = 0;
		}
		else if (startlvl+1>(int)(mapperData->zoneList.at(startzone)->getLevels()->count()))
		{
			startx = firstRoom->getX();
	 		starty = firstRoom->getY();
	 		startlvl = firstRoom->getLevel();
	 		startzone = mapperData->zoneList.findRef(firstRoom->getZone());
		}
		else if (!findRoomAt(startx,starty,startlvl,mapperData->zoneList.at(startzone)))
	 	{
	 		startx = firstRoom->getX();
	 		starty = firstRoom->getY();
	 		startlvl = firstRoom->getLevel();;
	 		startzone = firstRoom->getLevel();

	 	}
	}
	else
	{
		startx = -2;
		starty = -2;
		startlvl = firstRoom->getLevel();
	}
	loginRoom = NULL;

	showLevel(startx,starty,startlvl,mapperData->zoneList.at(startzone));

	return 0;
}

int CMapWidget::importMap_1_3(QFile *file)
{
	CMapLevelList *levelElement;
	struct roomHeaderTyp_1_3 in_room_header;
	struct roomTyp_1_3       in_room;
	struct pathHeaderTyp_1_3 in_path_header;
	struct pathTyp_1_3       in_path;
	struct mainHeaderTyp_1_3 in_main_header;
	struct textHeaderTyp_1_3 in_text_header;
	CMapRoom *room;
	int roomCount,pathCount;
	signed lvlCount;
	QString str;				
	QString family;
	int weight;
	bool italic;
	int level;
	int size;
	int red,green,blue;
	int x,y;
	int startx=-2;
	int starty=-2;
	int startlvl=-2;
	CMapRoom *firstRoom = NULL;
	
	eraseMap();

	// Create a zone because the old map format does not support it
	CMapZone *zone = new CMapZone("Unamed Zone",-1,-1,-1,NULL);
	mapperData->zoneList.append(zone);
	currentZone = zone;
	currentLevelList = zone->getLevels();
	
	file->readBlock((char *)&in_main_header, sizeof(in_main_header));
	for ( lvlCount=0 ; lvlCount<in_main_header.numLevels; lvlCount++)
	{
		// Add new level
		levelElement = new CMapLevelList();
		currentLevelList->append( levelElement );

		// Read in the room header
		file->readBlock((char *)&in_room_header,sizeof(in_room_header));
		for ( roomCount=0; roomCount < in_room_header.numRooms; roomCount++)
		{
			// Read the room
			file->readBlock((char *)&in_room,sizeof(in_room));

			// Create the new room
			//room = new CMapRoom(in_room.type);
			room = new CMapRoom(zone);
			room->setX(in_room.x);
			room->setY(in_room.y);				
			room->setLevel(lvlCount);
			room->setLowerColour(mapperData->lowerRoomColour);
			room->setDefaultColour(mapperData->defaultRoomColour);
			room->setHigherColour(mapperData->higherRoomColour);			
			room->setUseDefaultCol(in_room.useDefaultCol);
			room->setLoginColour(mapperData->loginColour);
			room->setSelectColour(mapperData->selectedColour);
			room->setCurrentColour(mapperData->currentColour);

			room->setColour(QColor::QColor(in_room.red,in_room.green,in_room.blue));

			// Add the room to the list
			levelElement->getRoomList()->append( room );
			elementList.append(room);
			
			room->setLabel(readStr(file));
			room->setDescription(readStr(file));
			
			if (!firstRoom) firstRoom = room;
		}
				
	}
	
	file->readBlock((char *)&in_path_header,sizeof(in_path_header));
	for (pathCount=0; pathCount<in_path_header.numPaths ; pathCount++)
	{
		file->readBlock((char *)&in_path,sizeof(in_path));
		createPath(in_path.srcX,in_path.srcY,in_path.srcLevel,zone,in_path.srcDir,in_path.destX,in_path.destY,in_path.destLevel,zone,in_path.destDir);
	}

	// Write out the text header
	file->readBlock((char *)&in_text_header,sizeof(in_text_header));

	// Save all the text elements
	for ( int textCount = 0; textCount < in_text_header.numText; textCount++)
	{
 		str = readStr(file);

		level=readInt(file);
		x=readInt(file);
		y=readInt(file);
				
		// read font details
		family = readStr(file);
		size=readInt(file);
		weight =readInt(file);
		italic =readInt(file);
		QFont font(family,size,weight,italic);				
		
		// Read the dummy colour value (yet to be implemented)
		red = readInt(file);
		green = readInt(file);
		blue = readInt(file);
		
		QColor color(red,green,blue);

		if (str)
			createText(str,x,y,font,color,level,zone,false);
	}

	// Close the map, it has now been saved
	file->close();

	// Setup the characters starting positions

	CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
	if (prof)
	{
	 	startx = prof->getMapStartX();
	 	starty =  prof->getMapStartY();
	 	startlvl = prof->getMapStartLevel();
	 	
	 	if (!findRoomAt(startx,starty,startlvl,zone))
	 	{
	 		startx = -2;
	 		starty = -2;
	 		startlvl = firstRoom->getLevel();;
	 	}
	}
	else
	{
		startx = -2;
		starty = -2;
		startlvl = firstRoom->getLevel();
	}

	loginRoom = NULL;

	mapperData->speedwalkList.clear();
	emit updateSpeedwalkList(NULL);

	showLevel(startx,starty,startlvl,zone);
	
	return 0;
}

/** import a map from a KCMapFile */
int CMapWidget::importKCMap(QString filename)
{
	CMapLevelList *levelElement;
	CMapRoom *room;
	CMapRoom *firstRoom = NULL;
	CMapZone *zone;
	int major, minor;
	int zone_count;
	int num_zones;
	int level_count;
	int num_levels;
	int room_count;
	int num_rooms;
	int path_count;
	int num_paths;
	int num_text;
	int num_speedwalks;
	int x, y, level, zoneNum;
	int startx, starty, startlvl, startzone;

	int count;

  if (!QFile::exists(filename)) return -1;

  eraseMap();

  KCMapFile kcmfile(filename);
  kcmfile.readVersion(major, minor);
  if (major==0 && minor==1)
  {
	  // Create a zone because the old map format does not support it
	  zone = new CMapZone("Unamed Zone",-1,-1,-1,NULL);
	  mapperData->zoneList.append(zone);
	  currentZone = zone;
	  currentLevelList = zone->getLevels();

    num_levels = kcmfile.readCount("levels");
    fprintf(stderr, "importKCMap: loading 0.1 file with %d levels\n", num_levels);

    count = 0;
	  for (level_count=0 ; level_count<num_levels; level_count++)
	  {
		  // Add new level
		  levelElement = new CMapLevelList();
		  currentLevelList->append( levelElement );
    }
  }
  else
  {
    // we know about zones
    num_zones = kcmfile.readCount("zones");
    fprintf(stderr, "importKCMap: loading %d.%d file with %d zones\n", major, minor, num_zones);
    for (zone_count=0; zone_count < num_zones; zone_count++)
    {
      kcmfile.setZoneGroup(zone_count+1);
      int parent = kcmfile.readNumEntry("parent zone", -1);
      if (parent==-1)
      {
        zone = new CMapZone("",0,0,0, NULL);
      }
      else
      {
        int lev = kcmfile.readNumEntry("level", 0);
        zone = new CMapZone("",0,0,0, mapperData->zoneList.at(parent));
        mapperData->zoneList.at(parent)->getLevels()->at(lev)->getZoneList()->append(zone);
      }
      zone->read(&kcmfile);
      mapperData->zoneList.append(zone);

      num_levels = kcmfile.readNumEntry("levels");
      fprintf(stderr, "importKCMap: zone %d has %d levels\n", (zone_count+1), num_levels);

      for (level_count=0; level_count<num_levels; level_count++)
      {
  			// Add new level
	  		levelElement = new CMapLevelList();
		  	zone->getLevels()->append( levelElement );
      }
    }
  }


  // read all rooms
	num_rooms = kcmfile.readCount("rooms");
	fprintf(stderr, "importKCMap: loading %d rooms\n", num_rooms);
	for ( room_count=0; room_count < num_rooms; room_count++)
	{
		// Read the room
    kcmfile.setRoomGroup(room_count+1);
    zone = mapperData->zoneList.at(kcmfile.readNumEntry("zone", 0));
		levelElement = zone->getLevels()->at(kcmfile.readNumEntry("level", 0));		

		room = new CMapRoom(zone);
		room->setCurrentColour(mapperData->currentColour);
		room->setLowerColour(mapperData->lowerRoomColour);
		room->setDefaultColour(mapperData->defaultRoomColour);
		room->setHigherColour(mapperData->higherRoomColour);			
		room->setLoginColour(mapperData->loginColour);
		room->setSelectColour(mapperData->selectedColour);
		room->setSpecialColour(mapperData->specialColour);
		room->read(&kcmfile);

		// Add the room to the list
		levelElement->getRoomList()->append( room );
		elementList.append(room);
			
		if (!firstRoom) firstRoom = room;
	}				
	
	//read the paths
	num_paths = kcmfile.readCount("paths");
	fprintf(stderr, "importKCMap: loading %d paths\n", num_paths);
	for (path_count=0; path_count<num_paths ; path_count++)
	{
	  kcmfile.setPathGroup(path_count+1);
	  int destX, destY, destL, destZone;
	  direction srcDir, destDir;
	  x = kcmfile.readNumEntry("srcX", -1);
	  y = kcmfile.readNumEntry("srcY", -1);
	  level = kcmfile.readNumEntry("srcLevel", 0);
	  srcDir =(direction) kcmfile.readUnsignedNumEntry("srcDir", 0);
    zoneNum = kcmfile.readNumEntry("srcZone", 0);
	  	
	  destX = kcmfile.readNumEntry("destX", -1);
	  destY = kcmfile.readNumEntry("destY", -1);
	  destL = kcmfile.readNumEntry("destLevel", 0);
	  destDir = (direction) kcmfile.readUnsignedNumEntry("destDir", 0);
    destZone = kcmfile.readNumEntry("destZone", 0);
	
		CMapPath *path =createPath(x, y, level, mapperData->zoneList.at(zoneNum), srcDir,
		                           destX, destY, destL, mapperData->zoneList.at(destZone), destDir);
		if (path != NULL)
		{
      path->read(&kcmfile);
    }
		
	}
	
	// read the texts
	num_text = kcmfile.readCount("texts");
	fprintf(stderr, "importKCMap: loading %d texts\n", num_text);
	for ( int text_count = 0; text_count < num_text; text_count++)
	{
	  kcmfile.setTextGroup(text_count+1);
    x = kcmfile.readNumEntry("X", -1);
    y = kcmfile.readNumEntry("Y", -1);
    level = kcmfile.readNumEntry("level", 0);
    zoneNum = kcmfile.readNumEntry("zone", 0);
    QFont font = kcmfile.readFontEntry("font");
    QColor col = kcmfile.readColorEntry("color", &mapperData->defaultTextColour);
    QString text_str = kcmfile.readEntry("text");

		if (text_str)
			createText(text_str, x, y, font, col,level, mapperData->zoneList.at(zoneNum),false);
	}
	
	// read the speedwalk list
	num_speedwalks = kcmfile.readCount("speedwalks");
  fprintf(stderr, "importKCMap: loading %d speedwalks\n", num_speedwalks);

	for ( int room_count=0; room_count < num_speedwalks; room_count++)
	{
		// Read the speedwalk
		kcmfile.setSpeedwalkGroup(room_count+1);
		x = kcmfile.readNumEntry("X", -1);
		y = kcmfile.readNumEntry("Y", -1);
		zoneNum = kcmfile.readNumEntry("zone", 0);
		level = kcmfile.readNumEntry("level", 0);

		room = findRoomAt(x, y, level, mapperData->zoneList.at(zoneNum));			

		if (room)
		{
			mapperData->speedwalkList.append(room);
		}
	}

	emit updateSpeedwalkList(NULL);
	
	// Setup the characters starting positions
	CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
	if (prof)
	{
	 	startx = prof->getMapStartX();
	 	starty =  prof->getMapStartY();
	 	startlvl = prof->getMapStartLevel();
	 	startzone = prof->getMapStartZone();
		if (startzone+1>(int)(mapperData->zoneList.count()))
		{
			startx = -2;
	 		starty = -2;
	 		startlvl = firstRoom->getLevel();;
	 		startzone = 0;
		}
		else if (!findRoomAt(startx,starty,startlvl,mapperData->zoneList.at(startzone)))
	 	{
	 		startx = -2;
	 		starty = -2;
	 		startlvl = firstRoom->getLevel();;
	 		startzone = 0;

	 	}
	}
	else
	{
		startx = -2;
		starty = -2;
		startlvl = firstRoom->getLevel();
	}
	loginRoom = NULL;

	showLevel(startx,starty,startlvl,mapperData->zoneList.at(startzone));

	return 0;
}

QString CMapWidget::readStr(QFile *f)
{
	int len;
	QString str;
	char c;
	
	len = readInt(f);

	if (len == 0)
	{
		str = "";
		return str;
	}
		
	for (int i=0;i<len;i++)
	{
		f->readBlock((char *)&c,sizeof(char));

		str = str + c;
	}
	return str;
}

int CMapWidget::readInt(QFile *f)
{
	int i;
	
	f->readBlock((char *)&i,sizeof(int));
	
	return i;
}

void CMapWidget::writeStr(QFile *f,QString str)
{
	writeInt(f,str.length());
	f->writeBlock((const char*)str,strlen(str));
}

void CMapWidget::writeInt(QFile *f,int i)
{
	f->writeBlock((char *)&i,sizeof(int));
}

/** Turn on and off the display of the higher map */
void CMapWidget::setViewHigherMap(bool visiable)
{
	bViewHigherMap = visiable;

	redraw();
}

/** Turn on and off the display of the lower map */
void CMapWidget::setViewLowerMap(bool visiable)
{
	bViewLowerMap = visiable;

	redraw();
}


/** This method is used to load a map */
void CMapWidget::loadMap(void)
{
	QDir localKmudDir;

	// if it's not there... create it
 	localKmudDir.setCurrent(QDir::homeDirPath());
 	localKmudDir.mkdir(".kde",false);
 	localKmudDir.mkdir(".kde/share",false);
 	localKmudDir.mkdir(".kde/share/apps",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud",false);
 	localKmudDir.mkdir(".kde/share/apps/kmud/maps",false);

	// Setup maps path
 	localKmudDir.setPath(QDir::homeDirPath()+"/.kde/share/apps/kmud/maps/");

	// Check to see if we can write to the map direcrtory
 	if (localKmudDir.exists() && localKmudDir.isReadable())
 	{
		if (importMap(localKmudDir.path() + QString::QString("/") + mapName)!=0)
			createNewMap();
	}
}

QScrollView *CMapWidget::getMapView(void)
{
	return mapView;
}

bool CMapWidget::getViewOverview(void)
{
	return bOverview;
}

void CMapWidget::changeStatusBar(QString msg)
{
    emit statusMsg((const char *)msg);
}

QList<CMapElement> *CMapWidget::getElementList(void)
{
	return &elementList;
}

bool CMapWidget::getCtrlPressed(void)
{
	return bCtrlPressed;
}

void CMapWidget::createNewMap(void)
{
	
	elementList.clear();
	elementListUpper.clear();
	elementListLower.clear();

	// Create empty rooms and paths elememts
	CMapZone *zone = new CMapZone("Unamed Zone",-1,-1,-1,NULL);
	mapperData->zoneList.append(zone);
	currentZone = zone;
	currentLevelList = zone->getLevels();
	
	CMapLevelList *levelElement = new CMapLevelList();
	currentLevelList->append( levelElement );
	currentMap = levelElement->getRoomList();
	currentLevelNum = currentLevelList->count() - 1;

	resize(ROOM_SIZE*6,ROOM_SIZE*6);

	xMax = 0;
	yMax = 0;

	createRoom(3,3,currentLevelNum,currentZone);

	currentMap->first()->setCurrentRoom(true);
	currentRoom = currentMap->first();	
	currentRoom->setLoginRoom(true);
	loginRoom = currentRoom;

	checkSize(xMax,yMax);	
}

QList<CMapRoom> *CMapWidget::getCurrentMap(void)
{
	return currentMap;
}

/** This method is used to erase a map from memory */
void CMapWidget::eraseMap(void)
{
	CMapRoom *room;
	CMapLevelList *map;

	// Loop to add the paths to the bottom of the file for each room
	for ( CMapZone *zone = mapperData->zoneList.first(); zone != 0 ; zone = mapperData->zoneList.next())
	{
		for ( map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
			{
				room->pathList.setAutoDelete(true);
				room->pathList.clear();
				room->connectingPaths.clear();
			}
			map->getRoomList()->setAutoDelete(true);
			map->getRoomList()->clear();

			map->getTextList()->setAutoDelete(true);
			map->getTextList()->clear();
		}
		zone->getLevels()->setAutoDelete(true);		
		zone->getLevels()->clear();		
	}

	mapperData->zoneList.setAutoDelete(true);
	mapperData->zoneList.clear();

	mapperData->speedwalkList.clear();
	
	elementList.clear();
	elementListUpper.clear();
	elementListLower.clear();

	xMax = 0;
	yMax = 0;

	currentMap = NULL;
	currentLevelNum = 0;
	currentRoom = NULL;
	
	emit updateSpeedwalkList(NULL);
}

/** the showLevel method is used to display a level on the map */
void CMapWidget::showLevel(int startx,int starty,signed int level,CMapZone *zone)
{
	CMapRoom *room;
	bool bFound = false;
	CMapLevelList *map;
	
	currentLevelNum = level;

	map = zone->getLevels()->at(level);
	currentZone = zone;
	currentLevelList = zone->getLevels();
	currentMap = map->getRoomList();
	elementList.clear();
	
	emit zoneChanged(zone->getName());
	emit enableZoneUp(zone->getZone()!=NULL);
	emit enableLevelDown(level!=0);
	emit enableLevelUp((int)(currentLevelList->count())-1>level);

	xMax = 0;
	yMax = 0;
		
	room=map->getRoomList()->first();
	room=map->getRoomList()->next();

	// Add the rooms to the list of elements to to be displayed on the current level
	for ( room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
	{
		// Set the start room
		if (room->getX() == startx && room->getY() == starty)
		{
			currentRoom = room;
			currentRoom->setCurrentRoom(true);
			if (loginRoom == NULL)
			{
				loginRoom = currentRoom;
				currentRoom->setLoginRoom(true);			
			}
			bFound = true;			
		}

		elementList.append(room);

		// Add the paths to the list of elements to to be displayed on the current level
		for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
			elementList.append(path);
		
		if (room->getHiX()>xMax) xMax = room->getHiX();		
		if (room->getHiY()>yMax) yMax = room->getHiY();
		
	}

	if ((startx == -2 || starty == -2) && startx != -1 && starty != -1)
	{
		if (!bFound)
		{
			currentRoom = currentMap->first();
			currentRoom->setCurrentRoom(true);
		}
		loginRoom = currentRoom;
		currentRoom->setLoginRoom(true);
	}

	if (currentRoom == NULL)
	{
		currentRoom = currentMap->first();
		currentRoom->setCurrentRoom(true);		
	}

	if (loginRoom == NULL)
	{
		loginRoom = currentRoom;
		currentRoom->setLoginRoom(true);
	}	

	// Add the zones to the list of elements to to be displayed on the current level
	for ( CMapZone *zone2 = map->getZoneList()->first(); zone2 !=0;zone2 = map->getZoneList()->next())
	{		
		elementList.append(zone2);
		
		if (zone2->getHiX()>xMax)
		{
			xMax = zone2->getHiX();		
		}
		if (zone2->getHiY()>yMax)
		{
			yMax = zone2->getHiY();
		}
	}

	// Add the text to the list of elements to to be displayed on the current level
	for ( CMapText *text=map->getTextList()->first(); text!=0; text=map->getTextList()->next())
	{
		elementList.append(text);

		if (text->getHiX()>xMax) xMax = text->getHiX();		
		if (text->getHiY()>yMax) yMax = text->getHiY();
	}

	// Get the elements for the upper level
	elementListUpper.clear();
	if ((int)(currentLevelList->count())-1 >= level+1)
	{
		for ( room=currentLevelList->at(level+1)->getRoomList()->first(); room != 0; room=currentLevelList->at(level+1)->getRoomList()->next() )
		{
	
			elementListUpper.append(room);
	
			for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
				elementListUpper.append(path);
	
			if (room->getHiX()>xMax) xMax = room->getHiX();
			if (room->getHiY()>yMax) yMax = room->getHiY();
		}
		
		for ( CMapZone *zone2=currentLevelList->at(level+1)->getZoneList()->first(); zone2 != 0;
					zone2=currentLevelList->at(level+1)->getZoneList()->next() )
		{
			elementListUpper.append(zone2);
			
			if (zone2->getHiX()>xMax) xMax = zone2->getHiX();
			if (zone2->getHiY()>yMax) yMax = zone2->getHiY();
		}
	}

	// Get the elements for the lower level
	elementListLower.clear();
	if (level-1>=0)
	{
		for ( room=currentLevelList->at(level-1)->getRoomList()->first(); room != 0;
					room=currentLevelList->at(level-1)->getRoomList()->next() )
		{
			elementListLower.append(room);
	
			for (CMapPath *path=room->pathList.first(); path!=0;path = room->pathList.next())
				elementListLower.append(path);
	
			if (room->getHiX()>xMax) xMax = room->getHiX();
			if (room->getHiY()>yMax) yMax = room->getHiY();
		}
		
		for (CMapZone *zone2=currentLevelList->at(level-1)->getZoneList()->first(); zone2 !=0;
		               zone2=currentLevelList->at(level-1)->getZoneList()->next())
		{
			elementListLower.append(zone2);

			if (zone2->getHiX()>xMax) xMax = zone2->getHiX();
			if (zone2->getHiY()>yMax) yMax = zone2->getHiY();		
		}
	}

	emit levelChange(level);

	// Resize the contents of the scroll view

	checkSize(xMax,yMax);	

	redraw();
}

/** Used to display and set the room properties for a element. Element should be a room */
void CMapWidget::roomProperties(CMapRoom *room)
{
	DlgMapRoomProperties d(this,"RoomProperties");

	for (CMapPath *path = room->pathList.first(); path !=0 ; path = room->pathList.next())
	{
		switch (path->getSrcDir())
  		{
			case NORTH     : d.addExit(mudProfile->getDirections()->north); break;
			case NORTHEAST : d.addExit(mudProfile->getDirections()->northeast); break;
			case EAST      : d.addExit(mudProfile->getDirections()->east); break;        		
			case SOUTHEAST : d.addExit(mudProfile->getDirections()->southeast); break;
			case SOUTH     : d.addExit(mudProfile->getDirections()->south); break;
			case SOUTHWEST : d.addExit(mudProfile->getDirections()->southwest); break;
			case WEST      : d.addExit(mudProfile->getDirections()->west); break;        		
			case NORTHWEST : d.addExit(mudProfile->getDirections()->northwest); break;        		
			case UP        : d.addExit(mudProfile->getDirections()->up); break;        		
			case DOWN      : d.addExit(mudProfile->getDirections()->down); break;        		
			case SPECIAL   : d.addExit(path->getSpecialCmd()); break;
		}
	}

	d.setDescription(room->getDescription());
	d.setLabel(room->getLabel());

	d.setUseDefaultCol(room->getUseDefaultCol());
	if (room->getUseDefaultCol())
		d.setColour(room->getDefaultColour());
	else
		d.setColour(room->getColour());
		
	d.setBeforeEnterCommand(room->getBeforeEnterCommand());
	d.setAfterEnterCommand(room->getAfterEnterCommand());
	d.setBeforeExitCommand(room->getBeforeExitCommand());			
	d.setAfterExitCommand(room->getAfterExitCommand());

	if (d.exec())
	{
		QString name;
    	
		if (d.getLabel()!=room->getLabel())
		{
			room->setLabel(d.getLabel());
			emit updateSpeedwalkList(room);
		}
		room->setDescription(d.getDescription());
		room->setColour(d.getColour());
		room->setUseDefaultCol(d.getUseDefaultCol());
		room->setBeforeEnterCommand(d.getBeforeEnterCommand());
		room->setAfterEnterCommand(d.getAfterEnterCommand());
		room->setBeforeExitCommand(d.getBeforeExitCommand());				
		room->setAfterExitCommand(d.getAfterExitCommand());						
	       	
		for (CMapPath *path = room->pathList.first(); path !=0 ; path = room->pathList.next())
		{
			switch (path->getSrcDir())
			{
				case NORTH     : name = mudProfile->getDirections()->north; break;
				case NORTHEAST : name = mudProfile->getDirections()->northeast; break;
				case EAST      : name = mudProfile->getDirections()->east; break;        		
				case SOUTHEAST : name = mudProfile->getDirections()->southeast; break;
				case SOUTH     : name = mudProfile->getDirections()->south; break;
				case SOUTHWEST : name = mudProfile->getDirections()->southwest; break;
				case WEST      : name = mudProfile->getDirections()->west; break;        		
				case NORTHWEST : name = mudProfile->getDirections()->northwest; break;        		
				case UP        : name = mudProfile->getDirections()->up; break;        		
				case DOWN      : name = mudProfile->getDirections()->down; break;        		
				case SPECIAL   : name = path->getSpecialCmd(); break;
			}
	        	
	        	
			bool FoundIt = false;	        	
			for (int i = d.getNumExits(); i>0;i--)
			{	       	
				if ((d.getExit(i-1))==name)
				{
					FoundIt = true;
					break;
				}
	       		
	       	}
       	
	       	if (!FoundIt)
	       		deletePath(path);
        }	       		       			       	
	}

	redraw();
}


/** This method is used to convert a direction into a offset */
void CMapWidget::directionToCord(direction dir, int distance,signed int *x,signed int *y)
{
	switch (dir)
	{
		case NORTH     : *x = 0;
	                         *y = -distance;
				 break;
		case EAST      : *x = distance;
        	                 *y = 0;
				 break;
		case SOUTH     : *x = 0;
				 *y = distance;
				 break;
		case WEST      : *x = -distance;
				 *y = 0;
				 break;
		case NORTHEAST : *x = distance;
				 *y = -distance;
				 break;
		case NORTHWEST : *x = -distance;
				 *y = -distance;
				 break;
		case SOUTHEAST : *x = distance;
				 *y = distance;
				 break;
		case SOUTHWEST : *x = -distance;
				 *y = distance;
				 break;
		case UP: break;
		case DOWN: break;
		case SPECIAL: break;
	}
}

/** Used to delete an element */
void CMapWidget::delElement(CMapElement *element)
{
	selected.element = element;
	slotDelElement();	
}


/** move the player relative to the current position of the player
  * This command has support for special paths, but can only move
  * anlong exsiting special paths. In other words, it is unable to
  * create special paths.
  */
void CMapWidget::movePlayerBy(direction dir,bool create,QString specialCmd)
{
	signed int x,y,incx,incy;
	CMapRoom *srcRoom;
	direction destDir;
	int movex,movey;
	signed int newLevel;
	CMapPath *opsitePath = NULL;
	
	CMapRoom *tmpRoom = currentRoom;
	
	CMapUndoElement *undoElem = new CMapUndoElement(currentRoom, dir, 0);

	if (currentLevelNum!=currentRoom->getLevel() || currentZone!=currentRoom->getZone())
	{
		showLevel(-1,-1,currentRoom->getLevel(),currentRoom->getZone());
	}

	currentRoom = tmpRoom;
	
	// Make the old room as not the current room
	currentRoom->setCurrentRoom(false);

	if (dir != UP && dir != DOWN)
	{
		// Find the destination of the path that was traveled and if no path
		// is exsits for the given direction create the room and path if necsarry
		CMapPath *path = currentRoom->getPathDirection(dir,specialCmd);

		if (path)
		{
			int oldLvl = currentRoom->getLevel();
			CMapZone *oldZone = currentRoom->getZone();
			currentRoom=path->getDestRoom();
			if (oldLvl!=currentRoom->getLevel() || oldZone!=currentRoom->getZone() ) showLevel(currentRoom->getX(),currentRoom->getY(),currentRoom->getLevel(),currentRoom->getZone());
		}
		else
		{		

			bool bFound = false;
			srcRoom = currentRoom;
			
			// Check to see if there is a path in the opsite direction that we should
			// be using				
			for (CMapPath *path2=srcRoom->connectingPaths.first();path2!=0;path2=srcRoom->connectingPaths.next())
			{

				if (path2->getDestDir() == dir)
				{
					bFound = true;
					x = path2->getSrcRoom()->getX();
					y = path2->getSrcRoom()->getY();
					opsitePath = path2;
					break;				
				}
			}

			if (!bFound)
			{
				directionToCord(dir,2,&incx,&incy);
				x = currentRoom->getX()+incx;
				y = currentRoom->getY()+incy;

				// Check to see if the map needs to be moved
				// and calulate the offset to move if it needs moving.
				if (x<3 || y<3)
				{
					if (x<3)
					{

						movex = 3 - x;

						x+=movex;
					}
					else
						movex = 0;
					
					if (y<3)
					{
						movey = 3- y;
						y+=movey;
					}
					else
						movey = 0;
		
					moveMap (movex,movey);
				}
			}

			// Check to see if the room already exsits
			CMapElement *elm = findElementAt(x,y,currentLevelNum,currentZone);
					
			if (!elm)
			{
				currentRoom = NULL;
			}
			else if (elm->getElementType()==ZONE)
			{
				currentRoom = srcRoom;
				return;
			}
			else
			{
				currentRoom = (CMapRoom *)elm;
			}


			// Create the room if it does not exsit
			if (!currentRoom)
			{
				if (create)
				{
					currentRoom=createRoom(x,y,currentLevelNum,currentZone);
					undoElem->setFlag(undoElem->ROOM);
				}
				else
				{
					currentRoom=srcRoom;
					emit elementsAutoCreated(undoElem);
					return;
				}				
			}
			
			switch (dir)
			{
				case NORTH : destDir = SOUTH; break;
				case SOUTH : destDir = NORTH; break;
				case EAST : destDir = WEST; break;
				case WEST : destDir = EAST; break;
				case NORTHWEST : destDir = SOUTHEAST; break;
				case NORTHEAST : destDir = SOUTHWEST; break;
				case SOUTHWEST : destDir = NORTHEAST; break;
				case SOUTHEAST : destDir = NORTHWEST; break;
				case UP: break;
				case DOWN: break;
				case SPECIAL: break;
			}
		
			// Create the new path to the room
			CMapPath *newPath = new CMapPath(dir,destDir,currentRoom,srcRoom);
			newPath->setLowerColour(mapperData->lowerPathColour);
			newPath->setDefaultColour(mapperData->defaultPathColour);
			newPath->setHigherColour(mapperData->higherPathColour);
			newPath->setSelectColour(mapperData->selectedColour);
			newPath->setEditColour(mapperData->editColour);
			
			undoElem->setFlag(undoElem->PATH);

			setPathCords(newPath);
			srcRoom->addPath (newPath );
			currentRoom->connectingPaths.append(newPath);
			elementList.append(newPath);
			
			if (opsitePath)
			{
			  newPath->setOpsitePath(opsitePath);
			  opsitePath->setOpsitePath(newPath);
			}
			
			// Make the path two way if the default path type is two way
			if (defaultTwoWayPath && bFound == false)
			{
				makePathTwoWay(newPath);
				undoElem->setFlag(undoElem->BIDIR);
			}
		}
		
		// Make the new room the current room
		currentRoom->setCurrentRoom(true);

		redraw();

	}
	else
	{
		// Find the destination of the path that was traveled and if no path
		// is exsits for the given direction create the room and path if necsarry
		CMapPath *path = currentRoom->getPathDirection(dir,specialCmd);

		if (path)
		{
			currentRoom=path->getDestRoom();
			showLevel(currentRoom->getX(),currentRoom->getY(),currentRoom->getLevel(),currentRoom->getZone());
		}
		else
		{		
			srcRoom = currentRoom;
			x = currentRoom->getX();
			y = currentRoom->getY();

			if (dir == UP)
			{
				destDir = DOWN;
				newLevel = currentLevelNum + 1;
			}
			else
			{
				destDir = UP;
				newLevel = currentLevelNum - 1;
			}

			for (CMapPath *path2=srcRoom->connectingPaths.first();path2!=0;path2=srcRoom->connectingPaths.next())
			{

				if (path2->getDestDir() == dir)
				{
					opsitePath = path2;
					break;				
				}
			}
	
			CMapElement *elm = findElementAt(x,y,newLevel,currentZone);
					
			if (!elm)
			{
				currentRoom = NULL;
			}
			else if (elm->getElementType()==ZONE)
			{
				currentRoom = srcRoom;
				return;
			}
			else
			{
				currentRoom = (CMapRoom *)elm;
			}

			// Create the room if it does not exsit
			if (!currentRoom)
			{
				if (create)
				{
					currentRoom=createRoom(x,y,newLevel,currentZone);

					if (newLevel < 0) newLevel = 0;					

          			undoElem->setFlag(undoElem->ROOM);

					if (currentLevelList->at(newLevel)->getRoomList()->count() == 1)
					{
		            	// only the new room exists so far
			            undoElem->setFlag(undoElem->LEVEL);
          			}
				}
				else
				{
					currentRoom=srcRoom;
					emit elementsAutoCreated(undoElem);
					return;
				}				
			}

			// Create the new path to the room
			CMapPath *newPath = new CMapPath(dir,destDir,currentRoom,srcRoom);
			newPath->setLowerColour(mapperData->lowerPathColour);
			newPath->setDefaultColour(mapperData->defaultPathColour);
			newPath->setHigherColour(mapperData->higherPathColour);
			newPath->setSelectColour(mapperData->selectedColour);
			newPath->setEditColour(mapperData->editColour);

			undoElem->setFlag(undoElem->PATH);
			
			newPath->setCords(0,0,0,0);
			srcRoom->addPath (newPath);

			currentRoom->connectingPaths.append(newPath);
			
			if (opsitePath)
			{
			  newPath->setOpsitePath(opsitePath);
			  opsitePath->setOpsitePath(newPath);
			}
			
			// Make the path two way if the default path type is two way
			if (defaultTwoWayPath)
			{
				
				CMapPath *newPath1 = makePathTwoWay(newPath);
				newPath1->setCords(0,0,0,0);
				undoElem->setFlag(undoElem->BIDIR);
			}
			
			showLevel(currentRoom->getX(),currentRoom->getY(),newLevel,currentRoom->getZone());
		}
	}
	
	emit elementsAutoCreated(undoElem);
}

/** move the player relative to the current position of the player
  * This command has support for special paths, but can only move
  * anlong exsiting special paths. In other words, it is unable to
  * create special paths.
  *
  * This is simlar to the above method but uses differe params.
  */
void CMapWidget::movePlayerBy(QString dirCmd,bool create)
{
	QString specialCmd = "";
	direction dir;
	
	dir = cmdToDir(dirCmd);
	if (dir == SPECIAL)
		specialCmd = dirCmd;

	movePlayerBy(dir,create,specialCmd);	
}

direction CMapWidget::cmdToDir(QString dirCmd)
{
	direction dir;
	directions *dirs= mudProfile->getDirections();	
	
	if (dirCmd == dirs->north)     { dir = NORTH; } else
	if (dirCmd == dirs->northeast) { dir = NORTHEAST; } else
	if (dirCmd == dirs->east)      { dir = EAST; } else
	if (dirCmd == dirs->southeast) { dir = SOUTHEAST; } else
	if (dirCmd == dirs->south)     { dir = SOUTH; } else
	if (dirCmd == dirs->southwest) { dir = SOUTHWEST; } else
	if (dirCmd == dirs->west)      { dir = WEST; } else
	if (dirCmd == dirs->northwest) { dir = NORTHWEST; } else
	if (dirCmd == dirs->up)        { dir = UP; } else
	if (dirCmd == dirs->down)      { dir = DOWN; } else		
		dir = SPECIAL;
		
	return dir;
}

/** Check to see if a string is a valid move command */
bool CMapWidget::validMoveCmd(QString dirCmd)
{
	bool result = false;	
	directions *dirs= mudProfile->getDirections();

	if (dirCmd.isEmpty()) return false;
	
	if (dirCmd == dirs->north)     { result = true; } else
	if (dirCmd == dirs->northeast) { result = true; } else
	if (dirCmd == dirs->east)      { result = true; } else
	if (dirCmd == dirs->southeast) { result = true; } else
	if (dirCmd == dirs->south)     { result = true; } else
	if (dirCmd == dirs->southwest) { result = true; } else
	if (dirCmd == dirs->west)      { result = true; } else
	if (dirCmd == dirs->northwest) { result = true; } else
	if (dirCmd == dirs->up)        { result = true; } else
	if (dirCmd == dirs->down)      { result = true; } else		
	{
		for (CMapPath *path=currentRoom->pathList.first(); path!=0; path=currentRoom->pathList.next())
		{
			if (path->getSpecialExit())
				if (path->getSpecialCmd()==dirCmd)
				{
					result = true;
					break;
				}
		}
	}
	
	return result;
}

/** This method is used to find a room at a given location in the current level of the map.
    Null is retured if the room can't be found otherwise a pointer is returned to the room. */
CMapRoom *CMapWidget::findRoomAt(int x, int y,signed int level,CMapZone *zone)
{
	CMapRoom *room;

	if (level > (int)(zone->getLevels()->count())-1) return NULL;

	QList<CMapRoom> *map = zone->getLevels()->at(level)->getRoomList();

	for ( room=map->first(); room != 0; room=map->next() )
	{
		if (x==room->getX() && y==room->getY())
		{			
			return room;
		}
	}

	return NULL;
}

CMapElement *CMapWidget::findElementAt(int x,int y,signed int level,CMapZone *zone)
{
	if (level > (int)(zone->getLevels()->count())-1)
		return NULL;

	CMapLevelList *lvl = zone->getLevels()->at(level);

	for ( CMapRoom *room=lvl->getRoomList()->first(); room != 0; room=lvl->getRoomList()->next() )
	{
		if (x==room->getX() && y==room->getY())
		{			
			return (CMapElement *) room;
		}
	}

	for ( CMapZone *zone2=lvl->getZoneList()->first(); zone2 != 0; zone2=lvl->getZoneList()->next() )
	{
		if (x==zone2->getX() && y==zone2->getY())
		{			
			return (CMapElement *)zone2;
		}
	}

	return NULL;	
}

/** Used to unslected all the elements on the map */
bool CMapWidget::unselectAll(void)
{
	bool result = false;
		
	for ( CMapZone *zone = mapperData->zoneList.first(); zone != 0; zone= mapperData->zoneList.next())
	{
		for ( CMapLevelList *map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			for ( CMapRoom *room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
			{
				for (CMapPath *path=room->pathList.first(); path != 0; path=room->pathList.next())
				{
					if (path->getSelected() || path->getEditMode())
					{
						result = true;
	        			path->setSelected(false);	
				        path->setEditMode(false);
					}
				}
				if (room->getSelected() || room->getEditMode())
				{
					result = true;
        			room->setSelected(false);	
			        room->setEditMode(false);
				}

			}

			for ( CMapText *text=map->getTextList()->first(); text!=0; text=map->getTextList()->next())
			{
				if (text->getSelected() || text->getEditMode())
				{
					result = true;
        			text->setSelected(false);	
			        text->setEditMode(false);
				}
			}
		}

		if (zone->getSelected() || zone	->getEditMode())
		{
			result = true;
   			zone->setSelected(false);	
	        zone->setEditMode(false);
		}
	}
	
	return result;
}

/** Set the current tool */
void CMapWidget::setTool(CMapToolBase *tool)
{	
	if (currentTool)
		currentTool->toolUnselected();

	currentTool = tool;
	currentTool->toolSelected();
}

/** Get the current zone */
CMapZone *CMapWidget::getCurrentZone(void)
{ return currentZone; }

/** Used to set the zone properties */
void CMapWidget::zoneProperties(CMapZone *zone)
{
	DlgMapZoneProperties d(this,"Properties");
		
	d.setName(zone->getName());
	
	if (d.exec())
	{
		if (d.getName()!=zone->getName())
		{
			zone->setName(d.getName());	
			emit updateSpeedwalkList(NULL);
		}
	}	
}

void CMapWidget::deleteLevel(int lvlNum,CMapZone *zone,bool recreate)
{
	QList<CMapLevelList> *levels = zone->getLevels();
	CMapLevelList *map = levels->at(lvlNum);

	for (CMapRoom *room =map->getRoomList()->last(); room !=0; room=map->getRoomList()->prev())
		deleteRoom(room,lvlNum,zone);

	for (CMapText *text = map->getTextList()->last(); text !=0; text=map->getTextList()->prev())
		deleteText(text,lvlNum,zone);

	for (CMapZone *zone2 = map->getZoneList()->last(); zone2 !=0; zone2=map->getZoneList()->prev())
	{

		deleteZone(zone2,lvlNum,zone);
	}

	levels->setAutoDelete(true);
	levels->remove(map);

	if (((int)(levels->count())-1 >= lvlNum) && ((int)(levels->count()) != 0) )
	{
		for ( CMapLevelList *map2=levels->at(lvlNum); map2 != 0; map2=levels->next())
		{
			for (CMapRoom *room=map2->getRoomList()->first(); room != 0; room=map2->getRoomList()->next() )
				room->setLevel(room->getLevel()-1);
		 	
		 	for (CMapText *text = map2->getTextList()->first(); text !=0; text=map2->getTextList()->next())
		 		text->setLevel(text->getLevel()-1);
		 		
		 	for (CMapZone *zone2 = map2->getZoneList()->first(); zone2 !=0; zone2=map2->getZoneList()->next())
		 		zone2->setLevel(zone2->getLevel()-1);
		 		
		}
	}

	if (levels->count()==0 && recreate)
	{
		createNewMap();
	}
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
void CMapWidget::mouseMoveEvent(QMouseEvent *e)
{
  if (bMouseDrag)
  {
    int dx, dy;
    dx = e->globalX() - nMouseDragPosX;
    dy = e->globalY() - nMouseDragPosY;
    nMouseDragPosX += dx;
    nMouseDragPosY += dy;
    mapView->scrollBy(dx*3, dy*3);
  }
  else
  {
  	currentTool->mouseMoveEvent(e);	
  }
}

void CMapWidget::mousePressEvent(QMouseEvent *e)
{
  emit mousePressed();
  	
	if (bOverview && width()>180 && height()>100)
	{
		int view_x,view_y;
		if (mapView->viewport()->width()>mapView->contentsWidth())
			view_x =mapView->contentsWidth();
		else
			view_x =mapView->viewport()->width();

		if (mapView->viewport()->height()>mapView->contentsHeight())
			view_y =mapView->contentsHeight();
		else
			view_y =mapView->viewport()->height();

		if (e->x()>=mapView->contentsX()+view_x-180 &&
		    e->x()<=mapView->contentsX()+view_x &&
		    e->y()>=mapView->contentsY()+view_y-100 &&
		    e->y()<=mapView->contentsY()+view_y)
		{
			
			int factor_x = width() / 180;
			int factor_y = height() / 100;

			mapView->center((e->x() -(mapView->contentsX()+view_x-180))*factor_x,(e->y()-(mapView->contentsY()+view_y-100))* factor_y);
			return;			
		}
	}

	if (e->button() ==RightButton)
	{
		slotContexMenu(e);
		return;
	}

	if (e->button() == MidButton)
	{
	  bMouseDrag = true;
	  nMouseDragPosX = e->globalX();
	  nMouseDragPosY = e->globalY();
	  QCursor* oldCursor = new QCursor(cursor());
	  setCursor(*mouseDragCursor);
	  delete mouseDragCursor;
	  mouseDragCursor = oldCursor;
	}
		
	currentTool->mousePressEvent(e);
}

void CMapWidget::slotContexMenu(QMouseEvent *e)
{
	bool found = false;
	
	for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
	{
		if (element->mouseInElement(e->x(),e->y(),getCurrentZone()))
		{
			if (element->getElementType()==ROOM)
			{					
				room_menu->setItemEnabled(ID_MAPPER_ROOM_ADD_SPEED,mapperData->speedwalkList.findRef((CMapRoom *)element)==-1);
				room_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));
			}

			if (element->getElementType()==ZONE)
			{					
				zone_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));
			}
								
			if (element->getElementType()==PATH)
			{

				CMapPath *path = (CMapPath *)element;
				bool twoWay = path->getOpsitePath();
				path_menu->setItemChecked(ID_MAPPER_PATH_TWOWAY,twoWay);
				path_menu->setItemChecked(ID_MAPPER_PATH_ONEWAY,!twoWay);				
				path_menu->setItemEnabled(ID_MAPPER_PATH_DELBEND,(path->mouseInPathSeg(e->x(),e->y(),getCurrentZone())!=-1));
				path_menu->setItemEnabled(ID_MAPPER_PATH_EDITBEND,(path->getBends()->count() > 0));
				path_menu->setItemEnabled(ID_MAPPER_PATH_ADDBEND,(path->getSrcRoom()->getZone()==path->getDestRoom()->getZone()));
				path_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));				
			}
			
			if (element->getElementType()==TEXT)
			{
				text_menu->popup(mapToGlobal(QPoint::QPoint(e->x(),e->y())));			
			}

			if (element->getLowX() < element->getHiX())
				selected.x = element->getLowX();	
			else
				selected.x = element->getHiX();	
				
			if (element->getLowY() < element->getHiY())
				selected.y = element->getLowY();	
			else
				selected.y = element->getHiY();	

			selected.element = element;
			selected.mouseX = e->x();
			selected.mouseY = e->y();
			found = true;
			break;
			
		}
 	}			
}

void CMapWidget::mouseReleaseEvent(QMouseEvent *e)
{
	if (e->button()==LeftButton)
	{
		currentTool->mouseReleaseEvent(e);
	}
	
	if (e->button() == MidButton)
	{
	  bMouseDrag = false;
	  QCursor* oldCursor = new QCursor(cursor());
	  setCursor(*mouseDragCursor);
	  delete mouseDragCursor;
	  mouseDragCursor = oldCursor;
	}
	
	emit statusMsg("Ready.");

}

/** Called when a key is pressed */
void CMapWidget::keyPressEvent(QKeyEvent *e)
{	
	if (e->key() == Key_Control) bCtrlPressed = true;
}

/** Called when a key is released */
void CMapWidget::keyReleaseEvent(QKeyEvent *e)
{	
	if (e->key() == Key_Control) bCtrlPressed = false;
}

void CMapWidget::slotSetCurrentPos(void)
{
	if (selected.element->getElementType() == ROOM)
	{
		currentRoom->setCurrentRoom(false);
		currentRoom=(CMapRoom *)selected.element;
		currentRoom->setCurrentRoom(true);
		selected.element = NULL;
		
		redraw();
	}
}


void CMapWidget::slotRoomAddToSpeedwalk(void)
{
	if (selected.element->getElementType() == ROOM)
	{
		mapperData->addSpeedwalkRoom((CMapRoom *)selected.element);
		emit updateSpeedwalkList((CMapRoom *)selected.element);
	}
}

void CMapWidget::setCurrentPos(CMapRoom* newPos)
{
	currentRoom->setCurrentRoom(false);
	currentRoom = newPos;
	currentRoom->setCurrentRoom(true);
	showLevel(currentRoom->getX(),currentRoom->getY(),currentRoom->getLevel(), currentRoom->getZone());	
}

void CMapWidget::slotSetLogin(void)
{
	if (selected.element->getElementType() == ROOM)
	{
		CMapRoom *room =(CMapRoom *)selected.element;

		loginRoom->setLoginRoom(false);
		loginRoom = room;
		loginRoom->setLoginRoom(true);

		CCharacterProfile* prof = doc->getCharacterProfile(doc->getCurrentCharacterID());
		if (prof)
		{
	 		prof->setMapStartX(loginRoom->getX());
	 		prof->setMapStartY(loginRoom->getY());
	 		prof->setMapStartLevel(loginRoom->getLevel());
	 		prof->setMapStartZone(mapperData->zoneList.findRef(loginRoom->getZone()));
	 		prof->writeData();
		}
				
		selected.element = NULL;

		redraw();
	}
}

/** Used to speedwalk a player to a given element which should be a room*/
void CMapWidget::walkPlayerTo(CMapRoom *room1)
{
	QQueue<CMapRoom> roomsToVisit;
	CMapRoom *destRoom;
	CMapRoom *srcRoom;
	CMapPath *path;
	CMapRoom *foundRoom;
	signed int time = 0;
	bool bFound = false;
	CMapPath *foundPath;	
	int speedWalkAbortCount = 0;

	if (currentRoom == room1) return;
	
	if (mapperData->speedwalkActive)
	{
		QMessageBox::information (this,i18n("Kmud Mapper"),
                                           i18n("Speedwalking is already in progress"));
		return;
	}

	mapperData->speedwalkActive = true;

	mapperData->pathToWalk.setAutoDelete(true);
	mapperData->pathToWalk.clear();
	
	// Reset the seach count for all the rooms
	for (CMapZone *zone = mapperData->zoneList.first(); zone!=0; zone = mapperData->zoneList.next())
		for (CMapLevelList *map=zone->getLevels()->first(); map !=0;map = zone->getLevels()->next())
			for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
			{
				room->setMoveTime(-1);
			}
		
	// Init things for the start room
	srcRoom = currentRoom;		
	destRoom = (CMapRoom *)room1;
	srcRoom->setMoveTime(time++);

	// enqueue starting room
	roomsToVisit.enqueue(srcRoom);
	CMapRoom* room;	
	
	// this is the Breadth First Search Algorithm
	while (!(roomsToVisit.isEmpty() || bFound))
	{
		foundRoom = roomsToVisit.dequeue();
		
		// for all neighbours of foundRoom
		for (path=foundRoom->pathList.first(); path != 0; path=foundRoom->pathList.next())
		{
			room = path->getDestRoom();
		
			if (room == destRoom)
			{
				bFound = true;
				break;
			}
	
			// neighbour has not been visited yet, enqueue it
			if (room->getMoveTime() == -1)
			{
				room->setMoveTime(time++);
				roomsToVisit.enqueue(room);
			}
		}
	}
		
	// Check to see if we were unable to find any paths
	if (!bFound)
	{
		roomsToVisit.clear();
		QMessageBox::information (this,i18n("Kmud Mapper"),
                                           i18n("The automapper was unable to find a path to requested room"));
				
		return;
	}
		
	speedWalkAbortCount=0;		
	directions *dirs= mudProfile->getDirections();
	
	// Trace steps that need to be taken backwards from the dest to the src room
	while(destRoom != srcRoom)
	{
		time = destRoom->connectingPaths.first()->getSrcRoom()->getMoveTime();
		foundRoom = destRoom->connectingPaths.first()->getSrcRoom();		
				
		// Find the room with the shortest time as this is the room we
		// should be moving to.		
		for (path=destRoom->connectingPaths.first();path!=0; path=destRoom->connectingPaths.next())
		{
			if (time == -1 || (path->getSrcRoom()->getMoveTime()<=time && path->getSrcRoom()->getMoveTime()!=-1))
			{
				time = path->getSrcRoom()->getMoveTime();
				foundRoom = path->getSrcRoom();
				foundPath = path;
			}
		}		
		
		switch (foundPath->getSrcDir())
		{
			case SPECIAL   : mapperData->pathToWalk.push(new QString(foundPath->getSpecialCmd())); break;
			case NORTH     : mapperData->pathToWalk.push(new QString(dirs->north)); break;
			case NORTHEAST : mapperData->pathToWalk.push(new QString(dirs->northeast)); break;
			case EAST      : mapperData->pathToWalk.push(new QString(dirs->east)); break;
			case SOUTHEAST : mapperData->pathToWalk.push(new QString(dirs->southeast)); break;
			case SOUTH     : mapperData->pathToWalk.push(new QString(dirs->south)); break;
			case SOUTHWEST : mapperData->pathToWalk.push(new QString(dirs->southwest)); break;
			case WEST      : mapperData->pathToWalk.push(new QString(dirs->west)); break;
			case NORTHWEST : mapperData->pathToWalk.push(new QString(dirs->northwest)); break;
			case UP        : mapperData->pathToWalk.push(new QString(dirs->up)); break;
			case DOWN      : mapperData->pathToWalk.push(new QString(dirs->down)); break;
		}

		destRoom=foundRoom;			
		// Check to make sure that tings are not stuck in a loop and abort
		// if the move limit is reached					
		
		speedWalkAbortCount++;
		if (speedWalkAbortActive && (speedWalkAbortCount == speedWalkAbortLimit))
		{
			QMessageBox::information (this,i18n("Kmud Mapper"),
                                           i18n("Speedwalk abort because move limit was reached"));
				
			return;
		}	
	}
		
				
	// Start walking path
	slotWalkPlayerAlongPath();
	
	// Tidy up
	roomsToVisit.clear();

}

/** Used to recersivly move the play along a speedwalk path */
void CMapWidget::slotWalkPlayerAlongPath(void)
{
	QString *dir = mapperData->pathToWalk.pop();
	emit movePlayer(*dir);

	// Walk the path
	if (!mapperData->pathToWalk.isEmpty())
		QTimer::singleShot( mapperData->speedwalkDelay * 100, this, SLOT(slotWalkPlayerAlongPath()) );
	else
	{
		mapperData->pathToWalk.setAutoDelete(true);
		mapperData->pathToWalk.clear();
		mapperData->speedwalkActive = false;
	}
}

/** Used to speed walk a player */
void CMapWidget::slotMovePlayerTo(void)
{
	// Check selected element is a room as we can only move to rooms
	if (selected.element->getElementType() == ROOM)
		walkPlayerTo((CMapRoom *)selected.element);
}


void CMapWidget::slotDelElement(void)
{
	if (selected.element->getElementType() == ROOM)
		deleteRoom((CMapRoom *)selected.element,currentLevelNum,currentZone);

	if (selected.element->getElementType() == ZONE)
		deleteZone((CMapZone *)selected.element,currentLevelNum,currentZone);		
		
	if (selected.element->getElementType() == PATH)
	{
		CMapPath *path = (CMapPath *)selected.element;
		for (CMapPath *path2 = path->getDestRoom()->pathList.first();path2!=0;path2=path->getDestRoom()->pathList.next())
		{			
			if (path2->getDestRoom()==path->getSrcRoom())
			{		
				deletePath(path2);
				break;
			}
		}
		
		deletePath(path);
	}

	if (selected.element->getElementType() == TEXT)
		deleteText((CMapText *)selected.element,currentLevelNum,currentZone);

	selected.element = NULL;

	redraw();
}

void CMapWidget::copyZone(CMapZone *parentZone,CMapZone *orgZone)
{
	int i;

	CMapZone *newZone = (CMapZone *)orgZone->copy();
	for (i=0;i<=(int)(orgZone->getLevels()->count())-1;i++)
	{
		createLevel(UP,newZone);
	}

	newZone->setZone(parentZone);
	orgZone->setSelected(true);
	elementClipboard.append(newZone);
	
	for (CMapLevelList *map=orgZone->getLevels()->first(); map !=0;map = orgZone->getLevels()->next())
	{
		for (CMapZone *zone=map->getZoneList()->first();zone !=0 ; zone = map->getZoneList()->next())
		{

			copyZone(newZone,zone);
		}
		
		for (CMapRoom *room=map->getRoomList()->first();room !=0 ; room = map->getRoomList()->next())
		{
			CMapRoom *newRoom = (CMapRoom *)room->copy();			
			newRoom->setOldZone(newRoom->getZone());
			newRoom->setZone(newZone);
			room->setSelected(true);
			elementClipboard.append(newRoom);	
		}
		
		for (CMapText *text=map->getTextList()->first();text !=0 ; text = map->getTextList()->next())
		{
			CMapText *newText = (CMapText *)text->copy();
			newText->setZone(newZone);			
			text->setSelected(true);
			elementClipboard.append(newText);
		}
	}
}

void CMapWidget::slotCopy(void)
{
	copiedZone = currentZone;
	
	elementClipboard.setAutoDelete(true);
	elementClipboard.clear();

	for (CMapElement *element=elementList.first(); element !=0; element =elementList.next())
	{
		if (element->getSelected())
		{
			CMapRoom *newRoom;
			
			switch (element->getElementType())
			{
				case PATH : break;
				case ZONE : copyZone(currentZone,(CMapZone *)element);break;
				case ROOM : newRoom = (CMapRoom *)element->copy();
			                newRoom->setOldZone(currentZone);
							newRoom->setZone(currentZone);
				            elementClipboard.append(newRoom);				
				default   : break;
			}
		}
	}
			
	for ( CMapZone *zone = mapperData->zoneList.first(); zone != 0; zone= mapperData->zoneList.next())
	{	
		for ( CMapLevelList *map=zone->getLevels()->first(); map != 0; map=zone->getLevels()->next())
		{
			for ( CMapRoom *room=map->getRoomList()->first(); room != 0; room=map->getRoomList()->next() )
			{
				for (CMapPath *oldPath=room->pathList.first(); oldPath != 0; oldPath=room->pathList.next())
				{
					CMapRoom *oldSrc = oldPath->getSrcRoom();
					CMapRoom *oldDest = oldPath->getDestRoom();

					if (oldSrc->getZone()!=copiedZone && oldDest->getZone()!=copiedZone)
						oldPath->setSelected(true);

					if (oldSrc->getSelected() && oldDest->getSelected() && oldPath->getSelected())
					{
						CMapRoom *newSrc = NULL;
						CMapRoom *newDest = NULL;

			        	for (CMapElement *elm = elementClipboard.first(); elm!=0; elm = elementClipboard.next())
						{
							if (elm->getElementType()==ROOM)
							{
								CMapRoom *foundRoom = (CMapRoom *)elm;
				
								if (foundRoom->getX() == oldSrc->getX() &&
							    	foundRoom->getY() == oldSrc->getY() &&
							    	foundRoom->getLevel() == oldSrc->getLevel() &&
							    	foundRoom->getOldZone() == oldSrc->getZone())
								{
									newSrc = foundRoom;
								}

								if (foundRoom->getX() == oldDest->getX() &&
							    	foundRoom->getY() == oldDest->getY() &&
							    	foundRoom->getLevel() == oldDest->getLevel() &&
							    	foundRoom->getOldZone() == oldDest->getZone())
								{
									newDest = foundRoom;
								}

	                        	if (newSrc && newDest) break;					
							}
						
						}

						if (newSrc && newDest)
						{	
							CMapPath *newPath = (CMapPath *)oldPath->copy();
			
							newPath->setSrcRoom(newSrc);
							newPath->setDestRoom(newDest);
							elementClipboard.append(newPath);									
						}
					}
				}
			}
		}
	}
}

void CMapWidget::slotCut(void)
{
	slotCopy();
	slotDelete();
}

CMapZone *CMapWidget::pasteZone(CMapZone *intoZone,CMapZone *clipZone,int inc)
{
	CMapElement *current = elementClipboard.current();

	CMapZone *newZone = createZone(clipZone->getX()+inc,clipZone->getY()+inc,clipZone->getLevel(),intoZone,inc > 0);

	if (!newZone) return NULL;
	
	newZone->setName(clipZone->getName());	

	for (CMapElement *element=elementClipboard.first(); element !=0; element =elementClipboard.next())
	{	
		CMapZone *zone;
		CMapRoom *room;
		CMapText *text;
		
		switch (element->getElementType())
		{
			case ZONE : zone = (CMapZone *)element;
			            if (zone->getZone() == clipZone)
			            {			            	
							pasteZone(newZone,zone,0);
			            }
			            break;
			case PATH : break;
			case ROOM : room= (CMapRoom *)element;
			            if (room->getZone() == clipZone)
						{							
				            CMapRoom *newRoom = createRoom(room->getX(),room->getY(),room->getLevel(),newZone);			
							newRoom->setColour(room->getColour());
							newRoom->setLabel(room->getLabel());
							newRoom->setDescription(room->getDescription());
							newRoom->setUseDefaultCol(room->getUseDefaultCol());								
							room->setNewRoom(newRoom);
				   		}
			            break;
			case TEXT : text = (CMapText *)element;
			            if (text->getZone() == clipZone)
			            {
			            	createText(text->getText(),text->getLowX(),text->getLowY(),text->getFont(),text->getColour(),text->getLevel(),newZone,false);
			            }
			            break;
			case OTHER: break;
		}
	}

	elementClipboard.find(current);

	return newZone;
}

void CMapWidget::slotPaste(void)
{
	unselectAll();	

	for (CMapElement *element=elementClipboard.first(); element !=0; element =elementClipboard.next())
	{		
		if (element->getElementType()==ROOM)
		{		
			CMapRoom *room= (CMapRoom *)element;
			if (room->getZone()==copiedZone)
			{
				CMapRoom *newRoom = createRoom(room->getX()+1,room->getY()+1,currentLevelNum,currentZone);
				if (newRoom)
				{
					newRoom->setColour(room->getColour());
					newRoom->setLabel(room->getLabel());
						newRoom->setDescription(room->getDescription());
					newRoom->setUseDefaultCol(room->getUseDefaultCol());	
					room->setNewRoom(newRoom);			
				}
			}
		}
		
		if (element->getElementType()==ZONE)
		{
			CMapZone *zone = (CMapZone *)element;
			if (copiedZone == zone->getZone())
			{
				pasteZone(currentZone,zone,1);			
			}
		}

		if (element->getElementType()==TEXT)
		{	
			CMapText *text = (CMapText *)element;
			if (copiedZone == text->getZone())
			{
				createText(text->getText(),text->getLowX(),text->getHiY() + ROOM_SIZE,text->getFont(),text->getColour(),currentLevelNum,currentZone,true);
			}
		}
	}

	for (CMapElement *element=elementClipboard.first(); element !=0; element =elementClipboard.next())
	{
		if (element->getElementType()==PATH)
		{
			CMapElement *currentElement = elementClipboard.current();
			CMapPath *path= (CMapPath *)element;     	

			CMapRoom *srcRoom = NULL;
			CMapRoom *destRoom = NULL;

			for (CMapElement *elm=elementClipboard.first(); elm!=0; elm=elementClipboard.next())
			{
				if (elm == path->getSrcRoom())
					srcRoom = path->getSrcRoom()->getNewRoom();
				if (elm == path->getDestRoom())
					destRoom = path->getDestRoom()->getNewRoom();
			}

			if (srcRoom && destRoom)
			{

				if (srcRoom->getElementType()!=ROOM || destRoom->getElementType()!=ROOM)
				{
					break;	

				}
				
				CMapPath *path2=createPath(srcRoom,destRoom,path->getSrcDir(),path->getDestDir());
   				if (path2)
	 				for (QPoint *point = path->getBends()->first(); point !=0 ; point= path->getBends()->next())
					{
						path2->addBend(point->x()+ROOM_SIZE,point->y()+ROOM_SIZE);
					}
			}		
			elementClipboard.find(currentElement);
		}
	}

	redraw();
}

void CMapWidget::slotDelete(void)
{
	CMapElement *element = elementList.last();

	// Delete All rooms that are selected
	while(element!=0)
	{
		if (element->getSelected())
		{
			switch (element->getElementType())
			{
				case ROOM : deleteRoom((CMapRoom*)element,currentLevelNum,currentZone);
				            element=elementList.last();
				            break;
				case ZONE : deleteZone((CMapZone*)element,currentLevelNum,currentZone);
				            element=elementList.last();
				            break;
				default   : element=elementList.prev();
				            break;
			}				
		}
		else
			element=elementList.prev();
	}

	element = elementList.last();

	// Delete all elements that are selected but are not rooms
	while(element!=0)
	{
		if (element->getElementType()!=ROOM && element->getElementType()!=OTHER && element->getSelected())
		{
			switch (element->getElementType())
			{
				case PATH : deletePath((CMapPath *)element);
							break;
				case TEXT : deleteText((CMapText*)element,currentLevelNum,currentZone); break;
			    default   : break;
			}
			elementList.last();
		}
		else
			element=elementList.prev();		
	}

	redraw();
}

void CMapWidget::slotSelectAll(void)
{
	for (CMapElement *element=elementList.last(); element!=0; element=elementList.prev())
		element->setSelected(true);

	redraw();
}

void CMapWidget::slotDeselectAll(void)
{
	unselectAll();

	redraw();
}

void CMapWidget::slotInvertSelection(void)
{
	for (CMapElement *element=elementList.last(); element!=0; element=elementList.prev())
		element->setSelected(!element->getSelected());
	
}

/** Used to change a path to one way */
void CMapWidget::slotPathSetOneWay(void)
{
	CMapPath *path = (CMapPath *)selected.element;	
	if (path->getOpsitePath())
	{
		makePathOneWay(path);
		redraw();
	}
}

/** Used to change a path to a twoway path */
void CMapWidget::slotPathSetTwoWay(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	if (!path->getOpsitePath())
	{
		makePathTwoWay(path);
		redraw();		
	}
}

/** Used to change the properties of the text elements */
void CMapWidget::slotTextProperties(void)
{

	DlgMapTextProperties d(this);
	
	CMapText *text = (CMapText *)selected.element;

	if (!text) return;

	d.setFont(text->getFont());
	d.setColour(text->getColour());
	d.setText(text->getText());


	if (d.exec())
	{
		text->setFont(d.getFont());
		text->setColour(d.getColour());
		text->setText(d.getText());
	 	QFontMetrics fm(d.getFont());
		int x = text->getLowX();
		int y = text->getLowY();
		text->setCords(x,y,x+fm.width(d.getText()),y-fm.height());	
		redraw();
	}									
}

/** Add a bend to a path */
void CMapWidget::slotPathAddBend(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	path->addBend(selected.mouseX,selected.mouseY);
	
	if (path->getOpsitePath())
		path->getOpsitePath()->addBend(selected.mouseX,selected.mouseY);
		
	redraw();
}

void CMapWidget::slotPathDelBend(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	int bendNum = path->mouseInPathSeg(selected.mouseX,selected.mouseY,getCurrentZone());
	
	if (bendNum == (int)(path->getBends()->count()))
		bendNum--;
	
	path->delBend(bendNum);	
	
	if (path->getOpsitePath())
		path->getOpsitePath()->delBend(bendNum);
		
	redraw();
}

/** Edit the bends of a path */
void CMapWidget::slotPathEditBends(void)
{
	CMapPath *path = (CMapPath *)selected.element;
	
	unselectAll();
	path->setEditMode(true);
	redraw();		
}

/** Used display the path properties */
void CMapWidget::slotPathProperties(void)
{
	DlgMapPathProperties d(this, "Properties");

	CMapPath *path = (CMapPath *)selected.element;
		
	d.setDirections(path->getSrcDir(),path->getDestDir());	
	d.setTwoWay(path->getOpsitePath());
	
	d.setNormalExit(!path->getSpecialExit());
	d.setSpecialCmd(path->getSpecialCmd());
	
	if (d.exec())
	{		
  		path->setSrcDir(d.getSrcDirection());
		path->setDestDir(d.getDestDirection());
		setPathCords(path);		
		path->setSpecialCmd(d.getSpecialCmd());
		path->setSpecialExit(!d.getNormalExit());
		
		CMapPath *path2 = path->getOpsitePath();
		if (path2)
		{
			if (!d.getTwoWay())
				makePathOneWay(path);
			else
			{
				path2->setSrcDir(d.getDestDirection());
				path2->setDestDir(d.getSrcDirection());
				setPathCords(path2);
			}					
		}
		else
		{
			if (d.getTwoWay())
				makePathTwoWay(path);
		}		
	}

	redraw();
}

void CMapWidget::slotZoneOpen(void)
{
	showLevel(-1,-1,0,(CMapZone*)selected.element);
}



void CMapWidget::slotZoneProperties(void)
{
	zoneProperties((CMapZone*)selected.element);
}

void CMapWidget::commandCallback(int id_)
{
	switch (id_)
	{
		ON_CMD(ID_MAPPER_ROOM_MOVE_LOC,    slotSetCurrentPos())
		ON_CMD(ID_MAPPER_ROOM_MOVE_PLAYER, slotMovePlayerTo())
		ON_CMD(ID_MAPPER_ROOM_DEL,         slotDelElement())
		ON_CMD(ID_MAPPER_ROOM_SET_LOGIN,   slotSetLogin())
		ON_CMD(ID_MAPPER_ROOM_ADD_SPEED,   slotRoomAddToSpeedwalk())
		ON_CMD(ID_MAPPER_ROOM_PROP,        slotRoomProperties())
		
		ON_CMD(ID_MAPPER_PATH_DEL,         slotDelElement())
		ON_CMD(ID_MAPPER_PATH_ONEWAY,      slotPathSetOneWay())
		ON_CMD(ID_MAPPER_PATH_TWOWAY,      slotPathSetTwoWay())
		ON_CMD(ID_MAPPER_PATH_PROP,        slotPathProperties())
		ON_CMD(ID_MAPPER_PATH_ADDBEND,     slotPathAddBend())
		ON_CMD(ID_MAPPER_PATH_DELBEND,     slotPathDelBend())
		ON_CMD(ID_MAPPER_PATH_EDITBEND,    slotPathEditBends())		
		
		ON_CMD(ID_MAPPER_TEXT_DEL,         slotDelElement())		
		ON_CMD(ID_MAPPER_TEXT_PROP,        slotTextProperties())
		
		ON_CMD(ID_MAPPER_ZONE_DEL,         slotDelElement())		
		ON_CMD(ID_MAPPER_ZONE_OPEN,        slotZoneOpen())				
		ON_CMD(ID_MAPPER_ZONE_PROP,        slotZoneProperties())				
	}		
}

void CMapWidget::statusCallback(int id_)
{
	switch (id_)
	{
		ON_STATUS_MSG(ID_MAPPER_ROOM_MOVE_LOC,    i18n("Set the current location to this room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_MOVE_PLAYER, i18n("Speed walk the player to this room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_DEL,         i18n("Delete the room"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_SET_LOGIN,   i18n("Set the players login position"))		
		ON_STATUS_MSG(ID_MAPPER_ROOM_ADD_SPEED,   i18n("Add to speedwalk list"))
		ON_STATUS_MSG(ID_MAPPER_ROOM_PROP,        i18n("Edit the room properties"))
		
		ON_STATUS_MSG(ID_MAPPER_PATH_DEL,         i18n("Delete the path"))
		ON_STATUS_MSG(ID_MAPPER_PATH_ONEWAY,      i18n("Make the path one way"))
		ON_STATUS_MSG(ID_MAPPER_PATH_TWOWAY,      i18n("Make the path two way"))
		ON_STATUS_MSG(ID_MAPPER_PATH_PROP,        i18n("Edit the paths properties"))
		ON_STATUS_MSG(ID_MAPPER_PATH_ADDBEND,     i18n("Add bend to the path"))
		ON_STATUS_MSG(ID_MAPPER_PATH_DELBEND,     i18n("Delete a segment from the path"))		
		ON_STATUS_MSG(ID_MAPPER_PATH_EDITBEND,    i18n("Edit the bends of the path"))				
		
		ON_STATUS_MSG(ID_MAPPER_TEXT_DEL,         i18n("Delete the text element"))
		ON_STATUS_MSG(ID_MAPPER_TEXT_PROP,        i18n("Edit the text properties"))		

		ON_STATUS_MSG(ID_MAPPER_ZONE_DEL,         i18n("Delete the zone element"))	
		ON_STATUS_MSG(ID_MAPPER_ZONE_OPEN,        i18n("Display the contents of the zone"))
		ON_STATUS_MSG(ID_MAPPER_ZONE_PROP,        i18n("Edit the zone properties"))		

 	}
}

void CMapWidget::slotStatusHelpMsg(const char *text)
{
	emit statusHelpMsg(text);
}

void CMapWidget::slotZoneUp(void)
{
	if (currentZone->getZone())
	{
	  int x = currentZone->getHiX();
	  int y = currentZone->getHiY();
	
		showLevel(-1,-1,currentZone->getLevel(),currentZone->getZone());
		mapView->center(x, y);
	}
}

void CMapWidget::slotLevelUp(void)
{
	if (currentLevelNum+1 < (int)(currentLevelList->count()))
		showLevel(-1,-1,currentLevelNum+1,currentZone);
}

void CMapWidget::slotLevelDown(void)
{
	if (currentLevelNum-1>=0)
		showLevel(-1,-1,currentLevelNum-1,currentZone);
}


void CMapWidget::slotRoomProperties(void)
{
	roomProperties((CMapRoom *)selected.element);
}



