
/* search.c
 *   Handles the search window popup
 */

#include <stdio.h>

#include <Xos.h>
#include <Intrinsic.h>
#include <StringDefs.h>

#include <Shell.h>

#include <Xaw/Command.h>
#include <Xaw/Label.h>
#include <Xaw/Form.h>
#include <Xaw/Box.h>
#include <Xaw/AsciiText.h>
#include <cursorfont.h>	/* should I be doing this? */

#include "defs.h"
#include "externs.h"
#include "game.h"
#include "widgets.h"
#include "search.h"
#include "convert.h"


enum {G_LABEL,G_INPUT,F_LABEL,F_INPUT,POUND_LABEL,POUND_INPUT,
		H_LABEL,H_INPUT,N_LABEL,N_INPUT,U_LABEL,U_INPUT,
		NUM_OF_N };

enum {
	NFORM_W, KANA_W, ENGLISH_W,
	SEARCH_ENG_L, SEARCH_ENG_W,
	SEARCH_KANA_L, SEARCH_KANA_W, 
	SEARCH_KANJI_L, SEARCH_KANJI_L2, SEARCH_KANJI_W,
	NUM_OF_W};


Widget search_popup;

Widget searchwidgets[NUM_OF_W];
Widget searchnumbers[NUM_OF_N];
Widget searchkanjiW,searchokuW; /* display current kanji here */

static TRANSLATION lastsearch=NULL;	/* holds index of last kanji searched */

#define MAXKANALENGTH 17
/* note.. if your compiler bitches about this initialization...
 * just remove  the "={{0,0}}".
 * It will probably get initialized to 0,0 anyways
 */
static XChar2b kanastring[MAXKANALENGTH]={{0,0}};

static char *searchAccel = 
 " <Key>Return:  do-find()";


static up = -1; /* this has to be visible by printsearch() */
		/* declares whether popup is up or not */

/* This gets called after clicking on the "search" button, or
 *  using the keyboard accelerator. It pops up the search window.
 */
void SearchCallback(Widget w,XtPointer client_data, XtPointer call_data)
{
	static Position rel_x,rel_y;
	Position x,y;

	if(up == -1){
		/* first time init.. */
		rel_x = GetXtNumber("search_popupx","Search_popupx");
		rel_y = GetXtNumber("search_popupy","Search_popupy");
		up = 0;
	}

	if(up == 0)
	{
		XtTranslateCoords(toplevel,rel_x,rel_y,&x,&y);
		XtVaSetValues(search_popup,
		      XtNx,x,
		      XtNy,y,
		      NULL);
		/*XtVaSetValues(searchwidgets[SEARCH_ENG_W],
		*	      XtNstring," Search String Here ",
		*	      NULL);
		*/
		XtPopup(search_popup,XtGrabNone);
		setstatus("Bringing up search window...");
		up = 1;
	} else
	{
		XtPopdown(search_popup);
		up = 0;
	}
}

/* printsearch:
*	Given an index, make search window print out appropriate stuff.
*	printsearch(0) means "clear windows".
*
*	This also sets the lastsearch static global!!
*/
void printsearch(TRANSLATION This)
{
	XChar2b blank = {0x0, 0x0};
	XChar2b *kanjiptr;
	int kindex=0;

	if(up != 1){
		/* pop up window, if it isnt already */
		SearchCallback(NULL,NULL,NULL);
	}



	if(This==NULL){
		/* BLANK EVERYTHING */
		XtVaSetValues(searchkanjiW,
			      XtNlabel,"                               ",
			      NULL);
		XtVaSetValues(searchwidgets[ENGLISH_W],XtNlabel,"",NULL);
		SetWidgetNumberval(searchnumbers[G_INPUT],0);
		SetWidgetNumberval(searchnumbers[F_INPUT],0);
		SetWidgetNumberval(searchnumbers[POUND_INPUT],0);
		SetWidgetNumberval(searchnumbers[H_INPUT],0);
		SetWidgetNumberval(searchnumbers[N_INPUT],0);
		SetWidgetNumberval(searchnumbers[U_INPUT],0);
		XtVaSetValues(searchkanjiW,XtNlabel,&blank,
			XtNwidth,KANJIWIDTH,XtNheight,KANJIWIDTH,NULL);
		
	} else {

		if(This->kanji == NULL)
		{
			kanjiptr = &blank;
		} else
		{
			kanjiptr = This->kanji;
		}
		XtVaSetValues(searchkanjiW,XtNlabel,kanjiptr,
			XtNwidth,KANJIWIDTH,XtNheight,KANJIWIDTH,NULL);

		SetWidgetNumberval(searchnumbers[G_INPUT],(int)This->grade_level);
		SetWidgetNumberval(searchnumbers[F_INPUT],(int)This->frequency);
		SetWidgetNumberval(searchnumbers[H_INPUT],(int)This->Hindex);
		SetWidgetNumberval(searchnumbers[N_INPUT],(int)This->Nindex);
		SetWidgetHexval(searchnumbers[U_INPUT],(int)This->Uindex);

		/* assume if kanji entry is ONE char long, that we
		 * can use that byte as old kdrill/kanjidic index
		 *  ('#x' window)
		 */

		if(This->kanji != NULL)
		  if(This->kanji[1].byte1 == 0)
		  {
			kindex = This->kanji[0].byte1;
			kindex = kindex <<8;
			kindex |= This->kanji[0].byte2;
			
		  }
		if(kindex == NOKANJI)
		{
			kindex = 0;
		}

		SetWidgetHexval(searchnumbers[POUND_INPUT], kindex);

		if(romajiswitch==1)
		{
			char stringbuff[100];
			/*translate all kana into romaji */
			/* translate to buffer, then set widget string*/
			kanatoromaji(This->pronunciation, stringbuff);
			XtVaSetValues(searchwidgets[KANA_W],
				      XtNencoding,XawTextEncoding8bit,
				      XtNfont,englishfont,
				      NULL);
			XtVaSetValues(searchwidgets[KANA_W],
				      XtNlabel,stringbuff,
				      NULL);
		
		} else
		{
			XtVaSetValues(searchwidgets[KANA_W],
			      XtNencoding, XawTextEncodingChar2b,
			      XtNfont,smallkfont,
			      NULL);
			XtVaSetValues(searchwidgets[KANA_W],
			      XtNlabel,This->pronunciation,NULL);
		}
		XtVaSetValues(searchwidgets[ENGLISH_W],
			      XtNlabel,This->english,NULL);
	}

	lastsearch = This;

}




/*
 * Kay, this is ugly.
 * This whole routine is because I'm too lazy to jump through the
 * incredible hoops of learning how to do X cut-n-paste properly.
 * So, I made a fake drop-target out of a asciiTextWidget, and
 * made it so we also get notified of a mouseclick.
 * Then, we have to do nasty decoding to JIS, and THEN
 * do a lookup ;-)
 *
 * We do this on button RELEASE, so that we can be reasonably sure
 * the widget has had the text inserted already.
 */
void Handle_searchPasted(Widget widget, XtPointer cl, XEvent *e, Boolean *cont)
{
	String str;
	int strparse;
	int searchchar;
	
	if(e->type != ButtonRelease)
		return;
		
	XtVaGetValues(widget,XtNstring,&str,NULL);

/*	puts("values of gotten string are:");*/
	strparse=0;
	do
	{
/*		printf("%.2x ",str[strparse]);*/
	} while(str[strparse++] != '\0');
/*	puts("");
	printf("strparse is now %d\n", strparse);*/

	if(strparse<2)
		return;

	if(str[0]&0xffffff00)
	{
		/* just PURE JIS (probably) */
		searchchar = (str[0]&0x7f) <<8;
		searchchar|= (str[1]&0x7f);
	}
	else
	{
		/* otherwise, assume iso2022 or whatever-encoded */
		/* leading 1b242942 is expected */
	
		if(strparse <7 )
		{
			puts("paste too short. bye-bye");
			XtVaSetValues(widget,XtNstring,"\0",NULL);
			return;
		}
		if(str[0] != 0x1b)
		{
			puts("oops. cannot recognise encoding");
			XtVaSetValues(widget,XtNstring,"\0",NULL);
			return;
		}

		searchchar = (str[4]&0x7f) <<8;
		searchchar|= (str[5]&0x7f);
	}

	XtVaSetValues(widget,XtNstring,"\0",NULL);

	if((searchchar<lowestkanji) ||
	   (searchchar>highestkanji))
	{
		setstatus("char unrecognized");
		return;
	}
	

	/*printf("Okay, potential char is now %.4x\n",searchchar); */
	printsearch(translations[searchchar]);
	
	
}


/* doenglishfind:
 *	Assume what we want to look for is in the search ENGLISH INPUT widget.
 *	First, read in that value.
 *	Start from last place searched, to allow for similar
 *	characters.
 *	Return pointer to kanji, or NULL if not found
 */
TRANSLATION doenglishfind()
{
	String str;
	TRANSLATION searchtarget = NULL;

	setstatus("Searching...");

	XtVaGetValues(searchwidgets[SEARCH_ENG_W],XtNstring,&str,NULL);

	if((str[0] == '\0') || (str[0] == '\n')){
		return NULL;
	}


	if(lastsearch == NULL){
		lastsearch = translations[lowestkanji];
		searchtarget = lastsearch;
	} else {
		searchtarget = lastsearch->nextk;
	}
	
	do {
		
		if(strstr(searchtarget->english,str) != NULL){
			return searchtarget;
		}
		searchtarget = searchtarget->nextk;
		if(searchtarget == NULL){
			searchtarget = translations[lowestkanji];
		}
	
	} while(searchtarget != lastsearch);

	return NULL;
	
}

/* DoFind
 *   (do-find)accelerator hook
 *	Farms out english search to doenglishearch().
 *
 *	Also 
 *    Handles <return> press for numerical input windows
 *	on the search popup.
 */
void 
DoFind(Widget w,XEvent *event,String *params,Cardinal *num_parags){
	TRANSLATION findtarget = NULL;

	setstatus("Searching...");

	if(w == searchwidgets[SEARCH_ENG_W]){
		findtarget = doenglishfind();

		if(findtarget == NULL){
			setstatus("No match found");
			Beep();
			printsearch(lastsearch);
		} else {
			printsearch(findtarget);
		}
		return;
		
	}

	if (w == searchnumbers[POUND_INPUT])
	{
		int number;
		/* want closest index match */
		number = GetWidgetNumberval(w);

		if((number <lowestkanji) || (number> highestkanji)){
			setstatus("input out of dict. range");
			Beep();
			printsearch(lastsearch);
			return;
		} 

		if(number > highestkanji){
			number = highestkanji;
		} else if (number <lowestkanji){
			number = lowestkanji;
		} else {
			/* find closest non-blank */
			while(translations[number] == NULL){
				number++;
				/* this should never be triggered, but... */
				if(number >highestkanji)
					number = lowestkanji;
			}
		}

		findtarget = translations[number];

		printsearch(findtarget);
	} else 	if (w == searchnumbers[F_INPUT])
	{
		/* want closest frequency match */
		/* "F:" input widget */
		int findfreq=-1;
		TRANSLATION cur_target = translations[lowestkanji];
		

		findfreq = GetWidgetNumberval(w);

		if(findfreq <1){
			setstatus("frequencies must be greater than 0");
			Beep();
			printsearch(lastsearch);
			return;			
		}

		cur_target = translations[lowestkanji];
		findtarget = cur_target;

		while(cur_target != NULL){
			if(cur_target->frequency == findfreq){
				findtarget = cur_target;
				break;
			}
			if((findfreq - (cur_target->frequency)) <
			   (findfreq - (findtarget->frequency)) ){
				findtarget = cur_target;
			}
			cur_target = cur_target->nextk;
			   
		}
		
		printsearch(findtarget);

	} else {
		/* find exact match for special index */
		char append='\0';
		char findstring[100];
		String str;

		
		if(w == searchnumbers[H_INPUT]){
			append = 'H';
		} else if(w == searchnumbers[N_INPUT]){
			append = 'N';
		} else if(w == searchnumbers[U_INPUT]){
			append = 'U';
		} else {
			puts("Where did that input come from??");
			return;
		}
		
		XtVaGetValues(w,XtNstring,&str,NULL);

		sprintf(findstring,"%c%.99s",append,str);

		/* FindIndex sets status appropriately on error */

		findtarget = translations[FindIndex(findstring)];
		printsearch(findtarget);
		
	}
	

}


/*************************************************************
 *    Widget Creation Below Here                             *
 *************************************************************/

/* make the G, F, # , etc,  for the search popup */
void makenumbers(Widget parent){
	searchnumbers[G_LABEL] = XtVaCreateWidget("0",labelWidgetClass,parent,
		       XtNlabel,"G:",      XtNborderWidth,0,
		       NULL);
	searchnumbers[G_INPUT] = XtVaCreateWidget("1",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[G_LABEL],
		       XtNstring,"   ",  XtNdisplayCaret,False,
		       XtNwidth,30,	/* leave this short */
		       NULL);
	searchnumbers[H_LABEL] = XtVaCreateWidget("6",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNlabel,"H:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[H_INPUT] = XtVaCreateWidget("7",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[H_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	searchnumbers[F_LABEL] = XtVaCreateWidget("2",labelWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[G_INPUT],
		       XtNfromHoriz,searchnumbers[H_INPUT],
		       XtNlabel,"F:",  XtNborderWidth,0,
		       NULL);
	searchnumbers[F_INPUT] = XtVaCreateWidget("3",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[F_LABEL],
		       XtNstring,"",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);
	searchnumbers[N_LABEL] = XtVaCreateWidget("8",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[G_INPUT],
		       XtNfromHoriz,searchnumbers[H_INPUT],
		       XtNlabel,"N:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[N_INPUT] = XtVaCreateWidget("9",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[N_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	searchnumbers[POUND_LABEL] = XtVaCreateWidget("4",labelWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[F_INPUT],
		       XtNfromHoriz,searchnumbers[N_INPUT],
		       XtNlabel,"#x:",  XtNborderWidth,0,
		       NULL);
	searchnumbers[POUND_INPUT] = XtVaCreateWidget("5",asciiTextWidgetClass,parent,
		       XtNfromHoriz,searchnumbers[POUND_LABEL],
		       XtNstring,"",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);
	searchnumbers[U_LABEL] = XtVaCreateWidget("10",labelWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[F_INPUT],
		       XtNfromHoriz,searchnumbers[N_INPUT],
		       XtNlabel,"Ux:",   XtNborderWidth,0,
		       NULL);
	searchnumbers[U_INPUT] = XtVaCreateWidget("11",asciiTextWidgetClass,parent,
		       XtNfromVert,searchnumbers[G_LABEL],
		       XtNfromHoriz,searchnumbers[U_LABEL],
		       XtNstring," ",  XtNwidth,INPUTWIDTH,
		       XtNeditType,XawtextEdit,
		       NULL);

	XtManageChildren(searchnumbers,NUM_OF_N);
}


/* make topstuff:
 * Make widgets dealing with "current kanji", at top of search popup
 */
void maketopsearchstuff(Widget parent){
	Widget numbersform;

	searchkanjiW = XtVaCreateManagedWidget("kanjilarge",
						  labelWidgetClass,
						  parent,
			XtNwidth,FULLWIDTH - 100,
			XtNheight,KANJIWIDTH,
			XtNlabel,"",
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,largekfont,
			NULL);
	
	numbersform = XtVaCreateManagedWidget("numform",
					formWidgetClass,
					parent,
			XtNfromVert,searchkanjiW,
       			NULL);

	makenumbers(numbersform);


}

/* define the label and input widgets for inputting english
 * and kana.
 * The kana is currently not operational...
 * For kana, the "label" is actually a button that will bring up the
 * special kana input window. User can then point-and-click.
 * Not implemented yet, however.
 */
void makesearchinputs(Widget parent)
{
	searchwidgets[SEARCH_ENG_L] =
		XtVaCreateWidget("searchenglishlabel",
			labelWidgetClass,
			parent,
			XtNjustify, XtJustifyRight,
			XtNlabel, "English search",
			XtNvertDistance, 20,
			XtNborderWidth, 0,
			XtNfromVert, searchwidgets[ENGLISH_W],
			NULL);
						       
	searchwidgets[SEARCH_ENG_W] =
		XtVaCreateWidget("searchenglish",
			asciiTextWidgetClass,
			parent, 
			XtNfromVert, searchwidgets[ENGLISH_W],
			XtNfromHoriz, searchwidgets[SEARCH_ENG_L],
			XtNvertDistance, 20,
			XtNfont, englishfont,
			XtNeditType, XawtextEdit,
			XtNwidth, FULLWIDTH - 100,
			NULL);

	/* and now the kana search input windows */

	searchwidgets[SEARCH_KANA_L] =
		XtVaCreateWidget("searchkanalabel",
			commandWidgetClass,
			parent,
			XtNjustify, XtJustifyRight,
			XtNlabel, "Kana search   ",
			/*XtNborderWidth, 0,*/
			XtNfromVert, searchwidgets[SEARCH_ENG_L],
			NULL);
						       
	searchwidgets[SEARCH_KANA_W] =
		XtVaCreateWidget("searchkana",
			labelWidgetClass,
			parent, 
			XtNwidth, FULLWIDTH - 100,
			XtNlabel,"\0",
			/*XtNlabel, "!z",*/ /* 0x217a */
			XtNfromVert, searchwidgets[SEARCH_ENG_L],
			XtNfromHoriz, searchwidgets[SEARCH_ENG_L],
			/*XtNfromHoriz, searchwidgets[SEARCH_KANA_L],*/
			XtNfont, smallkfont,
			XtNencoding, XawTextEncodingChar2b,
			XtNjustify, XtJustifyLeft,
	/*		XtNcursor, XC_tcross,/* show we can input something*/
			NULL);
	searchwidgets[SEARCH_KANJI_L] =
		XtVaCreateWidget("searchkanjilabel",
			commandWidgetClass,
			parent,
			XtNjustify, XtJustifyRight,
			XtNlabel, "Kanji search  ",
			/*XtNborderWidth, 0,*/
			XtNfromVert, searchwidgets[SEARCH_KANA_L],
			NULL);


	searchwidgets[SEARCH_KANJI_L2] =
		XtVaCreateWidget("searchkanjil2",
			labelWidgetClass,
			parent,
			XtNjustify, XtJustifyRight,
			XtNlabel, "(or paste char ->)",
			XtNborderWidth, 0,
			XtNfromVert, searchwidgets[SEARCH_KANA_L],
			XtNfromHoriz, searchwidgets[SEARCH_KANJI_L],
			NULL);

	searchwidgets[SEARCH_KANJI_W] =
		XtVaCreateWidget("searchkanji",
			asciiTextWidgetClass,
			parent, 
			XtNwidth, 50,
			XtNlabel, "",
			XtNfromVert, searchwidgets[SEARCH_KANA_L],
			XtNfromHoriz, searchwidgets[SEARCH_KANA_L],
			XtNfromHoriz, searchwidgets[SEARCH_KANJI_L2],
			XtNfont, smallkfont,
			XtNencoding, XawTextEncodingChar2b,
			XtNeditType, XawtextEdit,
			XtNjustify, XtJustifyLeft,
			XtNdisplayCaret, False,
			NULL);

	XtAddCallback(searchwidgets[SEARCH_KANA_L], 
		      XtNcallback, Showinputkana, NULL);
	XtAddCallback(searchwidgets[SEARCH_KANJI_L], 
		      XtNcallback, Showinputkanji, NULL);

	XtAddEventHandler(searchwidgets[SEARCH_KANA_W],
			  KeyPressMask,False,
			  Handle_romajikana,NULL);
	
	XtAddEventHandler(searchwidgets[SEARCH_KANJI_W],
			  ButtonReleaseMask, False,
			  Handle_searchPasted,NULL);
			  

}


/* match XChar2b
 *	Attempt to find occurence of fragment in xchar2b string
 *	return 1 on match, 0 on fail.
 *
 *	Note that we try to allow match of kanagana, by hiragana.
 */
int matchkana(XChar2b *fullstring, XChar2b *fragment)
{

	/* cheat and presume if byte1==0, we've hit terminator */
	if(fragment->byte1 == 0)
	{
		return 1;
	}

	while(fullstring->byte1 != 0)
	{
		XChar2b  *stringparse,*fragparse;

		/* find first possible match.
		 * continue fromthere.
		 * if we fail later on, continue from that first point
		 */
		if(fullstring->byte2 != fragment->byte2)
		{
			fullstring++;
			continue;
		}
		if((fullstring->byte1 == fragment->byte1) ||
		   ((fullstring->byte1 == 0x25) && 
		    (fragment->byte1 == 0x24)) )
		{
			
			stringparse = fullstring+1;
			fragparse = fragment+1;

			while(1)
			{
				if(fragparse->byte1 == 0)
				{
					/* matched to the end of fragment! */
					return 1;
				}

				if(fragparse->byte2 != stringparse->byte2)
					break;
				if(fragparse->byte1 != stringparse->byte1)
				{
					/* special check for kanatana
					 * vs hiragana
					 */
					if(stringparse->byte1 != 0x25)
						break;
					if(fragparse->byte1 != 0x24)
						break;
					
				}

				fragparse++;
				stringparse++;
			}

		}
	
		fullstring++;
	}
	return 0;
}



/* dokanafind()
 *	find match to XChar2b string in kana translations
 */

void dokanafind(XChar2b *target)
{
	/*TRANSLATION findtmp,startsearch;*/
	TRANSLATION searchtarget = NULL;

	setstatus("Searching...");

	if(lastsearch == NULL)
	{
		lastsearch = translations[lowestkanji];
		searchtarget = lastsearch;
	} else {
		searchtarget = lastsearch->nextk;
	}

	do
	{
		if(matchkana(searchtarget->pronunciation, target))
		{
			printsearch(searchtarget);
			/* printsearch sets lastsearch */
			return;
		}
		searchtarget = searchtarget->nextk;
		if(searchtarget == NULL)
		{
			searchtarget = translations[lowestkanji];
		}
	} while(searchtarget != lastsearch);

	setstatus("No match found");
	Beep();

	return;
}

/* dokanjifind()
 *	find match to "Four Corner" encoding in kanjidic translations
 */

void dokanjifind(int target)
{
	/*TRANSLATION findtmp,startsearch;*/
	TRANSLATION searchtarget = NULL;

	setstatus("Searching...");

	if(lastsearch == NULL)
	{
		lastsearch = translations[lowestkanji];
		searchtarget = lastsearch;
	} else {
		searchtarget = lastsearch->nextk;
	}

	do
	{
		if(searchtarget->Qindex == target)
		{
			printsearch(searchtarget);
			/* printsearch sets lastsearch */
			return;
		}

		searchtarget = searchtarget->nextk;

		if(searchtarget == NULL)
		{
			searchtarget = translations[lowestkanji];
		}
	} while(searchtarget != lastsearch);

	setstatus("No match found");
	Beep();

	return;
}

/* doskipfind()
 *	find match to "SKIP" encoding in kanjidic translations
 *	Indentical to dokanjifind, except look at Qindex instead of Sindex
 */

void doskipfind(int target)
{
	/*TRANSLATION findtmp,startsearch;*/
	TRANSLATION searchtarget = NULL;

	setstatus("Searching...");

	if(lastsearch == NULL)
	{
		lastsearch = translations[lowestkanji];
		searchtarget = lastsearch;
	} else {
		searchtarget = lastsearch->nextk;
	}

	do
	{
		if(searchtarget->Sindex == target)
		{
			printsearch(searchtarget);
			/* printsearch sets lastsearch */
			return;
		}

		searchtarget = searchtarget->nextk;

		if(searchtarget == NULL)
		{
			searchtarget = translations[lowestkanji];
		}
	} while(searchtarget != lastsearch);

	setstatus("No match found");
	Beep();

	return;
}


/* process_kinput
 *	accepts a single 2-byte char as input.
 *	Adjusts the SEARCH_KANA_W widget labelstring appropriately
 *	(allowing backspacing)
 *	Also handles the mirror widget on the popup window
 *
 *
 *	We call the convert routine "romajitokana" on our internal
 *	string, to handle romaji-to-kana stuffs.
 *	Note that we get called by both the point-n-click-kana window,
 *	AND by the kepress-event handler for the window.
 */
void process_kinput(XChar2b in_char)
{
	XChar2b *kparse = kanastring;
	int kanalength;

	for(kanalength=0; kparse->byte1 != 0; kanalength++)
	{
		kparse++;
	}
	
	/* handle backspace */
	if((in_char.byte1 == 0x22) && (in_char.byte2 == 0x2b) )
	{
		if(kanalength==0)
			return;
		kanalength--;
		kanastring[kanalength].byte1 = 0;
		kanastring[kanalength].byte2 = 0;

		XtVaSetValues(searchwidgets[SEARCH_KANA_W],
		      XtNlabel, kanastring, NULL);
		XtVaSetValues(romajiinput,
		      XtNlabel, kanastring, NULL);

		return;
	
	}

	/* check for special "return" char */
	/* which menas "go find this string" */
	if((in_char.byte1 == paragraphglyph[0].byte1) &&
	   (in_char.byte2 == paragraphglyph[0].byte2))
	{
		if(kanalength == 0)
			return;
		/* accept.. search! */
		dokanafind(kanastring);
		return;
	}

	/* otherwise.. add to end! */

	if(kanalength<MAXKANALENGTH-1)
	{
		kanastring[kanalength].byte1 = in_char.byte1;
		kanastring[kanalength].byte2 = in_char.byte2;

		kanalength++;
		kanastring[kanalength].byte1 = 0;
		kanastring[kanalength].byte1 = 0;
	}
	romajitokana(kanastring);

	XtVaSetValues(searchwidgets[SEARCH_KANA_W],
		      XtNlabel, kanastring, NULL);
	XtVaSetValues(romajiinput,
		      XtNlabel, kanastring, NULL);
	
}

/* This is the only exported "make widgets" routine */
void MakeSearchPopup()
{
	 Widget searchform;
	 XtAccelerators Accel;

	search_popup = XtVaCreatePopupShell("kdrill_search",
		transientShellWidgetClass,
		toplevel,
		NULL);

	searchform = XtVaCreateManagedWidget("searchform",
					     formWidgetClass,
					     search_popup,
					     NULL);

	searchwidgets[NFORM_W] = XtVaCreateManagedWidget("northform",formWidgetClass,
			searchform,
			XtNborderWidth,2,
			NULL);

	maketopsearchstuff(searchwidgets[NFORM_W]);

	/* now make the long output display widgets */
	searchwidgets[KANA_W] = XtVaCreateWidget("searchkana",
						 labelWidgetClass,
						 searchform,
			XtNlabel,"                               ",
			XtNfromVert,searchwidgets[NFORM_W],
			XtNvertDistance,20,
			XtNencoding,XawTextEncodingChar2b,
			XtNfont,smallkfont,
			XtNwidth,FULLWIDTH,
			NULL);

	searchwidgets[ENGLISH_W] = XtVaCreateWidget("searchenglish",
						    labelWidgetClass,
						    searchform, 
			XtNlabel,"                                       ",
			XtNfromVert,searchwidgets[KANA_W],
			XtNfont,englishfont,
			XtNwidth,FULLWIDTH,
			NULL);

	makesearchinputs(searchform);



	XtManageChildren(searchwidgets,NUM_OF_W);

	Accel = XtParseAcceleratorTable(searchAccel);
	XtOverrideTranslations(searchwidgets[SEARCH_ENG_W],Accel);


	XtOverrideTranslations(searchnumbers[F_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[POUND_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[H_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[N_INPUT],Accel);
	XtOverrideTranslations(searchnumbers[U_INPUT],Accel);

	MakeKanainputPopup();
	MakeKanjiinputPopup();
}
