/* 
 * This file contains all callback functions for this client
 */

#include <errno.h>
#include <string.h>
#include <ctype.h>
#include "xfibs.h"
#include "callbacks.h"
#include "Fibw.h"
#include "FibwP.h"
#include "config.h"
#include "comms.h"
#include "parse.h"
#include "graphics.h"

String pending_list[50];	/* 50 entries should be enough */

struct gagnode {		/* See no evil, hear no evil... */
    char name[STRING_LENGTH];
    struct gagnode *next;
};
struct gagnode *firstgag;


struct hilitenode {
    char name[STRING_LENGTH];
    struct hilitenode *next;
};
struct hilitenode *firsthilite;

int	 number_of_pending_entries = 0;
int	 ready = 1;		/* Always ready before we connect */
int	 scroll = 1;		/* Scrolling input window? */
int	 away = 0;		/* Say 'away' after each command. (when away) */
int	 echo_input = 0;	/* Echo...     ...echo */
int	 blanklines = 0;	/* Allow blank lines in output window */
int	 login = 0;		/* Not used yet */
int	 password = 0;		/* Not used yet */
char	 board_desc[STRING_LENGTH];	/* Buffer incomplete boards */
int	 board_desc_pending = 0;

char	 logpath[STRING_LENGTH] = "";	/* Path to find log file */
FILE	*logfilefp = NULL;	/* Pointer to log file */
int	 logged_in = 0;		/* 1 if we are logged in */

char	 output_buffer[OUTBUFLENGTH];	/* output coming back from FIBS */
int	 first_time = 1;		/* flag to initialize buffer */

XtInputId socket_connection;

extern int print_ascii_board_to_file();

static void do_login()
{
    char cmd_string[STRING_LENGTH];

    if (user_name[0] != '\0') {
	sprintf(cmd_string, "%s\r\n", user_name);
	write(s, cmd_string, strlen(cmd_string));
    }
    else
	login = 1;
}

static void do_passwd()
{
    char cmd_string[STRING_LENGTH];

    if (pass_word[0] != '\0') {
	sprintf(cmd_string, "%s\r\n", pass_word);
	write(s, cmd_string, strlen(cmd_string));
    }
    else
	password = 1;
}


static int check_gag_list(name)
char *name;
{
    struct gagnode *p;

    if (!firstgag)
	return 0; /* empty gag-list */

    p = firstgag;
    do {
	if (strcmp(p->name, name) == 0)
	    return 1;
	p = p->next;
    } while (p);
    return 0;
}

static int check_hilite_list(line)
char *line;
{
    struct hilitenode *p;

    if (!firsthilite)
	return 0; /* empty hilite-list */

    p = firsthilite;
    do {
	if (strstr(line, p->name))
	    return 1;
	p = p->next;
    } while (p);
    return 0;
}


FILE *openLog(name)
char *name;
{
    char file[STRING_LENGTH];

    strcpy(file, logpath);
    if (*name == 0)
	/* Change this to xfibslog.date with todays date? */
	strcpy(name, "xfibs.logfile");

    strcat(file, name);
    /* If file already exists, this will append to the end */
    logfilefp = fopen(file, "a+");

    return(logfilefp);
}

void closeLog()
{
    if (logfilefp)
	fclose(logfilefp);
    logfilefp = NULL;
}


void output_text(line)
char *line;
{
    char	sourceptr[OUTBUFLENGTH];
    char	name[STRING_LENGTH];
    int		current_top;
    int		i, j, startpos, endpos;
    FibwWidget	fw = (FibwWidget) bgboard;
    
    /* If line is NULL, clear text field */
    if (!line) {
	XmTextSetString(outtext, "");
    }
    else {
	/* Check for non-printable characters, delete them */
	for (i = 0; i < strlen(line); i++)
	    if (!isascii(line[i])) {
		if (line[i] == '\007')	/* Bell */
		    XBell(XtDisplay(outtext), 100);
		for (j = i; j < strlen(line); j++)
		    line[j] = line[j + 1];
		    i--;
	    }
	/* Check to see if we're running out of space in the output buffer */
	startpos = XmTextGetLastPosition(outtext);
	if ((startpos + strlen(line) + 10) > OUTBUFLENGTH) {
	    /* Yes, need to make some more space */
	    current_top = startpos - XmTextGetTopCharacter(outtext);
	    strcpy(sourceptr, XmTextGetString(outtext));
	    /* remove some 10% of the text */
	    i = OUTBUFLENGTH/10;
	    while (sourceptr[i] != '\n')
		i++;
	    /* This works better than the old way - 'replace' from the start
	     * to the first \n after 10% with nothing!
	     */
	    XmTextReplace(outtext, 0, i, "");
	}

	startpos = XmTextGetLastPosition(outtext);
	i = strlen(line);
	endpos = startpos + i;

	XmTextInsert(outtext, startpos, line);
	XmTextSetInsertionPosition(outtext, XmTextGetLastPosition(outtext));

	/* should the line be highlighted ? */
	if (check_hilite_list(line)) 
	    XmTextSetHighlight(outtext, startpos, endpos, XmHIGHLIGHT_SELECTED);

	/* print copy of output line to logfile if logging is desired */
	if (logfilefp != NULL) {
	    if (!(fw->fibw.logStyle & LOGALL) &&
		!(fw->fibw.logStyle & LOGINPUT)) {
		if (!(fw->fibw.logStyle & LOGPLAY))
		    return;

		i = sscanf(line, "%s", name); 
		/* Just log stuff done by you, player or opponent */
		if ((strcmp(name, fw->fibw.fibs_rec.playername) != 0) &&
		    (strcmp(name, fw->fibw.fibs_rec.opponentname) != 0) &&
		    (strcmp(name, "You") != 0))
		    return;
	    }
	    fprintf(logfilefp, "%s", line);
	    fflush(logfilefp);
	}
    }
}


int parseInput(line, first)
char	*line;
int	 first;
{
    char	name1[STRING_LENGTH],
		name2[STRING_LENGTH],
		dummy1[STRING_LENGTH],
		dummy2[STRING_LENGTH],
		dummy3[STRING_LENGTH];	/* used to scan incoming string */
    int		value1, value2;		/* values for dice rolls etc. */
    int		i, n;			/* counters */
    int		add;			/* show in outtext window? */
    FibwWidget	fw = (FibwWidget) bgboard;

    /* 
     * Do some simple parsing. Basically all I do is check the 
     * incoming line from FIBS whether it corresponds to certain
     * known ones. Like "Please move 3 pieces" is checked by looking 
     * for "Please move" at the start of each line. It that phrase is
     * found the number 3 is fetched and used in the mouse moving routine
    */
    add = 1;
    
    /* If it's a simple '> ' mark the server as ready */
    /* Or if its a "You're marked away..."	    */
    if ((strlen(line) > 1) && (strcmp(&line[strlen(line) - 2], "> ") == 0)) {
	if (!logged_in) {
	    do_login_cmds();
	    logged_in = 1;
	}
	if (!ready) {
	    /* The NULL indicates that we'll just try to send 
	     * something if there are commands pending
	    */
	    ready = 1;
	    XUndefineCursor(XtDisplay(form), XtWindow(form));
	    XUndefineCursor(XtDisplay(textform), XtWindow(textform));
	    add_to_pending_list((String)NULL);
	}
	if ((strstr(line, "away")) && (!away)) {
	    away = 1;
	    add = 0;
	}
	else
	    add = 0;
    }

    /* Delete the '> ' at the beginning of line, the rest may be usefull */
    if (strlen(line) > 2 && strncmp(line, "> ", 2) == 0)
	line += 2;

    /* is it a board layout? */
    if (strncmp(line, "board:", 6) == 0) {
	FibwSetInfo(fw, line, fi_board);
	if ((fw->fibw.logStyle & LOGBOARDS) && logfilefp) {
	    print_ascii_board_to_file(logfilefp, fw->fibw.lastboard);
	    fflush(logfilefp);
	}
	add = 0;
    }

    /* login time? */
    if (strncmp(line, "login:", 6) == 0) {
	/* do the right thing... */
	do_login();
	add = (login) ? 1 : 0;
    }
    
    /* prompting for password? */
    if (strncmp(line, "password:", 6) == 0) {
	do_passwd();
	add = (password) ? 1 : 0;
    }

    /* Are we back ? */
    if (strncasecmp(line, "Welcome back", 12) == 0) {
	away = 0;
	add = 1;
    }

    /* is it a dice roll */
    n = sscanf(line, "%s%s%i%s%i", name1, dummy1, &value1, dummy2, &value2);
    if ((n == 5) && ((strcmp(dummy1, "rolls") == 0) || 
		  (strcmp(dummy1, "roll") == 0))) {
	/* got a roll */
	FibwDiceRolled(fw, name1, (char *)NULL,
		value1, value2);
	add = 1;
    }

    /* it might be the first roll... */
    n = sscanf(line, "%s%s%i%s%s%s%i", name1, dummy1, &value1, dummy2,
				   name2, dummy3, &value2);
    if ((n == 7) && (strcmp(dummy1, "rolled") == 0)) {
	/* got first roll */
	FibwDiceRolled(fw, name1, name2, value1, value2);
	add = 1;
    }
    
    /* 
     * May we move some pieces? pleeeasse...?  
     * Get the number of moves...
    */
    n = sscanf(line, "%s%s%i", dummy1, dummy2, &value1); 
    if ((n == 3) && (strcmp(dummy1, "Please") == 0) &&
	    (strcmp(dummy2, "move") == 0)) {
	FibwSetMoves(fw, value1);
	add = 1;
    }

    /*
     * If it's the first roll, the line has a somewhat different look
     * Like: "It's your turn. Please move ..."
     * I guess I have to catch that one also :)
    */
    if (strncasecmp(line, "It's your turn to move", 22) == 0) {
	/* As it is the first roll, we get to move 2 pieces.. */
	FibwSetMoves(fw, 2);
	add = 1;
    }

    /* Is someone doubling ? */
    n = sscanf(line, "%s", name1);
    if (strstr(line, "double. Please wait") ||
	strstr(line, "doubles. Type")) {
	/* Yup, he/she/we are doubling */
	FibwDouble(fw, name1);
	add = 1;
    }

    /* Rejection? */
    if (strstr(line, "gives up.")) {
	/* Got it! */
	FibwReject(fw, name1);
	add = 1;
    }

    /* or acceptance? */
    if (strstr(line, "accept the double") ||
	strstr(line, "accepts the double")) { 
	/* Got that too... */
	FibwAccept(fw, name1);
	add = 1;
    }

    /* Opponent resigns */
    if (strstr(line, "wants to resign")) {
	/* Yup, he's a coward... :) */
	FibwResign(fw, name1);
	add = 1;
    }

    /* Started to play or watching?  Ask for a board */
    if (strstr(line, "has joined you") ||
	(strncmp(line, "You're now watching", 15) == 0)) {
	strcpy (dummy1, "board");
	add_to_pending_list(dummy1);
	add = 1;
    }

    /* May we roll ? */
    if (strncasecmp(line, "It's your turn.", 15) == 0) {
	FibwRoll(fw);
	add = 1;
    }

    /* Next game ?? */
    if (strncmp(line, "Type 'join' if you want", 23) == 0) {
	FibwJoin(fw, (char *)NULL);
	add = 1;
    }

    /* Got invited ? */
    n = sscanf(line, "%s%s%s",dummy1,dummy2,name1);
    if ((strcmp(dummy1, "Type") == 0) && 
	(strcmp(&dummy2[1], "join") == 0)) {
	name1[strlen(name1) - 1] = '\0';
	FibwJoin(fw, name1);
	add=1;
    }
    
    /* Checking for gagged shouts... */
    n = sscanf(line, "%s%s", name1, dummy1);
    if ((n == 2) && (strcmp(dummy1, "shouts:") == 0)) {
	if (check_gag_list(name1)) {
	    add = 0;	/* Got one! */
	}
	else
	    add = 1;	/* Let him through... */
    }
    
    /* add text to output window */
    if (add) {
	if (blanklines == 1) {	/* if doing blank lines */
	    if (first == 1 && logged_in == 1) {
		/* print extra when necessary */
		if (strlen(line) > 0 && line[0] != '\n')
		    output_text("\n");
	    }

	    output_text(line);          /* print regular line */
	}

	else {
	    if (line[0] != '\n') {
		/* print extra when necessary */
		if (first == 1 && away == 1)
		    output_text("\n");
		output_text(line);
	    }
	}
    }
    return (add);
}

/* Called when text is received from FIBS */
/*ARGSUSED*/
void incoming_text(unused, fid, id)
XtPointer	 unused;
int		*fid;
XtInputId	*id;
{
    /*
     * I probably have to clean up a bit... I seem to have added a lot 
     * of variables during the development :) 
    */
    char	*buf, *storedbufptr;	/* pointer to start of incoming text */
    char	*line;			/* pointer to current line */
    char	*cp;			/* yet another pointer */
    int		nbytes;			/* number of incoming bytes */
    int		i, n;			/* counters */
    int		not_through_yet=1;	/* still more to parse? */
    int		first = 0;		/* flag multiple line inputs */
    FibwWidget	fw = (FibwWidget) bgboard;

    buf = (char *)XtMalloc(sizeof(char)*LINELENGTH);
    storedbufptr = buf; /* used to free it later on */

    /* Incomplete lines are kept in output_buffer */
    if (first_time) {
	output_buffer[0] = '\0';
	first_time = 0;
    }

    nbytes = read(*fid, buf, LINELENGTH-1);
    
    if (nbytes < 0) {
	strcpy(buf, "Network problems: ");
	strcat(buf, Strerror(errno));
	i = strlen(buf);
	buf[i] = '\n';
	buf[i + 1] = '\0';
	output_text(buf);
	s = 0;
	XtRemoveInput(socket_connection);
	logged_in = 0;
	XtFree((char *)storedbufptr);
	return;
    }
    if (nbytes == 0) { /* Server has closed connection... */
	strcpy(buf, "Connection closed by foreign host...\r\n");
	output_text(buf);
	s = 0;
	XtRemoveInput(socket_connection);
	XtFree((char *)storedbufptr);
	return;
    }

    buf[nbytes] = '\0';	/* Add a \0 at the end, just to be certain */

    /* when logged in, buildup output until prompt is found, then process */
    if (logged_in == 1) {
	strcat(output_buffer, buf);	/* append string, no prompt go back */
	if (!((strlen(output_buffer) > 1) &&
	    (strcmp(&output_buffer[strlen(output_buffer) - 2], "> ") == 0))) {
	    XtFree((char *)storedbufptr);
	    return;
	}
	buf = output_buffer;		/* setup to process total string */
    }

    /* split the input into lines */
    while ((not_through_yet) && (strlen(buf) > 0)) {
	/* Are we at the start of an incomplete board description? */
	if (strncmp(buf, "board:",6) == 0) {
	    n = 0;
	    for (i = 0; i < strlen(buf); i++)
		if (buf[i] == ':')
		    n++;
	    if (n < 52){
		/* Yes, check buffer description */
		strncpy(board_desc, buf, strlen(buf));
		board_desc_pending = 1;
#if 0
		printf("Incomplete description, n = %d\n", n);
#endif
		break;	/* We must be at the end of the buffer */
	    }
	}

	/* See if there are any /r/n-pairs... */
	if (cp = strstr(buf, "\r\n")) {	/* yes, there are... */
	    line = buf;		/* set line to point to start of buffer */
	    cp[0] = '\n'; cp++;	/* end the line with \n\0	       */
	    cp[0] = '\0'; cp++;
	    buf = cp;		/* buffer points to remaing data	*/
	}
	else {
	    line = buf;		/* set line to point to start of buffer */
	    line[strlen(line)] = '\0';
	    not_through_yet = 0;
	}

	if (parseInput(line, first))
	    first++;
#ifdef WAS
	/* 
	 * Do some simple parsing. Basically all I do is check the 
	 * incoming line from FIBS whether it corresponds to certain
	 * known ones. Like "Please move 3 pieces" is checked by looking 
	 * for "Please move" at the start of each line. It that phrase is
	 * found the number 3 is fetched and used in the mouse moving routine
	*/
	add = 1;
	
	/* If it's a simple '> ' mark the server as ready */
	/* Or if its a "You're marked away..."	    */
	if ((strlen(line) > 1) && (strcmp(&line[strlen(line) - 2], "> ") == 0)) {
	    if (!logged_in) {
		do_login_cmds();
		logged_in = 1;
	    }
	    if (!ready) {
		/* The NULL indicates that we'll just try to send 
		 * something if there are commands pending
		*/
		ready = 1;
		XUndefineCursor(XtDisplay(form), XtWindow(form));
		XUndefineCursor(XtDisplay(textform), XtWindow(textform));
		add_to_pending_list((String)NULL);
	    }
	    if ((strstr(line, "away")) && (!away)) {
		away = 1;
		add = 0;
	    }
	    else
		add = 0;
	}
   
	/* Delete the '> ' at the beginning of line, the rest may be usefull */
	if (strlen(line) > 2 && strncmp(line, "> ", 2) == 0)
	    line += 2;

	/* is it a board layout? */
	if (strncmp(line, "board:", 6) == 0) {
	    FibwSetInfo(fw, line, fi_board);
	    if ((fw->fibw.logStyle & LOGBOARDS) && logfilefp) {
		print_ascii_board_to_file(logfilefp, fw->fibw.lastboard);
		fflush(logfilefp);
	    }
	    add = 0;
	}

	/* login time? */
	if (strncmp(line, "login:", 6) == 0) {
	    /* do the right thing... */
	    do_login();
	    add = (login) ? 1 : 0;
	}
	
	/* prompting for password? */
	if (strncmp(line, "password:", 6) == 0) {
	    do_passwd();
	    add = (password) ? 1 : 0;
	}

	/* Are we back ? */
	if (strncasecmp(line, "Welcome back", 12) == 0) {
	    away = 0;
	    add = 1;
	}

	/* is it a dice roll */
	n = sscanf(line, "%s%s%i%s%i", name1, dummy1, &value1, dummy2, &value2);
	if ((n == 5) && ((strcmp(dummy1, "rolls") == 0) || 
		      (strcmp(dummy1, "roll") == 0))) {
	    /* got a roll */
	    FibwDiceRolled(fw, name1, (char *)NULL,
		    value1, value2);
	    add = 1;
	}

	/* it might be the first roll... */
	n = sscanf(line, "%s%s%i%s%s%s%i", name1, dummy1, &value1, dummy2,
				       name2, dummy3, &value2);
	if ((n == 7) && (strcmp(dummy1, "rolled") == 0)) {
	    /* got first roll */
	    FibwDiceRolled(fw, name1, name2, value1, value2);
	    add = 1;
	}
	
	/* 
	 * May we move some pieces? pleeeasse...?  
	 * Get the number of moves...
	*/
	n = sscanf(line, "%s%s%i", dummy1, dummy2, &value1); 
	if ((n == 3) && (strcmp(dummy1, "Please") == 0) &&
		(strcmp(dummy2, "move") == 0)) {
	    FibwSetMoves(fw, value1);
	    add = 1;
	}

	/*
	 * If it's the first roll, the line has a somewhat different look
	 * Like: "It's your turn. Please move ..."
	 * I guess I have to catch that one also :)
	*/
	if (strncasecmp(line, "It's your turn to move", 22) == 0) {
	    /* As it is the first roll, we get to move 2 pieces.. */
	    FibwSetMoves(fw, 2);
	    add = 1;
	}

	/* Is someone doubling ? */
	n = sscanf(line, "%s", name1);
	if (strstr(line, "double. Please wait") ||
	    strstr(line, "doubles. Type")) {
	    /* Yup, he/she/we are doubling */
	    FibwDouble(fw, name1);
	    add = 1;
	}

	/* Rejection? */
	n = sscanf(line, "%s", name1);
	if (strstr(line, "gives up.")) {
	    /* Got it! */
	    FibwReject(fw, name1);
	    add = 1;
	}

	/* or acceptance? */
	n = sscanf(line, "%s", name1);
	if (strstr(line, "accept the double") ||
	    strstr(line, "accepts the double")) { 
	    /* Got that too... */
	    FibwAccept(fw, name1);
	    add = 1;
	}

	/* Opponent resigns */
	n = sscanf(line, "%s", name1);
	if (strstr(line, "wants to resign")) {
	    /* Yup, he's a coward... :) */
	    FibwResign(fw, name1);
	    add = 1;
	}

	/* May we roll ? */
	if (strncasecmp(line, "It's your turn.", 15) == 0) {
	    FibwRoll(fw);
	    add = 1;
	}

	/* Next game ?? */
	if (strncmp(line, "Type 'join' if you want", 23) == 0) {
	    FibwJoin(fw, (char *)NULL);
	    add = 1;
	}

	/* Started watching?  Ask for a board */
	if (strncmp(line, "You're now watching", 15) == 0) {
	    strcpy (dummy1, "board");
	    add_to_pending_list(dummy1);
	    add = 1;
	}
	/* Got invited ? */
	n = sscanf(line, "%s%s%s",dummy1,dummy2,name1);
	if ((strcmp(dummy1, "Type") == 0) && 
	    (strcmp(&dummy2[1], "join") == 0)) {
	    name1[strlen(name1) - 1] = '\0';
	    FibwJoin(fw, name1);
	    add=1;
	}
	
	/* Checking for gagged shouts... */
	n = sscanf(line, "%s%s", name1, dummy1);
	if ((n == 2) && (strcmp(dummy1, "shouts:") == 0)) {
	    if (check_gag_list(name1)) {
		add = 0;	/* Got one! */
	    }
	    else
		add = 1;	/* Let him through... */
	}
	
	/* add text to output window */
	if (add) {
	    first++;

	    if (blanklines == 1) {	/* if doing blank lines */
		if (first == 1 && logged_in == 1) {
		    /* print extra when necessary */
		    if (strlen(line) > 0 && line[0] != '\n')
			output_text("\n");
		}

		output_text(line);          /* print regular line */
	    }

	    else {
		if (line[0] != '\n') {
		    /* print extra when necessary */
		    if (first == 1 && away == 1)
			output_text("\n");
		    output_text(line);
		}
	    }
	}
#endif
    }
    /* reset buffer to empty */
    output_buffer[0] = '\0';

    XtFree((char *)storedbufptr);
}


void add_to_gag_list(name)
char *name;
{
    struct gagnode *p;
    
    if (*name == '\0')
	return;

    if (!firstgag) { 
	firstgag=XtNew(struct gagnode);
	strcpy(firstgag->name,name);
	firstgag->next= (struct gagnode *)NULL;
	return;
    }
    
    p= firstgag;

    while(p->next) { /*search for last node */
	p=p->next;
    }

    /* allocate space for another node */
    p->next=XtNew(struct gagnode);
    strcpy(p->next->name,name);
    p->next->next=NULL;
}


void remove_from_gag_list(name)
char *name;
{
    struct gagnode *p;
    struct gagnode *last;

    if (!firstgag)
	return; /* empty gag-list */

    if (*name == '\0')
	return;

    p = firstgag;
    last = p;

    do {
	if (strcmp(p->name, name) == 0) { /* found what we where looking for */
	    /* is it the only node in the list? */
	    if ((p == firstgag) &&(p->next == (struct gagnode *)NULL)) {
		XtFree((char *)firstgag);
		firstgag = (struct gagnode *)NULL;
		return;
	    }
	    /* is it the first node in the list? */
	    if (p == firstgag) {
		firstgag = p->next; /* update starting point of list */
		XtFree((char *)p);
		return;
	    }
	    /* it's in the middle somewhere, or at the end */
	    last->next=p->next;
	    XtFree((char *)p);
	    return;
	}

	/* Haven't found right name yet, keep looking */
	last = p;
	p = p->next;
    } while (p);
}


static void list_gag_list()
{
    struct gagnode *p;
    char line[STRING_LENGTH];

    if (!firstgag) {
	strcpy(line, "No-one gagged\n");
	output_text(line);
	return; /* empty gag-list */
    }

    p = firstgag;
    do {
	if (p->name)
	    strcpy(line, "gagged: ");
	    strcat(line, p->name);
	    strcat(line, "\n");
	    output_text(line);
	p=p->next;
    } while (p);
}


void add_to_hilite_list(name)
char *name;
{
    struct hilitenode *p;
    
    if (*name == '\0')
	return;

    if (!firsthilite) { 
	firsthilite = XtNew(struct hilitenode);
	strcpy(firsthilite->name, name);
	firsthilite->next = (struct hilitenode *)NULL;
	return;
    }
    
    p = firsthilite;

    while(p->next) { /*search for last node */
	p = p->next;
    }

    /* allocate space for another node */
    p->next = XtNew(struct hilitenode);
    strcpy(p->next->name, name);
    p->next->next = NULL;
}


void remove_from_hilite_list(name)
char *name;
{
    struct hilitenode *p;
    struct hilitenode *last;

    if (!firsthilite)
	return; /* empty hilite-list */

    if (*name == '\0')
	return;

    p = firsthilite;
    last = p;

    do {
	if (strcmp(p->name, name) == 0) { /* found what we where looking for */
	    /* is it the only node in the list? */
	    if ((p == firsthilite) &&(p->next == (struct hilitenode *)NULL)) {
		XtFree((char *)firsthilite);
		firsthilite = (struct hilitenode *)NULL;
		return;
	    }
	    /* is it the first node in the list? */
	    if (p == firsthilite) {
		firsthilite = p->next; /* update starting point of list */
		XtFree((char *)p);
		return;
	    }
	    /* it's in the middle somewhere, or at the end */
	    last->next = p->next;
	    XtFree((char *)p);
	    return;
	}

	/* Haven't found right name yet, keep looking */
	last = p;
	p = p->next;
    } while (p);
}


static void list_hilite_list()
{
    struct hilitenode *p;
    char line[STRING_LENGTH];

    if (!firsthilite) {
	strcpy(line, "Nothing highlighted\n");
	output_text(line);
	return; /* empty hilite-list */
    }

    p = firsthilite;
    do {
	if (p->name)
	    strcpy(line, "highlighted: ");
	    strcat(line, p->name);
	    strcat(line, "\n");
	    output_text(line);
	p = p->next;
    } while (p);
}


void initPendingList()
{
    int i;

    number_of_pending_entries = 0;
    for (i = 0; i < 50; i++)
	pending_list[i]=(String)NULL;
}

static void connect_to_server()
{
    char buf[STRING_LENGTH];	/* I hope this will be enough... */

    if (s) {
	strcpy(buf, "We're already connected, my friend\n");
	output_text(buf);
	return;
    }
    socketinit();
    socket_connection = XtAppAddInput(app, s, (XtPointer)XtInputReadMask,
	    incoming_text,NULL);
    
    initPendingList();	/* Reset pending commands queue */
}


static void do_command(cmd_string)
char *cmd_string;
{
    char	someword[STRING_LENGTH];
    char	buf[STRING_LENGTH];	/* I hope this will be enough... */
    FibwWidget	fw = (FibwWidget) bgboard;

    if (fw->fibw.debug == 3)
	fprintf(stderr, "Sending: %s\n", cmd_string);

    sscanf(cmd_string, "%s", someword);
    if (strcasecmp(someword, "connect") == 0) { 
	connect_to_server();
	return;
    }
    
    if ((strcasecmp(someword, "disconnect") == 0) ||
	(strcasecmp(someword, "logout") == 0)){
	if (s) {
	    write(s, "bye\r\n", strlen("bye\r\n"));
	}
	logged_in = 0;		/* no longer logged in */
	return;
    }

    if ((strcasecmp(someword, "exit") == 0) ||
	(strcasecmp(someword, "quit") == 0) ||
	(strcasecmp(someword, "bye") == 0)) {
	if (s) {
	    write(s, "bye\r\n", sizeof("bye\r\n"));
	    closeLog();
	    socketclose();
	    XtRemoveInput(socket_connection);
	}
	exit(0);
    }

    if (!s) {
	strcpy(buf, "You must (Re)Connect to the server!\n");
	output_text(buf);
	return;
    }
    if (write(s, cmd_string, strlen(cmd_string)) < 0) {
	strcpy(buf, "Minor network problem: ");
	strcat(buf, Strerror(errno));
	buf[strlen(buf) + 1] = '\0';
	output_text(buf);
	fprintf(stderr, "Write to socket %d\n", s);
    }
    if (cmd_string[strlen(cmd_string)] != '\n')
	write(s, "\r\n", strlen("\r\n"));

    /* if echoing input, output it to screen, maybe after blank line */
    if (echo_input == 1) {
	if (blanklines == 1 && away == 0)
	    output_text("\n");

	output_text("> ");
	output_text(cmd_string);
	if (away == 0)
	    output_text("\n");
	else output_text(" ");
    }
}


/*ARGSUSED*/
void mnu_callback(w, command_string)
Widget	 w;
char	*command_string;
{
    char ellipsis_cmd[STRING_LENGTH];
    int cmd_len = strlen(command_string) - 3;

    if (strncmp(&command_string[cmd_len], "...", 3) == 0) {
	strncpy(ellipsis_cmd, command_string, cmd_len);
	ellipsis_cmd[cmd_len] = '\0'; /* rjohnson */
	XmTextInsert(intext, XmTextGetLastPosition(intext), ellipsis_cmd);
	/* put cursor after ellipsis command in input window correctly */
	XmTextSetInsertionPosition(intext, XmTextGetLastPosition(intext) +
					strlen(ellipsis_cmd));
    }
    else
	add_to_pending_list(command_string);
}



/* called when new text is entered in the input area */

/*ARGSUSED*/
void added_text(text_w, unused, cbs)
Widget		text_w;
XtPointer	unused;
XmTextVerifyCallbackStruct *cbs;
{
    int		i;
    char	sendbuffer[LINELENGTH];

    if (cbs->text->length >= 1) {
	if (cbs->text->ptr[cbs->text->length - 1] == '\n') {
	    strncpy(sendbuffer,XmTextGetString(intext),LINELENGTH);      
	    if (strlen(sendbuffer)>=LINELENGTH)
		sendbuffer[LINELENGTH-1] = '\0';

	    else
		sendbuffer[strlen(sendbuffer)+1]='\0';

	    /* get beginning of line... */
	    i = strlen(sendbuffer) - 1;
	    while ((i >= 0) && (sendbuffer[i] != '\n')) 
		i--;
	    /* Now send it */
	    if (!scroll) {
		cbs->doit = False;
		XmTextSetString (intext, "");
	    }
	    if ((login) || (password)) {
		do_command(&sendbuffer[i + 1]);
		login = 0;
	    }
	    else
		add_to_pending_list(&sendbuffer[i + 1]);
	}
	/* Snoopy's cntl-P -> send redo command to FIBS */
	if (cbs->text->ptr[0] == '') {
	    do_command("!!");
	    cbs->text->length = 0;
	}
    }
}

static void xfibsCommand(command)
char *command;
{
    int		i;
    int		value1, value2;
    char	opp[STRING_LENGTH], roll1[STRING_LENGTH];
    char	dummy[STRING_LENGTH], name[STRING_LENGTH];
    char	board[STRING_LENGTH];
    char	*parameter;
    FibwWidget	fw = (FibwWidget) bgboard;

    /* En/Disable scrolling input */
    if (strncasecmp(command, "scroll", 6) == 0)
	scroll = 1;
    else if (strncasecmp(command, "noscroll", 8) == 0)
	scroll = 0;

    /* En/Disable colour board */
    else if ((strncasecmp(command, "colour", 6) == 0) ||
	(strncasecmp(command, "color", 5) == 0)) {
	usecolor = 1;
	use_colors(fw);
    }
    else if ((strncasecmp(command, "nocolour", 8) == 0) ||
	(strncasecmp(command, "nocolor", 7) == 0)) {
	usecolor = 0;
	use_colors(fw);
    }

    /* Clear output screen */
    else if (strncasecmp(command, "cls", 3) == 0)
	output_text((char *)NULL);

    /* Commands to open & close logfile */
    else if (strncasecmp(command, "logfile", 7) == 0) {
	/* Check the file isn't already open */
	if (logfilefp == NULL) {
	    i = sscanf(command, "%s%s", dummy, name);
	    if (i != 2)
		strcpy(name, "");

	    if (!openLog(name))
		sprintf(dummy, "Error: couldn't open file %s\n", name);
	    else sprintf(dummy, "Opened file: %s\n", name);
	}
	else sprintf(dummy, "Error: %s already open.\n", name);

	/* Tell 'em what happened */
	output_text(dummy);
    }

    /* close log file */
    else if (strncasecmp(command, "nologfile", 9) == 0)
	closeLog();

    else if (strncasecmp(command, "logpath", 7) == 0) {
	i = sscanf(command, "%s%s", dummy, name);
	if (i != 2) {
	    sprintf(dummy, "logpath is \"%s\"\n", logpath);
	    output_text(dummy);
	}
	else
	    strcpy(logpath, name);
    }
    else if (strncasecmp(command, "nologpath", 9) == 0) {
	strcpy(logpath, "");
	sprintf(dummy, "logpath cleared\n");
	output_text(dummy);
    }

    /* Undo last move */
    else if (strncasecmp(command, "undo", 4) == 0)
	undoLastMove(fw);

    /* En/Disable board numbering */
    else if (strncasecmp(command, "number", 6) == 0)
	use_numbers(1);
    else if (strncasecmp(command, "nonumber", 8) == 0)
	use_numbers(0);

    /* De/Select score on edge of board */
    else if ((strncasecmp(command, "frame", 5) == 0) ||
	(strncasecmp(command, "noside", 6) == 0))
	use_board(1);
    else if ((strncasecmp(command, "noframe", 7) == 0) ||
	(strncasecmp(command, "side", 4) == 0))
	use_board(0);

    /* Do a pipcount for the brain-damaged */
    else if (strncasecmp(command, "pip", 3) == 0)
	pipCount(fw);

    /* Dump the current board to the logfile */
    else if (strncasecmp(command, "dump", 4) == 0) {
	if (logfilefp) {
	    print_ascii_board_to_file(logfilefp, fw->fibw.lastboard);
	    fflush(logfilefp);
	}
    }

    /* En/Disable debugging */
    else if (strncasecmp(command, "debug", 5) == 0) {
	i = sscanf(command, "%s%i", dummy, &value1);
	if (i == 1)
	    value1 = 1;
	fw->fibw.debug = value1;
    }
    else if (strncasecmp(command, "nodebug", 7) == 0)
	fw->fibw.debug = 0;

    /* Debug rolls */
    else if (strncasecmp(command, "roll", 4) == 0) {
	/* simulate a roll */
	i = sscanf(command, "%s%i%i%s%s", dummy, &value1, &value2, opp, roll1);
	if (i == 3)
	    FibwDiceRolled(fw, "You", (char *)NULL, value1, value2);
	else if (i == 4) {
	    if (strncasecmp(opp, "0", 1) == 0) {
		FibwDiceRolled(fw, "You", (char *)NULL, value1, value2);
		FibwSetMoves(fw, 1);
		return;
	    }
	    else FibwDiceRolled(fw, opp, (char *)NULL, value1, value2);
	}
	else FibwDiceRolled(fw, opp, roll1, value1, value2);
	if (value1 == value2)
	    FibwSetMoves(fw, 4);
	else FibwSetMoves(fw, 2);
    }

    /* What should we log? */
    else if (strncasecmp(command, "log", 3) == 0) {
	i = sscanf(command, "%s%s%s", dummy, name, roll1);
	value2 = 0;		/* Add in mode or overlay */
	if (i == 3) {
	    if (strcasecmp(name, "+") != 0)
		return;
	    strcpy(name, roll1);
	    i--;
	    value2 = 1;
	}
	if (i == 2) {
	    value1 = 0;
	    if (strcasecmp(name, "all") == 0)
		value1 = LOGALL;
	    else if (strcasecmp(name, "input") == 0)
		value1 = LOGINPUT;
	    else if (strcasecmp(name, "play") == 0)
		value1 = LOGPLAY;
	    else if (strcasecmp(name, "board") == 0)
		value1 = LOGBOARDS;
	    else if (strcasecmp(name, "none") == 0)
		fw->fibw.logStyle = LOGNONE;

	    if (value1) {
		if (value2)
		    fw->fibw.logStyle |= value1;
		else fw->fibw.logStyle = value1;
	    }
	}
    }

    /* Debug available moves */
    else if (strncasecmp(command, "avail", 5) == 0)
	debugAvail(fw);

    /* Debug first move */
    else if (strncasecmp(command, "move1", 5) == 0) {
	i = sscanf(command, "%s%i", dummy, &value1);
	debugFirst(fw, value1);
    }

    /* Debug board */
    else if (strncasecmp(command, "fibs", 4) == 0) {
	parameter = command + 5;
	parseInput(parameter, 0);
    }

    /* Debug board */
    else if (strncasecmp(command, "board", 5) == 0) {
	i = sscanf(command, "%s%s", dummy, board);
	if (i == 1)
	    debugBoard((char *)NULL);
	else
	    debugBoard(board);
    }

    /* gag commands... */
    else if (strncasecmp(command, "gag", 3) == 0) {
	i = sscanf(command, "%s%s", dummy, name);
	if (i == 2) 
	    add_to_gag_list(name);	/* add to gag list */
	else 
	    list_gag_list();
    }
    else if ((strncasecmp(command, "ungag", 5) == 0) ||
	(strncasecmp(command, "nogag", 6) == 0)) {
	i = sscanf(command, "%s%s", dummy, name);
	if (i == 2)
	    remove_from_gag_list(name);	/* remove from gag list */
    }

    /* highlight commands... */
    else if (strncasecmp(command, "hilite", 6) == 0) {
	i = sscanf(command, "%s%s", dummy, name);
	if (i == 2) 
	    add_to_hilite_list(name);	/* add to hilite list */
	else 
	    list_hilite_list();
    }
    else if (strncasecmp(command, "nohilite", 8) == 0) {
	i = sscanf(command, "%s%s", dummy, name);
	if (i == 2)
	    remove_from_hilite_list(name);
    }

    /* blank line suppression... */
    else if (strncasecmp(command, "echo", 4) == 0)
	echo_input = 1;
    else if (strncasecmp(command, "noecho", 6) == 0)
	echo_input = 0;

    /* blank line suppression... */
    else if (strncasecmp(command, "blanks", 6) == 0)
	blanklines = 1;
    else if (strncasecmp(command, "noblanks", 8) == 0)
	blanklines = 0;

    else {
	i = sscanf(command, "%s", name);
	sprintf(dummy, "Unknown command: %s.\n", name);

	/* Tell 'em what happened */
	output_text(dummy);
    }
}

void add_to_pending_list(cmd_string)
String cmd_string;
{
    int		i;
    FibwWidget	fw = (FibwWidget) bgboard;

    if (fw->fibw.debug == 3)
	fprintf(stderr, "Pending: %s\n", cmd_string);
    if (cmd_string) {
	/*  Catch !! as a FIBS command, any other is a system command */
	if (cmd_string[0] == '!' && cmd_string[1] != '!') {
	    system(&cmd_string[1]);
	    return;
	}

	/* Check for user-input to xfibs, rather than FIBS */
	if (cmd_string[0] == '/' || cmd_string[0] == '\\') {
	    /* Step over xfibs / or \. Just look at command! */
	    xfibsCommand(&cmd_string[1]);
	    return;
	}
    }

    /* 
     * If the command is "connect" or "exit" and there's no connection,
     * we just issue the command...
    */
    if ((!s) && ((strcasecmp(cmd_string, "connect") == 0) ||
		(strcasecmp(cmd_string, "exit") == 0)	  ||
		(strcasecmp(cmd_string, "bye") == 0)	  ||
		(strcasecmp(cmd_string, "quit") == 0))) {
	do_command(cmd_string);
	ready = 0;
	logged_in = 0;		/* no longer logged in */
	XDefineCursor(XtDisplay(form), XtWindow(form), cursor);
	XDefineCursor(XtDisplay(textform), XtWindow(textform), textCursor);
	return;
    }
       
    if (!s) return; /* If we ain't connected, we won't add any commands to
		       the list */
	
    if (cmd_string) {
	pending_list[number_of_pending_entries] = XtNewString(cmd_string);
	number_of_pending_entries++;
    }

    /* Is the server ready? */
    if (ready && number_of_pending_entries) {
	do_command(pending_list[0]);
	XtFree((char *)pending_list[0]);
	for (i = 0; i < number_of_pending_entries; i++) 
	    pending_list[i] = pending_list[i + 1];
	pending_list[number_of_pending_entries] = (String)NULL;
	number_of_pending_entries--;
	ready = 0;
	XDefineCursor(XtDisplay(form), XtWindow(form), cursor);
	XDefineCursor(XtDisplay(textform), XtWindow(textform), textCursor);
    }
}
