
/*-
# X-BASED RUBIK'S CUBE(tm)
#
#  Rubik.c
#
###
#
#  Copyright (c) 1994 - 99	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Rubik */

#include <stdio.h>
#include <stdlib.h>
#ifdef VMS
#include <unixlib.h>
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "RubikP.h"
#include "Rubik2dP.h"
#include "Rubik3dP.h"

#ifndef DATAFILE
#define DATAFILE "/usr/games/lib/rubik.data"
#endif

static void InitializeRubik(Widget request, Widget renew);
static void DestroyRubik(Widget old);
static Boolean SetValuesRubik(Widget current, Widget request, Widget renew);

static Bool checkFaceSquare(RubikWidget w, int face);
static void SetAllColors(RubikWidget w, Boolean init);
static void GetColor(RubikWidget w, int face, Boolean init);
static void MoveControlCb(RubikWidget w, int face, int direction);
static void CheckPolyhedrons(RubikWidget w);
static Boolean SelectPolyhedrons(RubikWidget w,
				 int x, int y, int *face, int *position);
static Boolean NarrowSelection(RubikWidget w,
			       int *face, int *position, int *direction);
static Boolean PositionPolyhedrons(RubikWidget w,
		     int x, int y, int *face, int *position, int *direction);
static void MoveNoPolyhedrons(RubikWidget w);
static void PracticePolyhedrons(RubikWidget w);
static void RandomizePolyhedrons(RubikWidget w);
static void MovePolyhedrons(RubikWidget w,
			    int face, int position, int direction);

/* rc : row or column */
static void ReadRC(RubikWidget w, int face, int dir, int h, int orient, int size);
static void RotateRC(RubikWidget w, int rotate, int orient, int size);
static void ReverseRC(RubikWidget w, int orient, int size);
static void WriteRC(RubikWidget w, int face, int dir, int h, int orient, int size);
static void RotateFace(RubikWidget w, int face, int direction);
static void DrawSquare(RubikWidget w, int face, int position, int offset);
static int  CheckMoveDir(RubikWidget w, int face,
			 int position1, int position2, int *direction);

RubikClassRec rubikClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		"Rubik",	/* class name */
		sizeof (RubikRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeRubik,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		NULL,		/* actions */
		0,		/* num actions */
		NULL,		/* resources */
		0,		/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyRubik,	/* destroy */
		NULL,		/* resize */
		NULL,		/* expose */
		(XtSetValuesFunc) SetValuesRubik,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		NULL,		/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubikWidgetClass = (WidgetClass) & rubikClassRec;

static RubikLoc slideNextRow[MAXFACES][MAXORIENT] =
{
	{
		{5, TOP},
		{3, RIGHT},
		{2, TOP},
		{1, LEFT}},
	{
		{0, RIGHT},
		{2, TOP},
		{4, LEFT},
		{5, BOTTOM}},
	{
		{0, TOP},
		{3, TOP},
		{4, TOP},
		{1, TOP}},
	{
		{0, LEFT},
		{5, BOTTOM},
		{4, RIGHT},
		{2, TOP}},
	{
		{2, TOP},
		{3, LEFT},
		{5, TOP},
		{1, RIGHT}},
	{
		{4, TOP},
		{3, BOTTOM},
		{0, TOP},
		{1, BOTTOM}}
};
static int  rowToRotate[MAXFACES][MAXORIENT] =
{
	{3, 2, 1, 5},
	{2, 4, 5, 0},
	{3, 4, 1, 0},
	{5, 4, 2, 0},
	{3, 5, 1, 2},
	{3, 0, 1, 4}
};

void
faceSizes(RubikWidget w, int face, int *sizeOfRow, int *sizeOfColumn)
{
	switch (face) {
		case 0:	/* TOP */
		case 4:	/* BOTTOM */
			*sizeOfRow = w->rubik.sizex;
			*sizeOfColumn = w->rubik.sizez;
			break;
		case 1:	/* LEFT */
		case 3:	/* RIGHT */
			*sizeOfRow = w->rubik.sizez;
			*sizeOfColumn = w->rubik.sizey;
			break;
		case 2:	/* FRONT */
		case 5:	/* BACK */
			*sizeOfRow = w->rubik.sizex;
			*sizeOfColumn = w->rubik.sizey;
			break;
		default:
			(void) printf("faceSizes: face %d\n", face);
	}
}

int
sizeFace(RubikWidget w, int face)
{
	int         sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return (sizeOfRow * sizeOfColumn);
}

int
sizeRow(RubikWidget w, int face)
{
	int         sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return sizeOfRow;
}

static void
InitializeRubik(Widget request, Widget renew)
{
	RubikWidget w = (RubikWidget) renew;
	int         face, orient;

	for (face = 0; face < MAXFACES; face++)
		w->rubik.cubeLoc[face] = NULL;
	for (orient = 0; orient < MAXORIENT; orient++)
		w->rubik.rowLoc[orient] = NULL;
	CheckPolyhedrons(w);
	InitMoves();
	ResetPolyhedrons(w);
	(void) SRAND(getpid());
	w->rubik.depth = DefaultDepthOfScreen(XtScreen(w));
	SetAllColors(w, True);
}

static void
DestroyRubik(Widget old)
{
	RubikWidget w = (RubikWidget) old;
	int         face;

	for (face = 0; face < MAXFACES; face++)
		XtReleaseGC(old, w->rubik.faceGC[face]);
	XtReleaseGC(old, w->rubik.borderGC);
	XtReleaseGC(old, w->rubik.puzzleGC);
	XtReleaseGC(old, w->rubik.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->rubik.select);
}

static      Boolean
SetValuesRubik(Widget current, Widget request, Widget renew)
{
	RubikWidget c = (RubikWidget) current, w = (RubikWidget) renew;
	Boolean     redraw = False, setColors = False;
	int         face;

	CheckPolyhedrons(w);
	for (face = 0; face < MAXFACES; face++) {
		if (strcmp(w->rubik.faceName[face], c->rubik.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->core.background_pixel != c->core.background_pixel ||
	    w->rubik.foreground != c->rubik.foreground ||
	    w->rubik.borderColor != c->rubik.borderColor ||
	    w->rubik.reverse != c->rubik.reverse ||
	    w->rubik.mono != c->rubik.mono ||
	    setColors) {
		SetAllColors(w, False);
		redraw = True;
	}
	if (w->rubik.orient != c->rubik.orient) {
		ResetPolyhedrons(w);
		redraw = True;
	} else if (w->rubik.practice != c->rubik.practice) {
		ResetPolyhedrons(w);
		redraw = True;
	}
	if (w->rubik.currentDirection == RUBIK_RESTORE) {
		SetStartPosition(w);
		w->rubik.currentDirection = RUBIK_IGNORE;
	} else if (w->rubik.currentDirection != RUBIK_IGNORE) {
		MovePolyhedrons(w, w->rubik.currentFace, w->rubik.currentPosition,
				w->rubik.currentDirection);
		w->rubik.currentDirection = RUBIK_IGNORE;
	}
	return redraw;
}

void
QuitRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}

void
SelectRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	int         control;

	if (SelectPolyhedrons(w, event->xbutton.x, event->xbutton.y,
		     &(w->rubik.currentFace), &(w->rubik.currentPosition))) {
		control = (int) (event->xkey.state & ControlMask);
		if (control || w->rubik.practice || !CheckSolved(w))
			DrawSquare(w, w->rubik.currentFace, w->rubik.currentPosition,
				   TRUE);
	} else {
		w->rubik.currentFace = RUBIK_IGNORE;
		w->rubik.currentDirection = RUBIK_IGNORE;
	}
}

void
ReleaseRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	int         control, face, position, count = -1, direction = 0;
	rubikCallbackStruct cb;

	if (w->rubik.currentFace == RUBIK_IGNORE)
		return;
	DrawSquare(w, w->rubik.currentFace, w->rubik.currentPosition, FALSE);
	control = (int) (event->xkey.state & ControlMask);
	if (!control && !w->rubik.practice && CheckSolved(w))
		MoveNoPolyhedrons(w);
	else if (SelectPolyhedrons(w, event->xbutton.x, event->xbutton.y,
				   &face, &position)) {
		control = (control) ? 1 : 0;
		if (face == w->rubik.currentFace)
			count = CheckMoveDir(w, face, w->rubik.currentPosition, position, &direction);
		if (count == 1) {
			MoveRubik(w, face, w->rubik.currentPosition, direction, control);
			if (!control && CheckSolved(w)) {
				cb.reason = RUBIK_SOLVED;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
		} else if (count == 0)
			MoveNoPolyhedrons(w);
	}
}

void
PracticeRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	PracticePolyhedrons(w);
}

void
PracticeRubikMaybe(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	if (!w->rubik.started)
		PracticePolyhedrons(w);
}

void
RandomizeRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	RandomizePolyhedrons(w);
}

void
RandomizeRubikMaybe(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	if (!w->rubik.started)
		RandomizePolyhedrons(w);
}

void
GetRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	FILE       *fp;
	char        c;
	int         i, sizex, sizey, sizez, orient, practice, moves;
	rubikCallbackStruct cb;

	if ((fp = fopen(DATAFILE, "r")) == NULL)
		(void) printf("Can not read %s for get.\n", DATAFILE);
	else {
		FlushMoves(w);
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &sizex);
		if (sizex >= MINCUBES) {
			for (i = w->rubik.sizex; i < sizex; i++) {
				cb.reason = RUBIK_INCX;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
			for (i = w->rubik.sizex; i > sizex; i--) {
				cb.reason = RUBIK_DECX;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
		} else
			(void) printf("%s corrupted: sizex %d should be between %d and MAXINT\n",
				      DATAFILE, sizex, MINCUBES);
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &sizey);
		if (sizey >= MINCUBES) {
			for (i = w->rubik.sizey; i < sizey; i++) {
				cb.reason = RUBIK_INCY;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
			for (i = w->rubik.sizey; i > sizey; i--) {
				cb.reason = RUBIK_DECY;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
		} else
			(void) printf("%s corrupted: sizey %d should be between %d and MAXINT\n",
				      DATAFILE, sizey, MINCUBES);
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &sizez);
		if (sizez >= MINCUBES) {
			for (i = w->rubik.sizez; i < sizez; i++) {
				cb.reason = RUBIK_INCZ;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
			for (i = w->rubik.sizez; i > sizez; i--) {
				cb.reason = RUBIK_DECZ;
				XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
			}
		} else
			(void) printf("%s corrupted: sizez %d should be between %d and MAXINT\n",
				      DATAFILE, sizez, MINCUBES);
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &orient);
		if (w->rubik.orient != (Boolean) orient) {
			cb.reason = RUBIK_ORIENT;
			XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
		}
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &practice);
		if (w->rubik.practice != (Boolean) practice) {
			cb.reason = RUBIK_PRACTICE;
			XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
		}
		while ((c = getc(fp)) != EOF && c != SYMBOL);
		(void) fscanf(fp, "%d", &moves);
		ScanStartPosition(fp, w);
		cb.reason = RUBIK_RESTORE;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
		ScanMoves(fp, w, moves);
		(void) fclose(fp);
		(void) printf("%s: sizex %d, sizey %d, sizez %d, orient %d, practice %d, moves %d.\n",
		     DATAFILE, sizex, sizey, sizez, orient, practice, moves);
	}
}

void
WriteRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	FILE       *fp;

	if ((fp = fopen(DATAFILE, "w")) == NULL)
		(void) printf("Can not write to %s.\n", DATAFILE);
	else {
		(void) fprintf(fp, "sizex%c %d\n", SYMBOL, w->rubik.sizex);
		(void) fprintf(fp, "sizey%c %d\n", SYMBOL, w->rubik.sizey);
		(void) fprintf(fp, "sizez%c %d\n", SYMBOL, w->rubik.sizez);
		(void) fprintf(fp, "orient%c %d\n", SYMBOL, (w->rubik.orient) ? 1 : 0);
		(void) fprintf(fp, "practice%c %d\n", SYMBOL, (w->rubik.practice) ? 1 : 0);
		(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
		PrintStartPosition(fp, w);
		PrintMoves(fp);
		(void) fclose(fp);
		(void) printf("Saved to %s.\n", DATAFILE);
	}
}

void
UndoRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	if (MadeMoves()) {
		int         face, position, direction, control;

		GetMove(&face, &position, &direction, &control);
		direction = (direction < MAXORIENT) ? (direction + MAXORIENT / 2) %
			MAXORIENT : 3 * MAXORIENT - direction;
		if (control)
			MoveControlCb(w, face, direction);
		else {
			rubikCallbackStruct cb;

			MovePolyhedrons(w, face, position, direction);
			cb.reason = RUBIK_UNDO;
			cb.face = face;
			cb.position = position;
			cb.direction = direction;
			XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
		}
	}
}

void
SolveRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	if ((w->rubik.sizex == 2 && w->rubik.sizey == 2 && w->rubik.sizez == 2) ||
	    (w->rubik.sizex == 3 && w->rubik.sizey == 3 && w->rubik.sizez == 3 && !w->rubik.orient))
		SolvePolyhedrons(w);
}

void
DecrementRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	if (w->rubik.sizex > MINCUBES) {
		cb.reason = RUBIK_DECX;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
	if (w->rubik.sizey > MINCUBES) {
		cb.reason = RUBIK_DECY;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
	if (w->rubik.sizez > MINCUBES) {
		cb.reason = RUBIK_DECZ;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
}

void
IncrementRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	cb.reason = RUBIK_INCX;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	cb.reason = RUBIK_INCY;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	cb.reason = RUBIK_INCZ;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

void
IncrementXRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizex <= MINCUBES)
			return;
		cb.reason = RUBIK_DECX;
	} else
		cb.reason = RUBIK_INCX;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

void
IncrementYRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizey <= MINCUBES)
			return;
		cb.reason = RUBIK_DECY;
	} else
		cb.reason = RUBIK_INCY;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

void
IncrementZRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	if (event->xbutton.state & (ShiftMask | LockMask)) {
		if (w->rubik.sizez <= MINCUBES)
			return;
		cb.reason = RUBIK_DECZ;
	} else
		cb.reason = RUBIK_INCZ;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

void
OrientizeRubik(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	rubikCallbackStruct cb;

	cb.reason = RUBIK_ORIENT;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

void
MoveRubikCcw(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput(w, event->xbutton.x, event->xbutton.y, CCW,
		       (int) (event->xbutton.state & ControlMask));
}

void
MoveRubikCw(RubikWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput(w, event->xbutton.x, event->xbutton.y, CW,
		       (int) (event->xkey.state & ControlMask));
}

void
MoveRubikInput(RubikWidget w, int x, int y, int direction, int control)
{
	int         face, position;

	if (!w->rubik.practice && !control && CheckSolved(w)) {
		MoveNoPolyhedrons(w);
		return;
	}
	if (!PositionPolyhedrons(w, x, y, &face, &position, &direction))
		return;
	control = (control) ? 1 : 0;
	MoveRubik(w, face, position, direction, control);
	if (!control && CheckSolved(w)) {
		rubikCallbackStruct cb;

		cb.reason = RUBIK_SOLVED;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
}

void
MoveRubik(RubikWidget w, int face, int position, int direction, int control)
{
	if (control)
		MoveControlCb(w, face, direction);
	else {
		rubikCallbackStruct cb;

		MovePolyhedrons(w, face, position, direction);
		cb.reason = RUBIK_MOVED;
		cb.face = face;
		cb.position = position;
		cb.direction = direction;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
	PutMove(face, position, direction, control);
}

static      Bool
checkFaceSquare(RubikWidget w, int face)
{
	int         sizeOfRow, sizeOfColumn;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	return (sizeOfRow == sizeOfColumn);
	/* Cubes can be made square with a 4x2 face where 90 degree turns
	 * should be permitted but that is kind of complicated for me.
	 * This can be done in 2 ways where the side of the cubies are
	 * the same size and one where one side (the side with half the
	 * number of cubies) is twice the size of the other.  The first is
	 * complicated because faces of cubies can go under other faces.
	 * The second way is similar to "banded cubes" where scotch tape
	 * restricts the moves of some cubes.  Here you have to keep track
	 * of the restrictions and show banded cubies graphically as one
	 * cube.
	 */
}

static void
SetAllColors(RubikWidget w, Boolean init)
{
	XGCValues   values;
	XtGCMask    valueMask;
	int         face;

	valueMask = GCForeground | GCBackground;

	if (w->rubik.reverse) {
		values.background = w->core.background_pixel;
		values.foreground = w->rubik.foreground;
	} else {
		values.foreground = w->core.background_pixel;
		values.background = w->rubik.foreground;
	}
	if (!init)
		XtReleaseGC((Widget) w, w->rubik.inverseGC);
	w->rubik.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->rubik.reverse) {
		values.background = w->rubik.foreground;
		values.foreground = w->core.background_pixel;
	} else {
		values.foreground = w->rubik.foreground;
		values.background = w->core.background_pixel;
	}
	if (!init)
		XtReleaseGC((Widget) w, w->rubik.puzzleGC);
	w->rubik.puzzleGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->rubik.depth < 2 || w->rubik.mono) {
		if (w->rubik.reverse) {
			values.background = w->rubik.foreground;
			values.foreground = w->core.background_pixel;
		} else {
			values.foreground = w->rubik.foreground;
			values.background = w->core.background_pixel;
		}
	} else {
		values.foreground = w->rubik.borderColor;
		values.background = w->core.background_pixel;
	}
	if (!init)
		XtReleaseGC((Widget) w, w->rubik.borderGC);
	w->rubik.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAXFACES; face++)
		GetColor(w, face, init);
}

static void
GetColor(RubikWidget w, int face, Boolean init)
{
	XGCValues   values;
	XtGCMask    valueMask;
	XColor      colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->rubik.reverse) {
		values.background = w->rubik.foreground;
	} else {
		values.background = w->core.background_pixel;
	}
	if (w->rubik.depth > 1 && !w->rubik.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				  DefaultColormap(XtDisplay(w), XtWindow(w)),
				w->rubik.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->rubik.faceColor[face] = colorCell.pixel;
			if (!init)
				XtReleaseGC((Widget) w, w->rubik.faceGC[face]);
			w->rubik.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
			return;
		} else {
			char        buf[121];

			(void) sprintf(buf, "Color name \"%s\" is not defined",
				       w->rubik.faceName[face]);
			XtWarning(buf);
		}
	}
	if (w->rubik.reverse) {
		values.background = w->rubik.foreground;
		values.foreground = w->core.background_pixel;
	} else {
		values.background = w->core.background_pixel;
		values.foreground = w->rubik.foreground;
	}
	if (!init)
		XtReleaseGC((Widget) w, w->rubik.faceGC[face]);
	w->rubik.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

static void
MoveControlCb(RubikWidget w, int face, int direction)
{
	rubikCallbackStruct cb;
	int         k, position, sizeOfRow, sizeOfColumn, sizeOnOppAxis;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	if (direction == TOP || direction == BOTTOM) {
		sizeOnOppAxis = sizeOfRow;
	} else {
		sizeOnOppAxis = sizeOfColumn;
	}
	for (k = 0; k < sizeOnOppAxis; k++) {
		position = k * sizeOfRow + k;
		MovePolyhedrons(w, face, position, direction);
		cb.reason = RUBIK_CONTROL;
		cb.face = face;
		cb.position = position;
		cb.direction = direction;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
}

static void
CheckPolyhedrons(RubikWidget w)
{
	if (w->rubik.sizex < MINCUBES) {
		char        buf[121];

		(void) sprintf(buf, "Number of Cubes on edge (x-axis) out of bounds, use %d..MAXINT",
			       MINCUBES);
		XtWarning(buf);
		w->rubik.sizex = DEFAULTCUBES;
	}
	if (w->rubik.sizey < MINCUBES) {
		char        buf[121];

		(void) sprintf(buf, "Number of Cubes on edge (y-axis) out of bounds, use %d..MAXINT",
			       MINCUBES);
		XtWarning(buf);
		w->rubik.sizey = DEFAULTCUBES;
	}
	if (w->rubik.sizez < MINCUBES) {
		char        buf[121];

		(void) sprintf(buf, "Number of Cubes on edge (z-axis) out of bounds, use %d..MAXINT",
			       MINCUBES);
		XtWarning(buf);
		w->rubik.sizez = DEFAULTCUBES;
	}
}

void
ResetPolyhedrons(RubikWidget w)
{
	int         face, position, orient, sizeOfFace;

	for (face = 0; face < MAXFACES; face++) {
		sizeOfFace = sizeFace(w, face);
		if (w->rubik.cubeLoc[face])
			(void) free((void *) w->rubik.cubeLoc[face]);
		if (!(w->rubik.cubeLoc[face] = (RubikLoc *)
		      malloc(sizeof (RubikLoc) * sizeOfFace)))
			XtError("Not enough memory, exiting.");
		if (startLoc[face])
			(void) free((void *) startLoc[face]);
		if (!(startLoc[face] = (RubikLoc *)
		      malloc(sizeof (RubikLoc) * sizeOfFace)))
			XtError("Not enough memory, exiting.");
		for (position = 0; position < sizeOfFace; position++) {
			w->rubik.cubeLoc[face][position].face = face;
			w->rubik.cubeLoc[face][position].rotation = STRT - MAXORIENT;
		}
	}
	for (orient = 0; orient < MAXORIENT; orient++) {
		if (w->rubik.rowLoc[orient])
			(void) free((void *) w->rubik.rowLoc[orient]);
		if (!(w->rubik.rowLoc[orient] = (RubikLoc *)
		      malloc(sizeof (RubikLoc) * MAXMAXSIZE)))
			XtError("Not enough memory, exiting.");
	}
	FlushMoves(w);
	w->rubik.started = False;
}

static      Boolean
SelectPolyhedrons(RubikWidget w, int x, int y, int *face, int *position)
{
	if (w->rubik.dim == 2)
		return SelectPolyhedrons2D((Rubik2DWidget) w, x, y,
					   face, position);
	else if (w->rubik.dim == 3)
		return SelectPolyhedrons3D((Rubik3DWidget) w, x, y,
					   face, position);
	return False;
}

static      Boolean
NarrowSelection(RubikWidget w, int *face, int *position, int *direction)
{
	if (w->rubik.dim == 2)
		return NarrowSelection2D((Rubik2DWidget) w, face, position, direction);
	else if (w->rubik.dim == 3)
		return NarrowSelection3D((Rubik3DWidget) w, face, position, direction);
	return False;
}

static      Boolean
PositionPolyhedrons(RubikWidget w, int x, int y, int *face, int *position, int *direction)
{
	if (!SelectPolyhedrons(w, x, y, face, position))
		return False;
	return NarrowSelection(w, face, position, direction);
}

static void
MoveNoPolyhedrons(RubikWidget w)
{
	rubikCallbackStruct cb;

	cb.reason = RUBIK_ILLEGAL;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void
PracticePolyhedrons(RubikWidget w)
{
	rubikCallbackStruct cb;

	cb.reason = RUBIK_PRACTICE;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
}

static void
RandomizePolyhedrons(RubikWidget w)
{
	rubikCallbackStruct cb;
	int         face, position, direction;
	int         avg = w->rubik.sizex + w->rubik.sizey + w->rubik.sizez / 3;
	int         big = w->rubik.sizex * w->rubik.sizey * w->rubik.sizez /
	avg * 3 + NRAND(2);

	if (big > 1000)
		big = 1000;
	if (w->rubik.practice)
		PracticePolyhedrons(w);
	cb.reason = RUBIK_RESET;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);

#ifdef DEBUG
	big = 3;
#endif

	while (big--) {
		face = NRAND(MAXFACES);
		position = NRAND(sizeFace(w, face));
		direction = NRAND(MAXORIENT);
		MoveRubik(w, face, position, direction, FALSE);
	}
	FlushMoves(w);
	cb.reason = RUBIK_RANDOMIZE;
	XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	if (CheckSolved(w)) {
		cb.reason = RUBIK_SOLVED;
		XtCallCallbacks((Widget) w, XtNselectCallback, &cb);
	}
}

/* Yeah this is big and ugly */
static void
SlideRC(int face, int direction, int h, int sizeOnOppAxis,
	int *newFace, int *newDirection, int *newH,
	int *rotate, Bool * reverse)
{
	*newFace = slideNextRow[face][direction].face;
	*rotate = slideNextRow[face][direction].rotation;
	*newDirection = (*rotate + direction) % MAXORIENT;
	switch (*rotate) {
		case TOP:
			*newH = h;
			*reverse = False;
			break;
		case RIGHT:
			if (*newDirection == TOP || *newDirection == BOTTOM) {
				*newH = sizeOnOppAxis - 1 - h;
				*reverse = False;
			} else {	/* *newDirection == RIGHT || *newDirect
					   ion == LEFT */
				*newH = h;
				*reverse = True;
			}
			break;
		case BOTTOM:
			*newH = sizeOnOppAxis - 1 - h;
			*reverse = True;
			break;
		case LEFT:
			if (*newDirection == TOP || *newDirection == BOTTOM) {
				*newH = h;
				*reverse = True;
			} else {	/* *newDirection == RIGHT || *newDirect
					   ion == LEFT */
				*newH = sizeOnOppAxis - 1 - h;
				*reverse = False;
			}
			break;
		default:
			(void) printf("SlideRC: rotate %d\n", *rotate);
	}
}


static void
MovePolyhedrons(RubikWidget w, int face, int position, int direction)
{
	int         newFace, newDirection, rotate, reverse = FALSE;
	int         h, k, newH = 0;
	int         i, j, sizeOfRow, sizeOfColumn, sizeOnAxis, sizeOnOppAxis;

	w->rubik.degreeTurn = (checkFaceSquare(w,
				   rowToRotate[face][direction])) ? 90 : 180;
	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	i = position % sizeOfRow;
	j = position / sizeOfRow;
	h = (direction == TOP || direction == BOTTOM) ? i : j;
	if (direction == TOP || direction == BOTTOM) {
		sizeOnAxis = sizeOfColumn;
		sizeOnOppAxis = sizeOfRow;
	} else {
		sizeOnAxis = sizeOfRow;
		sizeOnOppAxis = sizeOfColumn;
	}
	/* rotate sides CW or CCW or HALF */

	if (h == sizeOnOppAxis - 1) {
		newDirection = (direction == TOP || direction == BOTTOM) ?
			TOP : RIGHT;
		if (w->rubik.degreeTurn == 180)
			RotateFace(w, rowToRotate[face][newDirection], HALF);
		else if (direction == TOP || direction == RIGHT)
			RotateFace(w, rowToRotate[face][newDirection], CW);
		else		/* direction == BOTTOM || direction == LEFT */
			RotateFace(w, rowToRotate[face][newDirection], CCW);
	}
	if (h == 0) {
		newDirection = (direction == TOP || direction == BOTTOM) ?
			BOTTOM : LEFT;
		if (w->rubik.degreeTurn == 180)
			RotateFace(w, rowToRotate[face][newDirection], HALF);
		else if (direction == TOP || direction == RIGHT)
			RotateFace(w, rowToRotate[face][newDirection], CCW);
		else		/* direction == BOTTOM  || direction == LEFT */
			RotateFace(w, rowToRotate[face][newDirection], CW);
	}
	/* Slide rows */
	ReadRC(w, face, direction, h, 0, sizeOnAxis);
	if (w->rubik.degreeTurn == 180) {
		int         sizeOnDepthAxis;

		SlideRC(face, direction, h, sizeOnOppAxis,
			&newFace, &newDirection, &newH, &rotate, &reverse);
		sizeOnDepthAxis = sizeFace(w, newFace) / sizeOnOppAxis;
		ReadRC(w, newFace, newDirection, newH, 1, sizeOnDepthAxis);
		RotateRC(w, rotate, 0, sizeOnAxis);
		if (reverse == True)
			ReverseRC(w, 0, sizeOnAxis);
		face = newFace;
		direction = newDirection;
		h = newH;
		for (k = 2; k <= MAXORIENT + 1; k++) {
			SlideRC(face, direction, h, sizeOnOppAxis,
			  &newFace, &newDirection, &newH, &rotate, &reverse);
			if (k != MAXORIENT && k != MAXORIENT + 1)
				ReadRC(w, newFace, newDirection, newH, k,
				     (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			RotateRC(w, rotate, k - 2,
				 (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			if (k != MAXORIENT + 1)
				RotateRC(w, rotate, k - 1,
				     (k % 2) ? sizeOnAxis : sizeOnDepthAxis);
			if (reverse == True) {
				ReverseRC(w, k - 2,
				     (k % 2) ? sizeOnDepthAxis : sizeOnAxis);
				if (k != MAXORIENT + 1)
					ReverseRC(w, k - 1,
						  (k % 2) ? sizeOnAxis : sizeOnDepthAxis);
			}
			WriteRC(w, newFace, newDirection, newH, k - 2,
				(k % 2) ? sizeOnDepthAxis : sizeOnAxis);
			face = newFace;
			direction = newDirection;
			h = newH;
		}
	} else {
		for (k = 1; k <= MAXORIENT; k++) {
			SlideRC(face, direction, h, sizeOnOppAxis,
			  &newFace, &newDirection, &newH, &rotate, &reverse);
			if (k != MAXORIENT)
				ReadRC(w, newFace, newDirection, newH, k, sizeOnAxis);
			RotateRC(w, rotate, k - 1, sizeOnAxis);
			if (reverse == TRUE)
				ReverseRC(w, k - 1, sizeOnAxis);
			WriteRC(w, newFace, newDirection, newH, k - 1, sizeOnAxis);
			face = newFace;
			direction = newDirection;
			h = newH;
		}
	}
}

static void
ReadRC(RubikWidget w, int face, int dir, int h, int orient, int size)
{
	int         g, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	if (dir == TOP || dir == BOTTOM)
		for (g = 0; g < size; g++)
			w->rubik.rowLoc[orient][g] =
				w->rubik.cubeLoc[face][g * sizeOfRow + h];
	else			/* dir == RIGHT || dir == LEFT */
		for (g = 0; g < size; g++)
			w->rubik.rowLoc[orient][g] =
				w->rubik.cubeLoc[face][h * sizeOfRow + g];
}

static void
RotateRC(RubikWidget w, int rotate, int orient, int size)
{
	int         g;

	for (g = 0; g < size; g++)
		w->rubik.rowLoc[orient][g].rotation =
			(w->rubik.rowLoc[orient][g].rotation + rotate) % MAXORIENT;
}

static void
ReverseRC(RubikWidget w, int orient, int size)
{
	int         g;
	RubikLoc    temp;

	for (g = 0; g < size / 2; g++) {
		temp = w->rubik.rowLoc[orient][size - 1 - g];
		w->rubik.rowLoc[orient][size - 1 - g] = w->rubik.rowLoc[orient][g];
		w->rubik.rowLoc[orient][g] = temp;
	}
}

static void
WriteRC(RubikWidget w, int face, int dir, int h, int orient, int size)
{
	int         g, position, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	if (dir == TOP || dir == BOTTOM) {
		for (g = 0; g < size; g++) {
			position = g * sizeOfRow + h;
			w->rubik.cubeLoc[face][position] = w->rubik.rowLoc[orient][g];
			DrawSquare(w, face, position, FALSE);
		}
	} else {		/* dir == RIGHT || dir == LEFT */
		for (g = 0; g < size; g++) {
			position = h * sizeOfRow + g;
			w->rubik.cubeLoc[face][position] = w->rubik.rowLoc[orient][g];
			DrawSquare(w, face, position, FALSE);
		}
	}
}

static void
RotateFace(RubikWidget w, int face, int direction)
{
	int         position, i, j, sizeOfRow, sizeOfColumn, sizeOnPlane;
	RubikLoc   *faceLoc = NULL;

	faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
	sizeOnPlane = sizeOfRow * sizeOfColumn;
	if ((faceLoc = (RubikLoc *) malloc(sizeOnPlane * sizeof (RubikLoc))) ==
	    NULL)
		(void) fprintf(stderr,
		 "Could not allocate memory for rubik face position info\n");
	/* Read Face */
	for (position = 0; position < sizeOnPlane; position++)
		faceLoc[position] = w->rubik.cubeLoc[face][position];
	/* Write Face */
	for (position = 0; position < sizeOnPlane; position++) {
		i = position % sizeOfRow;
		j = position / sizeOfRow;
		if (direction == CW)
			w->rubik.cubeLoc[face][position] =
				faceLoc[(sizeOfRow - i - 1) * sizeOfRow + j];
		else if (direction == CCW)
			w->rubik.cubeLoc[face][position] =
				faceLoc[i * sizeOfRow + sizeOfColumn - j - 1];
		else		/* (direction == HALF) */
			w->rubik.cubeLoc[face][position] =
				faceLoc[sizeOfRow - i - 1 + (sizeOfColumn - j - 1) * sizeOfRow];

		w->rubik.cubeLoc[face][position].rotation =
			(w->rubik.cubeLoc[face][position].rotation +
			 direction - MAXORIENT) % MAXORIENT;
		DrawSquare(w, face, position, FALSE);
	}
	if (faceLoc != NULL)
		(void) free((void *) faceLoc);
}

void
DrawAllPolyhedrons(RubikWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < sizeFace(w, face); position++)
			DrawSquare(w, face, position, FALSE);
}

static void
DrawSquare(RubikWidget w, int face, int position, int offset)
{
	if (w->rubik.dim == 2)
		DrawSquare2D((Rubik2DWidget) w, face, position, offset);
	else if (w->rubik.dim == 3)
		DrawSquare3D((Rubik3DWidget) w, face, position, offset);
}

Boolean
CheckSolved(RubikWidget w)
{
	int         face, position;
	RubikLoc    test;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < sizeFace(w, face); position++) {
			if (!position) {
				test.face = w->rubik.cubeLoc[face][position].face;
				test.rotation = w->rubik.cubeLoc[face][position].rotation;
			} else if (test.face !=		/*face */
				   w->rubik.cubeLoc[face][position].face ||
				   (w->rubik.orient && test.rotation !=		/*STRT - MAXORIENT */
				  w->rubik.cubeLoc[face][position].rotation))
				return False;
		}
	return True;
}

static int
CheckMoveDir(RubikWidget w, int face, int position1, int position2, int *direction)
{
	int         count = 0;
	int         column1, column2, row1, row2, sizeOfRow;

	sizeOfRow = sizeRow(w, face);
	column1 = position1 % sizeOfRow;
	column2 = position2 % sizeOfRow;
	row1 = position1 / sizeOfRow;
	row2 = position2 / sizeOfRow;
	if (column1 == column2 && row1 != row2) {
		*direction = (row2 > row1) ? BOTTOM : TOP;
		count = 1;
	} else if (row1 == row2 && column1 != column2) {
		*direction = (column2 > column1) ? RIGHT : LEFT;
		count = 1;
	} else if (row1 == row2 && column1 == column2)
		count = 2;
	return count;
}


#ifdef DEBUG

void
PrintCube(RubikWidget w)
{
	int         face, position, sizeOfRow, sizeOfColumn;

	for (face = 0; face < MAXFACES; face++) {
		faceSizes(rp, face, &sizeOfRow, &sizeOfColumn);
		for (position = 0; position < sizeOfRow * sizeOfColumn; position++) {
			(void) printf("%d %d  ", w->rubik.cubeLoc[face][position].face,
				  w->rubik.cubeLoc[face][position].rotation);
			if (!((position + 1) % sizeOfRow))
				(void) printf("\n");
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

void
PrintRow(RubikWidget w, int orient, int size)
{
	int         i;

	(void) printf("Row %d:\n", orient);
	for (i = 0; i < size; i++)
		(void) printf("%d %d  ", w->rubik.rowLoc[orient][i].face,
			      w->rubik.rowLoc[orient][i].rotation);
	(void) printf("\n");
}

#endif
