/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 * database.h
 */

/* $Id: database.h,v 1.45 93/06/14 10:32:18 beust Exp Locker: beust $ */

/*
 ** Generic package to handle databases. No memory limit : chunks are 
 ** allocated dynamically whenever needed
 ** For each entry, two parallel arrays are maintained : one is a pointer
 ** to the entry itself (field entries) and one is a flag signalling that
 ** the corresponding entry is allocated (field activeEntries). When an
 ** entry is removed, the corresponding activeEntries[] is set to 0, thus
 ** telling the slot is free for re-use
 */

#ifndef DATABASE_H
#define DATABASE_H

#undef DATABASE_BY_COPY

/*
 ** Miscellaneous definitions
 */

#ifndef NEW
#define NEW(v, t) v = (t *) malloc(sizeof(t))
#endif
#ifndef True
#define True 1
#define False 0
#endif

/* Magic cookie */
#define DB_OCCUPIED 9

/*
 ** Defined types
 */

/* The generic type */
#ifndef Generic
#define Generic void *
#endif

/* Boolean type */
#ifndef Bool
#define Bool char
#endif

/* The main type */
typedef struct _DataBase *DataBase;

/*
 ** Function prototypes
 */

DataBase
DB_NewDataBase(int size, void *freeFunction);
/* Initialize a new database, which objects have size 'size' */
/* and which entries can be deallocated with freeFunction(). */
/* This function will receive the argument to be freed as the */
/* only parameter. It must free its content, but *not* the */
/* object itself. */

Bool
DB_EndOfDataBase(DataBase db);
/* True if we reached the end of the database */

void
DB_ClearDataBase(DataBase db);
/* Clear the database */

void
DB_DestroyDataBase(DataBase db);
/* Destroy the database, freeing all its entries */

Bool
DB_AddEntry(DataBase db, Generic entry);
/* Add the following entry into the database */
/* Alter the internal pointer */
/* Return True if the operation was successful */

Bool
DB_RemoveEntry(DataBase db, Generic entry);
/* Remove the specified entry, and free it via the freeFunction passed */
/* in parameter to DB_NewDataBase */
/* Return True if the operation was successful */

Bool
DB_RemoveNthEntry(DataBase db, int n);
/* Remove the nth DB_NextEntry from the database (0 = first entry) */
/* Return True if the operation was successful */

Bool
DB_ReplaceEntry(DataBase db, Generic oldEntry, Generic newEntry);
/* Replace the old entry with the new one */
/* CAREFUL : all allocated pointers in old will be freed, therefore new */
/* must contain newly allocated pointers */
/* Return True if the operation was successful */

void
DB_Rewind(DataBase db);
/* Rewind the database so that a subsequent NextEntry returns the */
/* first occupied element */
/* Alter the internal pointer */

Generic
DB_NextEntry(DataBase db);
/* Return the next entry, or NULL if we reached the end of the database */
/* Alter the internal pointer */

Generic
DB_NthEntry(DataBase db, int n);
/* Return the nth "DB_NextEntry" of the database (0 = first entry) */

int
DB_Count(DataBase db);
/* Return the number of entries in the databae */

void
DB_Sort(DataBase db, int (* compareFunction)(Generic a, Generic b));
/* Sort the database, so that */
/*     compareFunction(DB_NextEntry(db), DB_NextEntry(db)) < 0  */

void
DB_DisplayDataBase(DataBase db, void (* displayFunction)(Generic a));
/* Display the database by calling the displayFunction on each entry */
/* Internal pointer is not modified */

Generic
DB_LocateEntry(DataBase db,
	       Bool (* findFunction)(Generic a, void *userData),void *userData);
/* Locate an entry that matches a specified criterion. This criterion will */
/* be tested by findFunction() that must return either True or False. */
/* This function will be given the entry to test and a user-specified */
/* data that might be needed to achieve the test */

#endif
