/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kt_lib.h,v 1.1 1994/10/07 16:34:43 beust Exp beust $
 */

#ifndef __KT_LIB_H__
#define __KT_LIB_H__

#include <setjmp.h>     /* jmp_buf */
#include <stddef.h>     /* size_t */
#include <string.h>
#include <sys/types.h>      /* fd_set */
#include <X11/Xproto.h>
#ifndef DEBUG
#include <X11/ICE/ICE.h>
#include <X11/ICE/ICElib.h>
#include <X11/ICE/ICEconn.h>
#include <X11/ICE/ICEmsg.h>
#include <X11/ICE/ICEproto.h>
#include <X11/ICE/ICEutil.h>
#else
#include <X11/ICE/ICE.h>
#include <X11/ICE/ICElib.h>
#include <X11/ICE/ICEconn.h>
#include <X11/ICE/ICEmsg.h>
#include <X11/ICE/ICEproto.h>
#include <X11/ICE/ICEutil.h>
#endif



/*
** Use these files to store info about the server
*/
#define FILE_KTSERV_PORT "/tmp/ktserv.port"
#define FILE_KTSERV_PID "/tmp/ktserv.pid"


/*
**
** Types exported
**
*/

typedef struct _Kt_Context *
                                                    Kt_Context;
typedef struct _Kt_PeerConnection *
                                                    Kt_PeerConnection;
typedef struct _Kt_Message *
                                                    Kt_Message;
typedef struct _Kt_KRL *
                                                    Kt_KRL;
typedef enum { KT_OK, KT_ERR }
                                                    Kt_Status;
typedef enum { KT_QUEUE, KT_START }
                                                    Kt_Disposition;

typedef void (*
                                                    Kt_CallbackPeerMsg
)(Kt_Context, void *, Kt_Message);
typedef void (*
                                                    Kt_CallbackHandlerMsg
)(Kt_Context, void *, Kt_Message, CARD32);
typedef void (*
                                                    Kt_CallbackReplyMsg
)(Kt_Context, void *, Kt_Message, CARD32);
typedef void (*
                                                    Kt_CallbackServerMsg
)(Kt_Context, void *, Kt_Message, CARD32);
typedef void (*
                                                    Kt_CallbackAddInput
)(int fd, void *userData);
typedef void (*
                                                    Kt_CallbackRemoveInput
)(int fd, void *userData);

/*
** This is really ugly, but forced by ICE.
** If you write your own main loop, insert a setjmp() to this
** value at the point where you want execution to be resumed
** in case of a disconnection
*/
extern jmp_buf KoalaTalk_MainLoopJmp; /* really ugly, must take that off */

/*
 ** Prototypes
 */

/* File kt_lib.c */

/*
** First function to call before using the toolkit.
** addInput() and removeInput() will be called by Koalatalk to ask
** you to add file descriptors to your set. If you don't want to
** use this facility and only need to use kt_MainLoop(), you can
** safely pass 4 NULL here.
** Return : an opaque object to be used for subsequent calls
*/
Kt_Context
kt_Init(Kt_CallbackAddInput addInput, void *addInputUserData,
	Kt_CallbackRemoveInput removeInput, void *removeInputUserData);

/*
** To be called explicitly by the caller if it detected activity on
** a fd that belongs to Koalatalk.
*/
void
kt_HandleInput(Kt_Context context, fd_set fds);

/*
** Free all resources allocated for a Koalatalk context
*/
void
kt_UnInit(Kt_Context);

/*
** Tries to locate a server.
** serversFoundCount will contain the number of servers that
** could be found, or 0 if Koalatalk can't even make a guess.
** You will find the coordinates to try in
** serversFound[0..serversFoundCount-1], which you can pass directly
** to kt_OpenServerConnection()
** Note : this is just a guess, the coordinates returned might
** correspond to a dead server anyway
** Note 2 : each string serversFound[i] must be freed by the caller
** afterwards, and serversFound itself as well eventually
*/
void
kt_LocateServer(char ***serversFound,  /* array of possible servers */
		int *serversFoundCount /* number of servers found */);


/*
** Open a connection to a server
** Return : KT_OK if connection could be opened successfully, KT_ERR
** otherwise
*/
Kt_Status
kt_OpenServerConnection(Kt_Context context,
			char *serverLocation,/* server location (ICE format)*/
			char *identity);

/*
** Close a connection to the server
** Return : KT_OK if succesful, KT_ERR otherwise
*/
Kt_Status
kt_CloseServerConnection(Kt_Context context);

/*
** Return : the IceConn descriptor to the server
*/
IceConn
kt_ReturnIceServerConnection(Kt_Context context);

/*
** Return : a string containing my identity.
** Notes : do *not* modify this string, it is a pointer to
** Koalatalk's internal name space
*/
char *
kt_ReturnIdentity(Kt_Context context);


/*
** Add a user file descriptor. The specified callback will
** be called if ever data on this fd is ready. The callback
** will receive userData as its unique argument
*/
void
kt_AddUserFileDescriptor(Kt_Context context, int fd,
                         void (*callback)(),void *userData);

/*
** Return : the protocol used to talk with the server
*/
int
kt_ReturnIceServerProtocol(Kt_Context context);

/*
** Return the next message
*/
void
kt_NextMessage(Kt_Context context, Kt_Message *msg);


/*
** Call the appropriate callbacks for msg
** Return : True if at least a callback could be found, False otherwise
*/
Bool
kt_DispatchMessage(Kt_Context context, Kt_Message msg);


/*
** Define the handler to be called if the connection to the server
** is lost
*/
void
kt_SetServerErrorHandler(Kt_Context context, void *handler /* the error handler */);

/*
** Main loop to process the messages and invoke callbacks accordingly
*/
void
kt_MainLoop(Kt_Context context);

/* File kt_message.c */

/*
** Allocate a new message
** Return : an empty message
*/
Kt_Message
kt_MessageNew();

/*
** Return : the number of arguments in this message (first argument
** is #0
*/
int
kt_MessageArgCount(Kt_Message msg);

/*
** Allocate a new message, copy the content of 'msg' into it
** and return it
** Return : a copy of the parameter (which can be safely freed
** with kt_MessageFree() afterwards without interfering with the
** returned argument)
** Note : duplicating a message with a simple assignment will
** *not* work. If the duplicated message is freed, the copy you
** hold will become invalid as well
*/
Kt_Message
kt_MessageCopy(Kt_Message msg);

/*
** Free a message
*/
void
kt_MessageFree(Kt_Message msg);

/*
** Send a message to a peer
** Return : KT_OK if message was sent successfully, KT_ERR otherwise
*/
Kt_Status
kt_MessageSendToPeer(Kt_Context context,
		     char *peer, /* peer location (ICE format) */
		     Kt_Message msg);

/*
** Send a message to a krl without expecting an answer
** Return : KT_OK if message was sent successfully, KT_ERR otherwise
*/
Kt_Status
kt_MessageSend(Kt_Context context, Kt_KRL krl, Kt_Message msg);

/*
** Send a message and expect a reply
** Return : a serial number identifying uniquely the request
*/
CARD32
kt_MessageSendWithReply(Kt_Context context, Kt_KRL krl, Kt_Message msg,
			Kt_Disposition disposition,
			Kt_CallbackHandlerMsg callback, void *userData);

/*
** Reply to a request. The message 'reply' is sent to the sender
** of 'query'
** Return : KT_OK if message was sent successfully, KT_ERR otherwise
*/
Kt_Status
kt_MessageReply(Kt_Context context, Kt_Message query, Kt_Message reply);

/*
** Set the "operation" field for this message.
** Return : always KT_OK
** Note : the string is duplicated in Koalatalk's internal space,
** you don't have to worry about deallocating it
*/
Kt_Status
kt_MessageOperationSet(Kt_Message msg,
		       char *string);

/*
** Return : the "operation" field for this message.
** Note : the returned value must *not* be altered. It is a direct
** pointer to Koalatalk's internal space
*/
char *
kt_MessageOperationGet(Kt_Message msg);

/*
** Add a string parameter to a message ** Return : KT_OK if successful */
Kt_Status
kt_MessageStringAdd(Kt_Message msg,
		    char *string /* the parameter content */);

/*
** Add a bytes parameter to a message
** Return : KT_OK if successful
*/
Kt_Status
kt_MessageBytesAdd(Kt_Message msg,
		   void *bytes, /* pointer to the buffer */
		   size_t len /* size of the buffer */);

/*
** Retrieve the argPos'th argument of this message, which must be a string
** Note : returnedString must be freed with free() when no longer needed
** Return : KT_OK if successful, KT_ERR if argument type is not a string
** If something wrong occured, the returnedString will be an empty string
*/
Kt_Status
kt_MessageStringGet(Kt_Message msg,
		    int argPos, /* position of the argument to read */
		    char **returnedString /* where to store the returned value */);

/*
** Retrieve the argPos'th argument of this message, which must be of bytes type
** Note : returnedString must be freed with free() when no longer needed
** Return : KT_OK if successful, KT_ERR if argument type is not a bytes *
*/
Kt_Status
kt_MessageBytesGet(Kt_Message msg,
		   int argPos, char **returnedBytes, size_t *returnedLength);

/*
** Is argument #argPos in this message a string?
*/
Bool
kt_MessageArgIsString(Kt_Message msg, int argPos);

/*
** Ask the server to store this message and to send it (just like if
** sent with @code{kt_MessageSend()} if ever it loses its connection to me
** some other way than by @code{kt_CloseConnection()}.
** Return : KT_OK if ok, KT_ERR otherwise (krl might be 0)
*/
Kt_Status
kt_MessageSendOnExit(Kt_Context context, Kt_KRL krl, Kt_Message msg);

/*
** Return : the sender (ICE coordinates) of current message.
** This function *can* return an empty string (but never NULL), since
** kt_MessageNew() is invoked without any Kt_Context in it, and
** is therefore a "standalone" function.
** Do not modify or free the returned value, it belongs to Koalatalk.
*/
char *
kt_MessageSender(Kt_Message msg);


/* File kt_peer.c */

/*
** Open a direct connection to a peer
*/
Kt_PeerConnection
kt_PeerOpenConnection(Kt_Context context,
		      char *peerLocation /* location, in ICE format */);

/*
** Return : the IceConn descriptor to this peer
*/
IceConn
kt_PeerReturnIceConnection(Kt_PeerConnection peer);

/*
** Return : the protocol used to talk with this peer
*/
int
kt_PeerReturnIceProtocol(Kt_PeerConnection peer);

/*
** Specify an error handler for a specific peer
*/
void
kt_PeerSetErrorHandler(Kt_Context context, Kt_PeerConnection peer, void *handler);

/*
** Specify a callback to be called if a message arrives from a peer.
*/
void
kt_PeerAddMsgCallback(Kt_Context context, /* the connection context */
		      Kt_CallbackPeerMsg callback,  /* the callback */
		      void *userData  /* private date (closure) */);

/* File kt_krl.c */

/*
** Create a Koalatalk Resource Locator, in order to be able to send
** a message
** Return : the krl (must be freed by the caller or memory leak will ensue)
*/
Kt_KRL
kt_KRLNew(Kt_Context context, char *krlName);

/*
** Declare as an observer for a Koalatalk Resource. @var{callback} will
** receive a msg in argument, it must @strong{not} free it (it will be
** automatically freed by the toolkit). @var{krs} is the Koalatalk Resource
** Specification to be observed (a regular expression).
** Return : (CARD32) 0 if failure (not connected to any server)
*/
CARD32
kt_KRObserve(Kt_Context context, char *krs,   /* the kr specification */
	     Kt_CallbackServerMsg callback, void *userData);

/*
** Stop observing a Koalatalk Resource
** Return : KT_OK if the KR is now unobserved, KT_ERR otherwise
*/
Kt_Status
kt_KRUnObserve(Kt_Context context, CARD32 krId);

/*
** Declare as a handler for a Koalatalk Resource
** The callback will receive a msg in argument, it must *not* 
** free it (it will be automatically freed by the toolklit).
** Return : (CARD32) 0 if failure (not connected to any server)
*/
CARD32
kt_KRHandle(Kt_Context context, char *krs,
	    Kt_CallbackHandlerMsg callback, void *userData);

/*
** Free the memory allocated in a krl
*/
void
kt_KRLFree(Kt_KRL krl);


#endif /* __KT_LIB_H__ */
