/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kts_process.c,v 1.2 1994/12/15 13:38:51 beust Exp $
 */

#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>

#include "kts_main.h"

/*
** Set the identity for the given client
*/

static Bool
findId(Kts_Connection entry, int fd)
{
   if (entry -> fd == fd) return True;
   else return False;
}

/*
**---------------------------------------------------------------------------
** kts_packetDeclareKRL
** Declare a KRL for a client
*/

static Bool
findClientByFd(Kts_Connection entry, int fd)
{
   if (False == entry -> isVirtual) {
      if (fd == entry -> fd) return True;
      else return False;
   }
   else return False;
}

static void
kts_packetDeclareKRL(GV gv, CARD32 krlId, char *krlName,
		     int clientFd)
{
   Kts_Connection sender = NULL; DataBase db;
   Kts_DeclaredKRL dk;

   db = gv -> connections;
   sender = DB_LocateEntry(db,
			   (void *) ((unsigned long) findClientByFd),
			   (void *) ((unsigned long) clientFd));
                                /* alpha kludge to avoid warning */

   NEW(dk, Kts_DeclaredKRLRec);
   dk -> krlName = kt_strdup(krlName);
   dk -> krlId = krlId;
   DB_AddEntry(sender -> declaredKRL, dk);
}

/*
**---------------------------------------------------------------------------
** kts_setMessageSendOnExit
** Store this message as a message to send when this client dies
*/
static
kts_setMessageSendOnExit(GV gv, IceConn iceConn,
			 Kt_messageBuffer mb, CARD32 krlId)
{
   Kts_Connection sender = NULL;
   DataBase db;
   size_t newmbSize;
   int clientFd = IceConnectionNumber(iceConn);
   Kt_messageBuffer newmb;

   newmbSize = (mb -> length) << 3 + ICE_HEADER_SIZE;
   db = gv -> connections;
   sender = DB_LocateEntry(db,
			   (void *) ((unsigned long) findClientByFd),
			   (void *) ((unsigned long) clientFd));
   ASSERT(sender);

   {
      int n = ICE_HEADER_SIZE;
      newmbSize = (mb -> length) << 3;
      newmbSize = newmbSize + n;
   }
   newmb = sender -> msgSendOnExit;
   if (0 != newmb) SAFE_FREE(newmb);
   newmb = (Kt_messageBuffer) malloc(newmbSize);
   memcpy(newmb, mb, newmbSize);
   sender -> msgSendOnExit = newmb;
   sender -> msgSendOnExitKRLId = krlId;
}



/*
**---------------------------------------------------------------------------
** kts_packetHandleKR
** A handler just signalled itself to me. Put it into my database.
*/
static void
kts_packetHandleKR(GV gv, char *krSpec, CARD32 krSpecId, int clientFd,
		   char *iceCoord)
{
   Kt_handler handler;

   NEW(handler, Kt_handlerRec);
   handler -> callback = CALLBACK_UNKNOWN;
   handler -> userData = NULL;
   handler -> krs = kt_strdup(krSpec);
   handler -> iceCoord = kt_strdup(iceCoord);
   handler -> krSpecId = krSpecId;
   handler -> fd = clientFd;
   DB_AddEntry(gv -> handlers, handler);
}

/*
**---------------------------------------------------------------------------
** kts_packetObserveKR
** We received an OBSERVE packet. Record the Koalatalk Resource Spec.
** and its id. This id will be sent back to it when a matching packet
** arrives
*/

static void
kts_packetObserveKR(GV gv, char *krSpec, CARD32 krSpecId, int clientFd)
{
   Kts_Connection sender = NULL;
   DataBase db;
   Kts_InterestedInKR ir;

   db = gv -> connections;
   sender = DB_LocateEntry(db,
			   (void *) ((unsigned long) findClientByFd),
			   (void *) ((unsigned long) clientFd));

   NEW(ir, Kts_InterestedInKRRec);
   ir -> krSpec = kt_strdup(krSpec);
   ir -> krSpecId = krSpecId;
   DB_AddEntry(sender -> interestedInKR, ir);
}


/*
**---------------------------------------------------------------------------
** kts_packetUnobserveKR
** We received an UNOBSERVE packet. Remove the specified KRS from
** this client's database
*/

static Bool
findKR(Kts_DeclaredKRL entry, CARD32 krlId)
{
   if (entry -> krlId == krlId) return True;
   else return False;
}

static void
kts_packetUnobserveKR(GV gv, CARD32 krSpecId, int clientFd)
{
   Kts_Connection sender = NULL;
   DataBase db;
   Kts_InterestedInKR ir;

   db = gv -> connections;
   sender = DB_LocateEntry(db,
			   (void *) ((unsigned long) findClientByFd),
			   (void *) ((unsigned long) clientFd));
   ASSERT(sender);
   ir= DB_LocateEntry(sender -> interestedInKR,
		      (void *) ((unsigned long) findKR),
		      (void *) ((unsigned long) krSpecId));
   ASSERT(ir);
   DB_RemoveEntry(sender -> interestedInKR, ir);
}

/*
**---------------------------------------------------------------------------
** kts_processMessage
** Main function to interpret the messages
*/

    
/* callback of type IcePAprocessMsgProc */
void
kts_processMessage(IceConn iceConn, void *clientData,
		   int opCode, unsigned long length, Bool swap)
{
   GV gv = (GV) clientData;
   void *data;
   char *fillBuffer;

   ASSERT(iceConn);
   if (! IceValidIO(iceConn)) {
      char *s = IceConnectionString(iceConn);
      fprintf(stderr, "*** invalid iceConn '%s'\n", s);
      free(s);
      kts_removeRealClient(gv, IceConnectionNumber(iceConn));
      return;
   }

   switch(opCode) {
      case Ktp_NEW_CLIENT : {
	 size_t extraLength;
	 Ktm_NewClient header;
	 Ktm_NewClientReply reply;
	 Kts_Connection client;
	 int protocol;
	 CARD32 identityLength, krsLength;
	 
	 char id[64], op[64];
	 char *pdata;
	 unsigned long idLength;
	 int newId = gv -> serial++;
	 unsigned long lg[4];

	 DL(DEBUG_PACKETS, printf("processMessage: NEW_CLIENT\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_NewClientRec, header, data);

	 /* read the identity field */
	 identityLength = kt_decodeULONG(& header -> identityLength);
	 krsLength = kt_decodeULONG(& header -> krsLength);
	 pdata = (char *) & header -> identity;
	 memcpy(id, pdata, identityLength);
	 pdata += identityLength;
	 id[identityLength] = '\0';
	 /* read the operation field */
	 memcpy(op, pdata, krsLength);
	 pdata += krsLength;
	 op[krsLength] = '\0';

	 /*
         ** Fill in the coordinate field for this client.
         */
	 client = DB_LocateEntry(gv -> connections,
				 (void *) ((unsigned long) findClientByFd),
				 (void *) ((unsigned long) IceConnectionNumber(iceConn)));
	 client -> iceSenderCoord = kt_strdup(id);
	 DP(printf("associated fd %d with '%s'\n", client -> fd, id));

	 /*
	 ** Generate the reply (which will contain the unique id
	 ** affected to this client).
	 */
	 protocol = gv -> mainProtocol;
	 IceGetHeaderExtra(iceConn, protocol, Ktp_NEW_CLIENT_REPLY,
			   ICE_HEADER_SIZE, 0,
			   Ktm_NewClientReplyRec, reply, fillBuffer);
	 reply -> length = WORD64COUNT(sizeof(CARD32));
	 IceWriteData(iceConn, (reply -> length) << 3, (char *) & newId);
	 IceFlush(iceConn);
	 DP(printf("sending new client reply to fd %d\n",
		   IceConnectionNumber(iceConn)));

	 /*
	 ** Inform this new client of all know handlers
	 */
	 kts_sendHandlerInfo(gv, iceConn);
	 IceFlush(iceConn);

	 /*
	 ** If the krs field is non-null, then we must have messages
	 ** for this client.
	 */

	 DP(printf("   New client on fd=%d (op='%s'), total number of clients: %d\n", IceConnectionNumber(iceConn), op, DB_Count(gv -> connections)));

	 break;
      }

      case Ktp_KRL_DECLARE : {
	 Ktm_KRLDeclare header;
	 CARD32 senderId, l, krlId;
	 char *krlName;

	 DL(DEBUG_PACKETS, printf("processMessage: KRL_DECLARE\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_KRLDeclareRec, header, data);
	 senderId = kt_decodeULONG(& header -> id);
	 krlId = kt_decodeULONG(& header -> krlId);
	 l = kt_decodeULONG(& header -> nameLength);
	 krlName = (char *) &header -> name;
	 krlName[l] = '\0';
	 /*
	 ** Declare this new KRL for the client
	 */
	 kts_packetDeclareKRL(gv, krlId, krlName, IceConnectionNumber(iceConn));
/*
	 SAFE_FREE(krlName);    
*/
	 break;
      }

      case Ktp_MESSAGE_SENT : {
	 Ktm_MessageSent header;
	 Kt_Message msg;
	 char *string;
	 int l, *tab;

	 DL(DEBUG_PACKETS, printf("processMessage: MESSAGE_SENT\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_MessageSentRec, header, data);
	 header -> krlId = kt_decodeULONG(& header -> krlId);
	 header -> opLength = kt_decodeULONG(& header -> opLength);
	 kts_dispatchMessage(gv, (Kt_messageBuffer) header, header -> krlId,
			     IceConnectionNumber(iceConn));
	 break;
      }

      case Ktp_KR_OBSERVE : {
	 Ktm_KRObserve header;
	 CARD32 l, krSpecId;
	 char *krSpec;

	 DL(DEBUG_PACKETS, printf("processMessage: KR_OBSERVE\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_KRObserveRec, header, data);
	 krSpecId = kt_decodeULONG(& header -> krSpecId);
	 DP(printf("received an observe for id %x\n", krSpecId));
	 l = kt_decodeULONG(& header -> krSpecLength);
	 krSpec = (char *) &header -> krSpec;
	 krSpec[l] = '\0';
	 /*
	 ** Memorize this observation in our private database
	 */
	 kts_packetObserveKR(gv, krSpec, krSpecId, IceConnectionNumber(iceConn));
	 /*
         ** Check to see if we can get rid of some pending messages
         */
	 kts_queueSendMessages(gv, krSpec, iceConn, krSpecId);
/*
  SAFE_FREE(krSpec);
*/
	 break;
      }

      case Ktp_KR_UNOBSERVE : {
	 Ktm_KRUnObserve header;
	 CARD32 l, krSpecId;
	 char *krSpec;

	 DL(DEBUG_PACKETS, printf("processMessage: KR_UNOBSERVE\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_KRUnObserveRec, header, data);
	 krSpecId = kt_decodeULONG(& header -> krSpecId);
	 DP(printf("received an unobserve for id %x\n", krSpecId));
	 /*
	 ** Memorize this unobservation in our private database
	 */
	 kts_packetUnobserveKR(gv, krSpecId, IceConnectionNumber(iceConn));
/*
  SAFE_FREE(krSpec);
*/
	 break;
      }

      case Ktp_KR_HANDLE : {
	 Ktm_KRHandle header;
	 CARD32 lspec, krSpecId, lsender;
	 char *krSpec, *senderIceCoo, *p;

	 DL(DEBUG_PACKETS, printf("processMessage: KR_HANDLE\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_KRHandleRec, header, data);
	 p = data;
	 p += sizeof(CARD32);   /* skip id */
	 p += sizeof(CARD32);   /* skip opLength */
	 READLONG(krSpecId, p);
	 READLONG(lspec, p);
	 READLONG(lsender, p);

	 /*
	 ** Retrieve the specification in krSpec
	 */
	 krSpec = (char *) malloc(lspec + 1);
	 memcpy(krSpec, p, lspec); p += lspec;
	 krSpec[lspec] = '\0';

	 /*
	 ** Retrieve the name sender in cenderIceCoo
	 */
	 senderIceCoo = (char *) malloc(lsender + 1);
	 memcpy(senderIceCoo, p, lsender); p += lsender;
	 senderIceCoo[lsender] = '\0';
	 
	 /*
	 ** Memorize this observation in our private database
	 */
	 kts_packetHandleKR(gv, krSpec, krSpecId, IceConnectionNumber(iceConn),
		      senderIceCoo);

	 /*
	 ** And tell each of our client about this new handler
	 */
	 DP(printf("broadcasting info on kr='%s' ice='%s'\n", krSpec, senderIceCoo));
	 kts_broadcastNewHandler(gv, krSpec, IceConnectionNumber(iceConn),
				 krSpecId, senderIceCoo);
	 SAFE_FREE(krSpec); SAFE_FREE(senderIceCoo);
/*
  SAFE_FREE(krSpec);
*/
	 break;
      }

      case Ktp_LAUNCH : {
	 Ktm_Launch lpk;
	 CARD32 krlLength;
	 char *krl, *data, *p;

	 DL(DEBUG_PACKETS, printf("processMessage: LAUNCH\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_LaunchRec, lpk, data);
	 p = data;
	 p += sizeof(CARD32);   /* skip id */
	 p += sizeof(CARD32);   /* skip opLength */
	 READLONG(krlLength, p);
	 krl = (char *) malloc(krlLength + 1);
	 memcpy(krl, p, krlLength); p+= krlLength;
	 krl[krlLength] = '\0';
	 kts_checkVirtualClients(gv, krl, False);
	 free(krl);
	 break;
      }

      case Ktp_MESSAGE_SEND_ON_EXIT : {
	 Ktm_MessageSendOnExit header;
	 Kt_Message msg;
	 char *string;
	 int l, *tab;
	 CARD32 krSpecId;

	 DL(DEBUG_PACKETS, printf("processMessage: MESSAGE_SEND_ON_EXIT\n"));
	 IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
				Ktm_MessageSendOnExitRec, header, data);
	 kts_setMessageSendOnExit(gv, iceConn,
				  (Kt_messageBuffer) header,
				  kt_decodeULONG(& header -> krlId));
	 break;
      }


      default :
	 printf("unknown opcode %d\n", opCode);
   }
}
