/* Intel native support for ix86 DG/ux systems.
   Copyright (C) 1988, 1989, 1991, 1992, 1994, 1996 Free Software Foundation, Inc.
   Written by Takis Psarogiannakopoulos
   Department of Pure Mathematics, Cambridge University
   UK , e-mail <takis@dpmms.cam.ac.uk>


This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "frame.h"
#include "inferior.h"
#include "language.h"
#include "gdbcore.h"

#include <sys/types.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <signal.h>
#include <sys/user.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/dg_xtrace.h>
#include <sys/procfs.h>

#define user ptrace_user

#ifdef TARGET_HAS_HARDWARE_WATCHPOINTS
#include <sys/debugreg.h>
#endif

#include <sys/file.h>
#include "gdb_stat.h"
#include "floatformat.h"
#include "target.h"

extern int first_time_dg_attach;

#include <sys/regset.h>
#define SS      R_SS      /* 18 */
#define UESP    R_UESP
#define EFL     R_EFL     /* eflags */
#define CS      R_CS
#define EIP     R_EIP     /* pc */
#define ERR     R_ERR
#define TRAPNO  R_TRAPNO
#define EAX     R_EAX
#define ECX     R_ECX
#define EDX     R_EDX
#define EBX     R_EBX
#define ESP     R_ESP     /* sp */
#define EBP     R_EBP     /* fp  */
#define ESI     R_ES
#define EDI     R_EDI
#define DS      R_DS
#define ES      R_ES
#define FS      R_FS
#define GS      R_GS      /* 0 */

/* For SVR4 solib.c with DG/ux FLAVOR */
#include "bfd.h"
#include "elf-bfd.h"


/* For SVR4 style core files in ix86 DG/ux */
static void fetch_core_registers PARAMS ((char *, unsigned, int, CORE_ADDR));

/* blockend is the value for u ptrace_user 
 * u.pt_mcontext.gregs -u = first available 
 * general register which is GS (see below) 
 * from sys/regset.h
 *
 *define NGREG   19
 *define R_SS            18
 *define R_UESP          17
 *define R_EFL           16
 *define R_CS            15
 *define R_EIP           14
 *define R_ERR           13
 *define R_TRAPNO        12
 *define R_EAX           11
 *define R_ECX           10
 *define R_EDX           9
 *define R_EBX           8
 *define R_ESP           7
 *define R_EBP           6
 *define R_ESI           5
 *define R_EDI           4
 *define R_DS            3
 *define R_ES            2
 *define R_FS            1
 *define R_GS            0
 *
 */



/* Obtain the address of a register 
 * blockend is usually the offset
 * offsetof (struct ptrace_user, pt_mcontext.gregs)
 * Or for floating point info 
 * offsetof (struct ptrace_user, pt_mcontext.fpregs)
 */

#if !defined(offsetof)
#define offsetof(TYPE, MEMBER) ((unsigned long) &((TYPE *)0)->MEMBER)
#endif

/* For general geristers Info */
#define DGUX_GREGS_OFFSET \
    (offsetof (struct ptrace_user, pt_mcontext.gregs))
/* For floating point registers Info */
#define DGUX_FPREGS_OFFSET \
    (offsetof (struct ptrace_user, pt_mcontext.fpregs.fp_reg_set.fpchip_state.state))

/* this table must line up with REGISTER_NAMES in tm-i386dgux.h
 * remeber threee more registers here ie ESP,ERR,TRAPNO
 */

static int regmap[] =
{
  EAX, ECX, EDX, EBX,
  UESP, EBP, ESI, EDI,
  EIP, EFL, CS, SS,
  DS, ES, FS, GS,
  ESP, ERR, TRAPNO,
};



/* Return the address in the core dump or inferior of register REGNO.
   BLOCKEND is the address of the registers  */

CORE_ADDR
register_addr (regno, blockend)
     int regno;
     CORE_ADDR blockend;
{
  CORE_ADDR addr;

  if (regno < 0 || regno >= ARCH_NUM_REGS)
    error ("Invalid register number %d.", regno);

  REGISTER_U_ADDR (addr, blockend, regno);

  return addr;
}



int
i386dgux_register_u_addr (blockend, regnum)
     int blockend;
     int regnum;
{
   struct ptrace_user u;
   int ubase;
   unsigned int fpchip;
   unsigned int fpoffset;
   ubase = blockend;

   if (regnum >= FPCHIPSTATUS_REGNUM)
   {
      fpchip = (char *) &u.pt_mcontext.fpregs.fp_reg_set.fpchip_state.status - (char *) &u;
      return (fpchip);
   }

   if (regnum < FPC_REGNUM)
   {
      return (ubase + 4 * regmap[regnum]);
   }

   if (regnum >= FPC_REGNUM && regnum < FP0_REGNUM)
   {
      fpoffset = (char *) &u.pt_mcontext.fpregs.fp_reg_set.fpchip_state.state - (char *) &u;
      return (fpoffset + (4 * (regnum-FPC_REGNUM)));
   }

   if (regnum >= FP0_REGNUM && regnum < FPCHIPSTATUS_REGNUM)
   {
      fpoffset = (char *) &u.pt_mcontext.fpregs.fp_reg_set.fpchip_state.state - (char *) &u;
      return (fpoffset + 28 + (10 * (regnum - FP0_REGNUM)));
   }


}

/* Wait for child to do something.  Return pid of child, or -1 in case
   of error; store status through argument pointer OURSTATUS.  */


static int
proc_wait_dg (pid, status)
     int pid;
     int *status;
{
  /* USE wait3 */
  return wait3(status, 0 ,NULL);
}

extern void set_sigint_trap();
extern void set_sigio_trap ();
extern void clear_sigint_trap();
extern void clear_sigio_trap ();


int
child_wait (pid, ourstatus)
     int pid;
     struct target_waitstatus *ourstatus;
{
  union dg_xtrace_u dgxu;
  int status;
  int save_errno;
  int tpid=0;
  int result=0;
  int options=0;
  int w=0;

  do
  {
    set_sigint_trap();  /* Causes SIGINT to be passed on to the
				attached process. */
    set_sigio_trap ();

#if defined(DGDEBUG)
    printf("pid before  XTRACE  WAIT is %d\n",pid);
    printf("errno before XTRACE WAIT call is set to 0\n");
    printf("inferior_pid is %d\n",inferior_pid);
    printf("target_has_execution is %d\n", target_has_execution);
#endif

    /* Insist on XTRACE WAIT
     * if we doing attach
     * otherwise wait3 may
     * not catch the status if
     * the two processes are not
     * traditionally related.
     */

    if (attach_flag)
    {

       errno=0;
#if defined(DGDEBUG)
       printf("Using DG_XTRACE WAIT flavor.\n");
#endif
       result=DG_XTRACE_WAIT_FOR_TARGET (&tpid, &status, options, &dgxu);
       save_errno = errno;
       if (result !=0)
       {

         fprintf_unfiltered (gdb_stderr, "child_wait|Child process missing...\n");
         ourstatus->kind = TARGET_WAITKIND_SIGNALLED;
         ourstatus->value.sig = TARGET_SIGNAL_UNKNOWN;
         error("DG_XTRACE WAIT return -1 with errno %d\n",errno);
         pid= -1;
         continue;
       }
       pid=tpid;
    }
    else
    {
      /* USE wait3 */
#if defined(DGDEBUG)
      printf("Using wait3 flavor.\n");
#endif
      pid=wait3 (&status, 0, NULL);
      save_errno = errno;

    }

    w=status;
    clear_sigio_trap ();
    clear_sigint_trap();

#if defined(DGDEBUG)
    printf("pid returned from XTRACE WAIT is %d\n", pid);
    printf("wait_status w is %d\n",w);
    printf("WIFEXITED(w) is %d\n", WIFEXITED(w));
    printf("WEXITSTATUS(w) is %d\n", WEXITSTATUS(w));
    printf("WIFSIGNALED(w) is %d\n", WIFSIGNALED(w));
    printf("WTERMSIG(w) is %d\n", WTERMSIG(w));
    printf("WIFSTOPPED(w) is %d\n", WIFSTOPPED(w));
    printf("WSTOPSIG(w) is %d\n", WSTOPSIG(w));
    printf("WIFCONTINUED(w) is %d\n", WIFCONTINUED(w));
    printf("\n");
#endif

    /* This bit is only for wait3 case */
    if (pid == -1)
     {
        if (save_errno == EINTR)
          continue;
        fprintf_unfiltered (gdb_stderr, "child_wait|Child process unexpectedly missing: %s.\n",
				safe_strerror (save_errno));
        /* Claim it exited with unknown signal.  */
        ourstatus->kind = TARGET_WAITKIND_SIGNALLED;
        ourstatus->value.sig = TARGET_SIGNAL_UNKNOWN;
        return -1;
     }

  } while (pid != inferior_pid); /* Some other (xtraced !) child died or stopped */
  store_waitstatus (ourstatus, status);
  return pid;
}


int
i386dgux_kernel_u_size ()
{
   /* should be 2048 */
   struct ptrace_user u;
   return (sizeof(u));
}



/* For SVR4 shared libs with DGUX flavor
 * this code is from elfread.c
 */


char *
elf_interpreter (abfd)
     bfd *abfd;
{
  sec_ptr interp_sec;
  unsigned size;
  char *interp = NULL;

  interp_sec = bfd_get_section_by_name (abfd, ".interp");
  if (interp_sec)
    {
      size = bfd_section_size (abfd, interp_sec);
      interp = alloca (size);
      if (bfd_get_section_contents (abfd, interp_sec, interp, (file_ptr)0,
                                    size))
        {
          interp = savestring (interp, size - 1);
        }
      else
        {
          interp = NULL;
        }
    }
  return (interp);
}

/* For ix86 DG/ux */
void
i386dgux_extract_return_value(type, regbuf, valbuf)
     struct type *type;
     char regbuf[REGISTER_BYTES];
     char *valbuf;
{
/* On AIX, floating point values are returned in floating point registers.  */
#ifdef I386_AIX_TARGET
  if (TYPE_CODE_FLT == TYPE_CODE(type))
    {
      double d;
      /* 387 %st(0), gcc uses this */
      floatformat_to_double (&floatformat_i387_ext,
                             &regbuf[REGISTER_BYTE(FP0_REGNUM)],
                             &d);
      store_floating (valbuf, TYPE_LENGTH (type), d);
    }
  else
#endif /* I386_AIX_TARGET */
    {
      memcpy (valbuf, regbuf, TYPE_LENGTH (type));
    }
}


/* Get saved user PC for sigtramp from the pushed ucontext on the stack
   for all three variants of SVR4 sigtramps.  */


CORE_ADDR
i386dgux_sigtramp_saved_pc (frame)
     struct frame_info *frame;
{
  CORE_ADDR saved_pc_offset = 4;
  char *name = NULL;

  find_pc_partial_function (frame->pc, &name, NULL, NULL);
  if (name)
    {
      if (STREQ (name, "_sigreturn"))
        saved_pc_offset = 132 + 14 * 4;
      else if (STREQ (name, "_sigacthandler"))
        saved_pc_offset = 80 + 14 * 4;
      else if (STREQ (name, "sigvechandler"))
        saved_pc_offset = 120 + 14 * 4;
    }

  if (frame->next)
    return read_memory_integer (frame->next->frame + saved_pc_offset, 4);
  return read_memory_integer (read_register (SP_REGNUM) + saved_pc_offset, 4);
}


/* Print float regs as in mxdb */

typedef struct dguxf387
{
  unsigned short control;
  unsigned short r0;
  unsigned short status;
  unsigned short r1;
  unsigned short tag;
  unsigned short r2;
  unsigned long eip;
  unsigned short code_seg;
  unsigned short opcode;
  unsigned long operand;
  unsigned short operand_seg;
  unsigned short r3;
  unsigned char regs[8][10];
} dguxf387_t;

static void
i386dgux_print_f387_control (control)
     unsigned int control;
{
  printf_unfiltered ("(fctrl) control %s: ", local_hex_string(control));
  printf_unfiltered ("compute to ");
  switch ((control >> 8) & 3)
    {
    case 0: printf_unfiltered ("24 bits; "); break;
    case 1: printf_unfiltered ("(bad); "); break;
    case 2: printf_unfiltered ("53 bits; "); break;
    case 3: printf_unfiltered ("64 bits; "); break;
    }
  printf_unfiltered ("round ");
  switch ((control >> 10) & 3)
    {
    case 0: printf_unfiltered ("NEAREST; "); break;
    case 1: printf_unfiltered ("DOWN; "); break;
    case 2: printf_unfiltered ("UP; "); break;
    case 3: printf_unfiltered ("CHOP; "); break;
    }
  if (control & 0x3f)
    {
      printf_unfiltered ("mask:");
      if (control & 0x0001) printf_unfiltered (" INVALID");
      if (control & 0x0002) printf_unfiltered (" DENORM");
      if (control & 0x0004) printf_unfiltered (" DIVZ");
      if (control & 0x0008) printf_unfiltered (" OVERF");
      if (control & 0x0010) printf_unfiltered (" UNDERF");
      if (control & 0x0020) printf_unfiltered (" LOS");
      printf_unfiltered (";");
    }
  printf_unfiltered ("\n");
  if (control & 0xe080) warning ("reserved bits on: %s\n",
                                local_hex_string(control & 0xe080));
}

void
i386dgux_float_info()
{

  extern char registers[];
  unsigned int offset;
  union dg_xtrace_u dgxu;
  struct ptrace_user u;
  unsigned int total;
  char mess[128];
  char f387buf[114];

  /* 19 General Regs 4 bytes each
   *  = 76 Bytes
   * fpregset should be 380 bytes
   * but read only 112 = 10*4 
   * + 7*4 + 4
   */

  if (!target_has_registers)
  {
    error ("The program has no registers now.");
  }  

  if (inferior_pid)
  {
     /* if it is a core dont use this routine */
     if (target_has_execution)
     {
        i386dgxtrace_read_float_registers();
     }
     memcpy(&f387buf, &registers[REGISTER_BYTE(FPC_REGNUM)], 114);
     i386dgux_print_f387_status ((struct dguxf387 *)f387buf);
  }
  else
  {
      printf("float info: use *info all* to see float registers \n");
  }

}


i386dgux_print_f387_status (ep)
     struct dguxf387 *ep;
{
  int i;
  int top;
  int fpreg;
  unsigned char *p;

  if (ep->status != 0) 
  {
    printf_unfiltered ("(fstat) ");
    print_387_status_word (ep->status);
  }

  i386dgux_print_f387_control (ep->control);
  printf_unfiltered ("last exception: ");
  printf_unfiltered ("opcode %s; ", local_hex_string(ep->opcode));
  printf_unfiltered ("pc %s:", local_hex_string(ep->code_seg));
  printf_unfiltered ("%s; ", local_hex_string(ep->eip));
  printf_unfiltered ("operand %s", local_hex_string(ep->operand_seg));
  printf_unfiltered (":%s\n", local_hex_string(ep->operand));

  top = (ep->status >> 11) & 7;

  printf_unfiltered ("  regno   tag   msb            lsb   value\n");
  for (fpreg = 7; fpreg >= 0; fpreg--)
    {
      double val;

      printf_unfiltered ("%s fst%d: ", fpreg == top ? "=>" : "  ", fpreg);

      switch ((ep->tag >> (fpreg * 2)) & 3)
        {
        case 0: printf_unfiltered ("valid "); break;
        case 1: printf_unfiltered ("zero  "); break;
        case 2: printf_unfiltered ("trap  "); break;
        case 3: printf_unfiltered ("empty "); break;
        }
      for (i = 9; i >= 0; i--)
        printf_unfiltered ("%02x", ep->regs[fpreg][i]);
 
      floatformat_to_double (&floatformat_i387_ext, (char *)ep->regs[fpreg],
                               &val);
      printf_unfiltered ("  %g\n", val);
    }
  if (ep->r0)
    printf_unfiltered ("warning: reserved0 is %s\n", local_hex_string(ep->r0));
  if (ep->r1)
    printf_unfiltered ("warning: reserved1 is %s\n", local_hex_string(ep->r1));
  if (ep->r2)
    printf_unfiltered ("warning: reserved2 is %s\n", local_hex_string(ep->r2));
  if (ep->r3)
    printf_unfiltered ("warning: reserved3 is %s\n", local_hex_string(ep->r3));
  printf_unfiltered("See also *info all* ...\n"); 
}


#if defined(TAKIS_I387)
/* For floating point Register
 * and for studff that should
 * be in i387-tdep.c
 */
void
i387_to_double (from, to)
     char *from;
     char *to;
{
  long *lp;
  /* push extended mode on 387 stack, then pop in double mode
   *
   * first, set exception masks so no error is generated -
   * number will be rounded to inf or 0, if necessary
   */
  asm ("pushl %eax");           /* grab a stack slot */
  asm ("fstcw (%esp)");         /* get 387 control word */
  asm ("movl (%esp),%eax");     /* save old value */
  asm ("orl $0x3f,%eax");               /* mask all exceptions */
  asm ("pushl %eax");
  asm ("fldcw (%esp)");         /* load new value into 387 */

  asm ("movl 8(%ebp),%eax");
  asm ("fldt (%eax)");          /* push extended number on 387 stack */
  asm ("fwait");
  asm ("movl 12(%ebp),%eax");
  asm ("fstpl (%eax)");         /* pop double */
  asm ("fwait");

  asm ("popl %eax");            /* flush modified control word */
  asm ("fnclex");                       /* clear exceptions */
  asm ("fldcw (%esp)");         /* restore original control word */
  asm ("popl %eax");            /* flush saved copy */
}

void
double_to_i387 (from, to)
     char *from;
     char *to;
{
  /* push double mode on 387 stack, then pop in extended mode
   * no errors are possible because every 64-bit pattern
   * can be converted to an extended
   */
  asm ("movl 8(%ebp),%eax");
  asm ("fldl (%eax)");
  asm ("fwait");
  asm ("movl 12(%ebp),%eax");
  asm ("fstpt (%eax)");
  asm ("fwait");
}

#endif /* TAKIS_I387 */


/* Record that register REGNO contains VAL.
   This is used when the value is obtained from the inferior or core dump,
   so there is no need to store the value there.

   If VAL is a NULL pointer, then it's probably an unsupported register.  We
   just set it's value to all zeros.  We might want to record this fact, and
   report it to the users of read_register and friends.
*/

/* Needed fot the procfs stuff bellow
 * Also it is reentrant
 *
 */

#if defined(TAKIS_SUPPLY_REGISTER) 
void
supply_register (regno, val)
     int regno;
     char *val;
{
  bcopy (val, &registers[REGISTER_BYTE (regno)], REGISTER_RAW_SIZE (regno));
}
#endif /* TAKIS_SUPPLY_REGISTER */

/*
 * We need the following in ix86 DG/ux to be able to handle the code 
 * for core files of style SVR4
 * Actually this is the only style that will work for ix86 DG/ux
 * Modify the bfd simirarly. 
 *
 * For the i386 for example, the general register set is typically defined
 * by:

        typedef int gregset_t[19];              (for DG/ux in <sys/regset.h>)

        #define GS      0                       
        #define FS      1
        ...
        #define UESP    17
        #define SS      18

     and the floating point set by:

        typedef struct fpregset
          {
            union
              {
                struct fpchip_state     // fp extension state //
                {
                  int state[27];        // 287/387 saved state //
                  int status;           // status word saved at exception //
                } fpchip_state;
                struct fp_emul_space    // for emulators //
                {
                  char fp_emul[246];
                  char fp_epad[2];
                } fp_emul_space;
                int f_fpregs[62];       // union of the above //
              } fp_reg_set;
            long f_wregs[33];           // saved weitek state //
        } fpregset_t;

 *   These routines provide the packing and unpacking of gregset_t and
 *   fpregset_t formatted data.
 *
 */



/*  FIXME:  These routine absolutely depends upon (NUM_REGS - NUM_FREGS)
 *  being less than or equal to the number of registers that can be stored
 *  in a gregset_t.  Note that with the current scheme there will typically
 *  be more registers actually stored in a gregset_t that what we know
 *  about.  This is bogus and should be fixed.
 *  Given a pointer to a general register set in /proc format (gregset_t *),
 *   unpack the register contents and supply them as gdb's idea of the current
 *  register values.
 *
 */


void
supply_gregset (gregsetp)
     gregset_t *gregsetp;
{
  register int regi;
  register greg_t *regp = (greg_t *) gregsetp;
  extern int regmap[];

  for (regi = 0 ; regi <  NGREG ; regi++)
    {
      supply_register (regi, (char *) (regp + regmap[regi]));
    }
}

void
fill_gregset (gregsetp, regno)
     gregset_t *gregsetp;
     int regno;
{
  int regi;
  register greg_t *regp = (greg_t *) gregsetp;
  extern char registers[];
  extern int regmap[];

  for (regi = 0 ; regi < NGREG ; regi++)
    {
      if ((regno == -1) || (regno == regi))
        {
          *(regp + regmap[regi]) = *(int *) &registers[REGISTER_BYTE (regi)];
        }
    }
}

/*
 * For the i386 :
 * for DG/ux in <sys/regset.h> 

        typedef struct fpregset
          {
            union
              {
                struct fpchip_state     // fp extension state //
                {
                  int state[27];        // 287/387 saved state //
                  int status;           // status word saved at exception //
                } fpchip_state;
                struct fp_emul_space    // for emulators //
                {
                  char fp_emul[246];
                  char fp_epad[2];
                } fp_emul_space;
                int f_fpregs[62];       // union of the above //
              } fp_reg_set;
            long f_wregs[33];           // saved weitek state //
        } fpregset_t;


       int state[27] means int state[0] ... state[26]
       which we unpack as follows:
       fpregset->fp_reg_set.fpchip_state.state[0-6]=(7*4)=
        28 bytes = seven 4 bytes (int) floating registers
       We use the dguxf387 struct of ieeefp as a model:
        fctrl =  FPC_REGNUM
        fstat =  FPSTAT_REGNUM
        ftag  =  FPTAG_REGNUM
        fip   =  FIP_REGNUM
        fcs   =  FCS_REGNUM
        fopoff = FPOPOFF_REGNUM
        fopsel = FPOPSEL_REGNUM

       fpregset->fp_reg_set.fpchip_state.state[7-26]=(4*20)= 
       =80=8*10 = 8 floating point regs,10 bytes each(or 80 bits)
       ie fp_reg_set.fpchip_state.state[7 to 26] = 80 bytes
       Also we have the status:
       (from DG/ux fpchip_state above)
       fpchipstatus (the above "int status;") FPCHIPSTATUS_REGNUM
       last in the list of the registers for tm-i386dgux.h
 *   The routines below provide the packing and unpacking of
 *   fpregset_t formatted data.
 *
 */



/*  Given a pointer to a floating point register set in /proc format
 *  (fpregset_t *), unpack the register contents and supply them as gdb's
 *   idea of the current floating point register values.
 */

/* No emulation on Pentium machines ... */

void
supply_fpregset (fpregsetp)
 fpregset_t *fpregsetp;

{

  register int regi;
  char *from;
  char *ubase;

  supply_register (FPC_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[0]));
  supply_register (FPSTAT_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[1]));
  supply_register (FPTAG_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[2]));
  supply_register (FPIP_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[3]));
  supply_register (FPCS_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[4]));
  supply_register (FPOPOFF_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[5]));
  supply_register (FPOPSEL_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[6]));
  supply_register (FPCHIPSTATUS_REGNUM, (char *) &(fpregsetp -> fp_reg_set.fpchip_state.status));

  ubase = (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[7]);
  for (regi = FP0_REGNUM  ; regi < FPCHIPSTATUS_REGNUM ; regi++)
  {
      from = ubase + (10 * (regi - FP0_REGNUM));
      supply_register (regi, from);
  }
}


/* Given a pointer to a floating point register set in /proc format
 * (fpregset_t *), update the register specified by REGNO from gdb's idea
 * of the current floating point register set.  If REGNO is -1, update
 * them all.
 */

void
fill_fpregset (fpregsetp, regno)
fpregset_t *fpregsetp;
int regno;
{
  int regi;
  char *to;
  char *from;
  char *ubase;
  extern char registers[];
  ubase = (char *) &(fpregsetp -> fp_reg_set.fpchip_state.state[7]);
  for (regi = FP0_REGNUM ; regi < FPCHIPSTATUS_REGNUM ; regi++)
  {
      if ((regno == -1) || (regno == regi))
      {
          from = (char *) &registers[REGISTER_BYTE (regi)];
          to = ubase + (10 * (regi - FP0_REGNUM));
          memcpy (to, from, REGISTER_RAW_SIZE (regi));
      }
  }
  if ((regno == -1) || (regno == FPC_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[0] = *(int *) &registers[REGISTER_BYTE (FPC_REGNUM)];
  }

  if ((regno == -1) || (regno == FPSTAT_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[1] = *(int *) &registers[REGISTER_BYTE (FPSTAT_REGNUM)];
  }

  if ((regno == -1) || (regno == FPTAG_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[2] = *(int *) &registers[REGISTER_BYTE (FPTAG_REGNUM)];
  }

  if ((regno == -1) || (regno == FPIP_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[3] = *(int *) &registers[REGISTER_BYTE (FPIP_REGNUM)];
  }

  if ((regno == -1) || (regno == FPCS_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[4] = *(int *) &registers[REGISTER_BYTE (FPCS_REGNUM)];
  }

  if ((regno == -1) || (regno == FPOPOFF_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[5] = *(int *) &registers[REGISTER_BYTE (FPOPOFF_REGNUM)];
  }

  if ((regno == -1) || (regno == FPOPSEL_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.state[6] = *(int *) &registers[REGISTER_BYTE (FPOPSEL_REGNUM)];
  }

  if ((regno == -1) || (regno == FPCHIPSTATUS_REGNUM))
  {
      fpregsetp -> fp_reg_set.fpchip_state.status = *(int *) &registers[REGISTER_BYTE (FPCHIPSTATUS_REGNUM)];
  }


}


/*
 *
 * From core-regset.c to handle SVR4 style core files in ix86 DG/ux system

GLOBAL FUNCTION

        fetch_core_registers -- fetch current registers from core file

SYNOPSIS

        void fetch_core_registers (char *core_reg_sect,
                                          unsigned core_reg_size,
                                          int which, CORE_ADDR reg_addr)

DESCRIPTION

        Read the values of either the general register set (WHICH equals 0)
        or the floating point register set (WHICH equals 2) from the core
        file data (pointed to by CORE_REG_SECT), and update gdb's idea of
        their current values.  The CORE_REG_SIZE parameter is ignored.

NOTES

        Use the indicated sizes to validate the gregset and fpregset
        structures.
 *
 */


static void
fetch_core_registers (core_reg_sect, core_reg_size, which, reg_addr)
     char *core_reg_sect;
     unsigned core_reg_size;
     int which;
     CORE_ADDR reg_addr;        /* Unused in this version */
{
  gregset_t gregset;
  fpregset_t fpregset;

  if (which == 0)
    {
      if (core_reg_size != sizeof (gregset))
        {
          warning ("wrong size gregset struct in core file");
        }
      else
        {
          memcpy ((char *) &gregset, core_reg_sect, sizeof (gregset));
          supply_gregset (&gregset);
        }
    }
  else if (which == 2)
    {
      if (core_reg_size != sizeof (fpregset))
        {
          warning ("wrong size fpregset struct in core file");
        }
      else
        {
          memcpy ((char *) &fpregset, core_reg_sect, sizeof (fpregset));
          supply_fpregset (&fpregset);
        }
    }
}


/* Register that we are able to handle ELF file formats using standard
   procfs "regset" structures.  */

static struct core_fns regset_core_fns =
{
  bfd_target_elf_flavour,
  fetch_core_registers,
  NULL
};

void
_initialize_core_regset ()
{
  add_core_fns (&regset_core_fns);
}




/*********************************************
 From here we can calculate the
 KERNEL_U_ADDR with nlist and -lelf

void
_i386dgux_initialize_kernel_u_addr ()
{
  struct nlist names[2];
  names[0].n_name = "_u";
  names[1].n_name = NULL;
  if (nlist ("/dgux", names) == 0)
  {
    kernel_u_addr = names[0].n_value;
  }
  else
  {
    fatal ("ix86 DG/ux:Unable to get kernel_u_addr user area address.");
  }
}
**********************************************/

  

#ifdef TARGET_HAS_HARDWARE_WATCHPOINTS

#if !defined (offsetof)
#define offsetof(TYPE, MEMBER) ((unsigned long) &((TYPE *)0)->MEMBER)
#endif

/* Record the value of the debug control register.  */
static int debug_control_mirror;

/* Record which address associates with which register.  */
static CORE_ADDR address_lookup[DR_LASTADDR - DR_FIRSTADDR + 1];

static int
i386_insert_aligned_watchpoint PARAMS ((int, CORE_ADDR, CORE_ADDR, int,
					   int));

static int
i386_insert_nonaligned_watchpoint PARAMS ((int, CORE_ADDR, CORE_ADDR, int,
					   int));

/* Insert a watchpoint.  */

int
i386_insert_watchpoint (pid, addr, len, rw)
     int pid;
     CORE_ADDR addr;
     int len;
     int rw;
{
  return i386_insert_aligned_watchpoint (pid, addr, addr, len, rw);
}

static int
i386_insert_aligned_watchpoint (pid, waddr, addr, len, rw)
     int pid;
     CORE_ADDR waddr;
     CORE_ADDR addr;
     int len;
     int rw;
{
  int i;
  int read_write_bits, len_bits;
  int free_debug_register;
  int register_number;
  
  /* Look for a free debug register.  */
  for (i = DR_FIRSTADDR; i <= DR_LASTADDR; i++)
    {
      if (address_lookup[i - DR_FIRSTADDR] == 0)
	break;
    }

  /* No more debug registers!  */
  if (i > DR_LASTADDR)
    return -1;

  read_write_bits = (rw & 1) ? DR_RW_READ : DR_RW_WRITE;

  if (len == 1)
    len_bits = DR_LEN_1;
  else if (len == 2)
    {
      if (addr % 2)
	return i386_insert_nonaligned_watchpoint (pid, waddr, addr, len, rw);
      len_bits = DR_LEN_2;
    }

  else if (len == 4)
    {
      if (addr % 4)
	return i386_insert_nonaligned_watchpoint (pid, waddr, addr, len, rw);
      len_bits = DR_LEN_4;
    }
  else
    return i386_insert_nonaligned_watchpoint (pid, waddr, addr, len, rw);
  
  free_debug_register = i;
  register_number = free_debug_register - DR_FIRSTADDR;
  debug_control_mirror |=
    ((read_write_bits | len_bits)
     << (DR_CONTROL_SHIFT + DR_CONTROL_SIZE * register_number));
  debug_control_mirror |=
    (1 << (DR_LOCAL_ENABLE_SHIFT + DR_ENABLE_SIZE * register_number));
  debug_control_mirror |= DR_LOCAL_SLOWDOWN;
  debug_control_mirror &= ~DR_CONTROL_RESERVED;
  
  ptrace (6, pid, offsetof (struct user, u_debugreg[DR_CONTROL]),
	  debug_control_mirror);
  ptrace (6, pid, offsetof (struct user, u_debugreg[free_debug_register]),
	  addr);

  /* Record where we came from.  */
  address_lookup[register_number] = addr;
  return 0;
}

static int
i386_insert_nonaligned_watchpoint (pid, waddr, addr, len, rw)
     int pid;
     CORE_ADDR waddr;
     CORE_ADDR addr;
     int len;
     int rw;
{
  int align;
  int size;
  int rv;

  static int size_try_array[16] = {
    1, 1, 1, 1,			/* trying size one */
    2, 1, 2, 1,			/* trying size two */
    2, 1, 2, 1,			/* trying size three */
    4, 1, 2, 1			/* trying size four */
  };

  rv = 0;
  while (len > 0)
    {
      align = addr % 4;
      /* Four is the maximum length for 386.  */
      size = (len > 4) ? 3 : len - 1;
      size = size_try_array[size * 4 + align];

      rv = i386_insert_aligned_watchpoint (pid, waddr, addr, size, rw);
      if (rv)
	{
	  i386_remove_watchpoint (pid, waddr, size);
	  return rv;
	}
      addr += size;
      len -= size;
    }
  return rv;
}

/* Remove a watchpoint.  */

int
i386_remove_watchpoint (pid, addr, len)
     int pid;
     CORE_ADDR addr;
     int len;
{
  int i;
  int register_number;

  for (i = DR_FIRSTADDR; i <= DR_LASTADDR; i++)
    {
      register_number = i - DR_FIRSTADDR;
      if (address_lookup[register_number] == addr)
	{
	  debug_control_mirror &=
	    ~(1 << (DR_LOCAL_ENABLE_SHIFT + DR_ENABLE_SIZE * register_number));
	  address_lookup[register_number] = 0;
	}
    }
  ptrace (6, pid, offsetof (struct user, u_debugreg[DR_CONTROL]),
	  debug_control_mirror);
  ptrace (6, pid, offsetof (struct user, u_debugreg[DR_STATUS]), 0);

  return 0;
}

/* Check if stopped by a watchpoint.  */

CORE_ADDR
i386_stopped_by_watchpoint (pid)
    int pid;
{
  int i;
  int status;

  status = ptrace (3, pid, offsetof (struct user, u_debugreg[DR_STATUS]), 0);
  ptrace (6, pid, offsetof (struct user, u_debugreg[DR_STATUS]), 0);

  for (i = DR_FIRSTADDR; i <= DR_LASTADDR; i++)
    {
      if (status & (1 << (i - DR_FIRSTADDR)))
	return address_lookup[i - DR_FIRSTADDR];
    }

  return 0;
}

#endif /* TARGET_HAS_HARDWARE_WATCHPOINTS */

