;;; dispicon.el --- Display an icon which associates to the file.

;; Author: Yuuichi Teranishi <teranisi@gohome.org>
;; Keywords: Windows, Icon

;; Copyright (C) 2004 Yuuichi Teranishi <teranisi@gohome.org>

;; This file is not part of GNU Emacs

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.
;;

;;; Commentary:
;;

;;; History:
;; 

;;; Code:

;;; User variables.
(defgroup dispicon nil
  "Display an Windows icon."
  :group 'image)

(defcustom dispicon-default-type 'large
  "Default type of dispicon."
  :type '(choice (const :tag "Large Icon" 'large)
		 (const :tag "Small Icon" 'small))
  :group 'dispicon)

(defcustom dispicon-default-size 32
  "Default size of dispicon (width and height)."
  :type 'integer
  :group 'dispicon)

(defcustom dispicon-default-depth 24
  "Default depth of dispicon."
  :type 'integer
  :group 'dispicon)

(defcustom dispicon-default-bgcolor nil
  "Default background color of dispicon.
nil means to use the default face background."
  :type 'integer
  :group 'dispicon)

(defcustom dispicon-program "dispicon.exe"
  "Program name of dispicon."
  :type 'file
  :group 'dispicon)

(defcustom dispicon-image-type 'bmp
  "Image type of `dispicon-program' output."
  :type 'symbol
  :group 'dispicon)

(defun dispicon-default-background ()
  "Obtain background color of default face."
  (let ((rgb (color-values (or (frame-parameter
				(selected-frame) 'background-color)
			       "White"))))
    (format "#%02X%02X%02X"
	    (* 255 (/ (float (nth 2 rgb)) 65535))
	    (* 255 (/ (float (nth 1 rgb)) 65535))
	    (* 255 (/ (float (nth 0 rgb)) 65535)))))

;;; dispicon API.

(defun dispicon (filename &optional type size depth bgcolor ignore-errors)
  "Obtain an icon which associates to FILENAME.
If optional TYPE is specified, icon type is changed.
It is a symbol of `small' or `large'.
If second optional SIZE is specified, it is used as width and height of
the icon.
If third optional DEPTH is specified, it is used as the depth of the icon.
If fourth optional BGCOLOR is specified, it is used as the default bgcolor.
If fifth optional IGNORE-ERRORS is specified, error is ignored."
  (let ((args `("-s" ,(number-to-string (or size dispicon-default-size))
		"-b" ,(or bgcolor dispicon-default-bgcolor
			  (dispicon-default-background))
		"-d" ,(number-to-string (or depth dispicon-default-depth))
		,filename))
	stat image icontype)
    (when (file-exists-p filename)
      (setq args (append '("-f") args)))
    (with-temp-buffer
      (set-buffer-multibyte nil)
      (setq icontype (or type dispicon-default-type))
      (when (eq icontype 'large)
	(setq args (append '("-l") args)))
      (setq stat (apply 'call-process
			dispicon-program nil (current-buffer) t
			args))
      (if (numberp stat)
	  (if (eq stat 0)
	      (setq image
		    (find-image `((:type ,dispicon-image-type
					 :data ,(buffer-string)
					 :ascent 90))))
	    (unless ignore-errors
	      (error "Error while processing dispicon")))
	(if (stringp stat)
	    (unless ignore-errors
	      (error stat)))))
    (propertize " " 'display image 'invisible t)))

(provide 'dispicon)

;;; dispicon.el ends here
