/*
 * Author:      William Chia-Wei Cheng (bill.cheng@acm.org)
 *
 * Copyright (C) 1999, William Chia-Wei Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgifcal" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgifcal" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgifcal", the right to sell or distribute derivative
 * works of "tgifcal", the right to distribute "tgifcal" for a fee, and
 * the right to include "tgifcal" or derivative works of "tgifcal" in a
 * for-sale product) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * @(#)$Header: /mm/src/tgif/v4/tgifcal/RCS/tgifcal.c,v 1.1 1999/05/19 18:42:50 william Exp $
 */

/*
 * tgifcal [-gamma gamma_value] [-fit] [-ps] month year image_file
 */

#include "../tgifdefs.h"

#include "tgifintf.h"

static char gszProgName[MAXPATHLENGTH];

static int gnDebug=0;

static char gszImageFile[MAXPATHLENGTH];
static char gszMonth[MAXSTRING];
static char gszYear[MAXSTRING];
static char gszThisMonth[MAXSTRING];
static char gszThisYear[MAXSTRING];

static char gszDate[6][7][3];

static int gnMonth=INVALID;
static int gnYear=INVALID;

static int gnNumRows=INVALID;

static char gszTmpFile[MAXPATHLENGTH];
static int gnRemoteURL=FALSE;

static char gszGamma[MAXSTRING];

static int gnFit=FALSE;
static int gnPS=FALSE;

static
void Usage()
{
   fprintf(stderr, "usage: %s %s month year image_file\n",
         gszProgName, "[-gamma gamma_value] [-fit] [-ps]");
   exit(-1);
}

static
void CheckMonth(buf)
   char *buf;
{
   if (*buf >= '0' && *buf <= '9') {
      gnMonth = atoi(buf);
      if (gnMonth >= 1 && gnMonth <= 12) {
         return;
      }
   }
   fprintf(stderr, "%s: illegal month value: use 1-12\n", gszProgName);
   Usage();
}

static
void CheckYear(buf)
   char *buf;
{
   if (*buf >= '0' && *buf <= '9') {
      gnYear = atoi(buf);
      if (gnYear >= 1 && gnYear <= 9999) {
         return;
      }
   }
   fprintf(stderr, "%s: illegal year value: use 1-9999\n", gszProgName);
   Usage();
}

static
void ProcessOptions(argc, argv)
   int argc;
   char *argv[];
{
   gnMonth = INVALID;
   *gszImageFile = *gszMonth = *gszYear = *gszGamma = '\0';

   for (argc--, argv++; argc > 0; argc--, argv++) {
      if (*argv[0] == '-') {
         if (strcmp(*argv, "-debug") == 0) {
            gnDebug++;
         } else if (strcmp(*argv, "-gamma") == 0) {
            double gamma=(double)0;

            argc--; argv++;
            if (argc <= 0) {
               fprintf(stderr, "%s: missing gamma value\n", gszProgName);
               Usage();
            }
            TgifStrCpyN(gszGamma, sizeof(gszGamma), *argv);
            TgifTrimBlanks(gszGamma);
            if (sscanf(gszGamma, "%lg", &gamma) != 1 || gamma < INT_TOL) {
               fprintf(stderr, "%s: invalid gamma value: '%s'\n",
                     gszProgName, *argv);
               Usage();
            }
            if (fabs(gamma-1.0) < INT_TOL) {
               *gszGamma = '\0';
            }
         } else if (strcmp(*argv, "-fit") == 0) {
            gnFit = TRUE;
         } else if (strcmp(*argv, "-ps") == 0) {
            gnPS = TRUE;
         } else {
            Usage();
         }
      } else if (*gszMonth == '\0') {
         TgifStrCpyN(gszMonth, sizeof(gszMonth), *argv);
      } else if (*gszYear == '\0') {
         TgifStrCpyN(gszYear, sizeof(gszYear), *argv);
      } else if (*gszImageFile == '\0') {
         TgifStrCpyN(gszImageFile, sizeof(gszImageFile), *argv);
      } else {
         Usage();
      }
   }
   CheckMonth(gszMonth);
   CheckYear(gszYear);

   if (*gszImageFile == '\0') {
      Usage();
   }
}

static
void SetProgramName(s)
   char *s;
{
   char *c_ptr=TgifStrRChr(s, '/');

   if (c_ptr == NULL) {
      strcpy(gszProgName, s);
   } else {
      strcpy(gszProgName, ++c_ptr);
   }
}

static
void LocalCleanUp()
{
   if (gnRemoteURL && *gszTmpFile != '\0') {
      unlink(gszTmpFile);
      *gszTmpFile = '\0';
   }
   CleanUpTgifInterface();
}

static
void LocalInit()
{
   if (!InitTgifInterface()) exit(-1);
}

static
int SetDate(buf, line_num, extra_skip)
   char *buf;
   int line_num, extra_skip;
{
   int col=0, row=line_num-3;

   for (col=0; col < 7; col++) {
      if (*buf == '\0') {
         *gszDate[row][col] = '\0';
      } else {
         if (buf[1] == '\0') {
            fprintf(stderr, "Malformed column %d of line %d in input.\n",
                  col+1, line_num);
            return FALSE;
         }
         if (col == 0) {
            gnNumRows++;
         }
         gszDate[row][col][0] = buf[0];
         gszDate[row][col][1] = buf[1];
         gszDate[row][col][2] = '\0';
         TgifTrimBlanks(gszDate[row][col]);
         buf = (&buf[2]);
      }
      if (*buf == ' ') {
         buf++;
      }
      if (*buf == ' ' && extra_skip) {
         buf++;
      }
   }
   return TRUE;
}

static
void DebugDate()
{
   int col=0, row=0;

   for (row=0; row < 6; row++) {
      for (col=0; col < 7; col++) {
         fprintf(stderr, "%2s ", *gszDate[row][col]=='\0' ? "  " :
               gszDate[row][col]);
      }
      fprintf(stderr, "\n");
   }
   fprintf(stderr, "gnNumRows = %1d\n", gnNumRows);
   fprintf(stderr, "gnMonth = %1d, gszThisMonth = '%s'\n", gnMonth,
         gszThisMonth);
   fprintf(stderr, "gnYear = %1d, gszThisYear = '%s'\n", gnYear, gszThisYear);
}

static
int ReadInput()
{
   char *buf=NULL, cmd[MAXPATHLENGTH];
   FILE *fp=NULL;
   int line_num=0, extra_skip=FALSE;

   sprintf(cmd, "cal %d %d", gnMonth, gnYear);
   if ((fp=(FILE*)popen(cmd, "r")) == NULL) {
      fprintf(stderr, "Fail to execute '%s'.\n", cmd);
      return FALSE;
   }
   memset(gszDate, 0, sizeof(gszDate));
   *gszThisMonth = *gszThisYear = '\0';
   gnNumRows = 0;

   while ((buf=TgifGetALine(fp)) != NULL) {
      line_num++;
      if (gnDebug) {
         fprintf(stderr, "%s\n", buf);
      }
      if (line_num == 1) {
         char *psz_month=strtok(buf, " \t\n\r,");

         if (psz_month != NULL) {
            char *psz_year=strtok(NULL, " \t\n\r,");

            if (psz_year != NULL) {
               strcpy(gszThisMonth, psz_month);
               strcpy(gszThisYear, psz_year);
            }
         }
         if (*gszThisMonth == '\0' || *gszThisYear == '\0') {
            fprintf(stderr, "Malformed line %d while executing '%s'.\n",
                  line_num, cmd);
            return FALSE;
         }
      } else if (line_num == 2) {
         /* throw away */
      } else {
         if (line_num == 3) {
            int len=strlen(buf);

            while (len > 0 && (buf[len-1] == '\r' || buf[len-1] == '\n' ||
                  buf[len-1] == '\t' || buf[len-1] == ' ')) {
               buf[--len] = '\0';
            }
            if (len >= 26) extra_skip = TRUE;
         }
         if (!SetDate(buf, line_num, extra_skip)) {
            return FALSE;
         }
      }
      TgifFree(buf);
   }
   pclose(fp);
   if (line_num <= 0) {
      Usage();
   }
   if (gnDebug > 0) {
      DebugDate();
   }
   return TRUE;
}

static
int ReadImage()
{
   char *psz=NULL, sz_ext[MAXSTRING];
   float x=(float)47, y=(float)81, w=(float)510, h=(float)340;

   if ((psz=TgifStrRChr(gszImageFile, '.')) == NULL) {
      fprintf(stderr, "Cannot determine file type for '%s'.\n", gszImageFile);
      return FALSE;
   }
   TgifStrCpyN(sz_ext, sizeof(sz_ext), ++psz);
   TgifStrLower(sz_ext);
   if (strcmp(sz_ext, "gif") == 0) {
      if (!TgifImportGIF(gnRemoteURL ? gszTmpFile : gszImageFile,
            &x, &y, &w, &h, gszGamma, gnFit)) {
         return FALSE;
      }
   } else if (strcmp(sz_ext, "jpg") == 0 || strcmp(sz_ext, "jpeg") == 0) {
      if (!TgifImportJPEG(gnRemoteURL ? gszTmpFile : gszImageFile,
            &x, &y, &w, &h, gszGamma, gnFit)) {
         return FALSE;
      }
   } else {
      fprintf(stderr, "Can only handle GIF or JPEG files.\n");
      return FALSE;
   }
   return TRUE;
}

static
int ProcessYear()
{
   float x=(float)556, baseline_y=(float)72;

   TgifTextJust("right");
   TgifText(&x, &baseline_y, "Times-Bold", 34, gszThisYear, "black");
   return TRUE;
}

static
int ProcessMonth()
{
   float x=(float)224, baseline_y=(float)72;

   TgifTextJust("center");
   TgifText(&x, &baseline_y, "Times-Bold", 72, gszThisMonth, "black");
   return TRUE;
}

static char *gszDayOfWeek[] = {
   "Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday",
   NULL
};

static
int ProcessDates()
{
   int row=0, col=0, start_x=46;
   float lty=(float)427, w=(float)73, h=(float)0;
   float big_ltx=(float)0, big_lty=(float)0, big_rbx=(float)0, big_rby=(float)0;

   switch (gnNumRows) {
   case 4: h = (float)62; break;
   case 5: h = (float)52; break;
   case 6: h = (float)44; break;
   default:
      if (gnNumRows < 4) {
         fprintf(stderr, "Too few lines in the input.\n");
      } else if (gnNumRows > 6) {
         fprintf(stderr, "Too many lines in the input.\n");
      }
      return FALSE;
   }
   big_ltx = ((float)start_x);
   big_lty = lty;
   big_rbx = ((float)start_x)+7*w;
   big_rby = lty+((float)(gnNumRows+1))*h;
   TgifBox(&big_ltx, &big_lty, &big_rbx, &big_rby, 0, 0, "6", "black");

   for (row=(-1); row < gnNumRows; row++, lty+=h) {
      float ltx=(float)start_x;

      if (row == (-1)) {
         for (col=0; col < 7; col++, ltx+=w) {
            float rbx=ltx+w, rby=lty+h;

            TgifBox(&ltx, &lty, &rbx, &rby, 0, 0, "1", "black");
            TgifCenterAttr(&ltx, &lty, &rbx, &rby, "Times-Bold", 14, 1,
                  0, "day_of_week=", gszDayOfWeek[col],
                  (col==0 ? "red" : "black"));
         }
      } else {
         for (col=0; col < 7; col++, ltx+=w) {
            float rbx=ltx+w, rby=lty+h;
            float attr_x=ltx+(float)6, attr_baseline_y=lty+(float)16;

            TgifBox(&ltx, &lty, &rbx, &rby, 0, 0, "1", "black");
            TgifAttr(&attr_x, &attr_baseline_y, "Times-Bold", 14, 0, 1,
                  0, "date=", gszDate[row][col], (col==0 ? "red" : "black"));
         }
      }
   }
   return TRUE;
}

static
void Process()
{
   *gszTmpFile = '\0';
   gnRemoteURL = FALSE;

   if (strstr(gszImageFile, "//") != NULL) {
      if (TgifDownloadURL(gszImageFile, gszTmpFile) && *gszTmpFile != '\0') {
         gnRemoteURL = TRUE;
      } else {
         return;
      }
   } else {
      if (*gszImageFile != DIR_SEP) {
         char dir[MAXPATHLENGTH], path[MAXPATHLENGTH];

         TgifGetCurDir(dir, sizeof(dir)-strlen(gszImageFile)-2);
         sprintf(path, "%s%c%s", dir, DIR_SEP, gszImageFile);
         strcpy(gszImageFile, path);
      }
      if (!TgifPathExists(gszImageFile)) {
         fprintf(stderr, "File '%s' does not exist.\n", gszImageFile);
         Usage();
      }
   }
   if (!ReadInput()) return;
   if (!ReadImage()) return;
   if (!ProcessYear()) return;
   if (!ProcessMonth()) return;
   if (!ProcessDates()) return;

   TgifGenerate(stdout, (gnPS ? "ps" : "obj"));
}


int main(argc, argv)
   int argc;
   char *argv[];
{
   SetProgramName(*argv);
   ProcessOptions(argc, argv);

   LocalInit();
   Process();
   LocalCleanUp();
   return 0;
}
