/***************************************************************************
                             rmcache.c
                             -------------------
    begin                : May 2001
    copyright            : (C) 2001 by Jorge Allyson Azevedo
                                       Milena Scanferla
                                       Magnos Martinello
                                       Daniel Sadoc
    email                : {allyson,milena,magnos,sadoc}@land.ufrj.br
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/******************************************************************************************************
 *																									  *
 * Reliable Multicast Protocol (rmcache.c)															  *
 *																									  *
 * Reliable Multicast cache functions.																  *
 *																									  *
 * use tab spacing = 4																				  *
 *																									  *
 ******************************************************************************************************/


#ifndef RMCACHE_C

#define RMCACHE_C

#include "rmstruct.h"

#include <stdlib.h>
#include <string.h>

#include <pthread.h>

#include "rmcache.h"


pthread_mutex_t   cache_mutex = PTHREAD_MUTEX_INITIALIZER;


/*---------------- Auxiliary functions to manipulate the cache ---------------------------------------*/

/***************************************************************************************
 *
 * int cacheAllocHeaderNode(CACHE **cache, MEMBER_ID *member)
 *
 * Allocates a cache header node.
 *
 * Arguments:	cache, the cache;
 *              member, the member to be inserted.
 *
 * Returns: 0 on error, 1 otherwise.
 *
 ***************************************************************************************/


int cacheAllocHeaderNode(CACHE **cache, MEMBER_ID *member)
{
	if ((*cache = (CACHE*)malloc(sizeof(CACHE)))==NULL)
	{
		fprintf(stderr,"Error: Allocating cache header node\n");
		return(0);
	}
	bzero((*cache),sizeof(CACHE));
	(*cache)->active = 1;
	(*cache)->sm_info.member_id = *member;
    (*cache)->sm_info.member_status.last_received_or_nak = -1;
    
    (*cache)->sm_info.member_status.first_rcv = -1;
    (*cache)->sm_info.member_status.last_rcv = -1;  
    (*cache)->sm_info.member_status.last_seq_rcv = -1;
    
	return (1);
}

/***************************************************************************************
 *
 * int cacheAllocMessageNode(CACHE_NODE **cache_node, DATA_PACKET *node_info)
 *
 * Allocates a cache message node.
 *
 * Arguments:	cache_node, the cache node to be created - it will be allocated here;
 *              node_info, information about the message node to be created.
 *
 * Returns: 0 on error, 1 otherwise.
 *
 ***************************************************************************************/

int cacheAllocMessageNode(CACHE_NODE **cache_node, DATA_PACKET *node_info)
{
	if ((*cache_node = (CACHE_NODE*)malloc(sizeof(CACHE_NODE)))==NULL)
	{
		fprintf(stderr,"Error allocating cache message node");
		return(0);
	}
	
	bzero((*cache_node),sizeof(CACHE_NODE));
	(*cache_node)->data_pckt.sn = node_info->sn;
	(*cache_node)->data_pckt.data_size = node_info->data_size;
	
	(*cache_node)->data_pckt.data = (BYTE*)malloc( (node_info->data_size+1)* sizeof(BYTE));
	memcpy((*cache_node)->data_pckt.data,node_info->data,(node_info->data_size+1));
	
	/*	Observation: 
	
	we allocate and copy data_size data bytes, plus one byte for the string termination caracter */
	
	return (1);
}

/***************************************************************************************
 *
 * int cacheFreeMessageNode(CACHE_NODE **node)
 *
 * Free a cache message node.
 *
 * Arguments:	cache_node, the cache node to be released.
 *
 * Returns: 1 allways.
 *
 ***************************************************************************************/

int cacheFreeMessageNode(CACHE_NODE **node)
{
	free((*node)->data_pckt.data);
	free(*node);
	*node=NULL;
	return (1);
}

/*---------------- Main functions to manipulate the cache --------------------------------------------*/

/***************************************************************************************
 *
 * void cacheInit(CACHE **cache)
 *
 * Initialize the cache.
 *
 * Arguments:	cache, the cache;
 *
 ***************************************************************************************/


void cacheInit(CACHE **cache)
{
	*cache = NULL;
}

/***************************************************************************************
 *
 * int cacheInsertMessage(CACHE **cache, MEMBER_ID *member, DATA_PACKET *data_packet)
 *
 * Inserts a message in the cache.
 *
 * Arguments:	cache, the cache;
 *				member, the member who originaly sent the message;
 *				data_packet, message data.
 *
 * Return value:	0,	on error;
 *					1,	if the inserted message was in sequence,
 *						i.e. the message received was the one expected;
 *					2,	if the inserted message was not is sequence.
 *
 ***************************************************************************************/


int cacheInsertMessage(CACHE **cache, MEMBER_ID *member, DATA_PACKET *data_packet)
{
	int count = 0;
	int isfirst = 0;
   	CACHE *memberHeaderNode;
   	CACHE_NODE *newMessageCacheNode, *auxMessageNode;
    
	
    memberHeaderNode = cacheLookForMember(cache, member);
    
    
    if (memberHeaderNode == NULL)
	{	 
		if ((memberHeaderNode = cacheInsertMember(cache, member)) == NULL)
        {
			return (0);
        }
	}
	
	
	if (memberHeaderNode->number_of_nodes == MEMBER_CACHE_SIZE)
	{
#ifdef	DEBUG2
		fprintf(stderr,"yes\n");
#endif	
		
		((memberHeaderNode->last)->previous)->next = NULL;
		
		auxMessageNode = (memberHeaderNode->last);
		
		(memberHeaderNode->last) = ((memberHeaderNode->last)->previous);
		
		cacheFreeMessageNode(&auxMessageNode);
		
		memberHeaderNode->number_of_nodes--;
	}
#ifdef	DEBUG2
	else fprintf(stderr,"no\n");
#endif	 
		 
	auxMessageNode = memberHeaderNode -> first;
	
	if (cacheAllocMessageNode(&newMessageCacheNode, data_packet)==(int)NULL)
    {
		return (0);
    }
	
	memberHeaderNode -> last_inserted = newMessageCacheNode;
	
	if (memberHeaderNode->number_of_nodes == 0)
	{
		// we will insert the first message node
		
		memberHeaderNode -> last = memberHeaderNode -> first = newMessageCacheNode;
        
        memberHeaderNode -> sm_info.member_status.last_received_or_nak = -1;
		
		memberHeaderNode->sm_info.member_status.first_rcv = data_packet->sn;	  
		
		isfirst = 1;
		
#ifdef DEBUG2
		fprintf(stderr,"we will insert the first message node\n");
#endif		  
	}
	else
	{
#ifdef DEBUG2
		fprintf(stderr,"a message node will be inserted\n");
#endif		  
		auxMessageNode = memberHeaderNode -> first;
		
		while (auxMessageNode != NULL)
		{
			if (auxMessageNode->data_pckt.sn <= newMessageCacheNode->data_pckt.sn)
				break;
			auxMessageNode = auxMessageNode -> next;
			count ++;
		}
		if (auxMessageNode)
		{
			if (auxMessageNode->data_pckt.sn == newMessageCacheNode->data_pckt.sn)
			{
				return (0);
			}
		}
		if (count == 0)
		{
			memberHeaderNode->first = newMessageCacheNode;
		}
#ifdef DEBUG2
		fprintf(stderr,"adjusting pointers\n");
#endif		  
		
		if (auxMessageNode != NULL)
		{		 
			newMessageCacheNode->next = auxMessageNode;
			newMessageCacheNode->previous = auxMessageNode->previous;
			
			if (auxMessageNode->previous!=NULL)
				auxMessageNode->previous->next = newMessageCacheNode;
			
			auxMessageNode->previous = newMessageCacheNode;
		}
		else
		{
#ifdef DEBUG2
			fprintf(stderr,"inserting in the end...");
			cacheShow(*cache);
#endif
			newMessageCacheNode -> previous = memberHeaderNode -> last;
			memberHeaderNode -> last -> next = newMessageCacheNode;
			memberHeaderNode -> last = newMessageCacheNode;
		}
#ifdef DEBUG			  
		if (newMessageCacheNode -> next == NULL)		  // is it necessary? no?  
		{	
			if (memberHeaderNode->last != newMessageCacheNode)
			{
			fprintf(stderr,"ATENTION!!!! memberHeaderNode->last != newMessageCacheNode \
				&& newMessageCacheNode->next == NULL");
			exit(1);
			}				
		}
#endif
		
	}
#ifdef DEBUG0
	fprintf(stderr,"node (%d) inserted in cache\n", data_packet->sn);    
#endif    
    
	memberHeaderNode->number_of_nodes++;	  	
	
	memberHeaderNode->sm_info.member_status.last_rcv = data_packet->sn;  
	

    if (isfirst == 1)
    {
        if (data_packet->sn !=0)
        {
           memberHeaderNode->sm_info.member_status.last_seq_rcv = -1;
           
           
           return(2);
        }
            
        else
        {
           memberHeaderNode->sm_info.member_status.last_seq_rcv = 0;
           
           
           return (1);
        }   
        
    }
	else if (data_packet->sn==memberHeaderNode->sm_info.member_status.last_seq_rcv+1 /* || isfirst == 1 */)
	{
	/* the message is in sequence - setting up the last sequence received field 
		in the member header node */
		
		/* check this ! */
		
		auxMessageNode = newMessageCacheNode;
		
		while (auxMessageNode->previous!=NULL)
		{
			if (!((auxMessageNode->previous)->data_pckt.sn == auxMessageNode->data_pckt.sn + 1 ))
				break;
			auxMessageNode = auxMessageNode->previous;
		}
		
		memberHeaderNode->sm_info.member_status.last_seq_rcv = auxMessageNode->data_pckt.sn;  
#ifdef DEBUG2		
		cacheShow(*cache);	  
#endif        

		return (1);  /* the packet is in sequence */
	}
	else 
	{
#ifdef DEBUG2		    
		cacheShow(*cache);	  
#endif        		
        
		return (2); /* the packet is not in sequence - it will be needed to send a nak */
	}
}

/***************************************************************************************
 *
 * CACHE *cacheInsertMember(CACHE **cache, MEMBER_ID *member)
 *
 * Inserts a member in the cache.
 *
 * Arguments:	cache, the cache;
 *				member, the member identification;
 *
 * Return value:	a pointer to the inserted member; NULL on error.
 *
 ***************************************************************************************/

CACHE *cacheInsertMember(CACHE **cache, MEMBER_ID *member)
{
	CACHE *auxHeaderNode = *cache, *newHeaderNode;
	
	if (cacheAllocHeaderNode( &newHeaderNode, member )==(int)NULL)
	{
		return (NULL);
	}
	
	if (cacheIsEmpty(cache))
	{
		// cache is empty
		
		*cache = newHeaderNode;
	}
	else  // inserts new node in the end of the list
	{
#ifdef DEBUG2
		fprintf(stderr,"searching where to insert new member node...\n");
#endif		  
		
		while (auxHeaderNode->next!=NULL)
		{
			auxHeaderNode = auxHeaderNode->next;
		}
		
#ifdef DEBUG2
		fprintf(stderr,"member node will be inserted...\n");
#endif		  
		auxHeaderNode->next = newHeaderNode;
	}
	return (newHeaderNode);
}


/***************************************************************************************
 *
 * int cacheIsEmpty(CACHE **cache)
 *
 * Checks if the cache is empty.
 *
 * Arguments:	cache, the cache;
 *
 * Return value:	1 if the cache is empty; 0 otherwise.
 *
 ***************************************************************************************/

int cacheIsEmpty(CACHE **cache)
{
	if (*cache==NULL)
		return (1);
	else
		return (0);
}


/***************************************************************************************
 *
 * int cacheOfAnyMemberIsFull(CACHE *cache)
 *
 * Checks if the cache space reserved for any member is full.
 *
 * Arguments:	cache, the cache;
 *
 * Return value:	1 if the the cache of any member is full; 0 otherwise.
 *
 ***************************************************************************************/

int cacheOfAnyMemberIsFull(CACHE *cache)
{
	CACHE *auxCacheHeaderNode = cache;
	while (auxCacheHeaderNode->number_of_nodes < MEMBER_CACHE_SIZE)
	{
		auxCacheHeaderNode = auxCacheHeaderNode->next;
		
		if (auxCacheHeaderNode == NULL)
			return (0);
	}
	return (!(auxCacheHeaderNode->number_of_nodes < MEMBER_CACHE_SIZE));
}

int memberIsEqual(MEMBER_ID *member1, MEMBER_ID *member2)
{
	if ((member1->pid==member2->pid)&&(!strcmp(member1->ip, member2->ip)))
		return (1);
	else
		return (0);
}

/***************************************************************************************
 *
 * CACHE* cacheLookForMember(CACHE **cache1, MEMBER_ID *member_id)
 *
 * Searches for a member in the cache.
 *
 * Arguments:	cache, the cache;
 *				member_id, the member identification.
 *
 * Return value:    a pointer to the requested member node, 
 *                  or NULL if the member does not exist.
 *
 ***************************************************************************************/


CACHE* cacheLookForMember(CACHE **cache1, MEMBER_ID *member_id)
{
	CACHE *auxHeaderNode=NULL;
    CACHE *cache=NULL;
    
    if (!cacheIsEmpty(cache1))
    {        
          
        cache = *cache1;
        auxHeaderNode = cache;

		while (!memberIsEqual(member_id,&auxHeaderNode->sm_info.member_id))
		{
			auxHeaderNode = auxHeaderNode->next;
			if (auxHeaderNode == NULL)
			{
				break;
			}
		}
	}
#ifdef DEBUG2    
    else
    {
        fprintf(stderr,"Cache is empty. (cacheLookForMember2)\n");        
    }
#endif    
    
    
	return (auxHeaderNode);
}

/***************************************************************************************
 *
 * int cacheDesactivateMember(CACHE *cache, MEMBER_ID *member)
 *
 * Desactivate a cache member, setting its activate bit to zero.
 *
 * Arguments:	cache, the cache;
 *				member_id, the member to be desactivated.
 *
 * Return value:    1 on success, 0 on fail.
 *
 ***************************************************************************************/


int cacheDesactivateMember(CACHE *cache, MEMBER_ID *member)
{
	CACHE* auxCacheHeaderNode = cacheLookForMember(&cache,member);
	if (auxCacheHeaderNode != NULL)
	{
		auxCacheHeaderNode -> active = 0;
		return (1);
	}
	else
		return (0);
}

/***************************************************************************************
 *
 * void cacheShow(CACHE *cache)
 *
 * Show the cache on the screen.
 *
 * Arguments:	cache, the cache;
 *
 ***************************************************************************************/

void cacheShow(CACHE *cache)
{
#ifdef CACHE_SHOW

	CACHE* auxCacheHeaderNode = cache;
	CACHE_NODE *auxMessageNode;
	char aux_string[MAX_IP_STRING_SIZE];
	int i;
	
	fprintf(stderr,"\n-----  cache --------------------------------------------------\n");
	
	while (auxCacheHeaderNode!=NULL)
	{
		for (i=0; i<MAX_IP_STRING_SIZE;i++)
			aux_string[i]=auxCacheHeaderNode->sm_info.member_id.ip[i];
		
		auxMessageNode = auxCacheHeaderNode->first; 	
		fprintf(stderr,"[%x %x %sactive {first:%d last:%d last_seq:%d}] (%d nodes) %s %d:\n",
			(unsigned int)auxCacheHeaderNode->first,
			(unsigned int)auxCacheHeaderNode->last,
			(auxCacheHeaderNode->active==1?"   ":"des"),
			auxCacheHeaderNode->sm_info.member_status.first_rcv,
			auxCacheHeaderNode->sm_info.member_status.last_rcv,
			auxCacheHeaderNode->sm_info.member_status.last_seq_rcv,
			auxCacheHeaderNode->number_of_nodes,
			aux_string, 
			auxCacheHeaderNode->sm_info.member_id.pid
			);
		while (auxMessageNode!=NULL)
		{
			fprintf(stderr," \t([%08x %08x %08x] sn: %d pck size: %d %s)\n",
				(unsigned int)auxMessageNode->previous,
				(unsigned int)auxMessageNode,
				(unsigned int)auxMessageNode->next,
				auxMessageNode->data_pckt.sn,
				auxMessageNode->data_pckt.data_size,
				(char*)auxMessageNode->data_pckt.data
				); 
			auxMessageNode = auxMessageNode->next;
		} 
		
		auxCacheHeaderNode = auxCacheHeaderNode->next;
	}
	fprintf(stderr,"\n-------------------------------------------------------------\n"); 

#endif
}



/***************************************************************************************
 *
 * void cacheGetSMInfo(CACHE *cache, SM_INFO **sm_table, int *nusers)
 *
 * Gets session message information from the cache.
 *
 * Arguments:	cache, 		the cache;
 *				sm_table, 	a pointer to where the destination info. will be stored - 
 *							its allocated here;
 *				nusers, 	returns the number of users in the cache - 
 *							it must be previously allocated.
 *
 ***************************************************************************************/

void cacheGetSMInfo(CACHE *cache, SM_INFO **sm_table, int *nusers)
{
	int i=0;
	CACHE* auxCacheHeaderNode = cache;
	
	*sm_table=NULL;
	*nusers=0;
	
	for (auxCacheHeaderNode=cache; auxCacheHeaderNode!=NULL; auxCacheHeaderNode=auxCacheHeaderNode->next)
	{		
		
		if ((*sm_table= (SM_INFO*)realloc(*sm_table,(i+1)*sizeof(SM_INFO)))==NULL)
		{
			fprintf(stderr,"Error allocating sm_info");
			exit(1);
		}
		
		memcpy((*sm_table)+i, &auxCacheHeaderNode->sm_info, sizeof(SM_INFO));
		
		i++; (*nusers)++;
	}
}

/***************************************************************************************
 *
 * DATA_PACKET *cacheLookForMessage(CACHE **cache1, MEMBER_ID *member_id,int message_sn)
 *
 * Searches for a message in the cache.
 *
 * Arguments:	cache1, 	the cache;
 *				member_id,	original sender of the message information;
 *				message_sn, message sequence number;
 *
 * Returns: 	a pointer to the data packet, or NULL when not found.
 *
 ***************************************************************************************/

DATA_PACKET *cacheLookForMessage(CACHE **cache1, MEMBER_ID *member_id,int message_sn)
{
	CACHE *auxMember;
	CACHE_NODE *auxMsg;
	DATA_PACKET *auxDatapkg=NULL;
    CACHE *cache;

    cache = *cache1;
	
	if ((auxMember=cacheLookForMember(&cache,member_id))!=NULL)
	{
		auxMsg=auxMember->first;
		
		while( (auxMsg!=NULL) && (auxMsg->data_pckt.sn != message_sn) )
		{
			auxMsg=auxMsg->next;
		}
		
		
		if (auxMsg != NULL && auxMsg->data_pckt.sn == message_sn)
		{
			auxDatapkg=&auxMsg->data_pckt;
		}
	} 
    
	return(auxDatapkg);
}

/*---------------- Auxiliary functions to manipulate the nak list ------------------------------------*/

/***************************************************************************************
 *
 * int cacheAllocNAKNode(NAK_LIST **nak_node, int sn)
 *
 * Allocates a new nak (not acknowledge) node, and fills it with specified info.
 *
 * Arguments:	nak_node, the node which will be created;
 *				sn, the sequence number of the nak node.
 *
 * Returns: 	0 on error, 1 on success.
 *
 ***************************************************************************************/


int cacheAllocNAKNode(NAK_LIST **nak_node, int sn)
{
	
	if ((*nak_node = (NAK_LIST*)malloc(sizeof(NAK_LIST)))==NULL)
	{
		fprintf(stderr,"Error allocating cache message node");
		return(0);
	}
	
	bzero((*nak_node),sizeof(NAK_LIST));
	(*nak_node)->sn = sn;
    (*nak_node)->hasReceived = 0;
	(*nak_node)->counter = 1; //at least one nak was send
	
	return (1);
}

/***************************************************************************************
 *
 * int cacheUpdateNakList(CACHE **cache1, MEMBER_ID *member, int sn)
 *
 * Updates the list of naks sent with respect to a specific member.
 *
 * Arguments:	cache1, the cache;
 *				member, the original sender of the message;
 *				sn, the sequence number of the message that we will wait for.
 *
 * Returns: 	0 on error, 1 on success.
 *
 ***************************************************************************************/

int cacheUpdateNakList(CACHE **cache1, MEMBER_ID *member, int sn)
{
	
	NAK_LIST *auxNakNode, *aux2, *new_nak_node;
	CACHE *memberHeaderNode;
    CACHE *cache;
    
    
    cache = *cache1;
	
	//FIXME    fprintf(stderr,"antes do cacheLookForMember:\nip: %s\npid: %d\n",member->ip,member->pid);	
	
	if((memberHeaderNode = cacheLookForMember(&cache, member))==NULL)
	{
		fprintf(stderr,"Error:cacheLookForMember failed at cacheUpdateNakList\n");
		return(0);
	}
	
	auxNakNode = memberHeaderNode -> nak_list;
	
	if(auxNakNode == NULL)
	{				 
		if(cacheAllocNAKNode(&new_nak_node,sn)== 0)
		{
			fprintf(stderr,"Error:cacheAllocNAKNode failed at cacheUpdateNakList\n");

			return (0);
		}
		memberHeaderNode -> nak_list = new_nak_node;				
	}
	else
	{	  
		while((auxNakNode) != NULL)
		{
			if(auxNakNode->sn != sn)
			{
				aux2=auxNakNode;				 
				auxNakNode = auxNakNode -> next;
			}  
			else 
			{  
				if( auxNakNode->counter >= MAX_NAK )
                {
					fprintf(stderr,"Error:Number of sent NAKs reached MAX_NAK at cacheUpdateNakList\n");

					return(0);
				}
#ifdef DONT_SEND_NAK_AGAIN_IMMEDIATLY
                else if ( !auxNakNode->hasReceived )
				{  
                    return(-1);
                } 
#endif                
                else
                {
                    auxNakNode->hasReceived = 0;
					auxNakNode->counter++;
					break;
				}
			}	  
		}	
		
		if(auxNakNode == NULL)
		{
			if(cacheAllocNAKNode(&new_nak_node,sn)== 0)
			{
				fprintf(stderr,"Error:cacheAllocNAKNode failed at cacheUpdateNakList\n");

				return (0);
			}
			aux2->next = new_nak_node;	   
		}
	}	

	return(1);
}

#ifdef DONT_SEND_NAK_AGAIN_IMMEDIATLY
void cacheUpdateNakListNakWasReceived(CACHE **cache1, MEMBER_ID *member, int sn)
{
	NAK_LIST *auxNakNode, *aux2;
    
	CACHE *memberHeaderNode, *cache;
    
    cache = *cache1;
    
	if((memberHeaderNode = cacheLookForMember(&cache, member))==NULL)
	{
		fprintf(stderr,"Error:cacheLookForMember failed at cacheUpdateNakListNakWasReceived\n");
        return;
	}
    
	auxNakNode = memberHeaderNode -> nak_list;

	while(auxNakNode != NULL)
	{
		if(auxNakNode->sn != sn)
		{
			aux2=auxNakNode;				 
			auxNakNode = auxNakNode -> next;
		}  
        else
        {
            auxNakNode->hasReceived = 1;
            break;
        }
    }

}
#endif

/*---------------- End of auxiliary functions to manipulate the nak list -----------------------------*/

#endif


