/*
 * Code in this file is derived from the public domain code in
 *              WWW/Library/Implementation/HTTP.c distributed with lynx-2.2,
 *              whose original author is Tim Berners-lee <timbl@info.cern.ch>.
 *
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/http.c,v 2.20 1995/05/15 01:22:06 william Exp $";
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "const.h"
#include "patchlevel.h"

#ifndef _NO_EXTERN
#include "http.e"
#endif /* !_NO_EXTERN */
#include "remote.e"
#include "tcp.e"
#include "util.e"
#include "version.e"

static char SZ_HTTP_VERSION[]="HTTP/1.0";
static char SZ_USER_AGENT[128];
static char SZ_USER_NAME[128];

static int debugHttp=FALSE;

void HttpFreeBuf(buf)
   char *buf;
{
   cfree(buf);
}

void HttpDebug(val)
   int val;
{
   debugHttp = val;
}

static int gnUserAgentNameInitialized=FALSE;

static
void InitUserAgentName()
{
   if (gnUserAgentNameInitialized) return;
   gnUserAgentNameInitialized = TRUE;

   GetClientID(SZ_USER_AGENT, sizeof(SZ_USER_AGENT));
   GetUserID(SZ_USER_NAME, sizeof(SZ_USER_NAME));
}

int HttpDoConnect(psz_host, us_port, pn_socket)
   char *psz_host;
   int us_port, *pn_socket;
{
   int rc, len=strlen(psz_host)+80;
   char *msg=(char*)calloc(len+1, sizeof(char));

   if (msg == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   sprintf(msg, "Making an HTTP connection to \"%s:%1d\"...", psz_host,
         us_port);
   ShowRemoteStatus(msg);

   rc = TcpDoConnect(psz_host, us_port, pn_socket);

   if (rc == TG_REMOTE_STATUS_OK) {
      sprintf(msg, "HTTP: connection to \"%s:%1d\" established.", psz_host,
            us_port);
   } else {
      sprintf(msg, "Fail to connect to HTTP server on \"%s:%1d\".", psz_host,
            us_port);
   }
   ShowRemoteStatus(msg);
   cfree(msg);

   return rc;
}

static char SZ_CONTENT_TYPE[]="Content-type:";
static char SZ_CONTENT_LENGTH[]="Content-length:";
static char SZ_DEF_GIF_NAME[]="/tmp/htclient.gif";
static char SZ_POST_CONTENT_TYPE[]="application/x-www-form-urlencoded";

static char *pszAccept[] = {
   "text/plain",
   "text/html",
   "application/x-tgif",
   "*/*",
   NULL
};

static
char *AppendAcceptStrings(buf)
   char *buf;
{
   char **s_ptr;
   int cur_len=strlen(buf);

   for (s_ptr=pszAccept; *s_ptr != NULL; s_ptr++) {
      int len=strlen(*s_ptr);
      int new_len=cur_len+len+2+8;

      if ((buf=(char*)realloc(buf, new_len+1)) == NULL) {
         return NULL;
      }
      sprintf(&buf[cur_len], "Accept: %s\r\n", *s_ptr);
      cur_len = new_len;
   }
   return buf;
}

static
char *AppendSimpleString(buf, name, value)
   char *buf, *name, *value;
{
   int cur_len=strlen(buf);

   if (name == NULL && value == NULL) {
      int new_len=cur_len+2;

      if ((buf=(char*)realloc(buf, new_len+1)) == NULL) {
         return NULL;
      }
      sprintf(&buf[cur_len], "\r\n");
   } else {
      int new_len=cur_len+strlen(name)+2+strlen(value)+2;

      if ((buf=(char*)realloc(buf, new_len+1)) == NULL) {
         return NULL;
      }
      sprintf(&buf[cur_len], "%s: %s\r\n", name, value);
   }
   return buf;
}

static
char *AppendUserAgentString(buf)
   char *buf;
{
   InitUserAgentName();
   return AppendSimpleString(buf, "User-Agent", SZ_USER_AGENT);
}

static
char *AppendFromString(buf)
   char *buf;
{
   InitUserAgentName();
   return AppendSimpleString(buf, "From", SZ_USER_NAME);
}

static
char *AppendPostContentTypeString(buf)
   char *buf;
{
   return AppendSimpleString(buf, "Content-type", SZ_POST_CONTENT_TYPE);
}

static
char *AppendPostContentLengthString(buf, content_length)
   char *buf;
   int content_length;
{
   char len_str[20];

   sprintf(len_str, "%1d", content_length);
   return AppendSimpleString(buf, "Content-length", len_str);
}

static
char *AppendPostContentString(buf, fp, content_length)
   char *buf;
   FILE *fp;
   int content_length;
{
   int cur_len=strlen(buf), bytes_read, total_read=0;
   int new_len=cur_len+content_length;
   char tmp_buf[512];

   if ((buf=(char*)realloc(buf, new_len+1)) == NULL) {
      return NULL;
   }
   while ((bytes_read=fread(tmp_buf, sizeof(char), sizeof(tmp_buf),
         fp)) > 0) {
      if (bytes_read+total_read > content_length) {
         bytes_read = content_length-total_read;
         fprintf(stderr, "Lines too long in AppendPostContentString().\n");
      }
      strncpy(&buf[cur_len+total_read], tmp_buf, bytes_read);
      total_read += bytes_read;
   }
   buf[cur_len+content_length] = '\0';
   return buf;
}

static
char *AppendCRLFString(buf)
   char *buf;
{
   return AppendSimpleString(buf, NULL, NULL);
}

int HttpDoWrite(n_socket, psz_path)
   int n_socket;
   char *psz_path;
{
   int status=0, total_sz=0;
   char *buf=(char*)calloc(strlen(psz_path)+5+2+31, sizeof(char));
   char msg[40];
   FILE *fp=NULL;

   if (buf == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   if (postingCGIQuery) {
      sprintf(buf, "POST %s %s\r\n", psz_path, SZ_HTTP_VERSION);
   } else {
      sprintf(buf, "GET %s %s\r\n", psz_path, SZ_HTTP_VERSION);
   }
   if ((buf=AppendAcceptStrings(buf)) == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   if ((buf=AppendUserAgentString(buf)) == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   if ((buf=AppendFromString(buf)) == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   if (postingCGIQuery && fnameForPostingCGIQuery != NULL) {
      int bytes_read;
      char tmp_buf[512];

      if ((fp=fopen(fnameForPostingCGIQuery, "r")) == NULL) {
         fprintf(stderr, "Fail to open '%s' for read.\n",
               fnameForPostingCGIQuery);
         return TG_REMOTE_STATUS_READ;
      }
      while ((bytes_read=fread(tmp_buf, sizeof(char), sizeof(tmp_buf),
            fp)) > 0) {
         total_sz += bytes_read;
      }
      rewind(fp);
      if ((buf=AppendPostContentTypeString(buf)) == NULL) {
         fclose(fp);
         fprintf(stderr, "Memory allocation failed.\n");
         return TG_REMOTE_STATUS_MEM;
      }
      if ((buf=AppendPostContentLengthString(buf, total_sz)) == NULL) {
         fclose(fp);
         fprintf(stderr, "Memory allocation failed.\n");
         return TG_REMOTE_STATUS_MEM;
      }
   }
   if ((buf=AppendCRLFString(buf)) == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   if (fp != NULL) {
      buf = AppendPostContentString(buf, fp, total_sz);
      fclose(fp);
      if (buf == NULL) {
         fprintf(stderr, "Memory allocation failed.\n");
         return TG_REMOTE_STATUS_MEM;
      }
   }
   sprintf(msg, "HTTP: sending requests...");
   ShowRemoteStatus(msg);

   status = TcpDoWrite(n_socket, buf, (int)strlen(buf));
   cfree(buf);

   if (status != TG_REMOTE_STATUS_OK) {
      sprintf(msg, "HTTP: fail to send requests.");
      ShowRemoteStatus(msg);
   }
   return status;
}

static
void HttpDumpResponse(buf)
   char *buf;
{
   char *c_ptr=strchr(buf, '\n'), *line_ptr=buf, *content_type=NULL;
   int content_length=(-1), gif87a=FALSE;
   int len1=strlen(SZ_CONTENT_TYPE), len2=strlen(SZ_CONTENT_LENGTH);
   FILE *fp=stdout;

   while (c_ptr != NULL) {
      char *prev_ptr=c_ptr;

      if (prev_ptr != line_ptr && *(--prev_ptr) == '\r') {
         *prev_ptr = '\0';
      } else {
         prev_ptr = NULL;
         *c_ptr = '\0';
      }
      fprintf(fp, "%s\n", line_ptr);
      if (content_type == NULL &&
            UtilStrNCaseCmp(line_ptr, SZ_CONTENT_TYPE, len1) == 0) {
         content_type = UtilStrDup(&line_ptr[len1]);
         if (content_type != NULL) {
            UtilTrimBlanks(content_type);
         }
      } else if (content_length == (-1) &&
            UtilStrNCaseCmp(line_ptr, SZ_CONTENT_LENGTH, len2) == 0) {
         char *tmp_ptr=UtilStrDup(&line_ptr[len2]);

         if (tmp_ptr != NULL) {
            UtilTrimBlanks(tmp_ptr);
            content_length = atoi(tmp_ptr);
            cfree(tmp_ptr);
         }
      }
      if (prev_ptr == NULL) {
         *c_ptr = '\n';
      } else {
         *prev_ptr = '\r';
      }
      line_ptr = &c_ptr[1];
      if (content_type != NULL && content_length != (-1)) {
         if (strcmp(content_type, "image/gif") == 0 &&
               UtilStrNCaseCmp(line_ptr, "GIF87a", 6) == 0) {
            gif87a = TRUE;
            break;
         }
      }
      c_ptr = strchr(line_ptr, '\n');
   }
   if (gif87a) {
      FILE *gif_fp;

      if ((gif_fp=fopen(SZ_DEF_GIF_NAME, "w")) == NULL) {
         fprintf(stderr, "Fail to open '%s' for write.\n", SZ_DEF_GIF_NAME);
      } else {
         if (fwrite(line_ptr, sizeof(char), content_length, gif_fp) !=
               content_length) {
            fprintf(stderr, "Write to '%s' failed.\n", SZ_DEF_GIF_NAME);
            fprintf(stderr, "Fail system may be full.\n");
         } else {
            fprintf(stderr, "GIF87a image written to '%s'.\n", SZ_DEF_GIF_NAME);
         }
         fclose(gif_fp);
      }
   } else if (line_ptr != NULL) {
      int len=strlen(line_ptr);

      if (len > 0 && line_ptr[len-1] == '\r') {
         line_ptr[len-1] = '\0';
         fprintf(fp, "%s\n", line_ptr);
         line_ptr[len-1] = '\r';
      } else {
         fprintf(fp, "%s\n", line_ptr);
      }
   }
}

char *HttpExtractText(buf, pn_buf_sz, pn_html, ppsz_content_type)
   char *buf, **ppsz_content_type;
   int *pn_buf_sz, *pn_html;
{
   char *c_ptr=strchr(buf, '\n'), *line_ptr=buf, *content_type=NULL;
   int content_length=(-1), text_type=FALSE;
   int len1=strlen(SZ_CONTENT_TYPE), len2=strlen(SZ_CONTENT_LENGTH);
   FILE *fp=stdout;

   if (pn_buf_sz != NULL) *pn_buf_sz = 0;
   if (pn_html != NULL) *pn_html = FALSE;
   if (ppsz_content_type != NULL) *ppsz_content_type = NULL;
   while (c_ptr != NULL) {
      char *prev_ptr=c_ptr;

      if (prev_ptr != line_ptr && *(--prev_ptr) == '\r') {
         *prev_ptr = '\0';
      } else {
         prev_ptr = NULL;
         *c_ptr = '\0';
      }
      if (debugHttp) fprintf(fp, "%s\n", line_ptr);
      if (*line_ptr == '\0') {
         /* empty line, end of header */
         if (prev_ptr == NULL) {
            *c_ptr = '\n';
         } else {
            *prev_ptr = '\r';
         }
         line_ptr = &c_ptr[1];
         break;
      }
      if (content_type == NULL &&
            UtilStrNCaseCmp(line_ptr, SZ_CONTENT_TYPE, len1) == 0) {
         content_type = UtilStrDup(&line_ptr[len1]);
         if (content_type != NULL) {
            UtilTrimBlanks(content_type);
            if (ppsz_content_type != NULL) {
               *ppsz_content_type = UtilStrDup(content_type);
            }
            if (UtilStrNCaseCmp(content_type, "text/", 5) == 0) {
               text_type = TRUE;
               if (strcmp(&content_type[5], "html") == 0) {
                  if (pn_html != NULL) *pn_html = TRUE;
               }
            }
         }
      } else if (content_length == (-1) &&
            UtilStrNCaseCmp(line_ptr, SZ_CONTENT_LENGTH, len2) == 0) {
         char *tmp_ptr=UtilStrDup(&line_ptr[len2]);

         if (tmp_ptr != NULL) {
            UtilTrimBlanks(tmp_ptr);
            content_length = atoi(tmp_ptr);
            cfree(tmp_ptr);
         }
      }
      if (prev_ptr == NULL) {
         *c_ptr = '\n';
      } else {
         *prev_ptr = '\r';
      }
      line_ptr = &c_ptr[1];
      c_ptr = strchr(line_ptr, '\n');
   }
   if (content_type != NULL) cfree(content_type);
   if (text_type) {
      int buf_len=strlen(line_ptr);
      char *return_buf;

      if (content_length == (-1)) {
         content_length = buf_len;
         return_buf = (char*)calloc(buf_len+1, sizeof(char));
      } else {
         return_buf = (char*)calloc(content_length+1, sizeof(char));
      }

      if (return_buf == NULL) {
         fprintf(stderr, "Memory allocation failed.\n");
         return NULL;
      }
      if (buf_len <= content_length) {
         memcpy(return_buf, line_ptr, content_length);
      } else {
         while (buf_len > content_length) {
            if (*line_ptr == '\r' || *line_ptr == '\n') {
               line_ptr++;
               buf_len--;
            } else {
               break;
            }
         }
         memcpy(return_buf, line_ptr, content_length);
      }
      return_buf[content_length] = '\0';
      if (pn_buf_sz != NULL) *pn_buf_sz = (content_length+1);
      return return_buf;
   } else if (content_length != (-1)) {
      char *return_buf=(char*)calloc(content_length+1, sizeof(char));

      if (return_buf == NULL) {
         fprintf(stderr, "Memory allocation failed.\n");
         return NULL;
      }
      memcpy(return_buf, line_ptr, content_length);
      return_buf[content_length] = '\0';
      if (pn_buf_sz != NULL) *pn_buf_sz = (content_length+1);
      return return_buf;
   }
   return NULL;
}

#define MIN_READ_SIZE 0x100

int HttpDoRead(n_socket, ppsz_buf, pn_buf_sz)
   int n_socket, *pn_buf_sz;
   char **ppsz_buf;
{
   int buf_sz=0x400, len=0, end_of_file=FALSE;
   int status=TG_REMOTE_STATUS_OK;
   char *buf=(char*)calloc(buf_sz, sizeof(char)), msg[40];

   if (pn_buf_sz != NULL) *pn_buf_sz = 0;
   *ppsz_buf = NULL;
   if (buf == NULL) {
      fprintf(stderr, "Memory allocation failed.\n");
      return TG_REMOTE_STATUS_MEM;
   }
   sprintf(msg, "HTTP: reading responses...");
   ShowRemoteStatus(msg);

   do {
      int bytes_read;

      if (buf_sz - len < MIN_READ_SIZE) {
         buf_sz += 0x400;
         if ((buf=(char*)realloc(buf, buf_sz)) == NULL) {
            fprintf(stderr, "Memory allocation failed.\n");
            status = TG_REMOTE_STATUS_MEM;
            break;
         }
      }
      bytes_read = read(n_socket, &buf[len], buf_sz-len-1);
      if (bytes_read <= 0) {
         if (bytes_read < 0 && (errno == ENOTCONN || errno == ECONNRESET ||
               errno == EPIPE)) {
            fprintf(stderr, "Network read error.\n");
            status = TG_REMOTE_STATUS_READ;
         } else if (bytes_read < 0) {
            fprintf(stderr, "Network error.\n");
            status = TG_REMOTE_STATUS_NET;
         }
         end_of_file = TRUE;
      } else {
         len += bytes_read;
      }
      if (status == TG_REMOTE_STATUS_OK && !end_of_file && UserAbortComm()) {
         if (buf != NULL) cfree(buf);
         ShowRemoteStatus("HTTP: aborted by the user.");
         return TG_REMOTE_STATUS_INTR;
      } else {
         sprintf(msg, "HTTP: %1d bytes...", len);
         ShowRemoteStatus(msg);
      }
   } while (status == TG_REMOTE_STATUS_OK && !end_of_file);
   if (status == TG_REMOTE_STATUS_OK) {
      buf[len] = '\0';
      *ppsz_buf = buf;
      if (pn_buf_sz != NULL) *pn_buf_sz = (len+1);
      sprintf(msg, "HTTP: responses received.");
   } else {
      if (buf != NULL) cfree(buf);
      sprintf(msg, "HTTP: error encountered in receiving responses.");
   }
   ShowRemoteStatus(msg);
   return status;
}
