/*
    Copyright (C) 1999 by  XCIN TEAM

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For any question or suggestion, please mail to xcin mailing-list:
    xcin@linux.org.tw, or the maintainer Tung-Han Hsieh: thhsieh@linux.org.tw
*/      

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/Xlocale.h>
#include "config.h"

#ifdef HAVE_WCHAR_H
#  include <wchar.h>
#endif

#ifdef HAVE_GETTEXT
#  include <libintl.h>
#  define _(STRING) gettext(STRING)
#  ifndef MSGLOCAT
#    define MSGLOCAT  "/tmp"
#  endif
#else
#  define _(STRING) STRING
#endif

#define PROGNAME  "testprog"

Display  *display;
int       screen;
Window    window;
GC	  gc;
char     *fontset_name;
XFontSet  fontset;
int       ef_height, ef_width, ef_ascent;

XIM       im;
XIC       ic;
char     *im_name;
wchar_t   input_buf[1024];
int       idx;
char      ic_focus;

void win_draw(void);

/*---------------------------------------------------------------------------

	If the wcs* functions is not available.

---------------------------------------------------------------------------*/

#ifndef HAVE_WCHAR_H

int
wcslen(wchar_t *s)
{
    wchar_t *cp = s;
    int len=0;

    while (*s) {
	len ++;
	s ++;
    }
    return len;
}
#endif

/*---------------------------------------------------------------------------

	XIM Functions.

---------------------------------------------------------------------------*/

static void
destroy_callback_func(XIM current_ic, XPointer client_data, XPointer call_data)
{
    ic = NULL;
    im = NULL;
    ic_focus = 0;

    printf("destroy_callback_func\n");
}

void
im_callback(Display *display, XPointer client_data, XPointer call_data)
{
    int i;
    XIMStyles *xim_styles = NULL;
    XIMStyle  input_style = 0;
    XIMValuesList im_values_list;
    XIMCallback destroy;

/*
 *  Open connection to IM server.
 */
    if (! (im = XOpenIM(display, NULL, NULL, NULL))) {
        printf(_("Cannot open the connection to XIM server.\n"));
	exit(1);
    }

/*
 *  Get IM values.
 */
/*
    if (XGetIMValues(im, XNQueryIMValuesList, &im_values_list, NULL) || 
	! im_values_list.count_values) {
	printf("no IM values list available.\n");
        XCloseIM(im);
        exit(1);
    }
    else {
	for (i=0; i<im_values_list.count_values; i++)
	    printf("supported: %s\n", im_values_list.supported_values[i]);
    }
*/
    if (XGetIMValues(im, XNQueryInputStyle, &xim_styles, NULL) || 
	! xim_styles) {
        printf("input method doesn't support any style\n");
        XCloseIM(im);
        exit(1);
    }
    else {
	input_style = (XIMPreeditNothing | XIMStatusNothing);
	if (input_style == xim_styles->supported_styles[0])
	    printf("input_style = Root\n");
    }

    destroy.callback = destroy_callback_func;
    destroy.client_data = NULL;
    XSetIMValues(im, XNDestroyCallback, &destroy, NULL);

/*
 *  Create IC.
 */
    ic = XCreateIC(im, 
		   XNInputStyle, XIMPreeditNothing|XIMStatusNothing,
		   XNClientWindow, window,
		   XNFocusWindow, window,
		   NULL);
    if(ic == NULL) {
        printf(_("Cannot create XIC.\n"));
        exit(1);
    }
}

void
xim_init(void)
{
    if (XRegisterIMInstantiateCallback(
		display, NULL, NULL, NULL, im_callback, NULL) != True) {
	printf(_("XRegisterIMInstantiateCallback false.\n"));
	exit(0);
    }
}

void
check_chars(char *buf, int buf_len)
{
    wchar_t cch[1024];
    int i;

    if (mbstowcs(cch, buf, 1024) <= 0)
	printf(_("Unprinted char.\n"));
    else {
	printf(_("Get char: %s\n"), buf);
	for (i=0; cch[i] && idx<1024-1; idx++, i++)
	    input_buf[idx] = cch[i];
    }
}

void
check_keysym(XKeyEvent *kev)
{
    KeySym keysym;
    unsigned int keystate;
    char keystr[65];
    int keystr_len;

    keystate = kev->state;
    keystr_len = XLookupString(kev, keystr, 64, &keysym, NULL);
    keystr[(keystr_len >= 0) ? keystr_len : 0] = 0;

    if (keysym == XK_Escape) {
	if (ic)
	    XDestroyIC(ic);
	if (im)
            XCloseIM(im);
	exit(0);
    }
    else if (keysym == XK_BackSpace) {
	if (idx)
	    idx --;
	input_buf[idx] = (wchar_t)0;
    }
    else if (keysym == XK_Return)
	XBell(display, 30);
    else if (keystr_len == 1)
	check_chars(keystr, keystr_len);
}

void
lookup_key(XKeyPressedEvent *event)
{
    static char *buf;
    static int buf_len, rlen;
    KeySym keysym;
    Status status;

    if (! buf) {
	buf_len = 1024;
	buf = malloc(buf_len * sizeof(char));
    }

    bzero(buf, buf_len);
    rlen = XmbLookupString(ic, event, buf, 1024, &keysym, &status);
    if (status == XBufferOverflow) {
	buf_len += 1024;
	buf = realloc(buf, buf_len * sizeof(char));
        bzero(buf, buf_len);
	rlen = XmbLookupString(ic, event, buf, 1024, &keysym, &status);
    }

    switch (status) {
    case XLookupNone:
	break;
    case XLookupKeySym:
	check_keysym(event);
	break;
    case XLookupChars:
	check_chars(buf, rlen);
	break;
    case XLookupBoth:
	check_keysym(event);
	break;
    }
}

/*--------------------------------------------------------------------------

	GUI functions

--------------------------------------------------------------------------*/

void
set_wm_property(int x, int y, int width, int height, int argc, char **argv)
{
    char *win_name = PROGNAME, *icon_name = PROGNAME;
    XTextProperty windowName, iconName;
    XSizeHints *size_hints;
    XWMHints *wm_hints;
    XClassHint *class_hints;

    if (! XStringListToTextProperty(&win_name, 1, &windowName) ||
        ! XStringListToTextProperty(&icon_name, 1, &iconName)) {
        printf(_("string text property error.\n"));
	exit(0);
    }

    if (! (size_hints = XAllocSizeHints()) ||
        ! (wm_hints = XAllocWMHints()) ||
        ! (class_hints = XAllocClassHint())) {
        printf(_("insufficient memory.\n"));
	exit(0);
    }

    size_hints->flags = USPosition | USSize;
    size_hints->x = x;
    size_hints->y = y;
    size_hints->width  = width;
    size_hints->height = height;

    wm_hints->flags = InputHint | StateHint;
    wm_hints->input = True;
    wm_hints->initial_state = NormalState;

    class_hints->res_name  = PROGNAME;
    class_hints->res_class = PROGNAME;

    XSetWMProperties(display, window, &windowName,
                &iconName, argv, argc, size_hints, wm_hints, class_hints);

    XFree(size_hints);
    XFree(wm_hints);
    XFree(class_hints);
    XFree(windowName.value);
    XFree(iconName.value);
}

void
create_fontset(void)
{
    int  i, fsize, charset_count, fontset_count=1;
    char *s1, *s2;
    char **charset_list, *def_string;
    XFontStruct **font_structs;

    if (! fontset_name) {
	printf(_("Error: you should specify the fontset.\n"));
	exit(0);
    }

/*
 *  Calculate the number of fonts.
 */
    s1 = fontset_name;
    while ((s2=strchr(s1, ',')) != NULL) {
	s2 ++;
	while (isspace(*s2))
	    s2++;
	if (*s2 && *s2 != ',') {
	    fontset_count ++;
	    s1 = s2;
	}
	else {
	    break;
	    *s1 = '\0';
	}
    }

/*
 *  Create fontset and extract font information.
 */
    fontset = XCreateFontSet(display, fontset_name,
                &charset_list, &charset_count, &def_string);
    if (charset_count || ! fontset) {
	printf(_("Error: cannot create fontset.\n"));
	exit(0);
    }
    if (fontset_count !=
	XFontsOfFontSet(fontset, &font_structs, &charset_list)) {
	printf(_("Error: fonts not consistant to fontset.\n"));
	exit(0);
    }
    for (i=0; i<fontset_count; i++) {
        fsize = font_structs[i]->max_bounds.width / 2;
        if (fsize > ef_width)
            ef_width = fsize;
        fsize = font_structs[i]->ascent + font_structs[i]->descent;
        if (fsize > ef_height) {
            ef_height = fsize;
            ef_ascent = font_structs[i]->ascent;
        }
    }
}

void
gui_init(int argc, char **argv)
{
    if (! (display = XOpenDisplay(NULL))) {
	printf(_("XOpenDisplay false.\n"));
	exit(0);
    }
    screen = DefaultScreen(display);
    window = XCreateSimpleWindow(display, RootWindow(display, screen), 
		50, 50, 400, 200, 1, BlackPixel(display, screen), 
		WhitePixel(display, screen));
    gc = XCreateGC(display, window, 0, NULL);
    XSetForeground(display, gc, BlackPixel(display, screen));
    XSetBackground(display, gc, WhitePixel(display, screen));
    set_wm_property(50, 50, 400, 200, argc, argv);
    XSelectInput(display, window, 
	(ExposureMask | StructureNotifyMask | KeyPressMask | FocusChangeMask));

    create_fontset();
}

void
win_draw(void)
{
    char *s = _("This is the Xi18n test program.");
    char str[256];
    int x, y;

    XClearWindow(display, window);
    x = 5;
    y = 5 + ef_ascent;
    XmbDrawImageString(display, window, fontset, gc, x, y, s, strlen(s));

    sprintf(str, "(window=0x%x)", window); 
    y += (ef_height + ef_ascent + 5);
    XmbDrawImageString(display, window, fontset, gc, x, y, str, strlen(str));

    y += (ef_height + ef_ascent + 5);
    XwcDrawImageString(display, window, fontset, gc, x, y, 
		input_buf, wcslen(input_buf));
}

/*--------------------------------------------------------------------------

	Main Program

--------------------------------------------------------------------------*/

void
locale_init(void)
{
    char buf[1024];

    if (setlocale(LC_CTYPE, "") == NULL) {
	printf(_("setlocale LC_CTYPE false.\n"));
	exit(0);
    }
#ifdef HAVE_GETTEXT
    if (setlocale(LC_MESSAGES, "") == NULL) {
	printf(_("setlocale LC_MESSAGES false.\n"));
	exit(0);
    }
    else {
	textdomain(PROGNAME);
	bindtextdomain(PROGNAME, MSGLOCAT);
    }
#endif

    if (XSupportsLocale() != True) {
	printf(_("XSupportsLocale false.\n"));
	exit(0);
    }

    if (im_name)
	snprintf(buf, 1024, "@im=%s", im_name);
    else
	bzero(buf, 1024);
    if (XSetLocaleModifiers(buf) == NULL) {
	printf(_("XSetLocaleModifiers false.\n"));
	exit(0);
    }
}

void
cmd_options(int argc, char **argv)
{
    int i;

    printf(_("Use \"-h\" option to see help.\n"));

    for (i=1; i<argc; i++) {
	if (! strcmp(argv[i], "-h")) {
	    printf(_("Usage: testprog -fontset <fontset> -im <im_name>\n"));
	    printf(_("Example:\n"));
	    printf(_("  export LC_CTYPE=<locale name>.\n"));
	    printf(_("  export LC_MESSAGES=<locale name>.\n"));
	    printf(_("  export XMODIFIERS=\"@im=<im_name>\".\n"));
	    printf(_("  zh_TW.Big5 fontset: \"-*-big5-0,-*-iso8859-1\".\n"));
	    printf(_("  zh_CN.GB2312 fontset: \"-*-gb2312.1980-0,-*-iso8859-1\".\n"));
	    exit(0);
	}
	else if (! strcmp(argv[i], "-fontset"))
	    fontset_name = strdup(argv[++i]);
	else if (! strcmp(argv[i], "-im"))
	    im_name = strdup(argv[++i]);
    }
}

main(int argc, char **argv)
{
    XEvent event;

    cmd_options(argc, argv);
    locale_init();
    gui_init(argc, argv);
    xim_init();
    XMapWindow(display, window);

    while (1) {
	XNextEvent(display, &event);
	if (XFilterEvent(&event, None))
	    continue;

        switch (event.type) {
        case Expose:
	    printf(_("Event: Expose.\n"));
            win_draw();
            break;

        case GraphicsExpose:
	    printf(_("Event: GraphicsExpose.\n"));
            win_draw();
            break;

	case ConfigureNotify:
	    printf(_("Event: ConfigureNotify.\n"));
	    break;

        case FocusIn:
            if (event.xany.window == window && ic) {
	        printf(_("Event: FocusIN.\n"));
                XSetICFocus(ic);
	        ic_focus = 1;
	    }
            break;

        case FocusOut:
            if (event.xany.window == window && ic) {
	        printf(_("Event: FocusOut.\n"));
                XUnsetICFocus(ic);
	        ic_focus = 0;
	    }
            break;

	case KeyPress:
	    printf(_("Event: KeyPress.\n"));
	    if (ic_focus)
	        lookup_key((XKeyPressedEvent *)&event);
	    else
		check_keysym((XKeyEvent *)&event);
	    win_draw();
	    break;
	}
    }
}
