#!/usr/bin/perl -w
#
# Disk subroutines brought by the MondoRescue project
#
# $Id$
#
# Copyright B. Cornec 2008-2015
# Provided under the GPL v2

package MondoRescue::Disk;

use strict 'vars';
use ProjectBuilder::Base;
use English;
use MondoRescue::File;

# Inherit from the "Exporter" module which handles exporting functions.

use Exporter;

# Export, by default, all the functions into the namespace of
# any code which uses this module.

our @ISA = qw(Exporter);
our @EXPORT = qw(mr_disk_list mr_disk_type mr_device_mounted);

=pod

=head1 NAME

MondoRescue::Disk, part of the mondorescue.org

=head1 DESCRIPTION

This modules provides disk related functions for the Mondorescue project

=head1 USAGE

=over 4

=item B<mr_disk_list>

This function uses fdisk to determine the list of all the disks on the system
It returns an array of strings giving all the devices

=cut

sub mr_disk_list {

my $fdisk = pb_check_req("fdisk",0);
my @disks;

open(FDISK, "LANG=C $fdisk -l 2>/dev/null |") || die "Unable to read from $fdisk";
while (<FDISK>) {
	chomp($_);
	my $i = $_;
	if ($i =~ /^Disk \//) {
		pb_log(2,"Found disk line: $i\n");
		$i =~ s|^Disk /([A-z0-9/_-]+):.*|/$1|;
		pb_log(2,"Pushing $i\n");
		push @disks,$i;
	}
}
close(FDISK);
pb_log(2,"Found the following disks: @disks\n");
return (@disks);
}

=item B<mr_disk_type>

This function uses fdisk to determine the type of disk whose device is passed as a parameter
It returns either msdos for MBR type or gpt for GPT type

=cut

sub mr_disk_type {

my $device = shift;
# MBR by default
my $type = "MBR";

my $fdisk = pb_check_req("fdisk",0);

open(FDISK, "LANG=C $fdisk -l $device 2>/dev/null |") || die "Unable to read from $fdisk";
while (<FDISK>) {
	if (($_ =~ /EFI GPT/) || ($_ =~ / GPT /) || ($_ =~ /Disk[\s]*label type: gpt/)) {
		$type= "GPT";
		last;
	}
}
close(FDISK);
pb_log(2,"Found a $type partition format\n");
return ($type);
}


=item B<mr_device_mounted>

This function returns the mount point if a device is mounted and undef if not.

=cut

sub mr_device_mounted {

my $device = shift;

return(1) if ((not defined $device) || ($device =~ "^/sys|^/proc"));

my $h = mr_file_read_all_link($device);
# Preserve that hash locally before earsing it to reuse another one
my $h2 = mr_file_copy_and_erase_hash($h);

foreach my $f (keys %$h2) {
	pb_log(2,"Working on device $f\n");
	# TODO: get them from a conf file - FreeBSD needs swapinfo for swap request
	foreach my $dev ("mount","cat /proc/swaps") {
		# Look for the device in the command result
		open(MOUNT,"$dev|") || die "Unable to execute the $dev command";
		while (<MOUNT>) {
			my ($mntp,$void) = split('\s',$_,2);
			$h = mr_file_read_all_link($mntp);
			foreach my $m (keys %$h) {
				pb_log(2,"== Working on mounted device $m\n");
				if ($m eq $f) {
					# Find the mountpoint and return it
					my ($void1,$mountpoint,$void2) = split('\s',$void);
					pb_log(2,"*** Found $m on $mountpoint\n") if (defined $mountpoint);
					return($mountpoint);
				}
			}
			my $h3 = mr_file_copy_and_erase_hash($h);
		}
		close(MOUNT);
	}
}

# Not found
return(undef);
}
