=begin
=TC_Assertions.rb
Contains tests for the Assertions module
=end

#<standard_header>
#
# Copyright (C) 2000-2001 Nathaniel Talbott
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#</standard_header>

require 'Lapidary/TestCase'

module Lapidary
	class TC_Assertions < TestCase
		def check(value, message="")
			if (!value)
				addFailedAssertion(message)
				throw :assertionFailed
			else
				addSuccessfulAssertion(message)
			end
		end
		def checkAssertions(expectFail, expectedMessage)
			@actualAssertionMessage = nil
			@actualAssertionCount = 0
			failed = true
			@catchAssertions = true
			catch (:assertionFailed) {
				yield
				failed = false
			}
			@catchAssertions = false
			check(expectFail == failed, (expectFail ? "Should have failed, but didn't" : "Should not have failed, but did with message <#{@actualAssertionMessage}>"))
			check(1 == @actualAssertionCount, "Should only have made one assertion")
			if (expectedMessage.kind_of?(String))
				check(@actualAssertionMessage == expectedMessage, "Should have the correct message. Expected <#{expectedMessage}> but was <#{@actualAssertionMessage}>")
			elsif (expectedMessage.kind_of?(Regexp))
				check(@actualAssertionMessage =~ expectedMessage, "The message should match correctly. Expected </#{expectedMessage.source}/> but was <#{@actualAssertionMessage}>")
			else
				check(false, "Incorrect expected message type in assertNothingFailed")
			end
		end
		def checkNothingFails(expectedMessage="", &proc)
			checkAssertions(false, expectedMessage, &proc)
		end
		def checkFails(expectedMessage="", &proc)
			checkAssertions(true, expectedMessage, &proc)
		end
		def testSuccessfulAssertBlock
			checkNothingFails {
				assertBlock {true}
			}
			checkNothingFails("successful assertBlock") {
				assertBlock("successful assertBlock") {true}
			}
			checkNothingFails("successful assertBlock") {
				assert_block("successful assertBlock") {true}
			}
			checkFails {
				assertBlock {false}
			}
			checkFails("failed assertBlock") {
				assertBlock("failed assertBlock") {false}
			}
		end
		def testSuccessfulAssert
			checkNothingFails {
				assert(true)
			}
			checkNothingFails("successful assert") {
				assert(true, "successful assert")
			}
			checkFails {
				assert(false)
			}
			checkFails("failed assert") {
				assert(false, "failed assert")
			}
		end
		def testAssertEqual
			checkNothingFails("Expected <string1> but was <string1>") {
				assertEqual("string1", "string1")
			}
			checkNothingFails("successful assertEqual. Expected <string1> but was <string1>") {
				assertEqual( "string1", "string1", "successful assertEqual")
			}
			checkNothingFails("successful assertEqual. Expected <string1> but was <string1>") {
				assert_equal("string1", "string1", "successful assertEqual")
			}
			checkFails("Expected <string1> but was <string2>") {
				assertEqual("string1", "string2")
			}
			checkFails("failed assertEqual. Expected <string1> but was <string2>") {
				assertEqual("string1", "string2", "failed assertEqual")
			}
		end
		def testAssertRaises
			checkNothingFails(%r{^Expected exception to be of type <RuntimeError> but was <RuntimeError: Error\n\t}m) {
				assertRaises(RuntimeError) {
					raise "Error"
				}
			}
			checkNothingFails(%r{^successful assertRaises. Expected exception to be of type <ArgumentError> but was <ArgumentError: Error\n\t}m) {
				assertRaises(ArgumentError, "successful assertRaises") {
					raise ArgumentError.new("Error")
				}
			}
			checkNothingFails(%r{^Expected exception to be of type <RuntimeError> but was <RuntimeError: Error\n\t}m) {
				assert_exception(RuntimeError) {
					raise "Error"
				}
			}
			checkNothingFails(%r{^successful assertRaises. Expected exception to be of type <RuntimeError> but was <RuntimeError: Error\n\t}m) {
				assert_raises(RuntimeError, "successful assertRaises") {
					raise "Error"
				}
			}
			checkFails("Expected exception of type <RuntimeError> but none was thrown") {
				assertRaises(RuntimeError) {
					1 + 1
				}
			}
			checkFails(%r{failed assertRaises. Expected exception to be of type <ArgumentError> but was <RuntimeError: Error\n\t}m) {
				assertRaises(ArgumentError, "failed assertRaises") {
					raise "Error"
				}
			}
		end
		def testAssertInstanceOf
			checkNothingFails("<string> should have been an instance of <String>. Expected <String> but was <String>") {
				assertInstanceOf(String, "string")
			}
			checkNothingFails("successful assertInstanceOf. <string> should have been an instance of <String>. Expected <String> but was <String>") {
				assertInstanceOf(String, "string", "successful assertInstanceOf")
			}
			checkNothingFails("successful assertInstanceOf. <string> should have been an instance of <String>. Expected <String> but was <String>") {
				assert_instance_of(String, "string", "successful assertInstanceOf")
			}
			checkFails("<string> should have been an instance of <Hash>. Expected <Hash> but was <String>") {
				assertInstanceOf(Hash, "string")
			}
			checkFails("failed assertInstanceOf. <string> should have been an instance of <Hash>. Expected <Hash> but was <String>") {
				assertInstanceOf(Hash, "string", "failed assertInstanceOf")
			}
		end
		def testAssertNil
			checkNothingFails("Expected <nil> but was <nil>") {
				assertNil(nil)
			}
			checkNothingFails("successful assertNil. Expected <nil> but was <nil>") {
				assertNil(nil, "successful assertNil")
			}
			checkNothingFails("successful assertNil. Expected <nil> but was <nil>") {
				assert_nil(nil, "successful assertNil")
			}
			checkFails("Expected <nil> but was <string>") {
				assertNil("string")
			}
			checkFails("failed assertNil. Expected <nil> but was <string>") {
				assertNil("string", "failed assertNil")
			}
		end
		def testAssertKindOf
			checkNothingFails("Expected <Array> to be kind_of?<Module>") {
				assertKindOf(Module, Array)
			}
			checkNothingFails("successful assertKindOf. Expected <string> to be kind_of?<Object>") {
				assertKindOf(Object, "string", "successful assertKindOf")
			}
			checkNothingFails("successful assertKindOf. Expected <string> to be kind_of?<Object>") {
				assert_kind_of(Object, "string", "successful assertKindOf")
			}
			checkFails("Expected <string> to be kind_of?<Class>") {
				assertKindOf(Class, "string")
			}
			checkFails("failed assertKindOf. Expected <string> to be kind_of?<Class>") {
				assertKindOf(Class, "string", "failed assertKindOf")
			}
		end
		def testAssertMatch
			checkNothingFails("Expected <string> to match regexp </string/>") {
				assertMatch(/string/, "string")
			}
			checkNothingFails("successful assertMatch. Expected <string> to match regexp </string/>") {
				assertMatch(/string/, "string", "successful assertMatch")
			}
			checkNothingFails("successful assertMatch. Expected <string> to match regexp </string/>") {
				assert_match(/string/, "string", "successful assertMatch")
			}
			checkFails("Expected <string> to match regexp </sling/>") {
				assertMatch(/sling/, "string")
			}
			checkFails("failed assertMatch. Expected <string> to match regexp </sling/>") {
				assertMatch(/sling/, "string", "failed assertMatch")
			}
		end
		def testAssertSame
			thing = "thing"
			checkNothingFails("Expected <#{thing}:#{thing.id}> to be equal? to <#{thing}:#{thing.id}>") {
				assertSame(thing, thing)
			}
			checkNothingFails("successful assertSame. Expected <#{thing}:#{thing.id}> to be equal? to <#{thing}:#{thing.id}>") {
				assertSame(thing, thing, "successful assertSame")
			}
			checkNothingFails("successful assertSame. Expected <#{thing}:#{thing.id}> to be equal? to <#{thing}:#{thing.id}>") {
				assert_same(thing, thing, "successful assertSame")
			}
			thing2 = "thing"
			checkFails("Expected <#{thing}:#{thing.id}> to be equal? to <#{thing2}:#{thing2.id}>") {
				assertSame(thing, thing2)
			}
			checkFails("failed assertSame. Expected <#{thing}:#{thing.id}> to be equal? to <#{thing2}:#{thing2.id}>") {
				assertSame(thing, thing2, "failed assertSame")
			}
		end
		def testAssertNothingRaised
			checkNothingFails {
				assertNothingRaised {
					1 + 1
				}
			}
			checkNothingFails("successful assertNothingRaised") {
				assertNothingRaised("successful assertNothingRaised") {
					1 + 1
				}
			}
			checkNothingFails("successful assertNothingRaised") {
				assert_nothing_raised("successful assertNothingRaised") {
					1 + 1
				}
			}
			checkFails(%r{Exception raised: <RuntimeError: Error\n\t}m) {
				assertNothingRaised {
					raise "Error"
				}
			}
			checkFails(%r{failed assertNothingRaised\. Exception raised: <RuntimeError: Error\n\t}m) {
				assertNothingRaised("failed assertNothingRaised") {
					raise "Error"
				}
			}
		end
		def testFlunk
			checkFails {
				flunk
			}
			checkFails("flunk message") {
				flunk("flunk message")
			}
		end
		def testAssertNotSame
			thing = "thing"
			thing2 = "thing"
			checkNothingFails("Expected <#{thing}:#{thing.id}> to not be equal? to <#{thing2}:#{thing2.id}>") {
				assertNotSame(thing, thing2)
			}
			checkNothingFails("message. Expected <#{thing}:#{thing.id}> to not be equal? to <#{thing2}:#{thing2.id}>") {
				assertNotSame(thing, thing2, "message")
			}
			checkFails("Expected <#{thing}:#{thing.id}> to not be equal? to <#{thing}:#{thing.id}>") {
				assertNotSame(thing, thing)
			}
			checkFails("message. Expected <#{thing}:#{thing.id}> to not be equal? to <#{thing}:#{thing.id}>") {
				assertNotSame(thing, thing, "message")
			}
		end
		def testAssertNotEqual
			checkNothingFails("Expected <string1> to be != to <string2>") {
				assertNotEqual("string1", "string2")
			}
			checkNothingFails("message. Expected <string1> to be != to <string2>") {
				assertNotEqual("string1", "string2", "message")
			}
			checkFails("Expected <string> to be != to <string>") {
				assertNotEqual("string", "string")
			}
			checkFails("message. Expected <string> to be != to <string>") {
				assertNotEqual("string", "string", "message")
			}
		end
		def testAssertNotNil
			checkNothingFails("Expected <string> to not be nil") {
				assertNotNil("string")
			}
			checkNothingFails("message. Expected <string> to not be nil") {
				assertNotNil("string", "message")
			}
			checkFails("Expected <nil> to not be nil") {
				assertNotNil(nil)
			}
			checkFails("message. Expected <nil> to not be nil") {
				assertNotNil(nil, "message")
			}
		end
		def testAssertDoesNotMatch
			checkNothingFails("Expected </sling/> to not match <string>") {
				assertDoesNotMatch(/sling/, "string")
			}
			checkNothingFails("message. Expected </sling/> to not match <string>") {
				assertDoesNotMatch(/sling/, "string", "message")
			}
			checkFails("Expected </string/> to not match <string>") {
				assertDoesNotMatch(/string/, "string")
			}
			checkFails("message. Expected </string/> to not match <string>") {
				assertDoesNotMatch(/string/, "string", "message")
			}
		end
		def testAssertThrows
			checkNothingFails("message. <:thing> should have been thrown") {
				assertThrows(:thing, "message") {
					throw :thing
				}
			}
			checkFails("message. Expected <:thing> to be thrown but <:thing2> was thrown") {
				assertThrows(:thing, "message") {
					throw :thing2
				}
			}
			checkFails("message. <:thing> should have been thrown") {
				assertThrows(:thing, "message") {
					1 + 1
				}
			}
		end
		def testAssertNothingThrown
			checkNothingFails("message. Nothing should have been thrown") {
				assertNothingThrown("message") {
					1 + 1
				}
			}
			checkFails("message. Nothing should have been thrown but <:thing> was thrown") {
				assertNothingThrown("message") {
					throw :thing
				}
			}
		end

		def addFailedAssertion(message)
			if (!@catchAssertions)
				super(message)
			else
				@actualAssertionCount += 1
				@actualAssertionMessage = message
			end
		end
		def addSuccessfulAssertion(message)
			if (!@catchAssertions)
				super(message)
			else
				@actualAssertionCount += 1
				@actualAssertionMessage = message
			end
		end
	end
end







