#!/usr/bin/env ruby

# :nodoc:

$:.push("../lib")
require 'ftools'
require 'test/unit'
require 'xmlconfigfile'

class TC_XmlConfigFile < Test::Unit::TestCase # :nodoc:
  def setup
    ENV['PARAM_1'] = 'param1'
    ENV['PARAM2'] = '4711'
    File::copy('original_files/config.xml', '.')
    @config = XmlConfigFile.new('config.xml')
  end

  def test_file_not_exist
    assert_raises(Errno::ENOENT) { XmlConfigFile.new('') }
  end

  def test_document_not_wellformed
    assert_raises(REXML::ParseException) { XmlConfigFile.new('notWellformedConfig.xml') }
  end
  
  def test_invalid_reload_period
    assert_raises(ArgumentError) { XmlConfigFile.new('config.xml', 'crap') }
    assert_raises(ArgumentError) { XmlConfigFile.new('config.xml', 0) }
    assert_raises(ArgumentError) { XmlConfigFile.new('config.xml', -1) }
    assert_raises(ArgumentError) { XmlConfigFile.new('config.xml', -1.5) }
  end

  def test_reload_valid_configuration
    config = XmlConfigFile.new('config.xml', 2)
    File::copy('original_files/changed_config.xml', 'config.xml')
    File::utime(0, 0, 'config.xml')
    sleep 3
    assert_equal('schmidt', config.get_parameter('/config/db/user'))
    assert_equal(4711, config.get_int_parameter('/config/environment-variable/@att'))

    # Check, if changes in environment are reflected correctly.
    ENV['PARAM_1'] = 'new'
    ENV['PARAM2'] = '4712'
    sleep 3
    assert_equal('schmidt', config.get_parameter('/config/db/user'))
    assert_equal(4711, config.get_int_parameter('/config/environment-variable/@att'))

    # We have to touch the file explicitly!
    now = Time::now
    File::utime(now, now, 'config.xml')
    sleep 3
    assert_equal('This is new (4712).', config.get_parameter('/config/environment-variable'))
    assert_equal(4712, config.get_int_parameter('/config/environment-variable/@att'))
  end

  def test_reload_invalid_configuration
    # An invalid configuration file will be ignored, i.e.
    # the last working configuration will be used. Additionally,
    # a message will be printed on STDERR.
    config = XmlConfigFile.new('config.xml', 2)
    File::copy('original_files/invalid_config.xml', 'config.xml')
    File::utime(0, 0, 'config.xml')
    sleep 3
    assert_equal('maik', config.get_parameter('/config/db/user'))

    # Wellformed changes are fine. This test ensures, that an
    # instance of class XmlConfigFile will still work after
    # reading an invalid configuration file.
    File::copy('original_files/changed_config.xml', 'config.xml')
    sleep 3
    assert_equal('schmidt', config.get_parameter('/config/db/user'))
  end

  def test_get_existing_string_parameter
    assert_equal('shop', @config.get_parameter('/config/db/name'))
    assert_equal('shop', @config['/config/db/name'])
    assert_equal('shop', @config.get_parameter('*/db/name'))
    assert_equal('shop', @config['*/db/name'])
    assert_equal('shop', @config.getParameter('/config/db/name'))
  end

  def test_get_unknown_string_parameter
    assert_equal(nil, @config.get_parameter('/config/xyz'))
    assert_equal(nil, @config['/config/xyz'])
  end

  def test_get_unknown_string_parameter_with_default
    assert_equal('guest', @config.get_parameter('/config/db/pwd', 'guest'))
    assert_equal('guest', @config['/config/db/pwd', 'guest'])
  end

  def test_get_strange_parameter_name
    assert_equal(nil, @config.get_parameter('?'))
    assert_equal(nil, @config['?'])
  end

  def test_get_existing_int_parameter
    assert_equal(42, @config.get_int_parameter('/config/int-amount'))
    assert_equal(12345678901234567890, @config.get_int_parameter('/config/bigint-amount'))
    assert_equal(-42, @config.get_int_parameter('/config/negative-int-amount'))
    assert_equal(100, @config.get_int_parameter('/config/timer/@value'))
    assert_equal(255, @config.get_int_parameter('/config/hex-int-amount'))
    assert_equal(254, @config.get_int_parameter('/config/octal-int-amount'))
  end

  def test_get_unknown_int_parameter
    assert_equal(nil, @config.get_int_parameter('/config/xyz'))
  end

  def test_get_unknown_int_parameter_with_default
    assert_equal(4711, @config.get_int_parameter('/config/unknown-amount', 4711))
  end
  
  def test_get_invalid_int_parameter
    assert_raises(ArgumentError) { @config.get_int_parameter('/config/db/name') }
  end

  def test_get_existing_float_parameter
    assert_equal(3.14, @config.get_float_parameter('/config/float-amount'))
    assert_equal(-3.14, @config.get_float_parameter('/config/negative-float-amount'))
  end

  def test_get_unknown_float_parameter
    assert_equal(nil, @config.get_float_parameter('/config/xyz'))
  end

  def test_get_invalid_float_parameter
    assert_raises(ArgumentError) { @config.get_float_parameter('/config/db/name') }
  end

  def test_get_unknown_float_parameter_with_default
    assert_equal(2.71, @config.get_float_parameter('/config/e', 2.71))
  end

  def test_get_existing_boolean_parameter
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-1'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-2'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-3'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-4'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-5'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-6'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-7'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-8'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-9'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-10'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-11'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-12'))
    assert_equal(true, @config.get_boolean_parameter('/config/bools/no-13'))
    assert_equal(false, @config.get_boolean_parameter('/config/bools/no-14'))
  end

  def test_get_unknown_boolean_parameter
    assert_equal(true, @config.get_boolean_parameter('/config/bools/not-exist', true))
    assert_equal(nil, @config.get_boolean_parameter('/config/bools/not-exist'))
  end

  def test_get_invalid_boolean_parameter
    assert_raises(ArgumentError) { @config.get_boolean_parameter('/config/db/name') }
  end

  def test_get_unknown_boolean_parameter_with_default
    assert_equal(true, @config.get_boolean_parameter('/config/xyz', true))
  end

  def test_user_defined_boolean_values
    # Try the german variant.
    @config.true_values  = ['wahr']
    @config.false_values = ['falsch']
    assert_raises(ArgumentError) { @config.get_boolean_parameter('/config/bools/no-1') }
    assert_equal(true, @config.get_boolean_parameter('/config/german-true'))
    assert_equal(false, @config.get_boolean_parameter('/config/german-false'))
    assert_equal(true, @config.get_boolean_parameter('/config/german-true/@case'))
    assert_equal(false, @config.get_boolean_parameter('/config/german-false/@case'))

    # Try the klingon variant.
    @config.true_values  = ["HIja'", "  HISlaH  "]
    @config.false_values = ["ghobe'"]
    assert_equal(true, @config.get_boolean_parameter('/config/klingon-true/@alternative'))
    assert_equal(true, @config.get_boolean_parameter('/config/klingon-true'))
    assert_equal(false, @config.get_boolean_parameter('/config/klingon-false'))
  end

  def test_get_parameters_without_attributes
    expectedResult = {
      'db.name' => 'shop',
      'db.user' => 'maik'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*'))
    
    expectedResult = {
      'db-name' => 'shop', 
      'db-user' => 'maik'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*', '-'))
    
    expectedResult = {
      'db-silly-name' => 'shop',
      'db-silly-user' => 'maik'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*', '-silly-'))
  end

  def test_get_parameters_with_attributes
    @config.expand_attributes = true
    expectedResult = {
      'db.name' => 'shop',
      'db.user' => 'maik',
      'db.name.role' => 'admin'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*'))
    
    expectedResult = {
      'db-name' => 'shop', 
      'db-user' => 'maik',
      'db-name-role' => 'admin'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*', '-'))
    
    expectedResult = {
      'db-silly-name' => 'shop',
      'db-silly-user' => 'maik',
      'db-silly-name-silly-role' => 'admin'
    }
    assert_equal(expectedResult, @config.get_parameters('/config/db/*', '-silly-'))
  end

  def test_get_parameters_with_empty_elements
    all = @config.get_parameters('//*')
    assert(all.has_key?('splash-screen'), "'splash-screen' element not found.")
    assert_equal(nil, all['splash-screen'])
    assert(all.has_key?('empty-element'), "'empty-element' element not found.")
    assert_equal(nil, all['empty-element'])
  end

  def test_get_parameter_array
    expectedResult = [
      { 
        'arr.db.name' => 'shop',
        'arr.db.user' => 'maik'
      },
      {
        'arr.db.name' => 'factory',
        'arr.db.user' => 'anna'
      },
      {
        'arr.db.name' => 'integ',
        'arr.db.user' => 'developer'
      }
    ]
    assert_equal(expectedResult, @config.get_parameter_array('/config/arr/db'))
    assert_equal(expectedResult, @config.get_parameter_array('//arr/db'))

    expectedResult = [
      { 
        'arr3.db.name' => 'addresses',
        'arr3.db.user' => 'scott',
        'arr3.db.pwd'  => 'tiger',
        'arr3.db.host' => 'example.com',
        'arr3.db.driver.vendor'  => 'MySql',
        'arr3.db.driver.version' => '3.23'
      },
      {
        'arr3.db.name' => 'addresses',
        'arr3.db.user' => 'production',
        'arr3.db.pwd'  => 'secret',
        'arr3.db.host' => 'example.com',
        'arr3.db.driver.vendor'  => 'Oracle',
        'arr3.db.driver.version' => '8.1'
      }
    ]
    assert_equal(expectedResult, @config.get_parameter_array('//arr3/db'))
  end

  def test_environment_variable_substitution
    assert_equal('This is param1 (4711).', @config.get_parameter('/config/environment-variable'))
    assert_equal(4711, @config.get_int_parameter('/config/environment-variable/@att'))
  end
end

